package main

import (
	"errors"
	"fmt"
	"math/rand"
	"os"

	"git.sr.ht/~charles/rq/builtins"
	"git.sr.ht/~charles/rq/internal"
	"git.sr.ht/~charles/rq/util"

	"github.com/alecthomas/kong"
	"github.com/jaswdr/faker"

	"github.com/pkg/profile"
)

// CLI represents the possible command line arguments for rq.
type CLI struct {
	Query        QueryCmd        `cmd:"" default:"withargs" help:"Run a Rego Query"`
	ListFormats  ListFormatsCmd  `cmd:"" help:"List supported input/output formats and exit"`
	Script       ScriptCmd       `cmd:"" help:"Run a Rego file as a script"`
	Version      VersionCmd      `cmd:"" help:"Print version information and exit"`
	Capabilities CapabilitiesCmd `cmd:"" help:"Access OPA capabilities information."`
}

// Globals represents command line arguments common to all rq subcommands.
type Globals struct {
}

// Common represents command line arguments common to rq subcommands that
// process Rego queries in some way. This represents the complete space of how
// rq can be asked to run a Rego query.
type Common struct {
	Input        string   `name:"input" short:"I" help:"Specify an input data source in the same format as --data. If the path is empty, data is read from standard in. The default is to read from standard in. Options expressed this way take precedence over options provided on the CLI."`
	InPlace      string   `name:"in-place" short:"p" help:"Similar to --input, but also sets --output to the same file, to simplify modifying files in-place."`
	InputFormat  string   `name:"input-format" short:"i" default:"" help:"Choose the format the input data should be parsed as. The empty string is treated as 'json'."`
	Output       string   `name:"output" short:"O" help:"Specify an output datasources, as with --input."`
	OutputFormat string   `name:"output-format" short:"o" default:"" help:"Choose the format the output data should be emitted in. The empty string is treated as 'json'."`
	NoTTYCheck   bool     `name:"no-tty-check" default:"false" help:"Disable checking if the input or output is a TTY. When this flag is asserted, rq will not bail out early when standard in is a TTY, and also won't disable colorization if standard out is not a TTY."`
	NoColor      bool     `name:"no-color" default:"false" help:"Disable colorizing output in any circumstance. Takes precedence over --force-color."`
	ForceColor   bool     `name:"force-color" default:"false" help:"Where possible, color output, even if the output is not a TTY."`
	Ugly         bool     `name:"ugly" default:"false" help:"Disable pretty-printing output. Implies --no-color."`
	Indent       string   `name:"indent" default:"\t" help:"Override the character(s) used for indentation."`
	Style        string   `name:"style" default:"native" help:"The Chroma style to use for colorized output, see: https://xyproto.github.io/splash/docs/"`
	CSVComma     string   `name:"csv-comma" short:"D" default:"" help:"Delimiter to use when parsing CSV input files."`
	CSVComment   string   `name:"csv-comment" default:"" help:"Comment character to use when parsing CSV input files."`
	CSVSkipLines int      `name:"csv-skip-lines" default:"0" help:"Number of leading lines to skip when parsing CSV input files."`
	CSVHeaders   bool     `name:"csv-headers" short:"H" default:"false" help:"Treat the first row of the CSV file as headers."`
	CSVNoInfer   bool     `name:"csv-no-infer" default:"false" help:"Disable automatic type inference for CSV fields."`
	Profile      bool     `name:"profile" default:"false" help:"enable runtime CPU profiling"`
	DataPaths    []string `name:"data" short:"d" sep:"none" help:"List of files to load into the 'data' package. You can prefix the path with '<format>:' where <format> is one of the formats -i knows about. The file data is loaded into data.<file base name> unless you prefix the path with 'some.rego.path='. If you want to specify both, the format goes first. Example: '--data yaml:data.foo.bar=/some/path'."`
	BundlePaths  []string `name:"bundle" short:"b" help:"Paths to bundle files or directories to load into Rego directly. This is equivalent to 'opa eval --bundle/-b'."`
	Raw          bool     `name:"raw" short:"R" help:"Shorthand for -o raw. Takes precedence over -o."`
	Save         string   `name:"save-bundle" short:"s" help:"Rather than executing the query, save all data and Rego modules to the specified path, overwriting any file that's already there."`
	Template     string   `name:"template" short:"t" help:"Template string to use as output.template. Also implies -o template. The same template functions are available as with the rq.template() builtin."`
	Check        bool     `name:"check" short:"c" help:"Exit with a nonzero exit code if the result of the query is nil, null, 0, false, an empty string, or any empty collection type."`
	V0Compatible bool     `name:"v0-compatible" help:"Evaluate Rego code in Rego v0 compatibility mode. See https://www.openpolicyagent.org/docs/latest/v0-compatibility/"`

	// FakeSeed is intentionally hidden because it is not supported -
	// seeding the fake data generator is fussy and does not seem to work
	// reliably across rq.fake(), rq.sfake(), rq.template(), and -o
	// template, which all utilize different code paths. In particular, -o
	// template is known to produce inconsistent outputs even when a seed
	// is specified with --fake-seed.
	//
	// This is still needed for testing, and might be useful for some
	// power-user use cases. Be aware that use of --fake-seed is *not*
	// supported at this time.
	FakeSeed int64 `name:"fake-seed" hidden:"" default:"-1" help:"Set a specific seed for fake data generation. If the value is -1, then the system time is used as the random seed instead."`
}

func main() {
	// Check if we might be being used as a shebang, which we'll know
	// because we will have exactly one argument which is an extant path,
	// in which case we act like the script subcommand was given.
	if len(os.Args) >= 2 {

		isSubcmd := (os.Args[1] == "query") || (os.Args[1] == "list-formats") || (os.Args[1] == "script") || (os.Args[1] == "version")

		if _, err := os.Stat(os.Args[1]); (err == nil) && (!isSubcmd) {
			os.Args = append([]string{os.Args[0], "script"}, os.Args[1:]...)
		}
	}

	// We intentionally use the same reference to common for those
	// subcommands that use it.
	common := &Common{}
	cli := CLI{}
	cli.Query.Common = common
	cli.Script.Common = common

	// Kong typically does not allow default subcommand to have arguments,
	// as discussed here: https://github.com/alecthomas/kong/issues/231
	//
	// However, in the case of rq, we can grantee no ambiguity, since
	// "input" and "data" aren't valid subcommands and any other
	// potentially valid subcommand wouldn't be a valid Rego query.

	parser, err := kong.New(&cli,
		kong.Name("rq"),
		kong.Description("A CLI tool for running Rego Queries"),
		kong.Bind(&Globals{}),
	)
	if err != nil {
		panic(err)
	}

	if len(os.Args) < 1 {
		panic("len(os.Args) < 1, this should never happen")
	}

	ctx, err := parser.Parse(os.Args[1:])
	if err != nil {
		panic(err)
	}

	if common.Profile {
		defer profile.Start(profile.ProfilePath(".")).Stop()
	}

	if common.FakeSeed >= 0 {
		internal.FakeOptions.Faker = faker.NewWithSeed(rand.NewSource(common.FakeSeed))
		util.TemplateFakeOptions.Faker = faker.NewWithSeed(rand.NewSource(common.FakeSeed))
	}

	if common.Ugly {
		common.NoColor = true
	}

	if os.Getenv("RQ_REGO_V0") != "" {
		common.V0Compatible = true
	}

	err = ctx.Run(common)

	be := builtins.ErrorFromBuiltin{}
	if errors.As(err, &be) {
		fmt.Println(be.Error())
		os.Exit(1)
	}

	ctx.FatalIfErrorf(err)

	if forceNonzeroExit {
		os.Exit(1)
	}
}
