package main

import (
	"testing"

	"github.com/open-policy-agent/opa/v1/ast"
	opabundle "github.com/open-policy-agent/opa/v1/bundle"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestParseScriptConfig(t *testing.T) {
	cases := []struct {
		input     string
		output    map[string][]string
		mustError bool
	}{
		{
			input:  "# rq: foo bar",
			output: map[string][]string{"foo": []string{"bar"}},
		},
		{
			input:  "# rq: foo bar\n",
			output: map[string][]string{"foo": []string{"bar"}},
		},
		{
			input:  "\t# rq: foo bar\r\n",
			output: map[string][]string{"foo": []string{"bar"}},
		},
		{
			input:     "\t# rq: bar\r\n",
			mustError: true,
			output:    map[string][]string{"foo": []string{"bar"}},
		},
		{
			input:  "\t# rq: foo bar baz quux\r\n",
			output: map[string][]string{"foo": []string{"bar baz quux"}},
		},
		{
			input:  "\t# rq: foo bar baz quux\r\n  \t#rq : second directive   \n# \n#hi: there\n",
			output: map[string][]string{"foo": []string{"bar baz quux"}, "second": []string{"directive"}},
		},
		{
			input:  "\t# rq: foo bar\n# rq: foo baz",
			output: map[string][]string{"foo": []string{"bar", "baz"}},
		},
		{
			input:  "# rq: output /dev/null # TODO(sr): switch to output-format null when",
			output: map[string][]string{"output": []string{"/dev/null # TODO(sr): switch to output-format null when"}},
		},
	}

	for _, c := range cases {
		directives, err := parseScriptConfig(c.input)
		if c.mustError {
			assert.NotNil(t, err)
			continue
		} else {
			assert.Nil(t, err)
		}

		assert.Equal(t, c.output, directives)
	}

}

func TestMergeBundles(t *testing.T) {
	b1 := &opabundle.Bundle{
		Manifest: opabundle.Manifest{
			Roots: &[]string{},
		},
		Data: map[string]interface{}{
			"hello": "world!",
			"b1":    "aaa",
		},
		Modules: []opabundle.ModuleFile{
			{
				Path: "/foo/bar.rego",
				Raw: []byte(`package bar
x := [1,2,3]
`),
			},
			{
				Path: "/foo/baz.rego",
				Raw: []byte(`package baz 
y := "zzz"
`),
			},
		},
	}

	b2 := &opabundle.Bundle{
		Manifest: opabundle.Manifest{
			Roots: &[]string{},
		},
		Data: map[string]interface{}{
			"hello": "rego",
			"b2":    "bbb",
		},
		Modules: []opabundle.ModuleFile{
			{
				Path: "/foo/bar.rego",
				Raw: []byte(`package bar
x := "hi"
`),
			},
		},
	}

	for i, m := range b1.Modules {
		c, err := ast.ParseModule(m.Path, string(m.Raw))
		require.Nil(t, err)
		b1.Modules[i].Parsed = c
	}

	for i, m := range b2.Modules {
		c, err := ast.ParseModule(m.Path, string(m.Raw))
		require.Nil(t, err)
		b2.Modules[i].Parsed = c
	}

	data := map[string]interface{}{
		"hello": "rq",
	}

	input := map[string]interface{}{
		"spam": "ham",
	}

	regoVersion := 1
	merged := mergeBundles([]*opabundle.Bundle{b1, b2}, data, &regoVersion)

	expects := []struct {
		query  string
		expect interface{}
		v0     bool
	}{
		{
			query:  "data.bar.x",
			expect: "hi",
		},
		{
			query:  "data.baz.y",
			expect: "zzz",
		},
		{
			query:  "data.b1",
			expect: "aaa",
		},
		{
			query:  "data.b2",
			expect: "bbb",
		},
		{
			query:  "data.hello",
			expect: "rq",
		},
		{
			query:  "input.spam",
			expect: "ham",
		},
	}

	for _, e := range expects {
		regoVersion := 1
		if e.v0 {
			regoVersion = 0
		}

		actual, err := runRegoQuery(input, e.query, data, merged, regoVersion)
		assert.Nil(t, err)
		assert.Equal(t, e.expect, actual)
	}
}
