/*
 *   Copyright 2007 by Aaron Seigo <aseigo@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "scripting/scriptengine.h"

#include <QDebug>
#include <kservice.h>
#include <kservicetypetrader.h>

#include "applet.h"
#include "dataengine.h"
#include "package.h"
#include "private/componentinstaller_p.h"
#include "scripting/appletscript.h"
#include "scripting/dataenginescript.h"

namespace Plasma
{

ScriptEngine::ScriptEngine(QObject *parent)
    : QObject(parent),
      d(0)
{
}

ScriptEngine::~ScriptEngine()
{
//    delete d;
}

bool ScriptEngine::init()
{
    return true;
}

Package ScriptEngine::package() const
{
    return Package();
}

QString ScriptEngine::mainScript() const
{
    return QString();
}

QStringList knownLanguages(Types::ComponentTypes types)
{
    QString constraintTemplate = "'%1' in [X-Plasma-ComponentTypes]";
    QString constraint;

    if (types & Types::AppletComponent) {
        // currently this if statement is not needed, but this future proofs
        // the code against someone initializing constraint to something
        // before we get here.
        if (!constraint.isEmpty()) {
            constraint.append(" or ");
        }

        constraint.append(constraintTemplate.arg("Applet"));
    }

    if (types & Types::DataEngineComponent) {
        if (!constraint.isEmpty()) {
            constraint.append(" or ");
        }

        constraint.append(constraintTemplate.arg("DataEngine"));
    }

    KService::List offers = KServiceTypeTrader::self()->query("Plasma/ScriptEngine", constraint);
    //qDebug() << "Applet::knownApplets constraint was '" << constraint
    //         << "' which got us " << offers.count() << " matches";

    QStringList languages;
    foreach (const KService::Ptr &service, offers) {
        QString language = service->property("X-Plasma-API").toString();
        if (!languages.contains(language)) {
            languages.append(language);
        }
    }

    return languages;
}

KService::List engineOffers(const QString &language, Types::ComponentType type)
{
    if (language.isEmpty()) {
        return KService::List();
    }

    QRegExp re("[^a-zA-Z0-9\\-_]");
    if (re.indexIn(language) != -1) {
#ifndef NDEBUG
        // qDebug() << "invalid language attempted:" << language;
#endif
        return KService::List();
    }

    QString component;
    switch (type) {
    case Types::AppletComponent:
        component = "Applet";
        break;
    case Types::DataEngineComponent:
        component = "DataEngine";
        break;
    default:
        return KService::List();
        break;
    }

    QString constraint = QString("[X-Plasma-API] == '%1' and "
                                 "'%2' in [X-Plasma-ComponentTypes]").arg(language, component);
    KService::List offers = KServiceTypeTrader::self()->query("Plasma/ScriptEngine", constraint);
    /* // qDebug() << "********************* loadingApplet with Plasma/ScriptEngine" << constraint
             << "resulting in" << offers.count() << "results";*/
    if (offers.isEmpty()) {
#ifndef NDEBUG
        // qDebug() << "No offers for \"" << language << "\"";
#endif
    }

    return offers;
}

ScriptEngine *loadEngine(const QString &language, Types::ComponentType type, QObject *parent,
    const QVariantList &args = QVariantList())
{
    KService::List offers = engineOffers(language, type);

    QString error;

    ScriptEngine *engine = 0;
    foreach (const KService::Ptr &service, offers) {
        switch (type) {
        case Types::AppletComponent:
            engine = service->createInstance<Plasma::AppletScript>(parent, args, &error);
            break;
        case Types::DataEngineComponent:
            engine = service->createInstance<Plasma::DataEngineScript>(parent, args, &error);
            break;
        default:
            return 0;
            break;
        }

        if (engine) {
            return engine;
        }

#ifndef NDEBUG
        // qDebug() << "Couldn't load script engine for language " << language
        //         << "! error reported: " << error;
#endif
    }

    // Try installing the engine. However, it's too late for this request.
    ComponentInstaller::self()->installMissingComponent("scriptengine", language);

    return 0;
}

AppletScript *loadScriptEngine(const QString &language, Applet *applet, const QVariantList &args)
{
    AppletScript *engine =
        static_cast<AppletScript *>(loadEngine(language, Types::AppletComponent, applet, args));

    if (engine) {
        engine->setApplet(applet);
    }

    return engine;
}

DataEngineScript *loadScriptEngine(const QString &language, DataEngine *dataEngine, const QVariantList &args)
{
    DataEngineScript *engine =
        static_cast<DataEngineScript *>(loadEngine(language, Types::DataEngineComponent, dataEngine, args));

    if (engine) {
        engine->setDataEngine(dataEngine);
    }

    return engine;
}

} // namespace Plasma

#include "moc_scriptengine.cpp"
