##
# ASG29 Electrical System
# adapted from: Cessna 172p Electrical System

# Initialize properties
var electrical	=	props.globals.initNode("/systems/electrical");
var serviceable	=	electrical.initNode("serviceable", 1, "BOOL");
var batt_prop	=	electrical.initNode("battery");
var output_prop	=	electrical.initNode("outputs");
var cb_prop	=	props.globals.initNode("/controls/circuit-breakers");
var sw_prop	=	props.globals.initNode("/controls/electric");

var sc7_volts	=	electrical.initNode( "volts" );	# used for ILEC SC-7 display

var com_ptt = props.globals.getNode("/instrumentation/comm[0]/ptt", 1);
var com_start = props.globals.getNode("/instrumentation/comm[0]/start", 1);
var vario_vol = props.globals.getNode("/instrumentation/ilec-sc7/volume", 1);
var vario_aud = props.globals.getNode("/instrumentation/ilec-sc7/audio", 1);
var vario_read = props.globals.getNode("/instrumentation/ilec-sc7/te-reading-mps", 1);
var turnbank_spin = props.globals.getNode("/instrumentation/turn-indicator/spin", 1);

# Array of circuit breakers
var circuit_breakers = {
	vario: cb_prop.initNode("vario", 1, "BOOL"),
	radio: cb_prop.initNode("radio", 1, "BOOL"),
	turn:  cb_prop.initNode("turn",  1, "BOOL"),
};

var switches = {
	vario: sw_prop.initNode("vario", 0, "INT"),
	radio: sw_prop.initNode("radio", 0, "INT"),
	turn:  sw_prop.initNode("turn",  0, "INT"),
};

var el_volts = {
	vario: 0.0,
	radio: 0.0,
	turn:  0.0,
};

var outputs = {
	vario:	output_prop.initNode("ilec-sc7", 0.0, "DOUBLE"),
	radio:	output_prop.initNode("comm[0]", 0.0, "DOUBLE"),
	flarm:	output_prop.initNode("flarm", 0.0, "DOUBLE"),
	turn:	output_prop.initNode("turn-coordinator", 0.0, "DOUBLE"),
};
	

var freeze_replay	=	props.globals.getNode("/sim/freeze/replay-state");

##
# Battery model class.
#

var BatteryClass = {
	# Constructor
	new: func( ideal_volts, ideal_amps, amp_hours, charge_amps, n ){
		var charge_prop	= batt_prop.getNode( "charge["~n~"]" );
		var charge	= nil;
		if( getprop("/systems/electrical/battery/charge["~n~"]") != nil ){			# If the battery charge has been set from a previous FG instance
			charge = charge_prop.getDoubleValue();
		} else {
			charge = 1.0;
			charge_prop = batt_prop.initNode("charge["~n~"]", 1.0, "DOUBLE");
		}
		var obj = {
			parents: [BatteryClass],
			ideal_volts: ideal_volts,
			ideal_amps: ideal_amps,
			amp_hours: amp_hours,
			charge_amps: charge_amps,
			charge: charge,
			charge_prop: charge_prop,
			n: n 
		};
		return obj;
	},
	# Passing in positive amps means the battery will be discharged.
	# Negative amps indicates a battery charge.
	apply_load: func( amps, dt ){
		var old_charge = me.charge_prop.getDoubleValue();
		if( freeze_replay.getBoolValue() ){
			return me.amp_hours * old_charge;
		}
		var amphrs_used = amps * dt / 3600.0;
		var fraction_used = amphrs_used / me.amp_hours;
		
		#	Debug
		#	print( "Drawing "~ amps ~ "A at " ~ dt ~"sec, these are "~ amphrs_used ~"Ah out of "~ me.amp_hours ~"Ah or "~ fraction_used );
		
		var new_charge = std.max(0.0, std.min(old_charge - fraction_used, 1.0));
		
		if (new_charge < 0.1 and old_charge >= 0.1)
			gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
		me.charge = new_charge;
		me.charge_prop.setDoubleValue( new_charge );
		return me.amp_hours * new_charge;
	},
	# Return output volts based on percent charged.  Currently based on a simple
	# polynomial percent charge vs. volts function.
	get_output_volts: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32 ) / 32;
		return me.ideal_volts * factor;
	},
	# Return output amps available.  This function is totally wrong and should be
	# fixed at some point with a more sensible function based on charge percent.
	# There is probably some physical limits to the number of instantaneous amps
	# a battery can produce (cold cranking amps?)
	get_output_amps: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32) / 32;
		return me.ideal_amps * factor;
	},
	# Set the current charge instantly to 100 %.
	reset_to_full_charge: func() {
		me.apply_load(-(1.0 - me.charge) * me.amp_hours, 3600);
	},
	# Get current charge
	get_charge: func() {
		return me.charge;
	}
};

############################
####	Battery Packs	####
############################

##	example glider battery: https://shop.segelflugbedarf24.de/Flugzeugausstattung/Akkus-Energieversorgung/Sonnenschein-Dryfit-A512-12V/6-5-Ah::731.html
##				http://www.sonnenschein.org/A500.htm	(A512-6.5S)
##				ideal volts: 12.0
##				ideal amps: 0.325 (max. 80 / 300 for 5 sec))
##				amp hours: 6.5
##				charge amps: 25
var batteries = [
	BatteryClass.new( 12.0, 0.325, 6.5, 25, 0),
	BatteryClass.new( 12.0, 0.325, 6.5, 25, 1),
	BatteryClass.new( 12.0, 0.325, 6.5, 25, 2),
	BatteryClass.new( 12.0, 0.325, 6.5, 25, 3),
];

var reset_battery = func {
	foreach(var b; keys(batteries)){
		batteries[b].reset_to_full_charge();
	}
}
var reset_circuit_breakers = func {
	foreach(var cb; keys(circuit_breakers)){
		circuit_breakers[cb].setBoolValue( 1 );
	}
}

##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_bus(dt);
	}
};

#	The ASG29 has 3 knobs that can select the power source for each instrument individually
#	Each instrument is also protected by a circuit breaker


#Load sources:
#	radio:		https://www.skyfox.com/becker-ar6201-022-vhf-am-sprechfunkgeraet-8-33.html
#	vario:		http://www.ilec-gmbh.com/ilec/manuals/SC7pd.pdf
#	turn:		https://www.airteam.eu/de/p/falcon-tb02e-2-1 (not the same but similar)
#	flarm:		http://flarm.com/wp-content/uploads/man/FLARM_InstallationManual_D.pdf
#	flarm display:	https://www.air-store.eu/Display-V3-FLARM

var update_bus = func (dt) {
	if( !serviceable.getBoolValue() ){
		return;
	}
	#	VARIO
	if( circuit_breakers.vario.getBoolValue() ){
		var vario_src = switches.vario.getIntValue();
		if( vario_src != 0 ){
			el_volts.vario = batteries[ vario_src - 1 ].get_output_volts();
			if( el_volts.vario > 9 ) {
				#Energy consumption:	25mA (medium volume) 60mA (max volume) -> guess: at 12V
				#			guess: base consumption 5mA (no volume)
				var vario_load = 0.06 / el_volts.vario;
				if(vario_aud.getValue() == 2 or (vario_aud.getValue() == 1 and vario_read.getValue() > 0)){
					vario_load += (vario_vol.getValue()*0.66) / el_volts.vario;
				}
				outputs.vario.setDoubleValue( el_volts.vario );
				sc7_volts.setDoubleValue( el_volts.vario );
				batteries[ vario_src - 1 ].apply_load( vario_load, dt );
			}
		} else {
			outputs.vario.setDoubleValue( 0.0 );
			sc7_volts.setDoubleValue( 0.0 );
		}
	} else {
		outputs.vario.setDoubleValue( 0.0 );
		sc7_volts.setDoubleValue( 0.0 );
	}
	#	RADIO	(assume: and FLARM)
	if( circuit_breakers.radio.getBoolValue() ){
		var radio_src = switches.radio.getIntValue();
		if( radio_src != 0 ){
			el_volts.radio = batteries[ radio_src - 1 ].get_output_volts();
			if( el_volts.radio > 9 ) {			
				if( com_ptt.getBoolValue() ){
					var radio_load = 19.2 / el_volts.radio; #transmitting: 1600mA at 12V
				}else{
					var radio_load = 1.02 / el_volts.radio; # 85mA at 12V (standby)
				}
				radio_load += 0.66 / el_volts.radio; #FLARM
				radio_load += 0.12 / el_volts.radio; #FLARM display
				outputs.radio.setDoubleValue( el_volts.radio );
				outputs.flarm.setDoubleValue( el_volts.radio );
				batteries[ radio_src - 1 ].apply_load( radio_load, dt );
			}
		} else {
			outputs.radio.setDoubleValue( 0.0 );
			outputs.flarm.setDoubleValue( 0.0 );
		}
	} else {
		outputs.radio.setDoubleValue( 0.0 );
		outputs.flarm.setDoubleValue( 0.0 );
	}
	#	TURN INDICATOR
	if( circuit_breakers.turn.getBoolValue() ){
		var turn_src = switches.turn.getIntValue();
		if( turn_src != 0 ){
			el_volts.turn = batteries[ turn_src - 1 ].get_output_volts();
			if( el_volts.turn > 9 ) {
				#Energy Consumption:
				#	starting ~9.9 / volts (approx)
				#	running ~7.8 / volts (approx)
				if( turnbank_spin.getValue() > 0.99 ){
					var turn_load = 7.8 / el_volts.turn;
				}else{
					var turn_load = 9.9 / el_volts.turn;
				}
				outputs.turn.setDoubleValue( el_volts.turn );
				batteries[ turn_src - 1 ].apply_load( turn_load, dt );
			}
		} else {
			outputs.turn.setDoubleValue( 0.0 );
		}
	} else {
		outputs.turn.setDoubleValue( 0.0 );
	}
}

##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

setlistener("/sim/signals/fdm-initialized", func {
	reset_circuit_breakers();
	
	system_updater.enable();
	print("Electrical System initialized");
});
