/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.util.file.classloader;

import java.io.File;
import java.io.FilenameFilter;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.HexFormat;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.nifi.util.security.MessageDigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClassLoaderUtils {
    static final Logger LOGGER = LoggerFactory.getLogger(ClassLoaderUtils.class);

    public static ClassLoader getCustomClassLoader(String modulePath, ClassLoader parentClassLoader, FilenameFilter filenameFilter) throws MalformedURLException {
        URL[] classpaths = ClassLoaderUtils.getURLsForClasspath(modulePath, filenameFilter, false);
        return ClassLoaderUtils.createModuleClassLoader(classpaths, parentClassLoader);
    }

    public static URL[] getURLsForClasspath(String modulePath, FilenameFilter filenameFilter, boolean suppressExceptions) throws MalformedURLException {
        return ClassLoaderUtils.getURLsForClasspath(modulePath == null ? Collections.emptySet() : Collections.singleton(modulePath), filenameFilter, suppressExceptions);
    }

    public static URL[] getURLsForClasspath(Set<String> modulePaths, FilenameFilter filenameFilter, boolean suppressExceptions) throws MalformedURLException {
        LinkedHashSet<String> modules = new LinkedHashSet<String>();
        if (modulePaths != null) {
            modulePaths.stream().flatMap(path -> Arrays.stream(path.split(","))).filter(path -> ClassLoaderUtils.isNotBlank(path)).map(String::trim).forEach(m -> modules.add((String)m));
        }
        return ClassLoaderUtils.toURLs(modules, filenameFilter, suppressExceptions);
    }

    private static boolean isNotBlank(String value) {
        return value != null && !value.isBlank();
    }

    protected static URL[] toURLs(Set<String> modulePaths, FilenameFilter filenameFilter, boolean suppressExceptions) throws MalformedURLException {
        LinkedList<URL> additionalClasspath = new LinkedList<URL>();
        if (modulePaths != null) {
            for (String modulePathString : modulePaths) {
                boolean isUrl = true;
                try {
                    additionalClasspath.add(URI.create(modulePathString).toURL());
                }
                catch (IllegalArgumentException | MalformedURLException e) {
                    isUrl = false;
                }
                if (isUrl) continue;
                try {
                    File modulePath = new File(modulePathString);
                    if (modulePath.exists()) {
                        File[] files;
                        additionalClasspath.add(modulePath.toURI().toURL());
                        if (!modulePath.isDirectory() || (files = modulePath.listFiles(filenameFilter)) == null) continue;
                        for (File classpathResource : files) {
                            if (classpathResource.isDirectory()) {
                                LOGGER.warn("Recursive directories are not supported, skipping {}", (Object)classpathResource.getAbsolutePath());
                                continue;
                            }
                            additionalClasspath.add(classpathResource.toURI().toURL());
                        }
                        continue;
                    }
                    throw new MalformedURLException("Path specified does not exist");
                }
                catch (MalformedURLException e) {
                    if (suppressExceptions) continue;
                    throw e;
                }
            }
        }
        return additionalClasspath.toArray(new URL[additionalClasspath.size()]);
    }

    public static String generateAdditionalUrlsFingerprint(Set<URL> urls, String classloaderIsolationKey) {
        StringBuilder formattedUrls = new StringBuilder();
        List<String> sortedUrls = urls.stream().map(Object::toString).sorted().collect(Collectors.toList());
        sortedUrls.forEach(url -> formattedUrls.append((String)url).append("-").append(ClassLoaderUtils.getLastModified(url)).append(";"));
        formattedUrls.append(classloaderIsolationKey);
        byte[] formattedUrlsBinary = formattedUrls.toString().getBytes(StandardCharsets.UTF_8);
        return HexFormat.of().formatHex(MessageDigestUtils.getDigest(formattedUrlsBinary));
    }

    private static long getLastModified(String url) {
        long lastModified = 0L;
        try {
            URI uri = new URI(url);
            if (uri.getScheme().equals("file")) {
                File file = new File(uri);
                lastModified = file.lastModified();
            }
        }
        catch (URISyntaxException e) {
            LOGGER.error("Error getting last modified date for {}", (Object)url);
        }
        return lastModified;
    }

    protected static ClassLoader createModuleClassLoader(URL[] modules, ClassLoader parentClassLoader) {
        return new URLClassLoader(modules, parentClassLoader);
    }
}

