/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.bigtable.data.v2.stub.metrics;

import com.google.api.Distribution;
import com.google.api.Metric;
import com.google.api.MetricDescriptor;
import com.google.api.MonitoredResource;
import com.google.cloud.bigtable.data.v2.stub.metrics.BuiltinMetricsConstants;
import com.google.cloud.opentelemetry.detection.DetectedPlatform;
import com.google.cloud.opentelemetry.detection.GCPPlatformDetector;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.monitoring.v3.Point;
import com.google.monitoring.v3.TimeInterval;
import com.google.monitoring.v3.TimeSeries;
import com.google.monitoring.v3.TypedValue;
import com.google.protobuf.util.Timestamps;
import io.opentelemetry.api.common.AttributeKey;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.sdk.metrics.data.AggregationTemporality;
import io.opentelemetry.sdk.metrics.data.DoublePointData;
import io.opentelemetry.sdk.metrics.data.HistogramData;
import io.opentelemetry.sdk.metrics.data.HistogramPointData;
import io.opentelemetry.sdk.metrics.data.LongPointData;
import io.opentelemetry.sdk.metrics.data.MetricData;
import io.opentelemetry.sdk.metrics.data.MetricDataType;
import io.opentelemetry.sdk.metrics.data.PointData;
import io.opentelemetry.sdk.metrics.data.SumData;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;

class BigtableExporterUtils {
    private static final Logger logger = Logger.getLogger(BigtableExporterUtils.class.getName());
    private static final String BIGTABLE_RESOURCE_TYPE = "bigtable_client_raw";
    private static final Set<AttributeKey<String>> BIGTABLE_PROMOTED_RESOURCE_LABELS = ImmutableSet.of(BuiltinMetricsConstants.BIGTABLE_PROJECT_ID_KEY, BuiltinMetricsConstants.INSTANCE_ID_KEY, BuiltinMetricsConstants.TABLE_ID_KEY, BuiltinMetricsConstants.CLUSTER_ID_KEY, BuiltinMetricsConstants.ZONE_ID_KEY);

    private BigtableExporterUtils() {
    }

    static String getDefaultTaskValue() {
        String jvmName = ManagementFactory.getRuntimeMXBean().getName();
        if (jvmName.indexOf(64) < 1) {
            String hostname = "localhost";
            try {
                hostname = InetAddress.getLocalHost().getHostName();
            }
            catch (UnknownHostException e) {
                logger.log(Level.INFO, "Unable to get the hostname.", e);
            }
            return "java-" + UUID.randomUUID() + "@" + hostname;
        }
        return "java-" + UUID.randomUUID() + jvmName;
    }

    static String getProjectId(PointData pointData) {
        return (String)pointData.getAttributes().get(BuiltinMetricsConstants.BIGTABLE_PROJECT_ID_KEY);
    }

    static List<TimeSeries> convertToBigtableTimeSeries(List<MetricData> collection, String taskId) {
        ArrayList<TimeSeries> allTimeSeries = new ArrayList<TimeSeries>();
        for (MetricData metricData : collection) {
            if (!metricData.getInstrumentationScopeInfo().getName().equals("bigtable.googleapis.com/internal/client/")) continue;
            metricData.getData().getPoints().stream().map(pointData -> BigtableExporterUtils.convertPointToBigtableTimeSeries(metricData, pointData, taskId)).forEach(allTimeSeries::add);
        }
        return allTimeSeries;
    }

    static List<TimeSeries> convertToApplicationResourceTimeSeries(Collection<MetricData> collection, String taskId, MonitoredResource applicationResource) {
        Preconditions.checkNotNull((Object)applicationResource, (Object)"convert application metrics is called when the supported resource is not detected");
        ArrayList<TimeSeries> allTimeSeries = new ArrayList<TimeSeries>();
        for (MetricData metricData : collection) {
            if (!metricData.getInstrumentationScopeInfo().getName().equals("bigtable.googleapis.com/internal/client/")) continue;
            metricData.getData().getPoints().stream().map(pointData -> BigtableExporterUtils.convertPointToApplicationResourceTimeSeries(metricData, pointData, taskId, applicationResource)).forEach(allTimeSeries::add);
        }
        return allTimeSeries;
    }

    @Nullable
    static MonitoredResource detectResource() {
        GCPPlatformDetector detector = GCPPlatformDetector.DEFAULT_INSTANCE;
        DetectedPlatform detectedPlatform = detector.detectPlatform();
        MonitoredResource monitoredResource = null;
        try {
            switch (detectedPlatform.getSupportedPlatform()) {
                case GOOGLE_COMPUTE_ENGINE: {
                    monitoredResource = BigtableExporterUtils.createGceMonitoredResource(detectedPlatform.getProjectId(), detectedPlatform.getAttributes());
                    break;
                }
                case GOOGLE_KUBERNETES_ENGINE: {
                    monitoredResource = BigtableExporterUtils.createGkeMonitoredResource(detectedPlatform.getProjectId(), detectedPlatform.getAttributes());
                }
            }
        }
        catch (IllegalStateException e) {
            logger.log(Level.WARNING, "Failed to create monitored resource for " + detectedPlatform.getSupportedPlatform(), e);
        }
        return monitoredResource;
    }

    private static MonitoredResource createGceMonitoredResource(String projectId, Map<String, String> attributes) {
        return MonitoredResource.newBuilder().setType("gce_instance").putLabels("project_id", projectId).putLabels("instance_id", BigtableExporterUtils.getAttribute(attributes, "instance_id")).putLabels("zone", BigtableExporterUtils.getAttribute(attributes, "availability_zone")).build();
    }

    private static MonitoredResource createGkeMonitoredResource(String projectId, Map<String, String> attributes) {
        return MonitoredResource.newBuilder().setType("k8s_container").putLabels("project_id", projectId).putLabels("location", BigtableExporterUtils.getAttribute(attributes, "gke_cluster_location")).putLabels("cluster_name", BigtableExporterUtils.getAttribute(attributes, "gke_cluster_name")).putLabels("namespace_name", (String)MoreObjects.firstNonNull((Object)System.getenv("NAMESPACE"), (Object)"")).putLabels("pod_name", (String)MoreObjects.firstNonNull((Object)System.getenv("HOSTNAME"), (Object)"")).putLabels("container_name", (String)MoreObjects.firstNonNull((Object)System.getenv("CONTAINER_NAME"), (Object)"")).build();
    }

    private static String getAttribute(Map<String, String> attributes, String key) {
        String value = attributes.get(key);
        if (value == null) {
            throw new IllegalStateException("Required attribute " + key + " does not exist in the attributes map " + attributes);
        }
        return value;
    }

    private static TimeSeries convertPointToBigtableTimeSeries(MetricData metricData, PointData pointData, String taskId) {
        TimeSeries.Builder builder = TimeSeries.newBuilder().setMetricKind(BigtableExporterUtils.convertMetricKind(metricData)).setValueType(BigtableExporterUtils.convertValueType(metricData.getType()));
        Metric.Builder metricBuilder = Metric.newBuilder().setType(metricData.getName());
        Attributes attributes = pointData.getAttributes();
        MonitoredResource.Builder monitoredResourceBuilder = MonitoredResource.newBuilder().setType(BIGTABLE_RESOURCE_TYPE);
        for (AttributeKey key : attributes.asMap().keySet()) {
            if (BIGTABLE_PROMOTED_RESOURCE_LABELS.contains(key)) {
                monitoredResourceBuilder.putLabels(key.getKey(), String.valueOf(attributes.get(key)));
                continue;
            }
            metricBuilder.putLabels(key.getKey(), String.valueOf(attributes.get(key)));
        }
        builder.setResource(monitoredResourceBuilder.build());
        metricBuilder.putLabels(BuiltinMetricsConstants.CLIENT_UID_KEY.getKey(), taskId);
        builder.setMetric(metricBuilder.build());
        TimeInterval timeInterval = TimeInterval.newBuilder().setStartTime(Timestamps.fromNanos((long)pointData.getStartEpochNanos())).setEndTime(Timestamps.fromNanos((long)pointData.getEpochNanos())).build();
        builder.addPoints(BigtableExporterUtils.createPoint(metricData.getType(), pointData, timeInterval));
        return builder.build();
    }

    private static TimeSeries convertPointToApplicationResourceTimeSeries(MetricData metricData, PointData pointData, String taskId, MonitoredResource applicationResource) {
        TimeSeries.Builder builder = TimeSeries.newBuilder().setMetricKind(BigtableExporterUtils.convertMetricKind(metricData)).setValueType(BigtableExporterUtils.convertValueType(metricData.getType())).setResource(applicationResource);
        Metric.Builder metricBuilder = Metric.newBuilder().setType(metricData.getName());
        Attributes attributes = pointData.getAttributes();
        for (AttributeKey key : attributes.asMap().keySet()) {
            metricBuilder.putLabels(key.getKey(), String.valueOf(attributes.get(key)));
        }
        metricBuilder.putLabels(BuiltinMetricsConstants.CLIENT_UID_KEY.getKey(), taskId);
        builder.setMetric(metricBuilder.build());
        TimeInterval timeInterval = TimeInterval.newBuilder().setStartTime(Timestamps.fromNanos((long)pointData.getStartEpochNanos())).setEndTime(Timestamps.fromNanos((long)pointData.getEpochNanos())).build();
        builder.addPoints(BigtableExporterUtils.createPoint(metricData.getType(), pointData, timeInterval));
        return builder.build();
    }

    private static MetricDescriptor.MetricKind convertMetricKind(MetricData metricData) {
        switch (metricData.getType()) {
            case HISTOGRAM: 
            case EXPONENTIAL_HISTOGRAM: {
                return BigtableExporterUtils.convertHistogramType(metricData.getHistogramData());
            }
            case LONG_GAUGE: 
            case DOUBLE_GAUGE: {
                return MetricDescriptor.MetricKind.GAUGE;
            }
            case LONG_SUM: {
                return BigtableExporterUtils.convertSumDataType(metricData.getLongSumData());
            }
            case DOUBLE_SUM: {
                return BigtableExporterUtils.convertSumDataType(metricData.getDoubleSumData());
            }
        }
        return MetricDescriptor.MetricKind.UNRECOGNIZED;
    }

    private static MetricDescriptor.MetricKind convertHistogramType(HistogramData histogramData) {
        if (histogramData.getAggregationTemporality() == AggregationTemporality.CUMULATIVE) {
            return MetricDescriptor.MetricKind.CUMULATIVE;
        }
        return MetricDescriptor.MetricKind.UNRECOGNIZED;
    }

    private static MetricDescriptor.MetricKind convertSumDataType(SumData<?> sum) {
        if (!sum.isMonotonic()) {
            return MetricDescriptor.MetricKind.GAUGE;
        }
        if (sum.getAggregationTemporality() == AggregationTemporality.CUMULATIVE) {
            return MetricDescriptor.MetricKind.CUMULATIVE;
        }
        return MetricDescriptor.MetricKind.UNRECOGNIZED;
    }

    private static MetricDescriptor.ValueType convertValueType(MetricDataType metricDataType) {
        switch (metricDataType) {
            case LONG_GAUGE: 
            case LONG_SUM: {
                return MetricDescriptor.ValueType.INT64;
            }
            case DOUBLE_GAUGE: 
            case DOUBLE_SUM: {
                return MetricDescriptor.ValueType.DOUBLE;
            }
            case HISTOGRAM: 
            case EXPONENTIAL_HISTOGRAM: {
                return MetricDescriptor.ValueType.DISTRIBUTION;
            }
        }
        return MetricDescriptor.ValueType.UNRECOGNIZED;
    }

    private static Point createPoint(MetricDataType type, PointData pointData, TimeInterval timeInterval) {
        Point.Builder builder = Point.newBuilder().setInterval(timeInterval);
        switch (type) {
            case HISTOGRAM: 
            case EXPONENTIAL_HISTOGRAM: {
                return builder.setValue(TypedValue.newBuilder().setDistributionValue(BigtableExporterUtils.convertHistogramData((HistogramPointData)pointData)).build()).build();
            }
            case DOUBLE_GAUGE: 
            case DOUBLE_SUM: {
                return builder.setValue(TypedValue.newBuilder().setDoubleValue(((DoublePointData)pointData).getValue()).build()).build();
            }
            case LONG_GAUGE: 
            case LONG_SUM: {
                return builder.setValue(TypedValue.newBuilder().setInt64Value(((LongPointData)pointData).getValue())).build();
            }
        }
        logger.log(Level.WARNING, "unsupported metric type");
        return builder.build();
    }

    private static Distribution convertHistogramData(HistogramPointData pointData) {
        return Distribution.newBuilder().setCount(pointData.getCount()).setMean(pointData.getCount() == 0L ? 0.0 : pointData.getSum() / (double)pointData.getCount()).setBucketOptions(Distribution.BucketOptions.newBuilder().setExplicitBuckets(Distribution.BucketOptions.Explicit.newBuilder().addAllBounds((Iterable)pointData.getBoundaries()))).addAllBucketCounts((Iterable)pointData.getCounts()).build();
    }
}

