/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.impl.prefetch;

import java.io.Closeable;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Future;
import org.apache.hadoop.fs.impl.prefetch.BoundedResourcePool;
import org.apache.hadoop.fs.impl.prefetch.BufferData;
import org.apache.hadoop.fs.impl.prefetch.PrefetchingStatistics;
import org.apache.hadoop.fs.impl.prefetch.Retryer;
import org.apache.hadoop.fs.impl.prefetch.Validate;
import org.apache.hadoop.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BufferPool
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(BufferPool.class);
    private final int size;
    private final int bufferSize;
    private BoundedResourcePool<ByteBuffer> pool;
    private Map<BufferData, ByteBuffer> allocated;
    private PrefetchingStatistics prefetchingStatistics;

    public BufferPool(int size, final int bufferSize, final PrefetchingStatistics prefetchingStatistics) {
        Validate.checkPositiveInteger(size, "size");
        Validate.checkPositiveInteger(bufferSize, "bufferSize");
        this.size = size;
        this.bufferSize = bufferSize;
        this.allocated = new IdentityHashMap<BufferData, ByteBuffer>();
        this.prefetchingStatistics = Objects.requireNonNull(prefetchingStatistics);
        this.pool = new BoundedResourcePool<ByteBuffer>(size){

            @Override
            public ByteBuffer createNew() {
                ByteBuffer buffer = ByteBuffer.allocate(bufferSize);
                prefetchingStatistics.memoryAllocated(bufferSize);
                return buffer;
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<BufferData> getAll() {
        Map<BufferData, ByteBuffer> map = this.allocated;
        synchronized (map) {
            return Collections.unmodifiableList(new ArrayList<BufferData>(this.allocated.keySet()));
        }
    }

    public synchronized BufferData acquire(int blockNumber) {
        BufferData data;
        int maxRetryDelayMs = 600000;
        int statusUpdateDelayMs = 120000;
        Retryer retryer = new Retryer(10, 600000, 120000);
        do {
            if (!retryer.updateStatus()) continue;
            if (LOG.isDebugEnabled()) {
                LOG.debug("waiting to acquire block: {}", (Object)blockNumber);
                LOG.debug("state = {}", (Object)this);
            }
            this.releaseReadyBlock(blockNumber);
        } while ((data = this.tryAcquire(blockNumber)) == null && retryer.continueRetry());
        if (data != null) {
            return data;
        }
        String message = String.format("Wait failed for acquire(%d)", blockNumber);
        throw new IllegalStateException(message);
    }

    public synchronized BufferData tryAcquire(int blockNumber) {
        return this.acquireHelper(blockNumber, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized BufferData acquireHelper(int blockNumber, boolean canBlock) {
        ByteBuffer buffer;
        Validate.checkNotNegative(blockNumber, "blockNumber");
        this.releaseDoneBlocks();
        BufferData data = this.find(blockNumber);
        if (data != null) {
            return data;
        }
        ByteBuffer byteBuffer = buffer = canBlock ? this.pool.acquire() : this.pool.tryAcquire();
        if (buffer == null) {
            return null;
        }
        buffer.clear();
        data = new BufferData(blockNumber, buffer.duplicate());
        Map<BufferData, ByteBuffer> map = this.allocated;
        synchronized (map) {
            Validate.checkState(this.find(blockNumber) == null, "buffer data already exists", new Object[0]);
            this.allocated.put(data, buffer);
        }
        return data;
    }

    private synchronized void releaseDoneBlocks() {
        for (BufferData data : this.getAll()) {
            if (!data.stateEqualsOneOf(BufferData.State.DONE)) continue;
            this.release(data);
        }
    }

    private synchronized void releaseReadyBlock(int blockNumber) {
        BufferData releaseTarget = null;
        for (BufferData data : this.getAll()) {
            if (!data.stateEqualsOneOf(BufferData.State.READY)) continue;
            if (releaseTarget == null) {
                releaseTarget = data;
                continue;
            }
            if (this.distance(data, blockNumber) <= this.distance(releaseTarget, blockNumber)) continue;
            releaseTarget = data;
        }
        if (releaseTarget != null) {
            LOG.warn("releasing 'ready' block: {}", (Object)releaseTarget);
            releaseTarget.setDone();
        }
    }

    private int distance(BufferData data, int blockNumber) {
        return Math.abs(data.getBlockNumber() - blockNumber);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void release(BufferData data) {
        Preconditions.checkNotNull(data, "data");
        BufferData bufferData = data;
        synchronized (bufferData) {
            Preconditions.checkArgument(this.canRelease(data), String.format("Unable to release buffer: %s", data));
            ByteBuffer buffer = this.allocated.get(data);
            if (buffer == null) {
                return;
            }
            buffer.clear();
            this.pool.release(buffer);
            this.allocated.remove(data);
        }
        this.releaseDoneBlocks();
    }

    @Override
    public synchronized void close() {
        for (BufferData data : this.getAll()) {
            Future<Void> actionFuture = data.getActionFuture();
            if (actionFuture == null) continue;
            actionFuture.cancel(true);
        }
        int currentPoolSize = this.pool.numCreated();
        this.pool.close();
        this.pool = null;
        this.allocated.clear();
        this.allocated = null;
        this.prefetchingStatistics.memoryFreed(currentPoolSize * this.bufferSize);
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(this.pool.toString());
        sb.append("\n");
        ArrayList<BufferData> allData = new ArrayList<BufferData>(this.getAll());
        Collections.sort(allData, (d1, d2) -> d1.getBlockNumber() - d2.getBlockNumber());
        for (BufferData data : allData) {
            sb.append(data.toString());
            sb.append("\n");
        }
        return sb.toString();
    }

    public synchronized int numCreated() {
        return this.pool.numCreated();
    }

    public synchronized int numAvailable() {
        this.releaseDoneBlocks();
        return this.pool.numAvailable();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private BufferData find(int blockNumber) {
        Map<BufferData, ByteBuffer> map = this.allocated;
        synchronized (map) {
            for (BufferData data : this.allocated.keySet()) {
                if (data.getBlockNumber() != blockNumber || data.stateEqualsOneOf(BufferData.State.DONE)) continue;
                return data;
            }
        }
        return null;
    }

    private boolean canRelease(BufferData data) {
        return data.stateEqualsOneOf(BufferData.State.DONE, BufferData.State.READY);
    }
}

