/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.IntBuffer;
import java.util.zip.Checksum;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.fs.ChecksumException;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.LimitedPrivate(value={"HDFS"})
@InterfaceStability.Unstable
public abstract class FSInputChecker
extends FSInputStream {
    public static final Logger LOG = LoggerFactory.getLogger(FSInputChecker.class);
    protected Path file;
    private Checksum sum;
    private boolean verifyChecksum = true;
    private int maxChunkSize;
    private byte[] buf;
    private byte[] checksum;
    private IntBuffer checksumInts;
    private int pos;
    private int count;
    private int numOfRetries;
    private long chunkPos = 0L;
    private static final int CHUNKS_PER_READ = 32;
    protected static final int CHECKSUM_SIZE = 4;

    protected FSInputChecker(Path file, int numOfRetries) {
        this.file = file;
        this.numOfRetries = numOfRetries;
    }

    protected FSInputChecker(Path file, int numOfRetries, boolean verifyChecksum, Checksum sum, int chunkSize, int checksumSize) {
        this(file, numOfRetries);
        this.set(verifyChecksum, sum, chunkSize, checksumSize);
    }

    protected abstract int readChunk(long var1, byte[] var3, int var4, int var5, byte[] var6) throws IOException;

    protected abstract long getChunkPosition(long var1);

    protected synchronized boolean needChecksum() {
        return this.verifyChecksum && this.sum != null;
    }

    @Override
    public synchronized int read() throws IOException {
        if (this.pos >= this.count) {
            this.fill();
            if (this.pos >= this.count) {
                return -1;
            }
        }
        return this.buf[this.pos++] & 0xFF;
    }

    @Override
    public synchronized int read(byte[] b, int off, int len) throws IOException {
        int nread;
        if ((off | len | off + len | b.length - (off + len)) < 0) {
            throw new IndexOutOfBoundsException();
        }
        if (len == 0) {
            return 0;
        }
        int n = 0;
        do {
            if ((nread = this.read1(b, off + n, len - n)) > 0) continue;
            return n == 0 ? nread : n;
        } while ((n += nread) < len);
        return n;
    }

    private void fill() throws IOException {
        assert (this.pos >= this.count);
        this.count = this.readChecksumChunk(this.buf, 0, this.maxChunkSize);
        if (this.count < 0) {
            this.count = 0;
        }
    }

    protected final synchronized int readAndDiscard(int len) throws IOException {
        int total;
        int rd;
        for (total = 0; total < len; total += rd) {
            if (this.pos >= this.count) {
                this.count = this.readChecksumChunk(this.buf, 0, this.maxChunkSize);
                if (this.count <= 0) break;
            }
            rd = Math.min(this.count - this.pos, len - total);
            this.pos += rd;
        }
        return total;
    }

    private int read1(byte[] b, int off, int len) throws IOException {
        int avail = this.count - this.pos;
        if (avail <= 0) {
            if (len >= this.maxChunkSize) {
                int nread = this.readChecksumChunk(b, off, len);
                return nread;
            }
            this.fill();
            if (this.count <= 0) {
                return -1;
            }
            avail = this.count;
        }
        int cnt = avail < len ? avail : len;
        System.arraycopy(this.buf, this.pos, b, off, cnt);
        this.pos += cnt;
        return cnt;
    }

    private int readChecksumChunk(byte[] b, int off, int len) throws IOException {
        this.pos = 0;
        this.count = 0;
        int read = 0;
        boolean retry = true;
        int retriesLeft = this.numOfRetries;
        do {
            --retriesLeft;
            try {
                read = this.readChunk(this.chunkPos, b, off, len, this.checksum);
                if (read > 0) {
                    if (this.needChecksum()) {
                        this.verifySums(b, off, read);
                    }
                    this.chunkPos += (long)read;
                }
                retry = false;
            }
            catch (ChecksumException ce) {
                LOG.info("Found checksum error: b[" + off + ", " + (off + read) + "]=" + StringUtils.byteToHexString(b, off, off + read), ce);
                if (retriesLeft == 0) {
                    throw ce;
                }
                if (this.seekToNewSource(this.chunkPos)) {
                    this.seek(this.chunkPos);
                    continue;
                }
                throw ce;
            }
        } while (retry);
        return read;
    }

    private void verifySums(byte[] b, int off, int read) throws ChecksumException {
        int leftToVerify = read;
        int verifyOff = 0;
        this.checksumInts.rewind();
        this.checksumInts.limit((read - 1) / this.maxChunkSize + 1);
        while (leftToVerify > 0) {
            this.sum.update(b, off + verifyOff, Math.min(leftToVerify, this.maxChunkSize));
            int expected = this.checksumInts.get();
            int calculated = (int)this.sum.getValue();
            this.sum.reset();
            if (expected != calculated) {
                long errPos = this.chunkPos + (long)verifyOff;
                throw new ChecksumException("Checksum error: " + this.file + " at " + errPos + " exp: " + expected + " got: " + calculated, errPos);
            }
            leftToVerify -= this.maxChunkSize;
            verifyOff += this.maxChunkSize;
        }
    }

    @Deprecated
    public static long checksum2long(byte[] checksum) {
        long crc = 0L;
        for (int i = 0; i < checksum.length; ++i) {
            crc |= (0xFFL & (long)checksum[i]) << (checksum.length - i - 1) * 8;
        }
        return crc;
    }

    @Override
    public synchronized long getPos() throws IOException {
        return this.chunkPos - Math.max(0L, (long)(this.count - this.pos));
    }

    @Override
    public synchronized int available() throws IOException {
        return Math.max(0, this.count - this.pos);
    }

    @Override
    public synchronized long skip(long n) throws IOException {
        if (n <= 0L) {
            return 0L;
        }
        this.seek(this.getPos() + n);
        return n;
    }

    @Override
    public synchronized void seek(long pos) throws IOException {
        if (pos < 0L) {
            throw new EOFException("Cannot seek to a negative offset");
        }
        long start = this.chunkPos - (long)this.count;
        if (pos >= start && pos < this.chunkPos) {
            this.pos = (int)(pos - start);
            return;
        }
        this.resetState();
        this.chunkPos = this.getChunkPosition(pos);
        int delta = (int)(pos - this.chunkPos);
        if (delta > 0) {
            FSInputChecker.readFully(this, new byte[delta], 0, delta);
        }
    }

    protected static int readFully(InputStream stm, byte[] buf, int offset, int len) throws IOException {
        int nread;
        int n = 0;
        do {
            if ((nread = stm.read(buf, offset + n, len - n)) > 0) continue;
            return n == 0 ? nread : n;
        } while ((n += nread) < len);
        return n;
    }

    protected final synchronized void set(boolean verifyChecksum, Checksum sum, int maxChunkSize, int checksumSize) {
        assert (!verifyChecksum || sum == null || checksumSize == 4);
        this.maxChunkSize = maxChunkSize;
        this.verifyChecksum = verifyChecksum;
        this.sum = sum;
        this.buf = new byte[maxChunkSize];
        this.checksum = new byte[32 * checksumSize];
        this.checksumInts = ByteBuffer.wrap(this.checksum).asIntBuffer();
        this.count = 0;
        this.pos = 0;
    }

    @Override
    public final boolean markSupported() {
        return false;
    }

    @Override
    public final void mark(int readlimit) {
    }

    @Override
    public final void reset() throws IOException {
        throw new IOException("mark/reset not supported");
    }

    private void resetState() {
        this.count = 0;
        this.pos = 0;
        if (this.sum != null) {
            this.sum.reset();
        }
    }
}

