/*
    Platypus - program for creating Mac OS X application wrappers around scripts
    Copyright (C) 2003-2010 Sveinbjorn Thordarson <sveinbjornt@simnet.is>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

// PlatypusController class is the controller class for the basic Platypus 
// main window interface.  Also delegate for the application, and for menus.

#import "PlatypusController.h"

#define	kWindowExpansionHeight		312

@implementation Platypus

/*****************************************
 - init function
*****************************************/
- (id)init
{
	if (self = [super init]) 
		recentItems = [[NSMutableArray alloc] initWithCapacity: 10];
    return self;
}

/*****************************************
 - dealloc for controller object
   release all the stuff we alloc in init
*****************************************/
-(void)dealloc
{
	[recentItems release];
	[defaultInterpreters release];
    [super dealloc];
}

/*****************************************
 - When application is launched by the user for the very first time
*****************************************/
+ (void)initialize 
{ 
	// create the user defaults here if none exists
    NSMutableDictionary *defaultPrefs = [NSMutableDictionary dictionary];
    
	// put default prefs in the dictionary
	
	// create default bundle identifier string from usename
	NSString *bundleId = [NSString stringWithFormat: @"org.%@.", NSUserName()];
	bundleId = [[bundleId componentsSeparatedByString: @" "] componentsJoinedByString: @""];//no spaces
	
	[defaultPrefs setObject: bundleId						forKey: @"DefaultBundleIdentifierPrefix"];
	[defaultPrefs setObject: @"Built-In"					forKey: @"DefaultEditor"];
	[defaultPrefs setObject: [NSNumber numberWithBool:NO]	forKey: @"ShowAdvancedOptions"];
	[defaultPrefs setObject: [NSArray array]				forKey: @"Profiles"];
	[defaultPrefs setObject: [NSNumber numberWithBool:NO]	forKey: @"RevealApplicationWhenCreated"];
	[defaultPrefs setObject: [NSNumber numberWithInt: DEFAULT_OUTPUT_TXT_ENCODING]
															forKey: @"DefaultTextEncoding"]; 
	[defaultPrefs setObject: NSFullUserName()				forKey: @"DefaultAuthor"];
	
    // register the dictionary of defaults
    [[NSUserDefaults standardUserDefaults] registerDefaults: defaultPrefs];
}

- (void)awakeFromNib
{	
	// we list ourself as an observer of changes to file system, for script
	[[[NSWorkspace sharedWorkspace] notificationCenter]
		addObserver: self selector: @selector(scriptFileDidChange) name: UKFileWatcherRenameNotification object: NULL];
	[[[NSWorkspace sharedWorkspace] notificationCenter]
		addObserver: self selector: @selector(scriptFileDidChange) name: UKFileWatcherDeleteNotification object: NULL];
	
	defaultInterpreters = [[NSArray alloc] initWithObjects:			
						   @"/bin/sh",
						   @"/usr/bin/perl",
						   @"/usr/bin/python",
						   @"/usr/bin/ruby",
						   @"/usr/bin/osascript",
						   @"/usr/bin/tclsh",
						   @"/usr/bin/expect",
						   @"/usr/bin/php", 
						   @"",
						   nil];
		
	[self createAppSupportFolders];
	
	//set up window
	if ([[NSUserDefaults standardUserDefaults] boolForKey:@"ShowAdvancedOptions"] == YES)
	{
		[showAdvancedArrow setState: YES];
		[self toggleAdvancedOptions: self];
	}
	[window center];
	[window registerForDraggedTypes: [NSArray arrayWithObjects:NSFilenamesPboardType, nil]];
	[window makeFirstResponder: appNameTextField];
		
	// if we haven't already loaded a profile via openfile delegate method
	// we set all fields to their defaults.  Any profile must contain a name
	// so we can be sure that one hasn't been loaded if the app name field is empty
	if ([[appNameTextField stringValue] isEqualToString: @""])
		[self clearAllFields: self];
}



/*****************************************
 - Handler for when app is done launching
 - Set up the window and stuff like that
*****************************************/
- (void)applicationDidFinishLaunching:(NSNotification *)aNotification
{
	//enable autocompletion for script path in the nstextfield
	[scriptPathTextField setAutocompleteStyle: STShellAutocomplete];
	[interpreterTextField setAutocompleteStyle: STShellAutocomplete];
	
	//load recent items and build menu
	[recentItems addObjectsFromArray: [[NSUserDefaults standardUserDefaults] objectForKey:@"RecentItems"]];
	
	//remove all files there that no longer exist
	int i;
	for (i = [recentItems count]-1; i > 0; i--)
	{
		if (![[NSFileManager defaultManager] fileExistsAtPath: [recentItems objectAtIndex: i]])
			[recentItems removeObjectAtIndex: i];
	}
	[self constructOpenRecentMenu: self];
	
	//load profiles
	[profilesControl constructProfilesMenu: self];
	
	//show window
	[window makeKeyAndOrderFront: self];
}

/*****************************************
 - Create Application Support folder and subfolders
******************************************/
- (void) createAppSupportFolders
{
	BOOL isDir;

	// app support folder
	if (! [[NSFileManager defaultManager] fileExistsAtPath: [APP_SUPPORT_FOLDER stringByExpandingTildeInPath] isDirectory: &isDir])
	{		
		if ( ! [[NSFileManager defaultManager] createDirectoryAtPath: [APP_SUPPORT_FOLDER stringByExpandingTildeInPath] attributes: NULL] )
			[STUtil alert: @"Error" subText: [NSString stringWithFormat: @"Could not create directory '%@'", [APP_SUPPORT_FOLDER stringByExpandingTildeInPath]]]; 
	}
	
	// profiles folder
	if (! [[NSFileManager defaultManager] fileExistsAtPath: [PROFILES_FOLDER stringByExpandingTildeInPath] isDirectory: &isDir])
	{		
		if ( ! [[NSFileManager defaultManager] createDirectoryAtPath: [PROFILES_FOLDER stringByExpandingTildeInPath] attributes: NULL] )
			[STUtil alert: @"Error" subText: [NSString stringWithFormat: @"Could not create directory '%@'", [PROFILES_FOLDER stringByExpandingTildeInPath]]]; 
	}
}

/*****************************************
 - Handler for application termination
************c*****************************/
- (void)applicationWillTerminate:(NSNotification *)aNotification
{
	//save recent items
	[[NSUserDefaults standardUserDefaults] setObject: recentItems  forKey:@"RecentItems"];
	//save window status
	[[NSUserDefaults standardUserDefaults] setBool: [showAdvancedArrow state]  forKey:@"ShowAdvancedOptions"];
	//delete possible temp file for dragging
	[[NSFileManager defaultManager] removeFileAtPath: @"/tmp/PlatypusIcon.icns" handler: NULL];
}

/*****************************************
 - Handler for dragged files and/or files opened via the Finder
   We handle these as scripts, not bundled files
*****************************************/

- (BOOL)application:(NSApplication *)theApplication openFile:(NSString *)filename
{
	BOOL	isDir = NO;
		
	if([[NSFileManager defaultManager] fileExistsAtPath: filename isDirectory: &isDir] && !isDir)
	{	
		if ([filename hasSuffix: PROFILES_SUFFIX]) //load as profile
			[profilesControl loadProfileFile: filename];
		else //load as script
			[self loadScript: filename];
		return(YES);
	}
	return(NO);
}

#pragma mark -

/*****************************************
 - Create a new script and open in default editor
*****************************************/

- (IBAction)newScript:(id)sender
{
	int			i = 0;
	int			randnum;
	NSString	*tempScript;

	// get a random number to append to script name in /tmp/
	do
	{
		randnum =  random() / 1000000;
		tempScript = [NSString stringWithFormat: @"%@Script.%d", [TEMP_FOLDER stringByExpandingTildeInPath], randnum];
	}
	while ([[NSFileManager defaultManager] fileExistsAtPath: tempScript]);
	
	//put shebang line in the new script text file
	NSString	*shebangStr = [NSString stringWithFormat: @"#!%@\n\n", [interpreterTextField stringValue]];
	
	//if this is a perl or shell script, we add a commented list of paths to the bundled files 
	if (([[interpreterTextField stringValue] isEqualToString: @"/usr/bin/perl"] || [[interpreterTextField stringValue] isEqualToString: @"/bin/sh"]) && [fileList numFiles] > 0)
	{
		shebangStr = [shebangStr stringByAppendingString: @"# The following files are bundled:\n#\n"];
		for (i = 0; i < [fileList numFiles]; i++)
		{
			if ([[interpreterTextField stringValue] isEqualToString: @"/bin/sh"])//shell script
				shebangStr = [shebangStr stringByAppendingString: [NSString stringWithFormat:@"# '%@'\n", [[fileList getFileAtIndex: i] lastPathComponent]]];
			else if ([[interpreterTextField stringValue] isEqualToString: @"/usr/bin/perl"])//perl script
				shebangStr = [shebangStr stringByAppendingString: [NSString stringWithFormat:@"# '%@'\n", [[fileList getFileAtIndex: i] lastPathComponent]]];
		}
		shebangStr = [shebangStr stringByAppendingString: @"#\n#\n\n"];
	}
	
	//write the default content to the new script
	[shebangStr writeToFile: tempScript atomically: YES];
	
	//load and edit the script
	[self loadScript: tempScript];
	[self editScript: self];

}

/*****************************************
 - Reveal script in Finder
*****************************************/

- (IBAction)revealScript:(id)sender
{
	if ([[scriptPathTextField stringValue] length] == 0)//make sure the script path is not an empty string
		return;
	//see if file exists
	if ([[NSFileManager defaultManager] fileExistsAtPath: [scriptPathTextField stringValue] ])
		[[NSWorkspace sharedWorkspace] selectFile:[scriptPathTextField stringValue] inFileViewerRootedAtPath:nil];
	else
		[STUtil alert:@"File does not exist" subText: @"No file exists at the specified path"];
}

/*****************************************
 - Open script in external editor
*****************************************/

- (IBAction)editScript:(id)sender
{
	//make sure the script path is not an empty string
	if ([[scriptPathTextField stringValue] length] == 0)
		return;
		
	//see if file exists
	if ([[NSFileManager defaultManager] fileExistsAtPath: [scriptPathTextField stringValue] ])
	{
		// if the default editor is the built-in editor, we pop down the editor sheet
		if ([[[NSUserDefaults standardUserDefaults] stringForKey:@"DefaultEditor"] isEqualToString: @"Built-In"])
		{
			[self openScriptInBuiltInEditor: [scriptPathTextField stringValue]];
		}
		else // open it in the external application
		{
			NSString *defaultEditor = [[NSUserDefaults standardUserDefaults] stringForKey:@"DefaultEditor"];
			if ([[NSWorkspace sharedWorkspace] fullPathForApplication: defaultEditor] != NULL)
				[[NSWorkspace sharedWorkspace] openFile: [scriptPathTextField stringValue] withApplication: defaultEditor];
			else
			{
				// Complain if editor is not found, set it to the built-in editor
				[STUtil alert: @"Application not found" subText: [NSString stringWithFormat: @"The application '%@' could not be found on your system.  Reverting to the built-in editor.", defaultEditor]];
				[[NSUserDefaults standardUserDefaults] setObject: @"Built-In"  forKey:@"DefaultEditor"];
				[self openScriptInBuiltInEditor: [scriptPathTextField stringValue]];
			}
		}
	}
	else
		[STUtil alert:@"File does not exist" subText: @"No file exists at the specified path"];
}


/*****************************************
 - Run the script in Terminal.app
*****************************************/
- (IBAction)runScript:(id)sender
{
	NSTask	*theTask = [[NSTask alloc] init];

	//open Terminal.app
	[[NSWorkspace sharedWorkspace] launchApplication: @"Terminal.app"];

	//the applescript command to run the script in Terminal.app
	NSString *osaCmd = [NSString stringWithFormat: @"tell application \"Terminal\"\n\tdo script \"%@ '%@'\"\nend tell", [interpreterTextField stringValue], [scriptPathTextField stringValue]];
	
	//initialize task -- we launc the AppleScript via the 'osascript' CLI program
	[theTask setLaunchPath: @"/usr/bin/osascript"];
	[theTask setArguments: [NSArray arrayWithObjects: @"-e", osaCmd, nil]];
	
	//launch, wait until it's done and then release it
	[theTask launch];
	[theTask waitUntilExit];
	[theTask release];
}

/*****************************************
 - Report on syntax of script
*****************************************/

- (IBAction)checkSyntaxOfScript: (id)sender
{
	NSTask			*interpreter;
	NSPipe			*outputPipe = [NSPipe pipe];
	NSFileHandle	*readHandle;

	if (![[NSFileManager defaultManager] fileExistsAtPath: [scriptPathTextField stringValue] ])//make sure it exists
		return;
	
	interpreter = [[NSTask alloc] init];

	//let's see if the script type is supported for syntax checking
	//if so, we set up the task's launch path as the script interpreter and set the relevant flags and arguments
	switch([[scriptTypePopupMenu selectedItem] tag])
	{
		case 0: //shell scripts - /bin/sh
			[interpreter setLaunchPath: [defaultInterpreters objectAtIndex: 0]];
			[interpreter setArguments: [NSArray arrayWithObjects: @"-n", [scriptPathTextField stringValue], nil]];
			break;
		case 1: //perl scripts - /usr/bin/perl
			[interpreter setLaunchPath: [defaultInterpreters objectAtIndex: 1]];
			[interpreter setArguments: [NSArray arrayWithObjects: @"-c", [scriptPathTextField stringValue], nil]];
			break;
		case 2: //python scripts -- use bundled syntax checking script
			[interpreter setLaunchPath: [[NSBundle mainBundle] pathForResource: @"pycheck" ofType: @"py"]];
			[interpreter setArguments: [NSArray arrayWithObjects: [scriptPathTextField stringValue], nil]];
			break;
		case 3: //ruby scripts - /usr/bin/ruby
			[interpreter setLaunchPath: [defaultInterpreters objectAtIndex: 3]];
			[interpreter setArguments: [NSArray arrayWithObjects: @"-c", [scriptPathTextField stringValue], nil]];
			break;
		case 7: //php scripts - /usr/bin/php
			[interpreter setLaunchPath: [defaultInterpreters objectAtIndex: 7]];
			[interpreter setArguments: [NSArray arrayWithObjects: @"-l", [scriptPathTextField stringValue], nil]];
			break;
		default:
			[STUtil sheetAlert: @"Syntax Checking Unsupported" subText: @"Syntax checking is not supported for the scripting language you have selected" forWindow: window];
			[interpreter release];
			return;
	}
	
	//direct the output of the task into a file handle for reading
	[interpreter setStandardOutput: outputPipe];
	[interpreter setStandardError: outputPipe];
	readHandle = [outputPipe fileHandleForReading];
	
	//launch task
	[interpreter launch];
	[interpreter waitUntilExit];
	
	//get output in string
	NSString *outputStr = [[[NSString alloc] initWithData: [readHandle readDataToEndOfFile] encoding: DEFAULT_OUTPUT_TXT_ENCODING] autorelease];
	
	if ([outputStr length] == 0) //if the syntax report string is empty, we report syntax as OK
		outputStr = [NSString stringWithString: @"Syntax OK"];
	
	//set syntax checked file's path in syntax chcker window
	[syntaxScriptPathTextField setStringValue: [scriptPathTextField stringValue]];
	[syntaxCheckerTextField setString: outputStr];
	
	[interpreter release];//release the NSTask

	//report the result
	[window setTitle: [NSString stringWithFormat: @"%@ - Syntax Check Report", PROGRAM_NAME]];
	[NSApp beginSheet:	syntaxCheckerWindow
						modalForWindow: window 
						modalDelegate:nil
						didEndSelector:nil
						contextInfo:nil];
	[NSApp runModalForWindow: syntaxCheckerWindow];
	[NSApp endSheet:syntaxCheckerWindow];
    [syntaxCheckerWindow orderOut:self];
}

- (IBAction)closeSyntaxCheckerWindow: (id)sender
{
	[window setTitle: PROGRAM_NAME];
	[NSApp stopModal];
}

-(void)scriptFileDidChange
{
	[scriptPathTextField updateTextColoring];
	[self controlTextDidChange: NULL];
}

#pragma mark -

/*********************************************************************
 - Create button was pressed: Verify that field values are valid
 - Then put up a sheet for designating location to create application
**********************************************************************/

- (IBAction)createButtonPressed: (id)sender
{
	if (![self verifyFieldContents])//are there invalid values in the fields?
		return;

	NSSavePanel *sPanel = [NSSavePanel savePanel];
	[sPanel setPrompt:@"Create"];
	[window setTitle: [NSString stringWithFormat: @"%@ - Select place to create app", PROGRAM_NAME]];
	[sPanel setAccessoryView: debugSaveOptionView];
	
	// development version checkbox, disable this option if secure bundled script is checked
	[developmentVersionCheckbox setIntValue: 0];
	[developmentVersionCheckbox setEnabled: ![encryptCheckbox intValue]];	
	
	// optimize application is enabled and on by default if ibtool is present
	[optimizeApplicationCheckbox setEnabled: [[NSFileManager defaultManager] fileExistsAtPath: IBTOOL_PATH]];
	[optimizeApplicationCheckbox setIntValue: [[NSFileManager defaultManager] fileExistsAtPath: IBTOOL_PATH]];
	
	//run save panel
    [sPanel beginSheetForDirectory: NULL file: [appNameTextField stringValue] modalForWindow: window modalDelegate: self didEndSelector: @selector(createApp:returnCode:contextInfo:) contextInfo: nil];

	[developmentVersionCheckbox setIntValue: 0];
}

/*************************************************
 - generate application bundle from data provided
**************************************************/
- (void)createApp:(NSSavePanel *)sPanel returnCode:(int)result contextInfo:(void *)contextInfo
{
	BOOL overwrite = NO;
	
	//restore window title
	[window setTitle: PROGRAM_NAME];

	//if user pressed cancel, we do nothing
	if (result != NSOKButton) 
		return;
	
	// we begin by making sure destination path ends in .app
	NSString *appPath = [sPanel filename];
	if (![appPath hasSuffix:@".app"])
		appPath = [appPath stringByAppendingString:@".app"];
	
	//check if app already exists, and if so, prompt if to replace
	if ([[NSFileManager defaultManager] fileExistsAtPath: appPath])
	{
		overwrite = [STUtil proceedWarning: @"Application already exists" subText: @"An application with this name already exists in the location you specified.  Do you want to overwrite it?" withAction: @"Overwrite"];
		if (!overwrite)
			return;
	}
	
	// create spec from controls and verify
	PlatypusAppSpec	*spec = [self appSpecFromControls];
	
	// we set this specifically -- extra-profile data
	[spec setProperty: appPath forKey: @"Destination"];
	[spec setProperty: [[NSBundle mainBundle] pathForResource: @"ScriptExec" ofType: NULL] forKey: @"ExecutablePath"];
	[spec setProperty: [[NSBundle mainBundle] pathForResource: @"MainMenu.nib" ofType: NULL] forKey: @"NibPath"];
	[spec setProperty: [NSNumber numberWithBool: [developmentVersionCheckbox intValue]] forKey: @"DevelopmentVersion"];
	[spec setProperty: [NSNumber numberWithBool: [optimizeApplicationCheckbox intValue]] forKey: @"OptimizeApplication"];	
	if (overwrite)
		[spec setProperty: [NSNumber numberWithBool: YES] forKey: @"DestinationOverride"];
	
	// verify that the values in the spec are OK
	if (![spec verify])
	{
		[STUtil alert: @"Spec verification failed" subText: [spec getError]];
		return;
	}
	
	// create the app from spec
	if (![spec create])
	{
		[STUtil alert: @"Creating from spec failed" subText: [spec getError]];
		return;
	}
	
	// reveal newly create app in Finder, if prefs say so
	if ([[NSUserDefaults standardUserDefaults] boolForKey:@"RevealApplicationWhenCreated"])
		[[NSWorkspace sharedWorkspace] selectFile: appPath inFileViewerRootedAtPath:nil];
		
	[developmentVersionCheckbox setIntValue: 0];
	[optimizeApplicationCheckbox setIntValue: 0];
}


/*************************************************
 - Create app spec and fill it w. data from controls
**************************************************/

-(id)appSpecFromControls
{
	PlatypusAppSpec *spec = [[[PlatypusAppSpec alloc] initWithDefaults] autorelease];
	
	[spec setProperty: [appNameTextField stringValue]		forKey: @"Name"];
	[spec setProperty: [scriptPathTextField stringValue]	forKey: @"ScriptPath"];
	
	// set output type to the name of the output type, minus spaces
	[spec setProperty: [outputTypePopupMenu titleOfSelectedItem]
															forKey: @"Output"];
	
	// icon
	if ([iconControl usesCustomIcnsFile] || ![iconControl hasCustomIcon])
	{
		[spec setProperty: [NSNumber numberWithBool: YES]	forKey: @"HasCustomIcon"];
		[spec setProperty: [iconControl getIcnsFilePath]	forKey: @"CustomIconPath"];
	}
	else
	{
		[spec setProperty: [NSNumber numberWithBool: NO]	forKey: @"HasCustomIcon"];
		[spec setProperty: [iconControl getImage]	forKey: @"Icon"];
	}
	
	// advanced attributes
	[spec setProperty: [interpreterTextField stringValue]	forKey: @"Interpreter"];
	[spec setProperty: [paramsControl paramsArray]			forKey: @"Parameters"];
	[spec setProperty: [versionTextField stringValue]		forKey: @"Version"];
	[spec setProperty: [signatureTextField stringValue]		forKey: @"Signature"];
	[spec setProperty: [bundleIdentifierTextField stringValue]
															forKey: @"Identifier"];
	[spec setProperty: [authorTextField stringValue]		forKey: @"Author"];
	
	// checkbox attributes
	[spec setProperty: [NSNumber numberWithBool: [paramsControl passAppPathAsFirstArg]] 
															forKey: @"AppPathAsFirstArg"];
	
	[spec setProperty: [NSNumber numberWithBool:[isDroppableCheckbox state]]					
															forKey: @"Droppable"];
	[spec setProperty: [NSNumber numberWithBool:[encryptCheckbox state]]
															forKey: @"Secure"];
	[spec setProperty: [NSNumber numberWithBool:[rootPrivilegesCheckbox state]]		
															forKey: @"Authentication"];
	[spec setProperty: [NSNumber numberWithBool:[remainRunningCheckbox state]]
															forKey: @"RemainRunning"];
	[spec setProperty: [NSNumber numberWithBool:[showInDockCheckbox state]]
															forKey: @"ShowInDock"];

	// bundled files
	[spec setProperty: [fileList getFilesArray]	forKey: @"BundledFiles"];
	
	// environment
	[spec setProperty: [envControl environmentDictionary]	forKey: @"Environment"];

	// file types
	[spec setProperty: (NSMutableArray *)[(SuffixList *)[typesControl suffixes] getSuffixArray]				forKey: @"Suffixes"];
	[spec setProperty: (NSMutableArray *)[(TypesList *)[typesControl types] getTypesArray]					forKey: @"FileTypes"];
	[spec setProperty: [typesControl role]																	forKey: @"Role"];

	//  text output text settings
	[spec setProperty: [NSNumber numberWithInt: (int)[textSettingsControl getTextEncoding]]							forKey: @"TextEncoding"];
	[spec setProperty: [[textSettingsControl getTextFont] fontName]													forKey: @"TextFont"];
	[spec setProperty: [NSNumber numberWithFloat: [[textSettingsControl getTextFont] pointSize]]					forKey: @"TextSize"];
	[spec setProperty: [STUtil hexFromColor: [textSettingsControl getTextForeground]]								forKey: @"TextForeground"];
	[spec setProperty: [STUtil hexFromColor: [textSettingsControl getTextBackground]]								forKey: @"TextBackground"];
	
	// status menu settings
	if ([[outputTypePopupMenu titleOfSelectedItem] isEqualToString: @"Status Menu"])
	{
		[spec setProperty: [statusItemSettingsControl displayType] forKey: @"StatusItemDisplayType"];
		[spec setProperty: [statusItemSettingsControl title] forKey: @"StatusItemTitle"];
		[spec setProperty: [[statusItemSettingsControl icon] TIFFRepresentation] forKey: @"StatusItemIcon"];		
	}
	
	return spec;
}

- (void) controlsFromAppSpec: (id)spec
{
	[appNameTextField setStringValue: [spec propertyForKey: @"Name"]];
	[scriptPathTextField setStringValue: [spec propertyForKey: @"ScriptPath"]];

	[versionTextField setStringValue: [spec propertyForKey: @"Version"]];
	[signatureTextField setStringValue: [spec propertyForKey: @"Signature"]];
	[authorTextField setStringValue: [spec propertyForKey: @"Author"]];
	
	[outputTypePopupMenu selectItemWithTitle: [spec propertyForKey: @"Output"]];
	[self outputTypeWasChanged: NULL];
	[interpreterTextField setStringValue: [spec propertyForKey: @"Interpreter"]];
	
	//icon
	if ([[spec propertyForKey: @"HasCustomIcon"] boolValue] == YES)
		[iconControl setIcnsPath: [spec propertyForKey: @"CustomIconPath"]];
	else
		[iconControl setImage: [[[NSImage alloc] initWithData: [spec propertyForKey: @"Icon"]] autorelease]  ];

	//checkboxes
	[rootPrivilegesCheckbox setState: [[spec propertyForKey: @"Authentication"] boolValue]];
	[isDroppableCheckbox setState: [[spec propertyForKey: @"Droppable"] boolValue]];
		[self isDroppableWasClicked: isDroppableCheckbox];
	[encryptCheckbox setState: [[spec propertyForKey: @"Secure"] boolValue]];
	[showInDockCheckbox setState: [[spec propertyForKey: @"ShowInDock"] boolValue]];
	[remainRunningCheckbox setState: [[spec propertyForKey: @"RemainRunning"] boolValue]];
	
	//file list
		[fileList clearList];
		[fileList addFiles: [spec propertyForKey: @"BundledFiles"]];

		//update button status
		[fileList tableViewSelectionDidChange: NULL];
	
	//suffix list
		[(SuffixList *)[typesControl suffixes] clearList];
		[(SuffixList *)[typesControl suffixes] addSuffixes: [spec propertyForKey: @"Suffixes"]];
	
	//types list
		[(TypesList *)[typesControl types] clearList];
		[(TypesList *)[typesControl types] addTypes: [spec propertyForKey: @"FileTypes"]];
		
		[typesControl tableViewSelectionDidChange: NULL];
		[typesControl setRole: [spec propertyForKey: @"Role"]];
	
	// environment
		[envControl set: [spec propertyForKey: @"Environment"]];
	
	// parameters
		[paramsControl set: [spec propertyForKey: @"Parameters"]];
		[paramsControl setAppPathAsFirstArg: [[spec propertyForKey: @"AppPathAsFirstArg"] boolValue]];
		 
	// text output settings
	[textSettingsControl setTextEncoding: [[spec propertyForKey: @"TextEncoding"] intValue]];
	[textSettingsControl setTextFont: [NSFont fontWithName: [spec propertyForKey: @"TextFont"] size: [[spec propertyForKey: @"TextSize"] intValue]]];
	[textSettingsControl setTextForeground: [STUtil colorFromHex: [spec propertyForKey: @"TextForeground"]]];
	[textSettingsControl setTextBackground: [STUtil colorFromHex: [spec propertyForKey: @"TextBackground"]]];

	// status menu settings
	if ([[spec propertyForKey: @"Output"] isEqualToString: @"Status Menu"])
	{
		if (![[spec propertyForKey: @"StatusItemDisplayType"] isEqualToString: @"Text"])
		{
			NSImage *icon = [[[NSImage alloc] initWithData: [spec propertyForKey: @"StatusItemIcon"]] autorelease];
			if (icon != NULL)
				[statusItemSettingsControl setIcon: icon];
		}
		[statusItemSettingsControl setTitle: [spec propertyForKey: @"StatusItemTitle"]];
		[statusItemSettingsControl setDisplayType: [spec propertyForKey: @"StatusItemDisplayType"]];
	}
	
	//update buttons
	[self controlTextDidChange: NULL];
	
	[self updateEstimatedAppSize];
	
	[bundleIdentifierTextField setStringValue: [spec propertyForKey: @"Identifier"]];
}

/*************************************************
 - Make sure that all fields contain valid values
**************************************************/

- (BOOL)verifyFieldContents
{
	BOOL			isDir;

	//file manager
	NSFileManager *fileManager = [NSFileManager defaultManager];

	//script path
	if ([[appNameTextField stringValue] length] == 0)//make sure a name has been assigned
	{
		[STUtil sheetAlert:@"Invalid Application Name" subText: @"You must specify a name for your application" forWindow: window];
		return NO;
	}

	//script path
	if (([fileManager fileExistsAtPath: [scriptPathTextField stringValue] isDirectory: &isDir] == NO) || isDir)//make sure script exists and isn't a folder
	{
		[STUtil sheetAlert:@"Invalid Script Path" subText: @"No file exists at the script path you have specified" forWindow: window];
		return NO;
	}
	//interpreter
	if ([fileManager fileExistsAtPath: [interpreterTextField stringValue]] == NO)//make sure interpreter exists
	{
		if (NO == [STUtil proceedWarning: @"Invalid Interpreter" subText: @"The specified interpreter does not exist on this system.  Do you wish to proceed anyway?" withAction: @"Proceed"])
			return NO;
	}
	//make sure typeslist contains valid values
	if ([(TypesList *)[typesControl types] numTypes] <= 0 && [(SuffixList *)[typesControl suffixes] numSuffixes] <= 0 && [isDroppableCheckbox state] == YES)
	{
		[STUtil sheetAlert:@"Invalid Types List" subText: @"The app has been set to be droppable but no file types are set.  Please modify the Types list to correct this." forWindow: window];
		return NO;
	}
	//make sure the signature is 4 characters
	if ([[signatureTextField stringValue] length] != 4)
	{
		[STUtil sheetAlert:@"Invalid App Signature" subText: @"The signature set for the application is invalid.  An app's signature must consist of four upper and/or lowercase ASCII characters." forWindow: window];
		return NO;
	}
	
	//make sure we have an icon
	if ([iconControl getImage] == NULL || ([iconControl usesCustomIcnsFile] && ![fileManager fileExistsAtPath: [iconControl getIcnsFilePath]]))
	{
		[STUtil sheetAlert:@"Missing Icon" subText: @"You must set an icon for your application." forWindow: window];
		return NO;
	}
	
	// let's be certain that the bundled files list doesn't contain entries that have been moved
	if(![fileList allPathsAreValid])
	{
		[STUtil sheetAlert: @"Moved or missing files" subText:@"One or more of the files that are to be bundled with the application have been moved.  Please rectify this and try again." forWindow: window];
		return NO;
	}
	
	return YES;
}


/*****************************************
 - Called when script type radio button is pressed
*****************************************/

- (IBAction)scriptTypeSelected:(id)sender
{
	[self setScriptType: [[sender selectedItem] tag] ];
}

- (void)selectScriptTypeBasedOnInterpreter
{
	int i;
	
	for (i = 0; i < [defaultInterpreters count]; i++)
	{
		if ([[interpreterTextField stringValue] isEqualToString: [defaultInterpreters objectAtIndex: i]])
		{
			[scriptTypePopupMenu selectItemAtIndex: i];
			return;
		}
	}
	[scriptTypePopupMenu selectItemAtIndex: 8];
}

/*****************************************
 - Updates data in interpreter, icon and script type radio buttons
*****************************************/

- (void)setScriptType: (int)typeNum
{	
	// set the script type based on the number which identifies each type
	[interpreterTextField setStringValue: [defaultInterpreters objectAtIndex: typeNum ] ];
	[scriptTypePopupMenu selectItemWithTag: typeNum];
	[self controlTextDidChange: NULL];
}

/********************************************************************
 - Parse the Shebang line (#!) to get the interpreter for the script
**********************************************************************/

- (NSArray *)getInterpreterFromShebang: (NSString *)path
{
	NSString	*script, *firstLine, *shebang, *interpreterCmd;
	NSArray		*lines, *words;
	
	// get the first line of the script
	script = [NSString stringWithContentsOfFile: path encoding: DEFAULT_OUTPUT_TXT_ENCODING error: nil];
	lines = [script componentsSeparatedByString: @"\n"];
	firstLine = [lines objectAtIndex: 0];
	
	// if the first line of the script is shorter than 2 chars, it can't possibly be a shebang line
	if ([firstLine length] <= 2)
		return [NSArray arrayWithObject: @""];
	
	// get first two characters of first line
	shebang = [firstLine substringToIndex: 2]; // first two characters should be #!
	if (![shebang isEqualToString: @"#!"])
		return [NSArray arrayWithObject: @""];
	
	// get everything that follows after the #!
	// seperate it by whitespaces, in order not to get also the params to the interpreter
	interpreterCmd = [firstLine substringFromIndex: 2];
	words = [interpreterCmd componentsSeparatedByString: @" "];
	return ([[words retain] autorelease]);
}


/*****************************************
 - Open sheet to select script to load
*****************************************/

- (IBAction)selectScript:(id)sender
{
	//create open panel
    NSOpenPanel *oPanel = [NSOpenPanel openPanel];
	[oPanel setPrompt:@"Select"];
    [oPanel setAllowsMultipleSelection:NO];
	[oPanel setCanChooseDirectories: NO];
	
	[window setTitle: [NSString stringWithFormat: @"%@ - Select script", PROGRAM_NAME]];
	
	//run open panel
    [oPanel beginSheetForDirectory:nil file:nil types:nil modalForWindow: window modalDelegate: self didEndSelector: @selector(selectScriptPanelDidEnd:returnCode:contextInfo:) contextInfo: nil];
		
}

- (void)selectScriptPanelDidEnd:(NSOpenPanel *)oPanel returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	if (returnCode == NSOKButton)
		[self loadScript: [oPanel filename]];
	[window setTitle: PROGRAM_NAME];
}

/*****************************************
 - Loads script data into platypus window
*****************************************/

- (void)loadScript:(NSString *)filename
{
	NSMutableArray		*shebangInterpreter;
	NSString			*interpreter;
	int					i;
	BOOL				isDir;

	//make sure file we're loading actually exists
	if (![[NSFileManager defaultManager] fileExistsAtPath: filename isDirectory: &isDir] || isDir)
		return;

	//set script path
	[scriptPathTextField setStringValue: filename];

	//set app name
	NSString *appName = [STUtil cutSuffix: [filename lastPathComponent]];
	[appNameTextField setStringValue: appName];
	
	//read shebang line
	shebangInterpreter = [NSMutableArray arrayWithArray: [self getInterpreterFromShebang: filename]];
	interpreter = [shebangInterpreter objectAtIndex: 0];
	
	//if no interpreter can be retrieved from shebang line
	if ([interpreter caseInsensitiveCompare: @""] == NSOrderedSame)
	{
		//try to determine type from suffix
		[self setScriptType: (int)[self getFileTypeFromSuffix: filename] ];
	}
	else
	{
		//see if interpreter matches a preset
		for (i = 0; i < [defaultInterpreters count]; i++)
		{
			if ([interpreter caseInsensitiveCompare: [defaultInterpreters objectAtIndex: i]] == NSOrderedSame)
				[self setScriptType: i];
		}
		
		//set shebang interpreter into interpreter field
		[interpreterTextField setStringValue: interpreter];
		[shebangInterpreter removeObjectAtIndex: 0];
		[paramsControl set: [NSArray arrayWithArray: shebangInterpreter]];
	}
	
	[self controlTextDidChange: NULL];
	[self updateEstimatedAppSize];
	
	//add to open recent menu
	if (![recentItems containsObject: filename])
	{
		if ([recentItems count] == 8)
			[recentItems removeObjectAtIndex: 0];

		[recentItems addObject: filename];
		[self constructOpenRecentMenu: NULL];
	}
}

/*****************************************
 - Toggles between advanced options mode
*****************************************/

- (IBAction)toggleAdvancedOptions:(id)sender
{
	NSRect winRect = [window frame];

	if ([showAdvancedArrow state] == NSOffState)
	{
		winRect.origin.y += kWindowExpansionHeight;
		winRect.size.height -= kWindowExpansionHeight;
		[showOptionsTextField setStringValue: @"Show Advanced Options"];
		[toggleAdvancedMenuItem setTitle: @"Show Advanced Options"];
		[signatureTextField setEditable: NO];
		[bundleIdentifierTextField setEditable: NO];
		[authorTextField setEditable: NO];
		[versionTextField setEditable: NO];
		
		[window setFrame: winRect display:TRUE animate: TRUE];
		
	}
	else if ([showAdvancedArrow state] == NSOnState)
	{
		winRect.origin.y -= kWindowExpansionHeight;
		winRect.size.height += kWindowExpansionHeight;
		[showOptionsTextField setStringValue: @"Hide Advanced Options"];
		[toggleAdvancedMenuItem setTitle: @"Hide Advanced Options"];
		[signatureTextField setEditable: YES];
		[bundleIdentifierTextField setEditable: YES];
		[authorTextField setEditable: YES];
		[versionTextField setEditable: YES];
		
		[window setFrame: winRect display:TRUE animate: TRUE];
	}
}

/*****************************************
 - called when [X] Is Droppable is pressed
*****************************************/

- (IBAction)isDroppableWasClicked:(id)sender
{
	[editTypesButton setHidden: ![isDroppableCheckbox state]];
	[editTypesButton setEnabled: [isDroppableCheckbox state]];
}

/*****************************************
 - called when [X] Is Droppable is pressed
*****************************************/

- (IBAction)outputTypeWasChanged:(id)sender
{
	// we don't show text output settings for output modes None and Web View
	if (![[outputTypePopupMenu titleOfSelectedItem] isEqualToString: @"None"] &&
		![[outputTypePopupMenu titleOfSelectedItem] isEqualToString: @"Web View"] && 
		![[outputTypePopupMenu titleOfSelectedItem] isEqualToString: @"Droplet"])
	{
		[textOutputSettingsButton setHidden: NO];
		[textOutputSettingsButton setEnabled: YES];
	}
	else
	{
		[textOutputSettingsButton setHidden: YES];
		[textOutputSettingsButton setEnabled: NO];
	}
	
	// disable options that don't make sense for status menu output mode
	if ([[outputTypePopupMenu titleOfSelectedItem] isEqualToString: @"Status Menu"])
	{
		// disable droppable & admin privileges
		[isDroppableCheckbox setIntValue: 0];
		[isDroppableCheckbox setEnabled: NO];
		[self isDroppableWasClicked: self];
		[rootPrivilegesCheckbox setIntValue: 0];
		[rootPrivilegesCheckbox setEnabled: NO];
		
		// force-enable "Remain running"
		[remainRunningCheckbox setIntValue: 1];
		[remainRunningCheckbox setEnabled: NO];
		
		// check Runs in Background as default for Status Menu output
		[showInDockCheckbox setIntValue: 1];
		
		// show button for special status item settings
		[statusItemSettingsButton setEnabled: YES];
		[statusItemSettingsButton setHidden: NO];
	}
	else
	{
		if ([[outputTypePopupMenu titleOfSelectedItem] isEqualToString: @"Droplet"])
		{
			[isDroppableCheckbox setIntValue: 1];
			[self isDroppableWasClicked: self];
		}
		
		// re-enable droppable
		[isDroppableCheckbox setEnabled: YES];
		[rootPrivilegesCheckbox setEnabled: YES];
		
		// re-enable remain running
		[remainRunningCheckbox setEnabled: YES];

		[showInDockCheckbox setIntValue: 0];
		
		// hide special status item settings
		[statusItemSettingsButton setEnabled: NO];
		[statusItemSettingsButton setHidden: YES];
	}
}

/*****************************************
 - called when (Clear) button is pressed 
 -- restores fields to startup values
*****************************************/

- (IBAction)clearAllFields:(id)sender
{
	//clear all text field to start value
	[appNameTextField setStringValue: @""];
	[scriptPathTextField setStringValue: @""];
	[versionTextField setStringValue: @"1.0"];
	[signatureTextField setStringValue: @"????"];
	
	[bundleIdentifierTextField setStringValue: [[NSUserDefaults standardUserDefaults] objectForKey:@"DefaultBundleIdentifierPrefix"]];
	[authorTextField setStringValue: [[NSUserDefaults standardUserDefaults] objectForKey:@"DefaultAuthor"]];
	
	//uncheck all options
	[isDroppableCheckbox setIntValue: 0];
	[self isDroppableWasClicked: isDroppableCheckbox];
	[encryptCheckbox setIntValue: 0];
	[rootPrivilegesCheckbox setIntValue: 0];
	[remainRunningCheckbox setIntValue: 1];
	[showInDockCheckbox setIntValue: 0];
	
	//clear file list
	[fileList clearFileList: self];
	
	//clear suffix and types lists to default values
	[typesControl setDefaultTypes: self];
	
	//set environment variables list to default
	[envControl resetDefaults: self];
	
	//set parameters to default
	[paramsControl resetDefaults: self];
	
	//set text ouput settings to default
	[textSettingsControl resetDefaults: self];
	
	//set status item settings to default
	[statusItemSettingsControl restoreDefaults: self];
	
	//set script type
	[self setScriptType: 0];
	
	//set output type
	[outputTypePopupMenu selectItemWithTitle: DEFAULT_OUTPUT_TYPE];
	[self outputTypeWasChanged: outputTypePopupMenu];
	
	//update button status
	[self controlTextDidChange: NULL];
	
	[appSizeTextField setStringValue: @""];
	
	[iconControl setDefaultIcon];
}

/*****************************************
 - Generate the shell command for the platypus
   command line tool based on the settings 
   provided in the graphical interface
*****************************************/

- (NSString *)commandLineStringFromSettings
{
	int i;
	NSString *checkboxParamStr = @"";
	NSString *iconParamStr = @"", *signatureString = @"", *versionString = @"", *authorString = @"";
	NSString *suffixesString = @"", *filetypesString = @"", *parametersString = @"", *environmentString = @"";
	NSString *textEncodingString = @"", *textOutputString = @"", *statusMenuOptionsString = @""; 
	
	// checkbox parameters
	if ([paramsControl passAppPathAsFirstArg])
		checkboxParamStr = [checkboxParamStr stringByAppendingString: @"F"];
	if ([rootPrivilegesCheckbox intValue])
		checkboxParamStr = [checkboxParamStr stringByAppendingString: @"A"];
	if ([encryptCheckbox intValue])
		checkboxParamStr = [checkboxParamStr stringByAppendingString: @"S"];
	if ([isDroppableCheckbox intValue])
		checkboxParamStr = [checkboxParamStr stringByAppendingString: @"D"];
	if ([showInDockCheckbox intValue])
		checkboxParamStr = [checkboxParamStr stringByAppendingString: @"B"];
	if ([remainRunningCheckbox intValue])
		checkboxParamStr = [checkboxParamStr stringByAppendingString: @"R"];
	
	if ([checkboxParamStr length] != 0)
		checkboxParamStr = [NSString stringWithFormat: @"-%@ ", checkboxParamStr];
	
	if (![[versionTextField stringValue] isEqualToString: @"1.0"])
		versionString = [NSString stringWithFormat:@" -V '%@' ", [versionTextField stringValue]];
	
	if (![[authorTextField stringValue] isEqualToString: NSFullUserName()])
		authorString = [NSString stringWithFormat: @" -u '%@' ", [authorTextField stringValue]];
	
	// if it's droppable, we need the Types and Suffixes
	if ([isDroppableCheckbox intValue])
	{
		//create suffixes param
		suffixesString = [[[typesControl suffixes] getSuffixArray] componentsJoinedByString:@"|"];
		suffixesString = [NSString stringWithFormat: @"-X '%@' ", suffixesString];
		
		//create filetype codes param
		filetypesString = [[(TypesList *)[typesControl types] getTypesArray] componentsJoinedByString:@"|"];
		filetypesString = [NSString stringWithFormat: @"-T '%@' ", filetypesString];
	}
	
	//create bundled files string
	NSString *bundledFilesCmdString = @"";
	NSArray *bundledFiles = [fileList getFilesArray];
	for (i = 0; i < [bundledFiles count]; i++)
	{
		bundledFilesCmdString = [bundledFilesCmdString stringByAppendingString: [NSString stringWithFormat: @"-f '%@' ", [bundledFiles objectAtIndex: i]]];
	}
	
	// create interpreter params arg
	if ([[paramsControl paramsArray] count])
	{
		parametersString = [[paramsControl paramsArray] componentsJoinedByString:@"|"];
		parametersString = [NSString stringWithFormat: @"-G '%@' ", parametersString];
	}
	
	// create environmental settings args
	for (i = 0; i < [[[envControl environmentDictionary] allKeys] count]; i++)
	{
		if (i == 0) { environmentString = [environmentString stringByAppendingString: @" -N '"]; }

		NSString *theKey = [[[envControl environmentDictionary] allKeys] objectAtIndex: i];
		environmentString = [environmentString stringByAppendingString: 
							[NSString stringWithFormat: @"%@=%@|", theKey, [[envControl environmentDictionary] objectForKey: theKey]]
							];
		
		if (i == [[[envControl environmentDictionary] allKeys] count]-1) 
		{ 
			environmentString = [environmentString substringToIndex: [environmentString length]-1];
			environmentString = [environmentString stringByAppendingString: @"' "]; 
		}
	}
	
	//  create args for text settings if progress bar/text window or status menu
	if (([[outputTypePopupMenu titleOfSelectedItem] isEqualToString: @"Text Window"] || 
		[[outputTypePopupMenu titleOfSelectedItem] isEqualToString: @"Progress Bar"] ||
		[[outputTypePopupMenu titleOfSelectedItem] isEqualToString: @"Status Menu"]))
	{
		NSString *textFgString = @"", *textBgString = @"", *textFontString = @""; 
		if (![[textSettingsControl getTextForeground] isEqual: [NSColor blackColor]])
			textFgString = [NSString stringWithFormat: @" -g '%@' ", [STUtil hexFromColor: [textSettingsControl getTextForeground]]];
	
		if (![[textSettingsControl getTextBackground] isEqual: [NSColor whiteColor]])
			textBgString = [NSString stringWithFormat: @" -b '%@' ", [STUtil hexFromColor: [textSettingsControl getTextBackground]]];
	
		if (![[textSettingsControl getTextFont] isEqual: [NSFont fontWithName: DEFAULT_OUTPUT_FONT size: DEFAULT_OUTPUT_FONTSIZE]])
			textFontString = [NSString stringWithFormat: @" -n '%@ %2.f' ", [[textSettingsControl getTextFont] fontName], [[textSettingsControl getTextFont] pointSize]];
		
		textOutputString = [NSString stringWithFormat: @"%@%@%@", textFgString, textBgString, textFontString];
	}
	
	//	text encoding	
	if ([textSettingsControl getTextEncoding] != DEFAULT_OUTPUT_TXT_ENCODING)
		textEncodingString = [NSString stringWithFormat: @" -E %d ", [textSettingsControl getTextEncoding]];
			
	//create custom icon string
	if ([iconControl usesCustomIcnsFile])
		iconParamStr = [NSString stringWithFormat: @" -i '%@' ", [iconControl getIcnsFilePath]];
	
	//only have signature param if it's not the usual '????'
	if (![[signatureTextField stringValue] isEqualToString: @"????"])
		signatureString = [NSString stringWithFormat: @"-s '%@' ", [signatureTextField stringValue]];
	
	//status menu settings, if output mode is status menu
	if ([[outputTypePopupMenu titleOfSelectedItem] isEqualToString: @"Status Menu"])
	{
		// -K kind
		statusMenuOptionsString = [statusMenuOptionsString stringByAppendingString: [NSString stringWithFormat: @"-K '%@' ", [statusItemSettingsControl displayType]]];
		
		// -L /path/to/image
		if (![[statusItemSettingsControl displayType] isEqualToString: @"Text"])
			statusMenuOptionsString = [statusMenuOptionsString stringByAppendingString: @"-L '/path/to/image' "];
		
		// -Y 'Title'
		if (![[statusItemSettingsControl displayType] isEqualToString: @"Icon"])
			statusMenuOptionsString = [statusMenuOptionsString stringByAppendingString: [NSString stringWithFormat: @"-Y '%@' ", [statusItemSettingsControl title]]];
	}	
	
	// finally, generate the command
	NSString *commandStr = [NSString stringWithFormat: 
	@"%@ %@%@-a '%@' -o '%@' -p '%@'%@%@ %@-I '%@' %@%@%@%@%@%@%@%@ -c '%@' 'MyApp.app'",
	CMDLINE_TOOL_PATH,
	checkboxParamStr,
	iconParamStr,
	[appNameTextField stringValue],
	[outputTypePopupMenu titleOfSelectedItem],
	[interpreterTextField stringValue],
	authorString,
	versionString,
	signatureString, 
	[bundleIdentifierTextField stringValue],
	suffixesString,
	filetypesString,
	bundledFilesCmdString,
	parametersString,
	environmentString,
	textEncodingString,
	textOutputString,
	statusMenuOptionsString,
	[scriptPathTextField stringValue],
	nil];

	return commandStr;
}

/*****************************************
 - Generate shell command and display in text field
*****************************************/

- (IBAction)showCommandLineString: (id)sender
{
	[commandTextField setFont:[NSFont userFixedPitchFontOfSize: 10.0]];
	[commandTextField setString: [self commandLineStringFromSettings]];
	[prefsControl updateCLTStatus: commandCLTStatusTextField];

	[window setTitle: [NSString stringWithFormat: @"%@ - Shell Command String", PROGRAM_NAME]];
	[NSApp beginSheet:	commandWindow
						modalForWindow: window 
						modalDelegate:nil
						didEndSelector:nil
						contextInfo:nil];
	[NSApp runModalForWindow: commandWindow];
	[NSApp endSheet:commandWindow];
    [commandWindow orderOut:self];
}

- (IBAction)closeCommandWindow: (id)sender
{
	[window setTitle: PROGRAM_NAME];
	[NSApp stopModal];
}


#pragma mark -

/*****************************************
 - Determine script type based on a file's suffix
*****************************************/

- (int)getFileTypeFromSuffix: (NSString *)fileName
{
	if ([fileName hasSuffix: @".sh"])
		return 0;
	else if ([fileName hasSuffix: @".pl"] || [fileName hasSuffix: @".perl"])
		return 1;
	else if ([fileName hasSuffix: @".py"])
		return 2;
	else if ([fileName hasSuffix: @".rb"] || [fileName hasSuffix: @".rbx"])
		return 3;
	else if ([fileName hasSuffix: @".scpt"] || [fileName hasSuffix: @".applescript"])
		return 4;
	else if ([fileName hasSuffix: @".tcl"])
		return 5;
	else if ([fileName hasSuffix: @".exp"] || [fileName hasSuffix: @".expect"])
		return 6;
	else if ([fileName hasSuffix: @".php"])
		return 7;
	else
		return 0;
}

/*****************************************
 - //return the bundle identifier for the application to be generated
 -  based on username etc.
*****************************************/

- (NSString *)generateBundleIdentifier
{
	NSString	*bundleId;
	//The format is "org.username.appname"
	bundleId = [NSString stringWithFormat: @"%@%@", [[NSUserDefaults standardUserDefaults] stringForKey:@"DefaultBundleIdentifierPrefix"], [appNameTextField stringValue]];
	bundleId = [[bundleId componentsSeparatedByString:@" "] componentsJoinedByString:@""];//no spaces
	return(bundleId);
}

#pragma mark -

/*****************************************
 - // set app size textfield to formatted str with app size
*****************************************/

- (void)updateEstimatedAppSize
{
	[appSizeTextField setStringValue: [NSString stringWithFormat: @"Estimated final app size: ~%@", [self estimatedAppSize]]];
}

/*****************************************
 - // Make a decent guess concerning final app size
*****************************************/

- (NSString *)estimatedAppSize
{
	UInt64 estimatedAppSize = 0;
	
	estimatedAppSize += 4096; // Info.plist
	estimatedAppSize += 4096; // InfoPlist.strings
	estimatedAppSize += 4096; // AppSettings.plist
	
	// if we want to know the size of the icon, let's assume default icon
	NSString *iconPath = [[NSBundle mainBundle] pathForResource: @"PlatypusDefault.icns" ofType: NULL];
	if ([iconControl usesCustomIcnsFile] || ![iconControl hasCustomIcon])
		iconPath = [iconControl getIcnsFilePath];

	estimatedAppSize += [STUtil fileOrFolderSize: iconPath]; // AppIcon.icns
	estimatedAppSize += [STUtil fileOrFolderSize: [scriptPathTextField stringValue]];
	estimatedAppSize += [STUtil fileOrFolderSize: [[NSBundle mainBundle] pathForResource: @"ScriptExec" ofType: NULL]];  // executable binary
	estimatedAppSize += [STUtil fileOrFolderSize: [[NSBundle mainBundle] pathForResource: @"MainMenu.nib" ofType: NULL]];  // bundled nib
	estimatedAppSize += [fileList getTotalSize];
		
	return [STUtil sizeAsHumanReadable: estimatedAppSize];
}

#pragma mark -

/*****************************************
 - Dragging and dropping for Platypus window
*****************************************/

- (BOOL)performDragOperation:(id <NSDraggingInfo>)sender
{
	NSPasteboard	*pboard = [sender draggingPasteboard];
	NSString		*filename;
	BOOL			isDir = FALSE;

    if ( [[pboard types] containsObject:NSFilenamesPboardType] ) 
	{
        NSArray *files = [pboard propertyListForType:NSFilenamesPboardType];
		filename = [files objectAtIndex: 0];//we only load the first dragged item
		if ([[NSFileManager defaultManager] fileExistsAtPath: filename isDirectory: &isDir] && !isDir)
		{
			if ([filename hasSuffix: PROFILES_SUFFIX])
				[profilesControl loadProfileFile: filename];
			else
				[self loadScript: filename];
	
			return YES;
		}
	}
	return NO;
}

- (NSDragOperation)draggingEntered:(id <NSDraggingInfo>)sender 
{
	// we accept dragged files
    if ( [[[sender draggingPasteboard] types] containsObject:NSFilenamesPboardType] ) 
		return NSDragOperationLink;

    return NSDragOperationNone;
}

#pragma mark -

/*****************************************
 - Delegate for when text changes in any of the Platypus text fields
*****************************************/

- (void)controlTextDidChange:(NSNotification *)aNotification
{
	BOOL	isDir, exists = NO, validName = NO;
	
	//app name or script path was changed
	if ([aNotification object] == NULL || [aNotification object] == appNameTextField || [aNotification object] == scriptPathTextField)
	{
		if ([[appNameTextField stringValue] length] > 0)
			validName = YES;
				
		if (exists = [scriptPathTextField hasValidPath])
		{
			// add watcher that tracks whether it exists
			[[UKKQueue sharedFileWatcher] addPathToQueue:  [scriptPathTextField stringValue]];
			exists = YES;
		}

		[editScriptButton setEnabled: exists];
		[revealScriptButton setEnabled: exists];
		
		//enable/disable create app button
		[createAppButton setEnabled: validName && exists];
		
		//update identifier
		if (validName)
			[bundleIdentifierTextField setStringValue: [self generateBundleIdentifier]];
		else
			[bundleIdentifierTextField setStringValue: [[NSUserDefaults standardUserDefaults] objectForKey:@"DefaultBundleIdentifierPrefix"]];
	}
	
	//bundle signature or "type code" changed
	if ([aNotification object] == signatureTextField || [aNotification object] == NULL)
	{
		NSRange	 range = { 0, 4 };
		NSString *sig = [[aNotification object] stringValue];
		
		if ([sig length] > 4)
		{
			[[aNotification object] setStringValue: [sig substringWithRange: range]];
		}
		else if ([sig length] < 4)
			[[aNotification object] setTextColor: [NSColor redColor]];
		else if ([sig length] == 4)
			[[aNotification object] setTextColor: [NSColor blackColor]];
	}
	
	//interpreter changed
	if ([aNotification object] == interpreterTextField || [aNotification object] == NULL)
	{
		[self selectScriptTypeBasedOnInterpreter];
		if ([[NSFileManager defaultManager] fileExistsAtPath: [interpreterTextField stringValue] isDirectory:&isDir] && !isDir)
			[interpreterTextField setTextColor: [NSColor blackColor]];
		else
			[interpreterTextField setTextColor: [NSColor redColor]];
	}
}

/*****************************************
 - Delegate for enabling and disabling menu items
*****************************************/
- (BOOL)validateMenuItem:(NSMenuItem*)anItem 
{
	BOOL isDir;

	//edit script
	if ([[anItem title] isEqualToString:@"Edit Script"] && (![[NSFileManager defaultManager] fileExistsAtPath: [scriptPathTextField stringValue] isDirectory:&isDir] || isDir))
		return NO;
		
	//reveal script
	if ([[anItem title] isEqualToString:@"Reveal Script"] && (![[NSFileManager defaultManager] fileExistsAtPath: [scriptPathTextField stringValue] isDirectory:&isDir] || isDir))
		return NO;
		
	//run script
	if ([[anItem title] isEqualToString:@"Run Script in Terminal"] && (![[NSFileManager defaultManager] fileExistsAtPath: [scriptPathTextField stringValue] isDirectory:&isDir] || isDir))
		return NO;
	
	//check script syntax
	if ([[anItem title] isEqualToString:@"Check Script Syntax"] && (![[NSFileManager defaultManager] fileExistsAtPath: [scriptPathTextField stringValue] isDirectory:&isDir] || isDir))
		return NO;
	
	//create app menu
	if ([[anItem title] isEqualToString:@"Create App"] && ![createAppButton isEnabled])
		return NO;
	
	return YES;
}

/*****************************************
 - Load the script of item selected in Open Recent Menu
*****************************************/
-(void) openRecentMenuItemSelected: (id)sender
{
	BOOL			isDir = NO;
	NSString		*filePath = [sender title];

	// reveal if Cmd key is down
	if(GetCurrentKeyModifiers() & cmdKey)
	{
		[[NSWorkspace sharedWorkspace] selectFile: filePath inFileViewerRootedAtPath:nil];
		return;
	}
	
	// otherwise load item
	if ([[NSFileManager defaultManager] fileExistsAtPath: filePath isDirectory: &isDir] && isDir == NO)
		[self loadScript: filePath];
	else
		[STUtil alert:@"Invalid item" subText: @"The file you selected no longer exists at the specified path."];
}

/*****************************************
 - Generate the Open Recent Menu
*****************************************/
- (void)menuWillOpen:(NSMenu *)menu
{
	[self constructOpenRecentMenu: NULL];
}

-(IBAction) constructOpenRecentMenu: (id)sender
{
	int i;
	
	//clear out all menu itesm
	while ([openRecentMenu numberOfItems])
		[openRecentMenu removeItemAtIndex: 0];

	if ([recentItems count] > 0)
	{
		//populate with contents of array
		for (i = [recentItems count]-1; i >= 0 ; i--)
		{
			NSString *filePath = [recentItems objectAtIndex: i];
			NSMenuItem *menuItem = [openRecentMenu addItemWithTitle: filePath action: @selector(openRecentMenuItemSelected:) keyEquivalent:@""];
			NSImage *icon = [[NSWorkspace sharedWorkspace] iconForFile: filePath];
			[icon setSize: NSMakeSize(16,16)];
			[menuItem setImage: icon];
			
			// if file no longer exists at path, color it red
			if (![[NSFileManager defaultManager] fileExistsAtPath: filePath])
			{
				NSDictionary *attr = [NSDictionary dictionaryWithObjectsAndKeys: [NSColor redColor], NSForegroundColorAttributeName, [NSFont systemFontOfSize: 14.0] , NSFontAttributeName, NULL];
				NSAttributedString *attStr = [[[NSAttributedString alloc] initWithString: filePath attributes: attr] autorelease];
				[menuItem setAttributedTitle: attStr];
		
			}
		}
		
		//add seperator and clear menu
		[openRecentMenu addItem: [NSMenuItem separatorItem]];
		[openRecentMenu addItemWithTitle: @"Clear Menu" action: @selector(clearRecentItems) keyEquivalent:@""];
		
	}
	else
	{
		[openRecentMenu addItemWithTitle: @"Empty" action: NULL keyEquivalent:@""];
		[[openRecentMenu itemAtIndex: 0] setEnabled: NO];
	}
}

/*****************************************
 - Clear the Recent Items menu
*****************************************/
-(void) clearRecentItems
{
	[recentItems removeAllObjects];
	[self constructOpenRecentMenu: NULL];
}

#pragma mark -

/*****************************************
 - Built-In script editor and associated functions
*****************************************/

- (void)openScriptInBuiltInEditor: (NSString *)path
{
	[editorText setFont:[NSFont userFixedPitchFontOfSize: 10.0]]; //set monospace font
	
	//update text notifying user of the path to the script he is editing
	[editorScriptPath setStringValue: [NSString stringWithFormat: @"Editing %@", path]];
	[editorText setString: [NSString stringWithContentsOfFile: path]];

	[window setTitle: [NSString stringWithFormat: @"%@ Built-In Script Editor", PROGRAM_NAME]];
	[NSApp beginSheet:	editorWindow
						modalForWindow: window 
						modalDelegate:nil
						didEndSelector:nil
						contextInfo:nil];
	[NSApp runModalForWindow: editorWindow];
	[NSApp endSheet: editorWindow];
    [editorWindow orderOut:self];	
}

- (IBAction)editorSave: (id)sender
{
	//see if we can write it
	if (![[NSFileManager defaultManager] isWritableFileAtPath: [scriptPathTextField stringValue]])
        [STUtil alert: @"Unable to save changes" subText: @"You don't the neccesary privileges to save this text file."];
	else //save it
		[[editorText string] writeToFile: [scriptPathTextField stringValue] atomically: YES];

	[window setTitle: PROGRAM_NAME];
	[NSApp stopModal];
}

- (IBAction)editorCancel: (id)sender
{
	[window setTitle: PROGRAM_NAME];
	[NSApp stopModal];
}

- (IBAction)editorCheckSyntax: (id)sender
{
	[[editorText string] writeToFile: [scriptPathTextField stringValue] atomically: YES];
	[self checkSyntaxOfScript: self];
}

#pragma mark -

/*****************************************
 - Open Platypus Help HTML file within app bundle
*****************************************/
- (IBAction) showHelp:(id)sender
{
	[[NSWorkspace sharedWorkspace] openURL:
	 [NSURL fileURLWithPath: [[NSBundle mainBundle] pathForResource: PROGRAM_DOCUMENTATION ofType:nil]]
	];
}

/*****************************************
 - Open 'platypus' command line tool man page in PDF
*****************************************/
- (IBAction) showManPage:(id)sender
{	
	[[NSWorkspace sharedWorkspace] openFile: [[NSBundle mainBundle] pathForResource: PROGRAM_MANPAGE ofType:nil]];
}

/*****************************************
 - Open Readme file
*****************************************/
- (IBAction) showReadme:(id)sender
{
	[[NSWorkspace sharedWorkspace] openURL: 
	[NSURL fileURLWithPath: [[NSBundle mainBundle] pathForResource: PROGRAM_README_FILE ofType:nil]]
	];
}

/*****************************************
 - Open Platypus website in default browser
*****************************************/
- (IBAction) openWebsite: (id)sender
{
	[[NSWorkspace sharedWorkspace] openURL: [NSURL URLWithString: PROGRAM_WEBSITE]];
}

/*****************************************
 - Open License.txt file
 *****************************************/
- (IBAction) openLicense: (id)sender
{
	[[NSWorkspace sharedWorkspace] openURL: 
	 [NSURL fileURLWithPath: [[NSBundle mainBundle] pathForResource: PROGRAM_LICENSE_FILE ofType:nil]]
	 ];
}

/*****************************************
- Open donations website
*****************************************/
- (IBAction) openDonations: (id)sender
{
	[[NSWorkspace sharedWorkspace] openURL: [NSURL URLWithString: PROGRAM_DONATIONS]];
}


@end
