/*******************************************************************************
 * The information contained in this file is confidential and proprietary to
 * QLogic Corporation.  No part of this file may be reproduced or
 * distributed, in any form or by any means for any purpose, without the
 * express written permission of QLogic Corporation.
 *
 * (c) COPYRIGHT 2015 QLogic Corporation, ALL RIGHTS RESERVED.
 *******************************************************************************/
/* **********************************************************
 * Copyright 2015 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/


#include "qfle3.h"
#include "qfle3_sm.h"

extern vmk_uint32 enable_fcoe_queue;
extern vmk_uint32 netpoll_count;
int
qfle3_remove_queue_filter(qfle3_adapter * adapter,
			  vmk_UplinkQueueID qid,
			  vmk_UplinkQueueFilterID fid);

static inline vmk_uint8 qfle3_fp_qzone_id(struct qfle3_fastpath *fp)
{
   return fp->cl_id;
}

static inline vmk_uint8 qfle3_fp_igu_sb_id(struct qfle3_fastpath *fp)
{
   return fp->adapter->igu_base_sb + fp->qid  + CNIC_SUPPORT(fp->adapter);
   
}

static inline vmk_uint8 qfle3_fp_fw_sb_id(struct qfle3_fastpath *fp)
{
   return fp->adapter->base_fw_ndsb + fp->qid + CNIC_SUPPORT(fp->adapter);

}

static vmk_uint8 qfle3_fp_cl_id(struct qfle3_fastpath *fp)
{
   return qfle3_fp_igu_sb_id(fp);
   
//   return fp->adapter->igu_base_sb + fp->qid;
}

vmk_uint32 qfle3_rx_ustorm_prods_offset(struct qfle3_fastpath *fp)
{
   struct qfle3_adapter *adapter = fp->adapter;
   vmk_uint32 offset = BAR_USTRORM_INTMEM;

   offset += USTORM_RX_PRODS_E2_OFFSET(fp->cl_qzone_id);

   QFLE3_DBG(QFLE3_DBG_START," rx prods offset %d\n", USTORM_RX_PRODS_E2_OFFSET(fp->cl_qzone_id));

   return offset;
}

VMK_ReturnStatus
qfle3_init_eth_fp(qfle3_adapter * adapter,
		  int req_type, struct qfle3_fastpath *fp)
{
   unsigned long q_type = 0;
   vmk_uint32 cids[QFLE3_MULTI_TX_COS] = {0,0,0};
   vmk_uint8 cos;
   
   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_init_eth_fp for fp %d\n", fp->qid);
   //fp->rx_queue = fp->qid;
   fp->cid = fp->qid;
   fp->cl_id = qfle3_fp_cl_id(fp);
   fp->fw_sb_id = qfle3_fp_fw_sb_id(fp);
   fp->igu_sb_id = qfle3_fp_igu_sb_id(fp);
   /* qZone id equals to FW (per path) client id */
   fp->cl_qzone_id  = qfle3_fp_qzone_id(fp);

   /* init shortcut */
   fp->ustorm_rx_prods_offset = qfle3_rx_ustorm_prods_offset(fp);

   /* Setup SB indices */
   fp->rx_cq_cons_sb = QFLE3_RX_SB_INDEX;

   fp->rx_bd_cons = 0;
   fp->rx_cq_cons = 0;

   /* Configure Queue State object */
   SET_BIT(ECORE_Q_TYPE_HAS_RX, &q_type);
   SET_BIT(ECORE_Q_TYPE_HAS_TX, &q_type);

//   fp->max_cos = 1;
//   /* init tx data */

//   //fp->tx_cid = cid;
//   //txdata->txq_index = txq_index;
//   fp->tx_cons_sb = QFLE3_TX_SB_INDEX_BASE;
//   fp->tx_ring_size = TX_BD_USABLE;
//   cids = fp->cid;
//   QFLE3_DBG(QFLE3_DBG_START, "created tx data cid %d, txq %d cl_id %d, fw sb id %d, igu sb id %d\n",
//         cids, fp->qid,fp->cl_id, fp->fw_sb_id, fp->igu_sb_id);
   for_each_cos_in_tx_queue(fp, cos) {
      qfle3_init_txdata(adapter, fp->txdata_ptr[cos],
                        CID_COS_TO_TX_ONLY_CID(fp->cid, cos, adapter),
                        FP_COS_TO_TXQ(fp, cos, adapter),
                        QFLE3_TX_SB_INDEX_BASE + cos, fp);
      cids[cos] = fp->txdata_ptr[cos]->cid;

   QFLE3_DBG(QFLE3_DBG_START, "created tx data cid %d, txq %d cl_id %d, fw sb id %d, igu sb id %d\n", 
            cids[cos], fp->qid,fp->cl_id, fp->fw_sb_id, fp->igu_sb_id);
   }

   //qfle3_init_sb(adapter, fp->status_block_ioa, QFLE3_VF_ID_INVALID, VMK_FALSE,
   qfle3_init_sb(adapter, fp->status_block_ioa, 0xFF, VMK_FALSE,
		 fp->fw_sb_id, fp->igu_sb_id);
   qfle3_update_fpsb_idx(fp);
   
   ecore_init_queue_obj(adapter, &QFLE3_SP_OBJ(adapter, fp).q_obj, fp->cl_id, cids,
			fp->max_cos, QFLE3_FUNC(adapter), QFLE3_SP(adapter, q_rdata),
			QFLE3_SP_MAPPING(adapter, q_rdata),q_type);

   /**
    * Configure classification DBs: Always enable Tx switching
    */
   //ecore_init_vlan_mac_fp_objs(fp, ECORE_OBJ_TYPE_RX_TX);

   /* Configure classification DBs */
   ecore_init_mac_obj(adapter, &QFLE3_SP_OBJ(adapter, fp).mac_obj, fp->cl_id,
		      fp->cid, QFLE3_FUNC(adapter), QFLE3_SP(adapter, mac_rdata),
		      QFLE3_SP_MAPPING(adapter, mac_rdata),
		      ECORE_FILTER_MAC_PENDING,
		      &adapter->sp_state, ECORE_OBJ_TYPE_RX_TX,
		      &adapter->macs_pool);
   if (QFLE3_SP_OBJ(adapter, fp).mac_obj.exe_queue.lock == NULL)
      return VMK_FAILURE;

   if (adapter->vxlan_filters_en) {
      ecore_init_vxlan_fltr_obj(adapter,
                                &QFLE3_SP_OBJ(adapter, fp).vxlan_filter_obj,
                                fp->cl_id, fp->cid, QFLE3_FUNC(adapter),
                                QFLE3_SP(adapter, mac_rdata),
                                QFLE3_SP_MAPPING(adapter, mac_rdata),
                                ECORE_FILTER_VXLAN_PENDING,
                                &adapter->sp_state, ECORE_OBJ_TYPE_RX,
                                &adapter->macs_pool,
                                &adapter->vlans_pool);
      if (QFLE3_SP_OBJ(adapter, fp).vxlan_filter_obj.exe_queue.lock == NULL)
         return VMK_FAILURE;
   }


   QFLE3_DBG(QFLE3_DBG_START,
	     "queue[%d]:  qfle3_init_sb(%p,%p)  cl_id %d  fw_sb %d  igu_sb %d\n",
	     fp->qid, adapter, fp->status_block.e2_sb, fp->cl_id, fp->fw_sb_id,
	     fp->igu_sb_id);
   return VMK_OK;
}

static int
qfle3_config_rss(struct qfle3_adapter *adapter,
		 const vmk_uint8 *raw_ind_tbl, const vmk_uint32 *rss_key_tbl)
{
   struct ecore_config_rss_params params = {NULL};
   int i, rc;
#if (ESX_DDK_VERSION >= 2017)
   vmk_uint32 j;
#endif

   if (raw_ind_tbl) {
      struct ecore_rss_config_obj *rss_obj;
      int *rss_idx_tbl;
      vmk_uint32 max_rssqs;

      if(raw_ind_tbl == adapter->defq_rss_raw_ind_tbl) {
      	rss_obj = &adapter->defq_rss_conf_obj;
         rss_idx_tbl = adapter->defq_rss_idx_tbl;
         max_rssqs = adapter->num_rssqs_def;
#if (ESX_DDK_VERSION >= 2017)
      } else {
           for (j=1; j<=QFLE3_DEVICE_MAX_RSS_ENGINE; j++){
              if(raw_ind_tbl == adapter->netq_rss_raw_ind_tbl[j])
              break;
           }
         if(j<=QFLE3_DEVICE_MAX_RSS_ENGINE){
            QFLE3_DBG(QFLE3_DBG_QUEUE, "Value of j is %d\n", j);
            rss_obj = &adapter->netq_rss_conf_obj[j];
            rss_idx_tbl = adapter->netq_rss_idx_tbl[j];
            max_rssqs = adapter->num_rssqs_nd;
         }else{
           QFLE3_DBG(QFLE3_DBG_QUEUE,"Trying ot configure RSS for %d beyond %d\n", j, QFLE3_DEVICE_MAX_RSS_ENGINE);
           return -1;
         }
      }
#else
      } else {
      	rss_obj = &adapter->netq_rss_conf_obj;
         rss_idx_tbl = adapter->netq_rss_idx_tbl;
         max_rssqs = adapter->num_rssqs_nd;
      }
#endif
      /* update RSS pool indirection table based on raw indirection
       * table provided and the list of rss queues
       */
      for (i = 0; i < T_ETH_INDIRECTION_TABLE_SIZE; i++) {
         vmk_uint8 offset;
         offset = raw_ind_tbl[i];
         if (offset >= max_rssqs) {
            QFLE3_ERR("Invalid rss ind tbl entry[%d]:%d >= %d\n",
                      i, offset, max_rssqs);
            return -1;
         }
         rss_obj->ind_table[i] = adapter->fp[rss_idx_tbl[offset]].cl_id;
      }
      QFLE3_DBG(QFLE3_DBG_QUEUE, "updated ind tbl :%d %d %d %d %d %d %d %d\n",
		rss_obj->ind_table[0], rss_obj->ind_table[1],
		rss_obj->ind_table[2], rss_obj->ind_table[3],
		rss_obj->ind_table[4], rss_obj->ind_table[5],
		rss_obj->ind_table[6], rss_obj->ind_table[7]);

      params.rss_obj = rss_obj;
      SET_BIT(RAMROD_COMP_WAIT, &params.ramrod_flags);
      SET_BIT(ECORE_RSS_MODE_REGULAR, &params.rss_flags);
      SET_BIT(ECORE_RSS_IPV4, &params.rss_flags);
      SET_BIT(ECORE_RSS_IPV4_TCP, &params.rss_flags);
      SET_BIT(ECORE_RSS_IPV4_UDP, &params.rss_flags);
      SET_BIT(ECORE_RSS_IPV6, &params.rss_flags);
      SET_BIT(ECORE_RSS_IPV6_TCP, &params.rss_flags);
      SET_BIT(ECORE_RSS_IPV6_UDP, &params.rss_flags);

      /* valid only for TUNN_MODE_VXLAN tunnel mode */
      SET_BIT(ECORE_RSS_IPV4_VXLAN, &params.rss_flags);
      SET_BIT(ECORE_RSS_IPV6_VXLAN, &params.rss_flags);

      if (QFLE3_IS_GENEVE_OFFLOAD_ENABLED(adapter)) {
         ECORE_SET_BIT(ECORE_RSS_TUNN_INNER_HDRS, &params.rss_flags);
      }

      /* Hash bits */
      params.rss_result_mask = MULTI_MASK;

      vmk_Memcpy(params.ind_table, rss_obj->ind_table,
		 sizeof(params.ind_table));
   }

   if (rss_key_tbl) {
      /* RSS keys */
      for (i = 0; i < T_ETH_RSS_KEY; i++)
         params.rss_key[i] = rss_key_tbl[i];

      SET_BIT(ECORE_RSS_SET_SRCH, &params.rss_flags);
   }

   rc = ecore_config_rss(adapter, &params);
   if (rc)
      QFLE3_ERR("Failed to config rss");
   return rc;
}

/*
 *-----------------------------------------------------------------------------
 *
 * BnxeRqAlloc --
 *
 *      Allocate a RX queue.
 *
 * Results:
 *      VMK_OK on success. Error code if otherwise.
 *      If successful, qid, and netpoll are updated.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

VMK_ReturnStatus
qfle3_rq_alloc(qfle3_adapter * adapter, // IN:  adapter
   vmk_UplinkQueueID * qid, // OUT: allocated qid
   vmk_NetPoll * netpoll,   // OUT: corresponding netpoll
   vmk_UplinkQueueFeature feat)
{
   vmk_UplinkSharedQueueData *sqd;
   struct qfle3_fastpath *rxq;
//#if (ESX_DDK_VERSION >= 2017)
//   struct qfle3_fastpath *rq;
//#endif
   vmk_uint32 i, flags, qidVal;
   //vmk_uint8 fp_idx;

   sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
#if (ESX_DDK_VERSION >= 2017)
   for (i = 0; i < adapter->num_rxqs_vmk; i++) {
#else
   for (i = 0; i < adapter->uplinkQueueInfo.maxRxQueues; i++) {
#endif
      if ((sqd[i].flags & VMK_UPLINK_QUEUE_FLAG_IN_USE) == 0) {
         break;
      }
   }
#if (ESX_DDK_VERSION >= 2017)
   if (i >= adapter->num_rxqs_vmk) {
#else
   if (i >= adapter->uplinkQueueInfo.maxRxQueues) {
#endif
      QFLE3_DBG(QFLE3_DBG_QUEUE, "No free RX queue found!");
      return VMK_FAILURE;
   }

   /*
    * rx data buffer size needs to be adjusted here
    */
   rxq = &adapter->fp[i];
   QFLE3_DBG(QFLE3_DBG_QUEUE, "allocating %s RX queue at %d\n",
   	Qfle3IsDefaultRxQueue(adapter, rxq) ? "default":"", i);

   if (feat & VMK_UPLINK_QUEUE_FEAT_LRO)
      rxq->tpa_enable = 1;

   rxq->num_rss = 1;
#if (ESX_DDK_VERSION >= 2017)
   if (feat & VMK_UPLINK_QUEUE_FEAT_RSS_GENERIC && (adapter->num_rssqs_nd>1)) {
#else
   if (feat & VMK_UPLINK_QUEUE_FEAT_RSS) {
#endif
      rxq->is_leading_rss = 1;
      rxq->num_rss = adapter->num_rssqs_nd;
      QFLE3_INFO("Rxq %d is leading RSS with %d RSS queues.\n", rxq->qid, rxq->num_rss);
   } else if (Qfle3IsDefaultRxQueue(adapter,rxq) && (adapter->num_rssqs_def>1)) {
      rxq->is_leading_rss = 1;
      rxq->is_defq_rss =  1;
      rxq->num_rss = adapter->num_rssqs_def;
      QFLE3_INFO("Rxq %d is Default Rxq with %d RSS queues.\n", rxq->qid, rxq->num_rss);
   }	

   if (qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_CREATE_Q, i, QFLE3_SM_PARAM_RXQ) != QFLE3_SMCMD_STATUS_COMPLETED)
      return VMK_FAILURE;
   
   qidVal = Qfle3GetQIDValFromSQD(adapter, &sqd[i]);
   vmk_UplinkQueueMkRxQueueID(qid, qidVal, qidVal);
   *netpoll = sqd[i].poll;

   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);

   flags = Qfle3IsDefaultRxQueue(adapter, rxq) ?
      VMK_UPLINK_QUEUE_FLAG_IN_USE | VMK_UPLINK_QUEUE_FLAG_DEFAULT :
      VMK_UPLINK_QUEUE_FLAG_IN_USE;
   sqd[i].flags &= ~(VMK_UPLINK_QUEUE_FLAG_DEFAULT | VMK_UPLINK_QUEUE_FLAG_IN_USE);
   sqd[i].flags |= flags;
   sqd[i].qid = *qid;
   QFLE3_SHARED_DATA_WRITE_END(adapter);
   QFLE3_DBG(QFLE3_DBG_QUEUE, "Marking RX queue %d IN_USE\n",i);

   return VMK_OK;
}

VMK_ReturnStatus
qfle3_rq_create(qfle3_adapter *adapter, vmk_uint32 id)
{
   VMK_ReturnStatus status = VMK_OK;
   vmk_uint32 num_rss;
   vmk_uint32 j;
   vmk_uint8 defq_rss = 0;
   vmk_uint8 fp_idx;
   struct qfle3_fastpath *rq, *rxq;
//   int qid = id;
#if (ESX_DDK_VERSION >= 2017)
   vmk_uint32 rss_engine_id;
   vmk_uint32 k;
#endif

   rxq = &adapter->fp[id];
   rxq->qid = id;

   if (rxq->fp_state != 0) {
      QFLE3_ERR("RQ seems to have already been created %ld",
         rxq - &adapter->fp[0]);
      return VMK_FAILURE;
   }
   rq = rxq;
   num_rss = rxq->num_rss;

   if (Qfle3IsDefaultRxQueue(adapter, rxq) && num_rss > 1)
      defq_rss = 1;
      QFLE3_DBG(QFLE3_DBG_QUEUE,"Number of Secondary queues requested for Primary QID: %d are  %d\n", rxq->qid, num_rss-1);
#if (ESX_DDK_VERSION >= 2017)
   if (rq->num_rss > 1){
      if (defq_rss){
         rss_engine_id = QFLE3_FUNC(adapter) * 2 * QFLE3_DEVICE_MAX_RSS_ENGINE;
         if(adapter->rss_engine[rss_engine_id])
            QFLE3_ERR("RSS Engine ID intended to configure for Default Queue already configured \n");
         QFLE3_DBG(QFLE3_DBG_QUEUE, "This is Default queue with RSS and Engine ID is %d\n", rss_engine_id);
         adapter->rss_engine[rss_engine_id] = -1;
      } else {
         for(k=1; k<=QFLE3_DEVICE_MAX_RSS_ENGINE; k++){
            rss_engine_id = QFLE3_FUNC(adapter) * 2 * QFLE3_DEVICE_MAX_RSS_ENGINE + k;
            if(adapter->rss_engine[rss_engine_id] == 0)
               break;
         }
         if(k>QFLE3_DEVICE_MAX_RSS_ENGINE){
            QFLE3_ERR("Trying to configure RSS beyond MAX Supported: %d\n", QFLE3_DEVICE_MAX_RSS_ENGINE);
            return VMK_FAILURE;
         }
         QFLE3_DBG(QFLE3_DBG_QUEUE, "This is Netqueue with RSS and Engine ID is %d and k is %d\n", rss_engine_id,k);
         adapter->rss_engine[rss_engine_id] = rxq->qid;
      } 
   rq->rss_engine_id = rss_engine_id;
   }
   for (j = 0; (j < num_rss) && (j < QFLE3_DEVICE_MAX_RSS_QUEUES); j++) {
#else
   for (j = 0; j < num_rss; j++) {
#endif
      if (j) {
         if (defq_rss){
            fp_idx = adapter->num_rxqs_drv - j;
#if (ESX_DDK_VERSION >= 2017)
            rq->sec_quque_ids[j] = fp_idx;
            QFLE3_DBG(QFLE3_DBG_QUEUE, "Q ID for the Secondary RSS Queue is %d\n", fp_idx);
#endif
         } else {
#if (ESX_DDK_VERSION >= 2017)
            for( fp_idx = adapter->num_rxqs_vmk - 1 +j; fp_idx < (adapter->num_rxqs_drv - (adapter->num_rssqs_def-1)); fp_idx++){
                rxq = &adapter->fp[fp_idx];
                if( rxq->q_in_use != 1)
                   break;
            }
            rxq->q_in_use = 1;
            rq->sec_quque_ids[j] = fp_idx;
            QFLE3_DBG(QFLE3_DBG_QUEUE, "Q ID for the Secondary RSS Queue is %d\n", fp_idx);
#else
            fp_idx = adapter->num_rxqs_vmk - 1 + j;
#endif         
         }
         rxq = &adapter->fp[fp_idx];
         rxq->qid = fp_idx;
         if (defq_rss)
            rxq->is_defq_rss = 1;
      }
      status = qfle3_alloc_fp_buffers(rxq);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to alloc buffers for rxq %d", rxq->qid);
         rxq->tpa_enable = VMK_FALSE;
         goto err_failed;
      }

      rxq->fp_state = QFLE3_SM_Q_CREATED;
   }
   return VMK_OK;
err_failed:
   while (j) {
      j--;
      if (j == 0)
         rxq = rq;
      else {
         if (defq_rss) {
            fp_idx = adapter->num_rxqs_drv - j;
         } else {
#if (ESX_DDK_VERSION >= 2017)
            fp_idx = rq->sec_quque_ids[j];
#else
            fp_idx = adapter->num_rxqs_vmk - 1 + j;
#endif
         }
         rxq = &adapter->fp[fp_idx];
#if (ESX_DDK_VERSION >= 2017)
         rxq->q_in_use = 0;
#endif
      }

//      qfle3_destroy_spinlock(QFLE3_SP_OBJ(adapter, rxq).mac_obj.exe_queue.lock);
//      qfle3_destroy_spinlock(QFLE3_SP_OBJ(adapter, rxq).vxlan_filter_obj.exe_queue.lock);
      qfle3_free_rx_bd_chain(rxq);
      qfle3_free_tpa_pool(rxq);
      qfle3_free_sge_chain(rxq);
   }
   return status;
}

VMK_ReturnStatus
qfle3_tq_alloc(qfle3_adapter * adapter, // IN:  adapter
   vmk_UplinkQueueID * qid, // OUT: allocated qid
   vmk_NetPoll * netpoll)   // OUT: corresponding netpoll
{
   vmk_UplinkSharedQueueData *sqd;
   struct qfle3_fastpath *txq;
   vmk_uint32 i, flags, qidVal;

   sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
   for (i = 0; i < adapter->uplinkQueueInfo.maxTxQueues; i++) {
      if ((sqd[i].flags & VMK_UPLINK_QUEUE_FLAG_IN_USE) == 0) {
         break;
      }
   }

   if (i >= adapter->uplinkQueueInfo.maxTxQueues) {
      QFLE3_DBG(QFLE3_DBG_QUEUE, "No free TX queue found!");
      return VMK_FAILURE;
   }

   qidVal = Qfle3GetQIDValFromSQD(adapter, &sqd[i]);
   /*
    * The user value (3rd argument) can be used directly indexing into
    * * adapter->txQueue[] array.
    */
   vmk_UplinkQueueMkTxQueueID(qid, qidVal,
      qidVal - adapter->uplinkQueueInfo.maxRxQueues);
   *netpoll = sqd[i].poll;

   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   txq = &adapter->fp[QFLE3_TX_Q_NUM_TO_FP_NUM(adapter, i)];
   QFLE3_DBG(QFLE3_DBG_QUEUE, "allocating %s TX queue at %d\n", 
   	Qfle3IsDefaultTxQueue(adapter, txq) ? "default" : "",i);
   
   flags = Qfle3IsDefaultTxQueue(adapter, txq) ?
      VMK_UPLINK_QUEUE_FLAG_IN_USE | VMK_UPLINK_QUEUE_FLAG_DEFAULT :
      VMK_UPLINK_QUEUE_FLAG_IN_USE;
   sqd[i].flags &= ~(VMK_UPLINK_QUEUE_FLAG_DEFAULT | VMK_UPLINK_QUEUE_FLAG_IN_USE);
   sqd[i].flags |= flags;
   sqd[i].qid = *qid;
   QFLE3_SHARED_DATA_WRITE_END(adapter);
   QFLE3_DBG(QFLE3_DBG_QUEUE, "Marking TX queue %d IN_USE\n",i);

   qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_CREATE_Q, i, QFLE3_SM_PARAM_TXQ);
   return VMK_OK;
}


VMK_ReturnStatus
qfle3_tq_create(qfle3_adapter *adapter, vmk_uint32 id)
{
   struct qfle3_fastpath *txq;
//   VMK_ReturnStatus status = VMK_OK;
   int qid = QFLE3_TX_Q_NUM_TO_FP_NUM(adapter, id);
   txq = &adapter->fp[qid];
   txq->qid = qid;
   
   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_tq_create for id %d, qid %d\n", id, qid);
   if (txq->sqd)
      if (!(txq->sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE)) {
         QFLE3_ERR("Trying to create TX Q without Allocating it first %ld", 
            txq - &adapter->fp[adapter->num_rxqs_drv]);
         return VMK_FAILURE;
      }

   if (txq->fp_state != 0) {
      QFLE3_ERR("TQ seesm to have already been created %ld", 
         txq - &adapter->fp[QFLE3_DEFAULT_TX_QID(adapter)]);
      return VMK_FAILURE;
   }
   
   txq->fp_state = QFLE3_SM_Q_CREATED;
   return VMK_OK;
}


/*
 * Netqueue Operations Implementation
 */
static VMK_ReturnStatus
qfle3_queue_alloc(vmk_AddrCookie driverData,
   vmk_UplinkQueueType qType,
   vmk_UplinkQueueID * qID, vmk_NetPoll * netpoll)
{

   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   VMK_ReturnStatus status;
   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.queueAlloc %s \n",
      (qType == VMK_UPLINK_QUEUE_TYPE_RX) ? "RX": "TX");
   
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_DBG(QFLE3_DBG_QUEUE, "Uplink is Resetting, Rejecting queue alloc. "
         "Recovery State:0x%x Error_status:0x%x\n", adapter->recovery_state, adapter->error_status);
      return VMK_FAILURE;
   }

   if (qType == VMK_UPLINK_QUEUE_TYPE_RX) {
      status = qfle3_rq_alloc(adapter, qID, netpoll, 0);
   } else if (qType == VMK_UPLINK_QUEUE_TYPE_TX) {
      status = qfle3_tq_alloc(adapter, qID, netpoll);
   } else {
      status = VMK_BAD_PARAM;
   }

   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.queueAlloc allocated %d \n", vmk_UplinkQueueIDVal(*qID));

   return status;

}

static VMK_ReturnStatus
qfle3_queue_alloc_with_attr(vmk_AddrCookie driverData,
   vmk_UplinkQueueType qType,
   vmk_uint16 num_attr,
   vmk_UplinkQueueAttr * attr,
   vmk_UplinkQueueID * qID, vmk_NetPoll * netpoll)
{
   int i;
   vmk_UplinkQueueFeature vmk_feat = 0;
   vmk_UplinkQueueFeature req_feat = 0;
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.qfle3_queue_alloc_with_attr "
      "num_attr :%u attrs: %p\n", num_attr, attr);

   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_DBG(QFLE3_DBG_QUEUE, "Uplink is Resetting, Rejecting queue alloc");
      return VMK_FAILURE;
   }

   if (qType == VMK_UPLINK_QUEUE_TYPE_TX) {
      return qfle3_tq_alloc(adapter, qID, netpoll);
   }

   if (qType != VMK_UPLINK_QUEUE_TYPE_RX) {
      return VMK_BAD_PARAM;
   }

   if (!attr || !num_attr) {
      goto alloc_rx_queue;
   }

   for (i = 0; i < num_attr; i++) {
      switch (attr[i].type) {
         case VMK_UPLINK_QUEUE_ATTR_PRIOR:
         QFLE3_WARN("VMK_UPLINK_QUEUE_ATTR_PRIOR isn't supported.\n");
         break;

         case VMK_UPLINK_QUEUE_ATTR_FEAT:
         vmk_feat = attr[i].args.features;
         if (vmk_feat & VMK_UPLINK_QUEUE_FEAT_LRO) {
            QFLE3_INFO("Feature LRO requested.\n");
            req_feat |= VMK_UPLINK_QUEUE_FEAT_LRO;
         }
#if (ESX_DDK_VERSION >= 2017)
         if (vmk_feat & VMK_UPLINK_QUEUE_FEAT_RSS_GENERIC) {
#else
         if ((vmk_feat & VMK_UPLINK_QUEUE_FEAT_RSS) ||
            (vmk_feat & VMK_UPLINK_QUEUE_FEAT_RSS_DYN)) {
#endif
            QFLE3_INFO("Feature %s requested.\n",
               (vmk_feat & VMK_UPLINK_QUEUE_FEAT_RSS_DYN)
               ? "Dynamic RSS" : "RSS");
            /* Set below feature so that we don't need to change
            checks for RSS feature in all functions if we RSS_DYN feature attr set. */
#if (ESX_DDK_VERSION >= 2017)
            req_feat |= VMK_UPLINK_QUEUE_FEAT_RSS_GENERIC;
#else
            req_feat |= VMK_UPLINK_QUEUE_FEAT_RSS;
#endif
         }

         /* Unsupported features */
#if (ESX_DDK_VERSION >= 2017)
         if (vmk_feat & ~(VMK_UPLINK_QUEUE_FEAT_LRO | VMK_UPLINK_QUEUE_FEAT_RSS_GENERIC |
                VMK_UPLINK_QUEUE_FEAT_RSS_DYN)) {
#else
         if (vmk_feat & ~(VMK_UPLINK_QUEUE_FEAT_LRO | VMK_UPLINK_QUEUE_FEAT_RSS |
            VMK_UPLINK_QUEUE_FEAT_RSS_DYN)) {
#endif
            QFLE3_WARN("Failed... "
               "Unsupported feature : 0x%x, Trying to allocate normal queue.\n",
               vmk_feat & ~(VMK_UPLINK_QUEUE_FEAT_LRO |
               VMK_UPLINK_QUEUE_FEAT_RSS |
               VMK_UPLINK_QUEUE_FEAT_RSS_DYN));
         }
         break;
         
         default:
         QFLE3_WARN("Invalid attribute type : %d\n",  attr[i].type);
      }
   }

alloc_rx_queue:
   return qfle3_rq_alloc(adapter, qID, netpoll, req_feat);
}

static VMK_ReturnStatus
qfle3_queue_realloc_with_attr(vmk_AddrCookie driverData,
			      vmk_UplinkQueueReallocParams * params)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.qfle3_queue_realloc_with_attr\n");

   return VMK_NOT_SUPPORTED;
}


VMK_ReturnStatus
qfle3_rq_free(qfle3_adapter * adapter, vmk_UplinkQueueID qid)
{
   struct qfle3_fastpath *rxq = Qfle3GetRxQueueFromQID(adapter, qid);
   int qidVal = vmk_UplinkQueueIDVal(qid);
   struct qfle3_fastpath *rq;
   vmk_uint32 /*i, */num_rss;
   vmk_uint8 defq_rss = 0;

//   QFLE3_DBG(QFLE3_DBG_QUEUE, "Freeing RQ %d, num_rss %d default q %d\n", qidVal, rxq->num_rss,
//   	Qfle3IsDefaultRxQueue(adapter, rxq));
   if (!(rxq->sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE)) {
      
      QFLE3_DBG(QFLE3_DBG_QUEUE, "RQ already freed %d\n", qidVal);
      return VMK_OK;
   }

   /*
    * update shared data
    */
   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   rxq->sqd->flags &= ~(VMK_UPLINK_QUEUE_FLAG_IN_USE | VMK_UPLINK_QUEUE_FLAG_DEFAULT);
   QFLE3_SHARED_DATA_WRITE_END(adapter);
   rq = rxq;

   num_rss = rxq->num_rss;
   if (Qfle3IsDefaultRxQueue(adapter, rxq) && num_rss > 1)
      defq_rss = 1;

   QFLE3_DBG(QFLE3_DBG_QUEUE, "Loop through %u RSS queues %d\n", num_rss, qidVal);

   qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_REMOVE_Q, qidVal, QFLE3_SM_PARAM_RXQ);
   return VMK_OK;
}

void qfle3_rq_release(qfle3_adapter *adapter, vmk_uint32 idx)
{
   vmk_uint32 i, num_rss;
   struct qfle3_fastpath *rxq = &adapter->fp[idx];
   struct qfle3_fastpath *rq;
   vmk_uint8 defq_rss = 0;
   vmk_uint8 fp_idx;

   if (rxq->fp_state != QFLE3_SM_Q_CREATED) {
      QFLE3_DBG(QFLE3_DBG_QUEUE, "Releasing RQ in the wrong state %d, Q idx %ld\n", rxq->fp_state, 
         rxq - &adapter->fp[0]);
   }
   rq = rxq;
   num_rss = rxq->num_rss;
   if (Qfle3IsDefaultRxQueue(adapter, rxq) && num_rss > 1)
      defq_rss = 1;

   for (i = 0; i < num_rss; i++) {
      if (i) {
         if (defq_rss)
            fp_idx = adapter->num_rxqs_drv - i;
         else {
#if (ESX_DDK_VERSION >= 2017)
            fp_idx = rq->sec_quque_ids[i];
#else
            fp_idx = adapter->num_rxqs_vmk - 1 + i;
#endif    
         }
         rxq = &adapter->fp[fp_idx];
      }

      qfle3_free_rx_bd_chain(rxq);
      qfle3_free_tpa_pool(rxq);
      qfle3_free_sge_chain(rxq);
      
#if (ESX_DDK_VERSION >= 2017)
      rxq->q_in_use = 0;
      adapter->rss_engine[rxq->rss_engine_id] = 0;
#endif
      if (rxq->is_defq_rss)
         rxq->is_defq_rss = 0;
      rxq->tpa_enable = 0;
      rxq->is_leading_rss = 0;
      rxq->num_rss = 0;

      rxq->fp_state = 0;
   }
}
VMK_ReturnStatus
qfle3_tq_free(qfle3_adapter * adapter, vmk_UplinkQueueID qid)
{
   struct qfle3_fastpath *txq = Qfle3GetTxQueueFromQID(adapter, qid);
   int qidVal = vmk_UplinkQueueIDVal(qid);
  // struct ecore_vlan_mac_obj *obj;
   
   QFLE3_DBG(QFLE3_DBG_QUEUE, "Freeing TQ   %d\n", qidVal);
   if (!(txq->sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE)) {
      
      QFLE3_DBG(QFLE3_DBG_QUEUE, "TQ already freed %d\n", qidVal);
	   return VMK_OK;
   }

   /*
    * update shared data
    */
   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   txq->sqd->flags &= ~(VMK_UPLINK_QUEUE_FLAG_IN_USE | VMK_UPLINK_QUEUE_FLAG_DEFAULT);
   QFLE3_SHARED_DATA_WRITE_END(adapter);
   qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_REMOVE_Q, qidVal, QFLE3_SM_PARAM_TXQ);
   return VMK_OK;
}



static VMK_ReturnStatus
qfle3_queue_free(vmk_AddrCookie driverData, vmk_UplinkQueueID qid)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.queueFree\n");
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_DBG(QFLE3_DBG_QUEUE, "Uplink is Resetting, Rejecting queue free");
      return VMK_FAILURE;
   }

   if (vmk_UplinkQueueIDType(qid) == VMK_UPLINK_QUEUE_TYPE_RX) {
      QFLE3_DBG(QFLE3_DBG_QUEUE, "RxQ, QueueIDVal:%d", vmk_UplinkQueueIDVal(qid));
      return qfle3_rq_free(adapter, qid);
   } else if (vmk_UplinkQueueIDType(qid) == VMK_UPLINK_QUEUE_TYPE_TX) {
      QFLE3_DBG(QFLE3_DBG_QUEUE, "TxQ, QueueIDVal:%d", vmk_UplinkQueueIDVal(qid));
      return qfle3_tq_free(adapter, qid);
   }
   QFLE3_ERR("Wrong QueueIDType(%x), qid(%u)", vmk_UplinkQueueIDType(qid), vmk_UplinkQueueIDVal(qid));
   return VMK_BAD_PARAM;

}
static inline int
qfle3_has_rx_work(struct qfle3_fastpath *fp)
{
   vmk_uint16 rx_cq_cons_sb;
   struct qfle3_adapter *adapter = fp->adapter;

   //vmk_LogMessage("rx_cq_cons_sb is %d, rx_cq_cons %d ", le16toh(*fp->rx_cq_cons_sb), fp->rx_cq_cons);
   /*
    * status block fields can change
    */
   rx_cq_cons_sb = le16toh(*fp->rx_cq_cons_sb);
   if ((rx_cq_cons_sb & RCQ_MAX) == RCQ_MAX)
      rx_cq_cons_sb++;
   return (fp->rx_cq_cons != rx_cq_cons_sb);
}

VMK_ReturnStatus
qfle3_rq_stop(qfle3_adapter * adapter, vmk_uint32 id)
{
   vmk_uint32 i,k, num_rss;
   vmk_UplinkSharedQueueData *sqd = NULL;
   vmk_UplinkSharedData *sd = NULL;
   struct qfle3_fastpath *rq;
   struct ecore_config_rss_params params = {NULL};
   struct ecore_vlan_mac_obj *obj;	
   vmk_int32 rc;
   vmk_uint8 defq_rss = 0;
   vmk_uint8 fp_idx = 0;
   vmk_uint32 offset;
   vmk_IntrCookie cookie=0;
   VMK_ReturnStatus status;


   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_rq_stop %d", id);
   struct qfle3_fastpath *rxq;
#if (ESX_DDK_VERSION >= 2017)
   int j;
#endif
   
   rq = &adapter->fp[id];
   rxq = rq;
   if (rq->fp_state != (QFLE3_SM_Q_STARTED)) {
      QFLE3_ERR("RQ already stopped %d, idx %d", rq->fp_state, id);
      return VMK_OK;
   }

   if (id < QFLE3_NUM_RX_VMK_ETH_QUEUES(adapter)) {
      sd = &adapter->uplinkSharedData;
      sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
      sqd += id;


      if (!(sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE)) {
         QFLE3_ERR("RX queue %d not allocated", id);
         return VMK_BAD_PARAM;
      }

      if (sqd->state == VMK_UPLINK_QUEUE_STATE_STOPPED) {
         QFLE3_ERR("RX queue %d already stopped", id);
         return VMK_BAD_PARAM;
      }
    
   }
   
   obj = &QFLE3_SP_OBJ(adapter, rq).mac_obj;
   qfle3_del_all_macs(adapter, obj,
                      ECORE_NETQ_ETH_MAC, VMK_TRUE);
   if (adapter->vxlan_filters_en) {
      obj = &QFLE3_SP_OBJ(adapter, rq).vxlan_filter_obj;
      qfle3_del_all_macs(adapter, obj,
                         ECORE_NETQ_ETH_MAC, VMK_TRUE);
   }

   num_rss = rq->num_rss;

   if (Qfle3IsDefaultRxQueue(adapter, rq) && num_rss > 1)
      defq_rss = 1;

   if (num_rss > 1) {
      if (defq_rss)
         params.rss_obj = &adapter->defq_rss_conf_obj;
#if (ESX_DDK_VERSION >= 2017)
      else{
         j = rq->rss_engine_id - ( QFLE3_FUNC(adapter) * 2 * QFLE3_DEVICE_MAX_RSS_ENGINE );
         if(j<=QFLE3_DEVICE_MAX_RSS_ENGINE)
            params.rss_obj = &adapter->netq_rss_conf_obj[j];
      }
#else
      else
         params.rss_obj = &adapter->netq_rss_conf_obj;
#endif
      SET_BIT(RAMROD_COMP_WAIT, &params.ramrod_flags);
      /* Hash bits */
      params.rss_result_mask = MULTI_MASK;
      vmk_Memset(params.ind_table, 0, sizeof(params.ind_table));
      
      if (adapter->recovery_state != QFLE3_RECOVERY_DONE){
         SET_BIT(RAMROD_DRV_CLR_ONLY, &params.ramrod_flags);
      }
      rc = ecore_config_rss(adapter, &params);

      if (rc) {
         QFLE3_WARN("Failed to stop RSS.\n");
      }
   }
#if (ESX_DDK_VERSION >= 2017)
   adapter->rss_engine[rq->rss_engine_id] = 0;
#endif
   for (i = 0; i < num_rss; i++) {
      fp_idx = id;
      if (i) {
         if (defq_rss)
            fp_idx = adapter->num_rxqs_drv - i;
         else {
#if (ESX_DDK_VERSION >= 2017)
            fp_idx = rxq->sec_quque_ids[i];
            QFLE3_DBG(QFLE3_DBG_QUEUE, "Q ID for the Secondary RSS Queue is %d\n", fp_idx);
#else
            fp_idx = adapter->num_rxqs_vmk - 1 + i;
            QFLE3_DBG(QFLE3_DBG_QUEUE, "Q ID for the Hidden RSS Queue is %d\n", fp_idx);
#endif
         }
         rq = &adapter->fp[fp_idx];
      }

      if (rq->is_defq_rss)
         rq->is_defq_rss = 0;
      QFLE3_STATS_LOCK(adapter);
      rq->collect_stats = 0;
      QFLE3_STATS_UNLOCK(adapter);
      qfle3_stop_queue(adapter, rq->qid);


      offset = 1 + CNIC_SUPPORT(adapter);
      offset += rq->qid;
      if (adapter->intr.intrType == QFLE3_IT_MSIX){
         cookie = adapter->intr.cookies[offset];

      }else {
         cookie = adapter->intr.cookies[0];
      }

      // set netpoll interrupt
      if (rq->netpoll) {
         if (!rq->pollDisabled) {
            vmk_NetPollDisable(rq->netpoll);
            vmk_NetPollFlushRx(rq->netpoll);
            netpoll_count--;
            rq->pollDisabled = VMK_TRUE;
            QFLE3_DBG(QFLE3_DBG_QUEUE, "disable netpoll for q_index %d count %d\n", rq->qid, netpoll_count);
         }
         status = vmk_NetPollInterruptUnSet(rq->netpoll);
         if (status != VMK_OK) {
            QFLE3_ERR("Failed to unset associated interrupt cookie #%d 0x%x"
               "with fp[%d] netpoll", offset, cookie, rq->qid);
         }
      }

      if (adapter->intr.intrType == QFLE3_IT_MSIX){

         QFLE3_DBG(QFLE3_DBG_QUEUE,"Disabling interrupt on vector # %d", offset);
            status = vmk_IntrDisable(cookie);
            if (status != VMK_OK) {
               QFLE3_ERR("Failed to disable intrCookie[%d] 0x%x (%x)", offset, cookie, status);
            }
         }
      // make sure the status block counter is zeroed out so incase an interrupt is 
      // in-flight, it won't cause us problems when we re-enable interrupt
      *rq->rx_cq_cons_sb = htole16(0);
      vmk_SpinlockLock(rq->fp_lock);
      vmk_SpinlockUnlock(rq->fp_lock);
      obj = &QFLE3_SP_OBJ(adapter, rq).mac_obj;
      qfle3_destroy_spinlock(obj->exe_queue.lock);
      obj->exe_queue.lock = NULL;
      obj = &QFLE3_SP_OBJ(adapter, rq).vxlan_filter_obj;
      qfle3_destroy_spinlock(obj->exe_queue.lock);
      obj->exe_queue.lock = NULL;
   rq->fp_state = QFLE3_SM_Q_CREATED;
   }

   sd = &adapter->uplinkSharedData;
   sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
   sqd += id;
   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   sqd->state = VMK_UPLINK_QUEUE_STATE_STOPPED;
   sd->queueInfo->activeRxQueues--;
   vmk_BitVectorClear(sd->queueInfo->activeQueues, Qfle3GetQIDValFromSQD(adapter, sqd));
   QFLE3_SHARED_DATA_WRITE_END(adapter);
   QFLE3_DBG(QFLE3_DBG_QUEUE, "active Rx queue Count %d", sd->queueInfo->activeRxQueues);
   for (k = 0; k < QFLE3_MAX_FILTERS_PER_QUEUE; k++) {
      rxq->rx_filters[k].flags = 0;
   }
   rxq->num_rx_filters = 0;
   return VMK_OK;
}

int qfle3_has_tx_work_unload(struct qfle3_fp_txdata *txdata)
{
	/* Tell compiler that consumer and producer can change */
	vmk_CPUMemFenceRead();
	return txdata->tx_pkt_prod != txdata->tx_pkt_cons;
}

int qfle3_clean_tx_queue(qfle3_adapter * adapter,
				       struct qfle3_fp_txdata *txdata)
{
   /* wait if there is tx work before stopping the tx queue, here we wait at most 3 seconds. */
   int cnt = 30;

   while (qfle3_has_tx_work_unload(txdata)) {
      if (!cnt) {
         QFLE3_ERR("timeout waiting for queue[%d]: txdata->tx_pkt_prod(%d) != txdata->tx_pkt_cons(%d)\n",
                   txdata->txq_index, txdata->tx_pkt_prod,
                   txdata->tx_pkt_cons);
         break;
      }
      cnt--;
      vmk_WorldSleep(100 * 1000);
   }

   return 0;
}

void qfle3_print_hw_place(struct qfle3_fastpath *fp)
{
  u8 cos;
  qfle3_adapter *adapter = fp->adapter;

  for_each_cos_in_tx_queue(fp, cos)
  {
    QFLE3_DBG(QFLE3_DBG_QUEUE, "txq[%d] prod %d, cons %d\n", 
         fp->qid, fp->txdata_ptr[cos]->tx_pkt_prod, fp->txdata_ptr[cos]->tx_pkt_cons);	 
  }
}

void qfle3_assert_tx_ring(struct qfle3_fastpath *fp)
{
  u8 cos;
  for_each_cos_in_tx_queue(fp, cos)
  {
      if(qfle3_has_tx_work_unload(fp->txdata_ptr[cos])) {
         QFLE3_ERROR("txq[%d] cos %d not clean, leak %d\n",
                   fp->qid, cos, fp->txdata_ptr[cos]->tx_pkt_prod - fp->txdata_ptr[cos]->tx_pkt_cons);
      }
      fp->txdata_ptr[cos]->queue_stuck = 0;
  }
}

static void qfle3_drain_tx_queue(qfle3_adapter * adapter,
					  struct qfle3_fastpath *fp)
{
	u8 cos;
	for_each_cos_in_tx_queue(fp, cos)
	{
		if (qfle3_clean_tx_queue(adapter, fp->txdata_ptr[cos]))
			return;
	}
}

VMK_ReturnStatus
qfle3_tq_stop (qfle3_adapter * adapter, vmk_uint32 id)
{
   vmk_UplinkSharedQueueData *sqd;
   vmk_UplinkSharedData *sd;
   struct qfle3_fastpath *tq;
   vmk_uint32 offset;
   vmk_IntrCookie cookie;
   VMK_ReturnStatus status = VMK_OK;
   struct ecore_vlan_mac_obj *obj;
   
   tq = QFLE3_GET_FP_FROM_TQID(adapter, id);

   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_tq_stop %d state %d", id, tq->fp_state);
   if (!(tq->fp_state & QFLE3_SM_Q_STARTED)) {
      QFLE3_ERR("TQ already stopped %d, idx %d", tq->fp_state, id);
      return VMK_OK;
   }

   if (id < QFLE3_NUM_TX_VMK_ETH_QUEUES(adapter)) {
      sd = &adapter->uplinkSharedData;
      sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
      sqd += id;

      if (!(sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE)) {
         QFLE3_ERR("TX queue %d not allocated", id);
   	  return VMK_BAD_PARAM;
      }

      vmk_SpinlockLock(tq->fp_lock);
      tq->soft_state.tq_stopped_by_tx = VMK_FALSE;
      vmk_SpinlockUnlock(tq->fp_lock);

      if (sqd->state != VMK_UPLINK_QUEUE_STATE_STOPPED) {
         QFLE3_DBG(QFLE3_DBG_QUEUE, "Stopping TX queue %d in kernel", id);

         /*
          * Notify uplink layer
          */
          
         QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
         sqd->state = VMK_UPLINK_QUEUE_STATE_STOPPED;
         sd->queueInfo->activeTxQueues--;
         vmk_BitVectorClear(sd->queueInfo->activeQueues, Qfle3GetQIDValFromSQD(adapter, sqd));
         QFLE3_SHARED_DATA_WRITE_END(adapter);
         vmk_UplinkQueueStop(adapter->uplink, sqd->qid);
         vmk_CPUMemFenceWrite();
      }
      
      QFLE3_DBG(QFLE3_DBG_QUEUE, "active Tx queue Count %d", sd->queueInfo->activeTxQueues);
  }

   QFLE3_STATS_LOCK(adapter);
   tq->collect_stats = 0;
   QFLE3_STATS_UNLOCK(adapter);

   if (adapter->recovery_state == QFLE3_RECOVERY_DONE){

      QFLE3_DBG(QFLE3_DBG_QUEUE, "Drain TX queue %d", tq->qid);
      //qfle3_print_hw_place(tq);
      qfle3_drain_tx_queue(adapter, tq);
      //qfle3_print_hw_place(tq);
      qfle3_stop_queue(adapter, tq->qid);
   } else {
      QFLE3_DBG(QFLE3_DBG_QUEUE, "SKipping Stopping queue in FW, no interrupt%d", tq->qid);

   }
   //qfle3_print_hw_place(tq);
   offset = 1 + CNIC_SUPPORT(adapter);
   offset += tq->qid;

   if (adapter->intr.intrType != QFLE3_IT_MSIX)
      cookie = adapter->intr.cookies[0];
   else {
      cookie = adapter->intr.cookies[offset];


   if (adapter->intr.intrType == QFLE3_IT_MSIX)

      QFLE3_DBG(QFLE3_DBG_QUEUE,"Disabling interrupt on vector # %d", offset);
      status = vmk_IntrDisable(cookie);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to disable intrCookie[%d] 0x%x (%x)", offset, cookie, status);
      }
   }
   if (tq->netpoll) {

      if (!tq->pollDisabled) {
         vmk_NetPollDisable(tq->netpoll);
         
         vmk_NetPollFlushRx(tq->netpoll);
         netpoll_count--;
         QFLE3_DBG(QFLE3_DBG_QUEUE,"disabled netpoll for q_index %d count %d\n", tq->qid, netpoll_count);
         tq->pollDisabled = VMK_TRUE;
      }

      vmk_DelayUsecs(5);
      status = vmk_NetPollInterruptUnSet(tq->netpoll);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to unset associated interrupt cookie #%d 0x%x"
            "with fp[%d] netpoll", offset, cookie, tq->qid);
      }

   }

   obj = &QFLE3_SP_OBJ(adapter, tq).mac_obj;
   qfle3_destroy_spinlock(obj->exe_queue.lock);
   obj->exe_queue.lock = NULL;
   obj = &QFLE3_SP_OBJ(adapter, tq).vxlan_filter_obj;
   qfle3_destroy_spinlock(obj->exe_queue.lock);
   obj->exe_queue.lock = NULL;

   tq->fp_state = QFLE3_SM_Q_CREATED;
   
   if (adapter->recovery_helper == QFLE3_RECOVERY_DONE){
      qfle3_assert_tx_ring(tq);
   }
   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_tq_stop %d state %d", id, tq->fp_state);
   return VMK_OK;
}

VMK_ReturnStatus
qfle3_tq_pause(qfle3_adapter *adapter, vmk_uint32 id)
{
   vmk_UplinkSharedQueueData *sqd;
   vmk_UplinkSharedData *sd;
   struct qfle3_fastpath *tq;

   sd = &adapter->uplinkSharedData;
   sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
   sqd += id;
   tq = QFLE3_GET_FP_FROM_TQID(adapter,id);

   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_tq_pause %d", id);

   if (!(sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE)) {
      QFLE3_ERR("TX queue %d not allocated", id);
	  return VMK_BAD_PARAM;
   }

   if (sqd->state != VMK_UPLINK_QUEUE_STATE_STARTED) {
      QFLE3_ERR("Tring to PAUSE TQ in wrong state %d, idx %d", tq->fp_state, id);
	  return VMK_BAD_PARAM;
   }
   if (tq->fp_state != (QFLE3_SM_Q_STARTED)) {
      QFLE3_ERR("Tring to PAUSE TQ in wrong state %d, idx %d", tq->fp_state, id);
      return VMK_BAD_PARAM;
   }
   tq->soft_state.tq_stopped_by_tx = VMK_TRUE;

   /*
    * Notify uplink layer
    */

   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   sqd->state = VMK_UPLINK_QUEUE_STATE_STOPPED;
   sd->queueInfo->activeTxQueues--;
   vmk_BitVectorClear(sd->queueInfo->activeQueues, Qfle3GetQIDValFromSQD(adapter, sqd));
   QFLE3_SHARED_DATA_WRITE_END(adapter);
   vmk_UplinkQueueStop(adapter->uplink, sqd->qid);

   QFLE3_DBG(QFLE3_DBG_QUEUE, "active Tx queue Count %d", sd->queueInfo->activeTxQueues);
   return VMK_OK;
}


static VMK_ReturnStatus
qfle3_queue_quiesce(vmk_AddrCookie driverData, vmk_UplinkQueueID qid)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   //struct qfle3_fastpath *fp;
   
   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.queueQuiesce\n");
   
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_DBG(QFLE3_DBG_QUEUE, "Uplink is Resetting, Rejecting queue quiesce");
      return VMK_FAILURE;
   }

   if (vmk_UplinkQueueIDType(qid) == VMK_UPLINK_QUEUE_TYPE_RX) {
      QFLE3_DBG(QFLE3_DBG_QUEUE, "RxQ, QueueIDVal:%d", vmk_UplinkQueueIDVal(qid));
      return qfle3_rx_queue_stop(adapter, qid);
   } else if (vmk_UplinkQueueIDType(qid) == VMK_UPLINK_QUEUE_TYPE_TX) {
      QFLE3_DBG(QFLE3_DBG_QUEUE, "TxQ, QueueIDVal:%d", vmk_UplinkQueueIDVal(qid));
      return qfle3_tx_queue_stop(adapter, qid);
   }
   QFLE3_ERR("Wrong QueueIDType(%x), qid(%u)", vmk_UplinkQueueIDType(qid), vmk_UplinkQueueIDVal(qid));
   return VMK_BAD_PARAM;

}

VMK_ReturnStatus
qfle3_tq_start(qfle3_adapter * adapter, vmk_uint32 id)
{
   vmk_UplinkSharedQueueData *sqd = NULL;

   vmk_UplinkSharedData *sd = NULL;
   struct qfle3_fastpath *tq;
   VMK_ReturnStatus status = VMK_OK;
   vmk_uint32 offset;
   vmk_IntrCookie cookie;
   struct ecore_vlan_mac_obj *obj;

   tq = QFLE3_GET_FP_FROM_TQID(adapter,id);
   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_tq_start %d  state 0x%x", id, tq->fp_state);

   if (tq->fp_state == QFLE3_SM_Q_STARTED){
      QFLE3_DBG(QFLE3_DBG_QUEUE, "TQ Already started 0x%x, idx %d", tq->fp_state, id);
      return VMK_OK;
   }

   if (tq->fp_state != QFLE3_SM_Q_CREATED){
      QFLE3_ERR("Tring to START TQ in wrong state 0x%x, idx %d", tq->fp_state, id);
      return VMK_BAD_PARAM;
   }
   if ((adapter->sm_state < QFLE3_SM_ACTIVATED)) {
      
      QFLE3_ERR("sm_state %s ", qfle3_sm_get_string(adapter->sm_state));
      QFLE3_ERR("Cannot start TX queue %d. Device not ready.  Cache the command for later execution", id);

      return VMK_BAD_PARAM;
   }
   
   status = qfle3_init_eth_fp(adapter, 0, tq);

   if (status != VMK_OK)
     return status;

   qfle3_assert_tx_ring(tq);
   qfle3_tq_init(tq);

   tq->rx_bd_prod = MAX_RX_AVAIL;
   tq->rx_cq_prod = MAX_RCQ_AVAIL;

   QFLE3_DBG(QFLE3_DBG_QUEUE,
      "RX fp[%d]: wrote prods bd_prod=%d cqe_prod=%d sge_prod=%d\n",
      tq->qid, tq->rx_bd_prod, tq->rx_cq_prod, tq->rx_sge_prod);

   qfle3_update_rx_prod(adapter, tq, tq->rx_bd_prod, tq->rx_cq_prod,
      tq->rx_sge_prod);

   offset = 1 + CNIC_SUPPORT(adapter);
   offset += tq->qid;
   if (adapter->intr.intrType == QFLE3_IT_MSIX){

      QFLE3_DBG(QFLE3_DBG_QUEUE, "Enabling interrupt on vector # %d", offset);
      cookie = adapter->intr.cookies[offset];
      status = vmk_IntrEnable(cookie);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to enable intrCookie[%d] 0x%x (%x)", offset, cookie, status);
         goto intr_fail;
      }
   }
   else {
      cookie = adapter->intr.cookies[0];
   }
   // set netpoll interrupt
   if (tq->netpoll) {
      status = vmk_NetPollInterruptSet(tq->netpoll, cookie);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to set associated interrupt cookie #%d 0x%x"
            "with fp[%d] netpoll", offset, cookie, tq->qid);
         goto pollintr_fail;
      }

      if (tq->pollDisabled) {
         vmk_NetPollEnable(tq->netpoll);
         netpoll_count++;
         tq->pollDisabled = VMK_FALSE;
         QFLE3_DBG(QFLE3_DBG_QUEUE, "enabled netpoll for q_index %d count %d\n", tq->qid,netpoll_count);
      }
   
   }

   status = qfle3_setup_queue(adapter, tq, 0);
   if (status != VMK_OK) {
      QFLE3_ERR("TX queue %d failed start", id);
      goto setup_fail;
   }
   

   QFLE3_STATS_LOCK(adapter);
   tq->collect_stats = 1;
   QFLE3_STATS_UNLOCK(adapter);

   /*
    * Notify uplink layer
    */
   
   sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
   sqd += id;
   sd = &adapter->uplinkSharedData;
   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   sqd->state = VMK_UPLINK_QUEUE_STATE_STARTED;
   sd->queueInfo->activeTxQueues++;
   vmk_BitVectorSet(sd->queueInfo->activeQueues, Qfle3GetQIDValFromSQD(adapter, sqd));
   QFLE3_SHARED_DATA_WRITE_END(adapter);
   
   vmk_UplinkQueueStart(adapter->uplink, sqd->qid);

   QFLE3_DBG(QFLE3_DBG_QUEUE, "active Tx queue Count %d", sd->queueInfo->activeTxQueues);
   
   QFLE3_DBG(QFLE3_DBG_QUEUE,"TX queue %d successfully started", id);
   tq->soft_state.tq_stopped_by_tx = VMK_FALSE;
   tq->fp_state = (QFLE3_SM_Q_STARTED);
   return VMK_OK;
setup_fail:

   if (tq->netpoll) {
      if (!tq->pollDisabled) {
         vmk_NetPollDisable(tq->netpoll);
         vmk_NetPollFlushRx(tq->netpoll);
         netpoll_count--;
         tq->pollDisabled = VMK_TRUE;
      }

      status = vmk_NetPollInterruptUnSet(tq->netpoll);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to unset associated interrupt cookie #%d 0x%x"
            "with fp[%d] netpoll", offset, cookie, tq->qid);
      }
   }
pollintr_fail:
   if (adapter->intr.intrType == QFLE3_IT_MSIX){
      status = vmk_IntrDisable(cookie);
   }
intr_fail:

   obj = &QFLE3_SP_OBJ(adapter, tq).mac_obj;
   qfle3_destroy_spinlock(obj->exe_queue.lock);
   obj->exe_queue.lock = NULL;
   obj = &QFLE3_SP_OBJ(adapter, tq).vxlan_filter_obj;
   qfle3_destroy_spinlock(obj->exe_queue.lock);
   obj->exe_queue.lock = NULL;
   return VMK_FAILURE;
}


VMK_ReturnStatus
qfle3_tq_resume(qfle3_adapter *adapter, vmk_uint32 id)
{
   vmk_UplinkSharedQueueData *sqd;
   vmk_UplinkSharedData *sd;
   struct qfle3_fastpath *tq;


   sd = &adapter->uplinkSharedData;
   sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
   sqd += id;
   tq = QFLE3_GET_FP_FROM_TQID(adapter,id);

   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_tq_resume %d state %d  ", id, tq->fp_state);

   if (!(sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE)) {
      QFLE3_ERR("TX queue %d not allocated", id);
      return VMK_BAD_PARAM;
   }
   
   if ( (tq->fp_state != QFLE3_SM_Q_STARTED) ) {
      QFLE3_ERR("Tring to RESUME TQ in wrong state %d, idx %d", tq->fp_state, id);
      return VMK_BAD_PARAM;
   }
   
   if (sqd->state == VMK_UPLINK_QUEUE_STATE_STARTED) {
      QFLE3_DBG(QFLE3_DBG_QUEUE, "TX queue %d already resumed", id);
      return VMK_OK;
   }
   

   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   sqd->state = VMK_UPLINK_QUEUE_STATE_STARTED;
   sd->queueInfo->activeTxQueues++;
   vmk_BitVectorSet(sd->queueInfo->activeQueues, Qfle3GetQIDValFromSQD(adapter, sqd));
   QFLE3_SHARED_DATA_WRITE_END(adapter);
   QFLE3_DBG(QFLE3_DBG_QUEUE, "active Tx queue Count %d", sd->queueInfo->activeTxQueues);
   /*
    * Notify uplink layer
    */
   vmk_UplinkQueueStart(adapter->uplink, sqd->qid);
   QFLE3_DBG(QFLE3_DBG_QUEUE,"TX queue %d successfully resumed", id);
   tq->soft_state.tq_stopped_by_tx = VMK_FALSE;
   return VMK_OK;
}



VMK_ReturnStatus
qfle3_rq_start(qfle3_adapter * adapter, vmk_uint32 id)
{
   vmk_uint32 i, num_rss;
   VMK_ReturnStatus status = VMK_OK;
   struct qfle3_fastpath *rq;
   vmk_uint8 defq_rss = 0;
   vmk_uint8 fp_idx;
   vmk_uint32 offset;
   vmk_IntrCookie cookie;
   vmk_UplinkSharedQueueData *sqd = NULL;
   vmk_UplinkSharedData *sd = NULL;
#if (ESX_DDK_VERSION >= 2017)
   struct qfle3_fastpath *rxq;
   vmk_uint32 k=0;
#else
   vmk_uint8 rss_engine_id=0;
#endif

   rq = &adapter->fp[id];
#if (ESX_DDK_VERSION >= 2017)
   rxq = rq;
#endif
   sd = &adapter->uplinkSharedData;
   sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
   sqd += id;

   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_rq_start %d", id);

   if (rq->fp_state == QFLE3_SM_Q_STARTED) {
      QFLE3_DBG(QFLE3_DBG_QUEUE, "RX queue %d Already Started", id);
      return VMK_OK;
   }   
   if (id < QFLE3_NUM_RX_VMK_ETH_QUEUES(adapter)) {
      sd = &adapter->uplinkSharedData;
      sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
      sqd += id;

      if (!(sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE)) {
         QFLE3_ERR("RX queue %d not allocated", id);
         return VMK_BAD_PARAM;
      }
      if (sqd->state == VMK_UPLINK_QUEUE_STATE_STARTED) {
         QFLE3_ERR("RX queue %d already started", id);
         return VMK_BAD_PARAM;
      }
   }
   if (rq->fp_state != QFLE3_SM_Q_CREATED) {
      QFLE3_ERR("RX queue %d not in CREATED state", id);
      return VMK_BAD_PARAM;
   }
   num_rss = rq->num_rss;
#if (ESX_DDK_VERSION >= 2017)
      if (Qfle3IsDefaultRxQueue(adapter, rq) && num_rss > 1) {
         defq_rss = 1;
      }
#else
   if (Qfle3IsDefaultRxQueue(adapter, rq) && num_rss > 1) {
      defq_rss = 1;
      rss_engine_id = QFLE3_FUNC(adapter) + (2 * E2_FUNC_MAX);
      QFLE3_DBG(QFLE3_DBG_QUEUE, "This is Default queue with RSS and Engine ID is %d\n",
               rss_engine_id);
   } else if ((!Qfle3IsDefaultRxQueue(adapter, rq)) && num_rss > 1) {
      rss_engine_id = QFLE3_FUNC(adapter);
      QFLE3_DBG(QFLE3_DBG_QUEUE, "This is Netqueue with RSS and Engine ID is %d\n", rss_engine_id);
   }
#endif
   for (i = 0; i < num_rss; i++) {
      if (i) {
         if (defq_rss)
            fp_idx = adapter->num_rxqs_drv - i;
         else {
#if (ESX_DDK_VERSION >= 2017)
            fp_idx =  rxq->sec_quque_ids[i];
            QFLE3_DBG(QFLE3_DBG_QUEUE, "Q ID for the Secondary RSS Queue is %d\n", fp_idx);
#else
            fp_idx = adapter->num_rxqs_vmk - 1 + i;
            QFLE3_DBG(QFLE3_DBG_QUEUE, "Q ID for the Hidden RSS Queue is %d\n", fp_idx);
#endif
         }
         rq = &adapter->fp[fp_idx];
      }
      offset = 1 + CNIC_SUPPORT(adapter);
      offset += rq->qid;
      if (adapter->intr.intrType == QFLE3_IT_MSIX){
         cookie = adapter->intr.cookies[offset];
      } else {
         cookie = adapter->intr.cookies[0];
      }

      rq->rx_bd_prod = MAX_RX_AVAIL;
      rq->rx_cq_prod = MAX_RCQ_AVAIL;

      status = qfle3_init_eth_fp(adapter, 0, rq);

      if (status != VMK_OK)
         return VMK_FAILURE;

      // make sure the status block counter is zeroed out so incase an interrupt is 
      // in-flight, it won't cause us problems when we re-enable interrupt
      *rq->rx_cq_cons_sb = htole16(0);

      QFLE3_DBG(QFLE3_DBG_QUEUE,
         "RX fp[%d]: wrote prods bd_prod=%d cqe_prod=%d sge_prod=%d\n",
         rq->qid, rq->rx_bd_prod, rq->rx_cq_prod, rq->rx_sge_prod);

      qfle3_update_rx_prod(adapter, rq, rq->rx_bd_prod, rq->rx_cq_prod,
         rq->rx_sge_prod);
#if (ESX_DDK_VERSION >= 2017)
      if (rq->num_rss > 1){
         if (defq_rss){
            ecore_init_rss_config_obj(adapter, &adapter->defq_rss_conf_obj, rxq->cl_id,
                             rxq->cid, QFLE3_FUNC(adapter),
                             rxq->rss_engine_id,
                             QFLE3_SP(adapter, defq_rss_rdata),
                             QFLE3_SP_MAPPING(adapter, defq_rss_rdata),
                             ECORE_FILTER_RSS_CONF_PENDING, &adapter->sp_state,
                             ECORE_OBJ_TYPE_RX);
          } else {
            k = rxq->rss_engine_id - QFLE3_FUNC(adapter) * 2 * QFLE3_DEVICE_MAX_RSS_ENGINE;
            if(k>QFLE3_DEVICE_MAX_RSS_ENGINE){
               QFLE3_ERR("Trying to configure RSS beyond MAX Supported: %d\n", QFLE3_DEVICE_MAX_RSS_ENGINE);
               return VMK_FAILURE;
            }
            ecore_init_rss_config_obj(adapter, &adapter->netq_rss_conf_obj[k], rxq->cl_id,
                             rxq->cid, QFLE3_FUNC(adapter),
                             rxq->rss_engine_id,
                             QFLE3_SP(adapter, rss_rdata[k]),
                             QFLE3_SP_MAPPING(adapter, rss_rdata[k]),
                             ECORE_FILTER_RSS_CONF_PENDING, &adapter->sp_state,
                             ECORE_OBJ_TYPE_RX);
       }
    }
#endif
      // set netpoll interrupt
      if (rq->netpoll) {
         status = vmk_NetPollInterruptSet(rq->netpoll, cookie);
         if (status != VMK_OK) {
            QFLE3_ERR("Failed to set associated interrupt cookie #%d 0x%x"
               "with fp[%d] netpoll", offset, cookie, rq->qid);
            goto netpoll_interrupt_fail;
         }

         if (rq->pollDisabled) {
            vmk_NetPollEnable(rq->netpoll);
            netpoll_count++;
            rq->pollDisabled = VMK_FALSE;
            QFLE3_DBG(QFLE3_DBG_QUEUE, "enabled netpoll for q_index %d count %d\n", rq->qid, netpoll_count);
         }
      }
   
      if (adapter->intr.intrType == QFLE3_IT_MSIX){

         QFLE3_DBG(QFLE3_DBG_QUEUE, "Enabling interrupt on vector # %d", offset);
         status = vmk_IntrEnable(cookie);
         if (status != VMK_OK) {
            QFLE3_ERR("Failed to enable intrCookie[%d] 0x%x (%x)", offset, cookie, status);
            goto interrupt_fail;
         }
      }
#if (ESX_DDK_VERSION >= 2017)
      rq->rss_engine_id = rxq->rss_engine_id;
#else
      rq->rss_engine_id = rss_engine_id;
#endif
      
      status = qfle3_setup_queue(adapter, rq, rq->is_leading_rss);
      if (status != VMK_OK) {
         QFLE3_ERR("RX queue:%d failed start.", id);
         //unwind and handle errors in case of RSS.
         goto setup_fail;
      }

      QFLE3_STATS_LOCK(adapter);
      rq->collect_stats = 1;
      QFLE3_STATS_UNLOCK(adapter);

      QFLE3_DBG(QFLE3_DBG_QUEUE, "RX queue setup_queue successful for %d ", rq->qid);
      if (num_rss > 1) {
         if (defq_rss)
            adapter->defq_rss_idx_tbl[i] = rq->qid;
         else {
#if (ESX_DDK_VERSION >= 2017)
            QFLE3_DBG(QFLE3_DBG_QUEUE, "Value of k is %d\n", k);
            adapter->netq_rss_idx_tbl[k][i] = rq->qid;   //Where j is a number from 1 to 3
                                                         //j = rxq->rss_engine_id - (QFLE3_FUNC(adapter) * 2 * QFLE3_DEVICE_MAX_RSS_ENGINE);
                                                         //WE will need to evaluate j from engine id while doing other operations on the queue
#else
            adapter->netq_rss_idx_tbl[i] = rq->qid;
#endif
        }
      }
   }

   if (num_rss > 1) {
         if (defq_rss)
            status = qfle3_config_rss(adapter,
                                      adapter->defq_rss_raw_ind_tbl,
                                      adapter->defq_rss_key_tbl);
         else {
#if (ESX_DDK_VERSION >= 2017)
            status = qfle3_config_rss(adapter,
                                      adapter->netq_rss_raw_ind_tbl[k],
                                      adapter->netq_rss_key_tbl[k]);
#else
            status = qfle3_config_rss(adapter,
                                      adapter->netq_rss_raw_ind_tbl,
                                      adapter->netq_rss_key_tbl);
#endif
      }
      if (status != VMK_OK) {
         QFLE3_ERR("RX queue %d failed RSS start", id);
//         return VMK_FAILURE;
      }
   }
   

   sd = &adapter->uplinkSharedData;
   sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
   sqd += id;
   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   sqd->state = VMK_UPLINK_QUEUE_STATE_STARTED;
   sd->queueInfo->activeRxQueues++;
   vmk_BitVectorSet(sd->queueInfo->activeQueues, Qfle3GetQIDValFromSQD(adapter, sqd));
   QFLE3_SHARED_DATA_WRITE_END(adapter);

   QFLE3_DBG(QFLE3_DBG_QUEUE, "active Rx queue Count %d", sd->queueInfo->activeRxQueues);
   QFLE3_DBG(QFLE3_DBG_QUEUE,"RX queue %d successfully started", id);
   rq = &adapter->fp[id];
   rq->fp_state = (QFLE3_SM_Q_STARTED);
   return VMK_OK;
setup_fail:
   if (adapter->intr.intrType == QFLE3_IT_MSIX){
      status = vmk_IntrDisable(cookie);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to Disable intrCookie[%d] 0x%x (%x)", offset, cookie, status);
      }
   }
interrupt_fail:
   if (rq->netpoll) {
      if (!rq->pollDisabled) {
         vmk_NetPollDisable(rq->netpoll);
         
         vmk_NetPollFlushRx(rq->netpoll);
         netpoll_count--;
         rq->pollDisabled = VMK_TRUE;
      }
      status = vmk_NetPollInterruptUnSet(rq->netpoll);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to unset associated interrupt cookie #%d 0x%x"
            "with fp[%d] netpoll", offset, cookie, rq->qid);
      }
   }
netpoll_interrupt_fail:
   qfle3_destroy_spinlock(QFLE3_SP_OBJ(adapter, rq).mac_obj.exe_queue.lock);
   QFLE3_SP_OBJ(adapter, rq).mac_obj.exe_queue.lock = NULL;
   qfle3_destroy_spinlock(QFLE3_SP_OBJ(adapter, rq).vxlan_filter_obj.exe_queue.lock);
   QFLE3_SP_OBJ(adapter, rq).vxlan_filter_obj.exe_queue.lock = NULL;
   return VMK_FAILURE;
}

VMK_ReturnStatus qfle3_rx_queue_start(qfle3_adapter *adapter, vmk_UplinkQueueID qID)
{
   vmk_UplinkSharedQueueData *sqd = NULL;
   vmk_UplinkSharedData *sd = NULL;
   vmk_uint32 id;
   struct qfle3_fastpath *rq;

   id = vmk_UplinkQueueIDVal(qID);
   
   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_rx_queue_start, QueueIDVal:%d", id);

   if (id < QFLE3_NUM_RX_VMK_ETH_QUEUES(adapter)) {
      sd = &adapter->uplinkSharedData;
      sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
      sqd += id;

      if (!(sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE)) {
         QFLE3_ERR("Uplink RX queue %d not allocated", id);
         return VMK_BAD_PARAM;
      }

      if (sqd->state == VMK_UPLINK_QUEUE_STATE_STARTED) {
         QFLE3_ERR("Uplink RX queue %d already started ", id);
         return VMK_OK;
      }
   } else {
      QFLE3_ERR("id %d out of range", id);
      return VMK_BAD_PARAM;
   }
   if (qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_START_Q, id, 
      QFLE3_SM_PARAM_RXQ) == QFLE3_SMCMD_STATUS_COMPLETED){

      rq = QFLE3_GET_FP_FROM_RQID(adapter, id);
      rq->soft_state.q_started_by_vmk = VMK_TRUE;

      return VMK_OK;
   }else
      return VMK_FAILURE;
}

VMK_ReturnStatus qfle3_tx_queue_start(qfle3_adapter *adapter, vmk_UplinkQueueID qID)
{
   vmk_UplinkSharedQueueData *sqd = NULL;
   vmk_UplinkSharedData *sd = NULL;
   vmk_uint32 id;
   struct qfle3_fastpath *tq;

   id = vmk_UplinkQueueIDVal(qID);

   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_tx_queue_start, QueueIDVal:%d", id);
   if (id < QFLE3_NUM_TX_VMK_ETH_QUEUES(adapter)) {
      sd = &adapter->uplinkSharedData;
      sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
      sqd += id;

      if (!(sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE)) {
         QFLE3_ERR("Uplink TX queue %d not allocated", id);
         return VMK_BAD_PARAM;
      }

      if (sqd->state == VMK_UPLINK_QUEUE_STATE_STARTED) {
         QFLE3_ERR("Uplink TX queue %d already started ", id);
         return VMK_OK;
      }

   } else {
      QFLE3_ERR("id %d out of range", id);
      return VMK_BAD_PARAM;
   }

   if (qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_START_Q, id, 
      QFLE3_SM_PARAM_TXQ) == QFLE3_SMCMD_STATUS_COMPLETED){
   
      tq = QFLE3_GET_FP_FROM_TQID(adapter, id);
      tq->soft_state.q_started_by_vmk = VMK_TRUE;
      tq->soft_state.tq_stopped_by_tx = VMK_FALSE;
      return VMK_OK;
   }else
      return VMK_FAILURE;
}

static VMK_ReturnStatus 
qfle3_queue_start(vmk_AddrCookie driverData, vmk_UplinkQueueID qID)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;

   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.queueStart\n");
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_DBG(QFLE3_DBG_QUEUE, "Uplink is Resetting, Rejecting queue start");
      return VMK_FAILURE;
   }

   if (vmk_UplinkQueueIDType(qID) == VMK_UPLINK_QUEUE_TYPE_RX) {
      
      QFLE3_DBG(QFLE3_DBG_QUEUE, "RxQ, QueueIDVal:%d", vmk_UplinkQueueIDVal(qID));
      return qfle3_rx_queue_start(adapter, qID);
      
   } else if (vmk_UplinkQueueIDType(qID) == VMK_UPLINK_QUEUE_TYPE_TX) {
   
      QFLE3_DBG(QFLE3_DBG_QUEUE, "TxQ, QueueIDVal:%d", vmk_UplinkQueueIDVal(qID));
      return qfle3_tx_queue_start(adapter, qID);
   }
   QFLE3_ERR("Wrong QueueIDType(%x), qid(%u)", vmk_UplinkQueueIDType(qID), vmk_UplinkQueueIDVal(qID));
   return VMK_BAD_PARAM;

}
VMK_ReturnStatus qfle3_rx_queue_stop(qfle3_adapter *adapter, vmk_UplinkQueueID qID)
{
   vmk_UplinkSharedQueueData *sqd = NULL;
   vmk_UplinkSharedData *sd = NULL;
   vmk_uint32 id;
   struct qfle3_fastpath *rq;

   id = vmk_UplinkQueueIDVal(qID);

   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_rx_queue_stop, QueueIDVal:%d", id);

   if (id < QFLE3_NUM_RX_VMK_ETH_QUEUES(adapter)) {
      sd = &adapter->uplinkSharedData;
      sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
      sqd += id;

      if (!(sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE)) {
         QFLE3_ERR("Uplink RX queue %d not allocated", id);
         return VMK_BAD_PARAM;
      }

      if (sqd->state != VMK_UPLINK_QUEUE_STATE_STARTED) {
         QFLE3_ERR("Uplink RX queue %d already Stopped ", id);
         return VMK_OK;
      }
   } else {
      QFLE3_ERR("id %d out of range", id);
   return VMK_BAD_PARAM;
   }
   if (qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_STOP_Q, id,
      QFLE3_SM_PARAM_RXQ) == QFLE3_SMCMD_STATUS_COMPLETED){

      rq  = QFLE3_GET_FP_FROM_RQID(adapter,id);
      rq->soft_state.q_started_by_vmk = VMK_FALSE;

      return VMK_OK;
   }else
      return VMK_FAILURE;
}

VMK_ReturnStatus qfle3_tx_queue_stop(qfle3_adapter *adapter, vmk_UplinkQueueID qID)
{
   vmk_UplinkSharedQueueData *sqd = NULL;
   vmk_UplinkSharedData *sd = NULL;
   vmk_uint32 id;
   struct qfle3_fastpath *tq;

   id = vmk_UplinkQueueIDVal(qID);
   QFLE3_DBG(QFLE3_DBG_QUEUE, "qfle3_tx_queue_stop, QueueIDVal:%d", id);

   if (id < QFLE3_NUM_TX_VMK_ETH_QUEUES(adapter)) {
      sd = &adapter->uplinkSharedData;
      sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
      sqd += id;

      if (!(sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE)) {
         QFLE3_ERR("Uplink TX queue %d not allocated", id);
         return VMK_BAD_PARAM;
      }

      if (sqd->state != VMK_UPLINK_QUEUE_STATE_STARTED) {
         QFLE3_ERR("Uplink TX queue %d already Stopped, try anyway! ", id);
      }
   } else {
      QFLE3_ERR("id %d out of range", id);
      return VMK_BAD_PARAM;
   }
   if (qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_STOP_Q, id,
      QFLE3_SM_PARAM_TXQ) == QFLE3_SMCMD_STATUS_COMPLETED){

      tq  = QFLE3_GET_FP_FROM_TQID(adapter,id);
      tq->soft_state.q_started_by_vmk = VMK_FALSE;
      tq->soft_state.tq_stopped_by_tx = VMK_FALSE;

      return VMK_OK;
   }else
      return VMK_FAILURE;
}

static void
qfle3_clear_rx_filter_entry(struct qfle3_fastpath *fp,
			    struct qfle3_rx_filter *rx_filter)
{
   if (rx_filter->flags & QFLE3_RX_FILTER_IS_MAC ) {
      vmk_Memset(rx_filter->mac, 0, VMK_ETH_ADDR_LENGTH);
      rx_filter->flags &= ~QFLE3_RX_FILTER_IS_MAC;
   } else if (rx_filter->flags & QFLE3_RX_FILTER_IS_VXLAN) {
      vmk_Memset(rx_filter->mac, 0, VMK_ETH_ADDR_LENGTH);
      vmk_Memset(rx_filter->inner_mac, 0, VMK_ETH_ADDR_LENGTH);
      rx_filter->vxlan_id = 0;
      rx_filter->flags &= ~QFLE3_RX_FILTER_IS_VXLAN;
   }
   rx_filter->flags &= ~QFLE3_RX_FILTER_ACTIVE;
   fp->num_rx_filters--;
}

static void
qfle3_add_rx_filter_entry(struct qfle3_fastpath *fp,
			  struct qfle3_rx_filter *rx_filter,
			  vmk_UplinkQueueFilter *vmk_filter)
{
   qfle3_adapter *adapter = fp->adapter;
   if (vmk_filter->class == VMK_UPLINK_QUEUE_FILTER_CLASS_MAC_ONLY)  {
      rx_filter->flags |= QFLE3_RX_FILTER_IS_MAC;
      vmk_Memcpy(rx_filter->mac,
                 vmk_filter->macFilterInfo->mac, VMK_ETH_ADDR_LENGTH);
   } else if (vmk_filter->class == VMK_UPLINK_QUEUE_FILTER_CLASS_VXLAN)  {
      rx_filter->flags |= QFLE3_RX_FILTER_IS_VXLAN;
      vmk_Memcpy(rx_filter->mac,
                 vmk_filter->vxlanFilterInfo->outerMAC, VMK_ETH_ADDR_LENGTH);
      vmk_Memcpy(rx_filter->inner_mac,
                 vmk_filter->vxlanFilterInfo->innerMAC, VMK_ETH_ADDR_LENGTH);
      rx_filter->vxlan_id = vmk_filter->vxlanFilterInfo->vxlanID;
   } else {
      QFLE3_ERR("Unkown filter type = %d", vmk_filter->class);
      return;
   }

   rx_filter->flags |= QFLE3_RX_FILTER_ACTIVE;
   fp->num_rx_filters++;
}

static vmk_Bool
qfle3_is_active_rx_filter_entry(struct qfle3_fastpath *fp,
	  struct qfle3_rx_filter *rx_filter,
	  vmk_UplinkQueueFilter *vmk_filter)
{
   if ((vmk_filter->class == VMK_UPLINK_QUEUE_FILTER_CLASS_MAC_ONLY) &&
      (rx_filter->flags & QFLE3_RX_FILTER_IS_MAC) &&
      !vmk_Memcmp(rx_filter->mac, vmk_filter->macFilterInfo->mac, VMK_ETH_ADDR_LENGTH)) {
         return VMK_TRUE;
   } else if ((vmk_filter->class == VMK_UPLINK_QUEUE_FILTER_CLASS_VXLAN) &&
      (rx_filter->flags & QFLE3_RX_FILTER_IS_VXLAN) &&
      (!vmk_Memcmp(rx_filter->mac, vmk_filter->vxlanFilterInfo->outerMAC, VMK_ETH_ADDR_LENGTH)) &&
      (!vmk_Memcmp(rx_filter->inner_mac, vmk_filter->vxlanFilterInfo->innerMAC, VMK_ETH_ADDR_LENGTH)) &&
      (rx_filter->vxlan_id == vmk_filter->vxlanFilterInfo->vxlanID)) {
         return VMK_TRUE;
   } else
      return VMK_FALSE;
}

int
qfle3_apply_queue_mac_filter(qfle3_adapter * adapter,
			     vmk_UplinkQueueID qid,
			     vmk_UplinkQueueFilter * filter,
			     vmk_UplinkQueueFilterID * fid, vmk_uint32 * pairHwQid)
{
   vmk_uint16 i;
   vmk_int32 rc;
   struct qfle3_fastpath *fp;
   vmk_uint8 *macaddr = NULL;
   vmk_uint16 filter_id = QFLE3_INVALID_FILTER_ID;
   unsigned long ramrod_flags = 0;
   unsigned long accept_flags = 0;
   vmk_uint32 fp_id = vmk_UplinkQueueIDVal(qid);
   struct ecore_vlan_mac_obj *obj;

   fp = &adapter->fp[fp_id];

   if (fp->fp_state != QFLE3_SM_Q_STARTED)
      return VMK_FAILURE;

   for (i = 0; i < adapter->filters_per_netq; i++) {
      if (QFLE3_IS_FILTER_ACTIVE(fp, i)) {
         if (qfle3_is_active_rx_filter_entry(fp, &fp->rx_filters[i], filter)) {
            QFLE3_INFO("NetQ apply duplicated RX filter for queue %d", fp_id);
            return VMK_FAILURE;
         }
      }
   }

   macaddr = (vmk_uint8 *)filter->macFilterInfo->mac;

   for (i = 0; i < adapter->filters_per_netq; i++) {
      if (!QFLE3_IS_FILTER_ACTIVE(fp, i)) {
         qfle3_add_rx_filter_entry(fp, &fp->rx_filters[i], filter);
         if (!QFLE3_IS_FILTER_ACTIVE(fp, i))
            return VMK_FAILURE;

         filter_id = i;
         break;
      }
   }

   if (VMK_UNLIKELY(filter_id == QFLE3_INVALID_FILTER_ID)) {
      if (fp_id) {
         QFLE3_ERR("NetQ could not add RX filter, no filters for queue %d\n", fp_id);
         return VMK_FAILURE;
      } else {
         QFLE3_DBG(QFLE3_DBG_QUEUE, "No filters available for default queue. MAC:"
			QFLE3_MAC_FMT "\n", QFLE3_MAC_PRN_LIST(macaddr));
         *fid = QFLE3_INVALID_FILTER_ID;
         return VMK_OK;
      }
   }

   if (fp->num_rx_filters == 1) {
      /* set to recv-unicast */
      ECORE_SET_BIT(ECORE_ACCEPT_UNICAST, &accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_ANY_VLAN, &accept_flags);
      ECORE_SET_BIT(RAMROD_RX, &ramrod_flags);
      if (adapter->flags & TX_SWITCHING)
         ECORE_SET_BIT(RAMROD_TX, &ramrod_flags);
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);

      if (fp_id == 0) {
         ECORE_SET_BIT(ECORE_ACCEPT_UNMATCHED, &accept_flags);
         ECORE_SET_BIT(ECORE_ACCEPT_ALL_MULTICAST, &accept_flags);
         ECORE_SET_BIT(ECORE_ACCEPT_BROADCAST, &accept_flags);
      }

      rc = qfle3_set_q_rx_mode(adapter, fp->cl_id, 0, accept_flags,
			       accept_flags, ramrod_flags);

      if (rc) {
         QFLE3_ERR("NetQ could not add RX filter, "
		   "rx mode failed: queue:%d mac: " QFLE3_MAC_FMT
                   "filter id:0x%x\n",
                   fp_id, QFLE3_MAC_PRN_LIST(macaddr),
                   filter_id);
         qfle3_clear_rx_filter_entry(fp, &fp->rx_filters[filter_id]);
         /* Always return failure, if Rx mode is not set. */
         return VMK_FAILURE;
      }
   }

   /* add MAC */
   ramrod_flags = 0;
   ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);
   obj = &QFLE3_SP_OBJ(adapter, fp).mac_obj;
#ifdef QFLE3_SRIOV
   if(!adapter->vfdb)
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated yet\n");
   else
      if(fp_id == 0){
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Executing qfle3_set_mac_one for" QFLE3_MAC_FMT " on fp_id 0 SRIOV is enabled\n", QFLE3_MAC_PRN_LIST(macaddr));
         qfle3_set_mac_one(adapter, macaddr, obj, VMK_TRUE, ECORE_ETH_MAC, &ramrod_flags);
   }
#endif

   rc = qfle3_set_mac_one(adapter, macaddr, obj, VMK_TRUE, ECORE_NETQ_ETH_MAC, &ramrod_flags);

   if (VMK_UNLIKELY(rc)) {
      QFLE3_ERR("NetQ could not add RX filter: queue:%d"
                " mac: " QFLE3_MAC_FMT
                "filter id:%d\n",
                fp_id,
                QFLE3_MAC_PRN_LIST(macaddr),
		filter_id);
      qfle3_clear_rx_filter_entry(fp, &fp->rx_filters[filter_id]);

      if (fp_id)
         return VMK_FAILURE;
      else {
         /* Ignore MAC push failure on defualt queue. */
         filter_id = QFLE3_INVALID_FILTER_ID;
      }
   }

   *fid = filter_id;

   QFLE3_DBG(QFLE3_DBG_QUEUE,"NetQ set RX filter: queue:%d mac: " QFLE3_MAC_FMT
      " filter id:%d\n",
      fp_id, QFLE3_MAC_PRN_LIST(macaddr), filter_id);

   return VMK_OK;
}

int
qfle3_apply_queue_vxlan_filter(qfle3_adapter * adapter,
			       vmk_UplinkQueueID qid,
			       vmk_UplinkQueueFilter * filter,
			       vmk_UplinkQueueFilterID * fid, vmk_uint32 * pairHwQid)
{
   vmk_uint16 i;
   vmk_int32 rc;
   struct qfle3_fastpath *fp;
   vmk_uint8 *outer_mac = NULL;
   vmk_uint8 *inner_mac = NULL;
   vmk_uint32 vxlan_id = 0;
   vmk_uint16 filter_id = 0xFFFF;
   unsigned long ramrod_flags = 0;
   unsigned long accept_flags = 0;
   vmk_uint32 fp_id = vmk_UplinkQueueIDVal(qid);
   struct ecore_vlan_mac_obj *obj;

   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps>>qfle3_apply_queue_vxlan_filter\n");

   fp = &adapter->fp[fp_id];

   if (fp->fp_state != QFLE3_SM_Q_STARTED)
      return VMK_FAILURE;

   for (i = 0; i < adapter->filters_per_netq; i++) {
      if (QFLE3_IS_FILTER_ACTIVE(fp, i)) {
         if (qfle3_is_active_rx_filter_entry(fp, &fp->rx_filters[i], filter)) {
            QFLE3_INFO("NetQ apply duplicated RX filter for queue %d", fp_id);
      return VMK_FAILURE;
   }
      }
   }

   for (i = 0; i < adapter->filters_per_netq; i++) {
      if (!QFLE3_IS_FILTER_ACTIVE(fp, i)) {
         qfle3_add_rx_filter_entry(fp, &fp->rx_filters[i], filter);
         if (!QFLE3_IS_FILTER_ACTIVE(fp, i))
            return VMK_FAILURE;

         filter_id = i;
         break;
      }
   }

   if (VMK_UNLIKELY(filter_id == 0xFFFF)) {
      QFLE3_ERR("NetQ could not add filter, no filters left for queue %d", fp_id);
      return VMK_FAILURE;
   }

   if (fp->num_rx_filters == 1) {
      /* set to recv-unicast */
      ECORE_SET_BIT(ECORE_ACCEPT_UNICAST, &accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_ANY_VLAN, &accept_flags);
      ECORE_SET_BIT(RAMROD_RX, &ramrod_flags);
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);

      if (fp_id == 0) {
	      ECORE_SET_BIT(ECORE_ACCEPT_UNMATCHED, &accept_flags);
         ECORE_SET_BIT(ECORE_ACCEPT_ALL_MULTICAST, &accept_flags);
         ECORE_SET_BIT(ECORE_ACCEPT_BROADCAST, &accept_flags);
      }

      rc = qfle3_set_q_rx_mode(adapter, fp->cl_id, 0, accept_flags,
			       accept_flags, ramrod_flags);

      if (rc) {
         QFLE3_ERR("NetQ could not add RX filter, "
		   "rx mode failed: queue:%d " 
                   "filter id:%d\n",
		   fp_id, filter_id);
         qfle3_clear_rx_filter_entry(fp, &fp->rx_filters[filter_id]);
         return VMK_FAILURE;
      }
   }

   /* add MAC */
   ramrod_flags = 0;
   ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);
   obj = &QFLE3_SP_OBJ(adapter, fp).vxlan_filter_obj;

   outer_mac = fp->rx_filters[filter_id].mac;
   inner_mac = fp->rx_filters[filter_id].inner_mac;
   vxlan_id= fp->rx_filters[filter_id].vxlan_id;

   rc = qfle3_set_vxlan_fltr_one(adapter, outer_mac, obj,
                                 VMK_TRUE, ECORE_NETQ_ETH_MAC,
                                 &ramrod_flags, inner_mac, vxlan_id);

   if (VMK_UNLIKELY(rc)) {
      QFLE3_ERR("NetQ could not add RX filter: queue:%d"
	        "outer mac: " QFLE3_MAC_FMT
                " inner mac: " QFLE3_MAC_FMT " vxlan_id:%d  filter id:%d\n",
		fp_id,
		QFLE3_MAC_PRN_LIST(outer_mac),
                QFLE3_MAC_PRN_LIST(inner_mac),
                vxlan_id,
		filter_id);
      qfle3_clear_rx_filter_entry(fp, &fp->rx_filters[filter_id]);
      return VMK_FAILURE;
   }

   *fid = filter_id;
   
   QFLE3_DBG(QFLE3_DBG_QUEUE, "NetQ set VXLAN filter: queue:%d outer mac: " QFLE3_MAC_FMT
      " inner mac: " QFLE3_MAC_FMT " vxlan id:%d filter id:%d\n",
      fp_id, QFLE3_MAC_PRN_LIST(outer_mac),QFLE3_MAC_PRN_LIST(inner_mac), vxlan_id, filter_id);

   return VMK_OK;
}


static VMK_ReturnStatus
qfle3_queue_apply_filter(vmk_AddrCookie driverData,
      vmk_UplinkQueueID qID,
      vmk_UplinkQueueFilter * qFilter,
      vmk_UplinkQueueFilterID * fid, vmk_uint32 * pairHWQID)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   int status = VMK_OK;
   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.queueApplyFilter %d \n", vmk_UplinkQueueIDVal(qID));

   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_INFO("Uplink is Resetting, Rejecting queue apply filter");
      return VMK_FAILURE;
   }
   switch (qFilter->class) {
   case VMK_UPLINK_QUEUE_FILTER_CLASS_MAC_ONLY:
      status = qfle3_apply_queue_mac_filter(adapter, qID, qFilter, fid, pairHWQID);
      break;
   case VMK_UPLINK_QUEUE_FILTER_CLASS_VXLAN:
      status = qfle3_apply_queue_vxlan_filter(adapter, qID, qFilter, fid, pairHWQID);
      break;
   case VMK_UPLINK_QUEUE_FILTER_CLASS_VLAN_ONLY:
   case VMK_UPLINK_QUEUE_FILTER_CLASS_VLANMAC:
   case VMK_UPLINK_QUEUE_FILTER_CLASS_NONE:
   default:
      QFLE3_ERR("Unsupported netq_filter class: 0x%x", qFilter->class);
      status = VMK_FAILURE;
      break;
   }
   return status;

}

int
qfle3_remove_queue_filter(qfle3_adapter * adapter,
			  vmk_UplinkQueueID qid,
			  vmk_UplinkQueueFilterID fid)
{
   vmk_int32 rc;
   struct qfle3_fastpath *fp;
   vmk_uint8 *macaddr = NULL;
   vmk_uint8 *outer_mac = NULL;
   vmk_uint8 *inner_mac = NULL;
   vmk_uint32 vxlan_id = 0;
   vmk_uint16 filter_id = 0xFFFF;
   unsigned long ramrod_flags = 0;
   struct ecore_vlan_mac_obj *obj;
   vmk_uint32 fp_id = vmk_UplinkQueueIDVal(qid);
   struct ecore_queue_state_params qstate = {NULL};
   fp = &adapter->fp[fp_id];

   if (fp->fp_state != QFLE3_SM_Q_STARTED)
      return VMK_FAILURE;

   filter_id = fid;

   if (!fp_id && filter_id == QFLE3_INVALID_FILTER_ID) {
      /* This filter was not pushed to adapter. Return success. */
      QFLE3_INFO("Removing s/w only filter\n");
      return VMK_OK;
   }

   if (!(fp->rx_filters[filter_id].flags & QFLE3_RX_FILTER_ACTIVE))
      return VMK_OK;

   macaddr = fp->rx_filters[filter_id].mac;

   if(fp_id && (fp->num_rx_filters == 1)) {
      /* set to recv-unicast */
      ECORE_SET_BIT(RAMROD_RX, &ramrod_flags);
      //ECORE_SET_BIT(RAMROD_TX, &ramrod_flags);
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);

      rc = qfle3_set_q_rx_mode(adapter, fp->cl_id, 0, 0, 0, ramrod_flags);

      if (rc) {
         QFLE3_ERR("NetQ could not remove RX filter, "
		   "rx mode failed: queue:%d mac: " QFLE3_MAC_FMT
		   "filter id:%d\n",
		   fp_id, QFLE3_MAC_PRN_LIST(macaddr),
		   filter_id);
         qfle3_clear_rx_filter_entry(fp, &fp->rx_filters[filter_id]);
         return VMK_OK;
      }
   }

   ramrod_flags = 0;
   ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);

   if ((fp->rx_filters[filter_id].flags & QFLE3_RX_FILTER_IS_MAC)) {
      obj = &QFLE3_SP_OBJ(adapter, fp).mac_obj;
      rc = qfle3_set_mac_one(adapter, macaddr, obj, VMK_FALSE,
                             ECORE_NETQ_ETH_MAC, &ramrod_flags);

      if (VMK_UNLIKELY(rc)) {
         QFLE3_ERR("NetQ could not remove RX filter: queue:%d"
                   " mac: " QFLE3_MAC_FMT " filter id:%d\n",
                   fp_id,QFLE3_MAC_PRN_LIST(macaddr), filter_id);
         qfle3_clear_rx_filter_entry(fp, &fp->rx_filters[filter_id]);
         return VMK_OK;
      }else{
         QFLE3_DBG(QFLE3_DBG_QUEUE, "NetQ removed RX filter: queue:%d"
                   " mac: " QFLE3_MAC_FMT " filter id:%d\n",
                   fp_id,QFLE3_MAC_PRN_LIST(macaddr), filter_id);
      }

   } else if ((fp->rx_filters[filter_id].flags & QFLE3_RX_FILTER_IS_VXLAN)) {
      obj = &QFLE3_SP_OBJ(adapter, fp).vxlan_filter_obj;
      outer_mac = fp->rx_filters[filter_id].mac;
      inner_mac = fp->rx_filters[filter_id].inner_mac;
      vxlan_id= fp->rx_filters[filter_id].vxlan_id;
      rc = qfle3_set_vxlan_fltr_one(adapter, outer_mac, obj, VMK_FALSE,
                                    ECORE_NETQ_ETH_MAC, &ramrod_flags,
                                    inner_mac, vxlan_id);
      if (VMK_UNLIKELY(rc)) {
         QFLE3_ERR("NetQ could not remove RX filter: queue:%d"
                   " outer mac: " QFLE3_MAC_FMT " inner mac: " QFLE3_MAC_FMT
                   " filter id:%d\n", fp_id, QFLE3_MAC_PRN_LIST(outer_mac),
                   QFLE3_MAC_PRN_LIST(inner_mac), filter_id);
         qfle3_clear_rx_filter_entry(fp, &fp->rx_filters[filter_id]);
         return VMK_OK;
      }else{
         QFLE3_DBG(QFLE3_DBG_QUEUE,"NetQ removed RX filter: queue:%d"
                   " outer mac: " QFLE3_MAC_FMT " inner mac: " QFLE3_MAC_FMT
                   " filter id:%d\n", fp_id, QFLE3_MAC_PRN_LIST(outer_mac),
                   QFLE3_MAC_PRN_LIST(inner_mac), filter_id);
      }
   }

   if (fp_id) {
      /* send empty-ramrod to flush packets lurking in the HW */
      qstate.q_obj = &QFLE3_SP_OBJ(adapter, fp).q_obj;
      ramrod_flags = 0;
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);
      qstate.ramrod_flags = ramrod_flags; /* wait for completion */
      qstate.cmd = ECORE_Q_CMD_EMPTY;
      if (ecore_queue_state_change(adapter, &qstate)) {
	 QFLE3_ERR("RX %d queue state not changed for fid: %d\n",
		   fp_id, filter_id);
	 return VMK_FAILURE;
      }
   }

   qfle3_clear_rx_filter_entry(fp, &fp->rx_filters[filter_id]);

   return VMK_OK;
}

static VMK_ReturnStatus
qfle3_queue_remove_filter(vmk_AddrCookie driverData,
			  vmk_UplinkQueueID qID, vmk_UplinkQueueFilterID fid)
{
   VMK_ReturnStatus status;
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.queueRemoveFilter\n");
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_INFO("Uplink is Resetting, Rejecting queue remove filter");
      return VMK_FAILURE;
   }
   status = qfle3_remove_queue_filter(adapter, qID, fid);
   return status;
}

static VMK_ReturnStatus
qfle3_queue_get_stats(vmk_AddrCookie driverData,
		      vmk_UplinkQueueID qID, struct vmk_UplinkStats *stats)
{

   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   vmk_uint32 q_index = vmk_UplinkQueueIDVal(qID);
   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.queueGetStats q_index %d\n", q_index);

   vmk_Memset(stats, 0, sizeof(vmk_UplinkStats));

   if ((q_index >= adapter->uplinkQueueInfo.maxRxQueues) &&
       (q_index <
	(adapter->uplinkQueueInfo.maxRxQueues +
	 adapter->uplinkQueueInfo.maxTxQueues))) {
      struct qfle3_fastpath *txq = &adapter->fp[q_index];
      QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.queueGetStats Tx queue %d\n", q_index);
      stats->txPkts = txq->tx_stats.tx_pkts;
   }

   return VMK_OK;
}

static VMK_ReturnStatus
qfle3_queue_toggle_feature(vmk_AddrCookie driverData,
			   vmk_UplinkQueueID qID,
			   vmk_UplinkQueueFeature qFeature, vmk_Bool setUnset)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.queueToggleFeature\n");
   return VMK_NOT_SUPPORTED;

}

static VMK_ReturnStatus
qfle3_queue_set_priority(vmk_AddrCookie driverData,
			 vmk_UplinkQueueID qID, vmk_UplinkQueuePriority priority)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.queueSetPriority\n");
   return VMK_NOT_SUPPORTED;

}

static VMK_ReturnStatus
qfle3_queue_set_coalesce_params(vmk_AddrCookie driverData,
				vmk_UplinkQueueID qID, vmk_UplinkCoalesceParams * params)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   QFLE3_DBG(QFLE3_DBG_QUEUE, "QueueOps.queueSetCoalesceParams\n");
   return VMK_NOT_SUPPORTED;

}
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
vmk_UplinkMultiQueueOps qfle3_multiqueue_ops = {
   .queueOps = {
      .queueAlloc = qfle3_queue_alloc,
      .queueAllocWithAttr = qfle3_queue_alloc_with_attr,
      .queueReallocWithAttr = qfle3_queue_realloc_with_attr,
      .queueFree = qfle3_queue_free,
      .queueQuiesce = qfle3_queue_quiesce,
      .queueStart = qfle3_queue_start,
      .queueApplyFilter = qfle3_queue_apply_filter,
      .queueRemoveFilter = qfle3_queue_remove_filter,
      .queueGetStats = qfle3_queue_get_stats,
      .queueToggleFeature = qfle3_queue_toggle_feature,
      .queueSetPriority = qfle3_queue_set_priority,
      .queueSetCoalesceParams = qfle3_queue_set_coalesce_params,
   },
   .queueSetCount = NULL,
};
#else
vmk_UplinkQueueOps qfle3_multiqueue_ops = {
   .queueAlloc = qfle3_queue_alloc,
   .queueAllocWithAttr = qfle3_queue_alloc_with_attr,
   .queueReallocWithAttr = qfle3_queue_realloc_with_attr,
   .queueFree = qfle3_queue_free,
   .queueQuiesce = qfle3_queue_quiesce,
   .queueStart = qfle3_queue_start,
   .queueApplyFilter = qfle3_queue_apply_filter,
   .queueRemoveFilter = qfle3_queue_remove_filter,
   .queueGetStats = qfle3_queue_get_stats,
   .queueToggleFeature = qfle3_queue_toggle_feature,
   .queueSetPriority = qfle3_queue_set_priority,
   .queueSetCoalesceParams = qfle3_queue_set_coalesce_params,
};

#endif
void
qfle3_tq_init(struct qfle3_fastpath *fp)
{
   qfle3_adapter *adapter = fp->adapter;
   vmk_uint8 cos;
   vmk_IOA busaddr;
   vmk_uint32 j;
   vmk_int32 tx_bd_num_pages = TX_BD_NUM_PAGES;
   vmk_int32 tx_bd_total = TX_BD_TOTAL;
   if (IS_FCOE_FP(fp)||IS_OOO_FP(fp)|| IS_FWD_FP(fp)){
      tx_bd_num_pages = FCOE_NUM_TX_PAGES_DEF;
      tx_bd_total = TX_BD_TOTAL_PER_PAGE * FCOE_NUM_TX_PAGES_DEF;
   }
   
   for_each_cos_in_tx_queue(fp, cos) {
      struct qfle3_fp_txdata * txdata = fp->txdata_ptr[cos];
      QFLE3_DBG(QFLE3_DBG_QUEUE, "qid %d, cos %d, txbuf_chain %p", fp->qid, cos, txdata->txbuf_chain);
      vmk_Memset(txdata->txbuf_chain, 0, tx_bd_total * sizeof(qfle3_txbuf_info));
      vmk_Memset(txdata->tx_chain, 0, (BCM_PAGE_SIZE * tx_bd_num_pages));
      /*
       * link together the tx bd chain pages
       */
      for (j = 1; j <= tx_bd_num_pages; j++) {
         /*
          * index into the tx bd chain array to last entry per page
          */
         struct eth_tx_next_bd *tx_next_bd =
            &txdata->tx_chain[TX_BD_TOTAL_PER_PAGE * j - 1].next_bd;
         /*
          * point to the next page and wrap from last page
          */
         busaddr = (txdata->tx_chain_ioa + (BCM_PAGE_SIZE * (j % tx_bd_num_pages)));
         tx_next_bd->addr_hi = htole32(U64_HI(busaddr));
         tx_next_bd->addr_lo = htole32(U64_LO(busaddr));
      }
      QFLE3_DBG(QFLE3_DBG_QUEUE, "qid %d, cos %d, cid %d, tx_cons_sb %p", fp->qid, cos, txdata->cid, txdata->tx_cons_sb);
      *txdata->tx_cons_sb = htole16(0);
      SET_FLAG(txdata->tx_db.data.header.header, DOORBELL_HDR_DB_TYPE, 1);
      txdata->tx_db.data.zero_fill1 = 0;
      txdata->tx_db.data.prod = 0;

      txdata->tx_pkt_prod = 0;
      txdata->tx_pkt_cons = 0;
      txdata->tx_bd_prod = 0;
      txdata->tx_bd_cons = 0;
      txdata->prev_tx_pkt_cons = 0;
      txdata->queue_stuck = 0;
   // fp->eth_q_stats.tx_pkts = 0;
}
}

void
qfle3_tq_init_all(qfle3_adapter * adapter)
{
   int i, fp_num;
   struct qfle3_fastpath *txq;
   for (i = 0; i < QFLE3_NUM_TX_ETH_QUEUES(adapter); i++) {
      fp_num = QFLE3_TX_Q_NUM_TO_FP_NUM(adapter, i);
      txq = &adapter->fp[fp_num];
//      if (txq->state)
	 qfle3_tq_init(&adapter->fp[fp_num]);
   }
}

void
qfle3_calculate_filters_count(qfle3_adapter * adapter)
{
   vmk_uint32 total_filters = 0;
   vmk_uint32 filter_cnt= 0;

   total_filters = adapter->macs_pool.check(&adapter->macs_pool);
   if (total_filters > 3)
      total_filters -= 3;
   else
      total_filters = 0;

   adapter->total_filters = total_filters;

   filter_cnt = total_filters / adapter->num_rxqs_vmk;

   if (qfle3_multi_rx_filters == 0) {
      filter_cnt = 1;
      QFLE3_DBG(QFLE3_DBG_LOAD,  "Multiple RX filters disabled, forced to "
		"use only 1 filter.\n");
   } else if (qfle3_multi_rx_filters > 0) {

      if (qfle3_multi_rx_filters > filter_cnt)
	 QFLE3_DBG(QFLE3_DBG_LOAD,  "Forced RX filter setting %d higher "
		   "than checked value %d\n",
		   qfle3_multi_rx_filters,
		   filter_cnt);
      filter_cnt = min(filter_cnt, qfle3_multi_rx_filters);
   }
   adapter->filters_per_netq = filter_cnt;
}

VMK_ReturnStatus
qfle3_init_shared_queue_info(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status;
   vmk_UplinkSharedQueueInfo *sqi;
   vmk_UplinkSharedQueueData *sqd;
   struct qfle3_fastpath *txq;
   struct qfle3_fastpath *rxq;
   vmk_ServiceAcctID serviceID;
   vmk_NetPollProperties pollProp;
   int i, tx_qid, rx_qid;

   adapter->uplinkSharedData.queueInfo = sqi = &adapter->uplinkQueueInfo;
   sqi->supportedQueueTypes = VMK_UPLINK_QUEUE_TYPE_TX | VMK_UPLINK_QUEUE_TYPE_RX;
   sqi->supportedRxQueueFilterClasses = VMK_UPLINK_QUEUE_FILTER_CLASS_MAC_ONLY;
   sqi->defaultRxQueueID = 0;
   sqi->defaultTxQueueID = 0;
   sqi->activeRxQueues = 0;
   sqi->activeTxQueues = 0;
   sqi->maxTxQueues = adapter->num_txqs_vmk;
#if (ESX_DDK_VERSION >= 2017)
   sqi->maxRxQueues = adapter->num_rxqs_drv;
   QFLE3_DBG(QFLE3_DBG_QUEUE, "sqi->maxRxQueues: %d\n", sqi->maxRxQueues);
#else
   sqi->maxRxQueues = adapter->num_rxqs_vmk;
#endif
   sqi->activeQueues = vmk_BitVectorAlloc(qfle3_mod_info.heapID,
					  sqi->maxTxQueues + sqi->maxRxQueues);
   if (sqi->activeQueues == NULL) {
      QFLE3_ERR("Failed to allocate bit vector");
      status = VMK_FAILURE;
      goto err;
   }

   if (adapter->vxlan_filters_en) {
     sqi->supportedRxQueueFilterClasses |=
         VMK_UPLINK_QUEUE_FILTER_CLASS_VXLAN;
   }

   vmk_BitVectorZap(sqi->activeQueues);
   /* Commenting due to Recovery Changes */
 //  qfle3_calculate_filters_count(adapter);

   sqi->maxTotalDeviceFilters = adapter->total_filters;

   sqi->queueData = adapter->uplinkQueueData;

   vmk_ServiceGetID(VMK_SERVICE_ACCT_NAME_NET, &serviceID);

   /*
    * Netpoll allocation:
    */
   sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
   rxq = &adapter->fp[0];
#if (ESX_DDK_VERSION >= 2017)
   for (rx_qid = 0; rx_qid < adapter->num_rxqs_vmk; rx_qid++, sqd++, rxq++) {
#else
   for (rx_qid = 0; rx_qid < sqi->maxRxQueues; rx_qid++, sqd++, rxq++) {
#endif
      rxq->sqd = sqd;
      sqd->flags = VMK_UPLINK_QUEUE_FLAG_UNUSED;
      sqd->type = VMK_UPLINK_QUEUE_TYPE_RX;
      sqd->qid = VMK_UPLINK_QUEUE_INVALID_QUEUEID;
      sqd->state = VMK_UPLINK_QUEUE_STATE_STOPPED;
      sqd->supportedFeatures = VMK_UPLINK_QUEUE_FEAT_NONE;
      sqd->activeFeatures = 0;

      /*  Default queue will get all the filters remaining after
	  non default queue allocation. */
      /*just tempary let it be*/
      //sqd->maxFilters = adapter->filters_per_netq;
      /* Hardcoding since qfle3_calculate_filters_count has been moved to be
         executed later and setting sqd->maxFilters to 0 would lead to crash in 
         ASSERT bora/vmkernel/net/netqueue_bal.c:1424  
         ASSERT((nQueues == 1) || nFiltersPerQueue); */
      sqd->maxFilters = 24;
      if (rx_qid) {
         if (adapter->lro_enable) {
          sqd->supportedFeatures |= VMK_UPLINK_QUEUE_FEAT_LRO;
         }
#if (ESX_DDK_VERSION >= 2017)
         if (adapter->num_rss_engines > 0) {
            sqd->supportedFeatures |= VMK_UPLINK_QUEUE_FEAT_RSS_GENERIC;
#else
         if (adapter->num_rssqs_nd > 1) {
          sqd->supportedFeatures |= VMK_UPLINK_QUEUE_FEAT_RSS;
          sqd->supportedFeatures |= VMK_UPLINK_QUEUE_FEAT_RSS_DYN;
#endif
         }
         if ((sqd->maxFilters > 1)) {
          sqd->supportedFeatures |= VMK_UPLINK_QUEUE_FEAT_DYNAMIC;
         }
      }

      sqd->activeFilters = 0;
      sqd->dmaEngine = adapter->dmaEngine_rxbuf;      //???
      sqd->priority = VMK_VLAN_PRIORITY_MINIMUM;
      vmk_Memset(&sqd->coalesceParams, 0, sizeof(vmk_UplinkCoalesceParams));

      if ((adapter->intr.intrType != QFLE3_IT_MSIX) && (rx_qid != 0)) {
         sqd->poll = NULL;
         continue;
      }
      pollProp.poll = qfle3_netpoll_rxcb;
      pollProp.priv.ptr = rxq;
      pollProp.deliveryCallback = NULL;
      pollProp.features = VMK_NETPOLL_NONE;
      status = vmk_NetPollCreate(&pollProp, serviceID, vmk_ModuleCurrentID, &sqd->poll);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to create netpoll for rq[%d] (%x)", rx_qid, status);
         goto err;
      }
      rxq->netpoll = sqd->poll;
      rxq->pollDisabled = VMK_TRUE;
      
      QFLE3_DBG(QFLE3_DBG_QUEUE, "netpoll created for rq[%d] (%x)", rx_qid, status);
   }

   //Create netpolls for non leading RSS queues including RSS queues
   //associated with both netqueue and default queue
   for (i = QFLE3_NUM_RX_VMK_ETH_QUEUES(adapter); 
      i < QFLE3_NUM_RX_ETH_QUEUES(adapter); i++) {
      rxq = &adapter->fp[QFLE3_RX_Q_NUM_TO_FP_NUM(adapter,i)];
      pollProp.poll =   qfle3_netpoll_rxcb;
      pollProp.priv.ptr = rxq;
      pollProp.deliveryCallback = NULL;
      pollProp.features = VMK_NETPOLL_NONE;
      status = vmk_NetPollCreate(&pollProp, serviceID, vmk_ModuleCurrentID, &rxq->netpoll);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to create netpoll for rss[%d] (%x)", i, status);
         goto err;
      }
      rxq->pollDisabled = VMK_TRUE;
      QFLE3_DBG(QFLE3_DBG_QUEUE, "netpoll created for non-visible rxq[%d] (%x)", i, status);
   }


   sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
   for (tx_qid = 0; tx_qid < sqi->maxTxQueues; tx_qid++, sqd++, txq++) {
      txq = QFLE3_GET_FP_FROM_TQID(adapter,tx_qid);
      txq->sqd = sqd;
      sqd->flags = VMK_UPLINK_QUEUE_FLAG_UNUSED;
      sqd->type = VMK_UPLINK_QUEUE_TYPE_TX;
      sqd->qid = VMK_UPLINK_QUEUE_INVALID_QUEUEID;
      sqd->state = VMK_UPLINK_QUEUE_STATE_STOPPED;
      sqd->supportedFeatures = VMK_UPLINK_QUEUE_FEAT_NONE;
      sqd->activeFeatures = 0;
      sqd->maxFilters = 0;
      sqd->activeFilters = 0;
      sqd->dmaEngine = adapter->dmaEngine;
      sqd->priority = VMK_VLAN_PRIORITY_MINIMUM;

      sqd->poll = NULL;
      vmk_Memset(&sqd->coalesceParams, 0, sizeof(vmk_UplinkCoalesceParams));

      if ((adapter->intr.intrType != QFLE3_IT_MSIX) && (tx_qid != 0)) {
         sqd->poll = NULL;
         continue;
      }
      pollProp.poll = qfle3_netpoll_txrxcb;
      pollProp.priv.ptr = txq;
      pollProp.deliveryCallback = NULL;
      pollProp.features = VMK_NETPOLL_NONE;
      status = vmk_NetPollCreate(&pollProp, serviceID, vmk_ModuleCurrentID, &sqd->poll);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to create netpoll for tq[%d] (%x)", tx_qid+QFLE3_NUM_RX_ETH_QUEUES(adapter), status);
         goto err;
      }

      QFLE3_DBG(QFLE3_DBG_QUEUE, "netpoll created for txq[%d] fp[%d](%x)", tx_qid, tx_qid + QFLE3_NUM_RX_ETH_QUEUES(adapter),status);
      txq->netpoll = sqd->poll;
      txq->pollDisabled = VMK_TRUE;
   }
   //Create netpolls for cnic queues;
   for (i = QFLE3_NUM_ETH_QUEUES(adapter); i < QFLE3_NUM_ETH_QUEUES(adapter)+ QFLE3_NUM_CNIC_QUEUES(adapter); i++) {
      txq = QFLE3_GET_FP_FROM_QID(adapter,i);
      pollProp.poll = qfle3_netpoll_txrxcb;
      pollProp.priv.ptr = txq;
      pollProp.deliveryCallback = NULL;
      pollProp.features = VMK_NETPOLL_NONE;
      status = vmk_NetPollCreate(&pollProp, serviceID, vmk_ModuleCurrentID, &txq->netpoll);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to create netpoll for cnicq[%d] (%x)", i, status);
         goto err;
      }
      txq->pollDisabled = VMK_TRUE;
      QFLE3_DBG(QFLE3_DBG_QUEUE|QFLE3_DBG_CNIC, "netpoll created for cnicq[%d] (%x)", i, status);
   }

   return VMK_OK;

  err:
   qfle3_destroy_shared_queue_info(adapter);
   return status;
}

void
qfle3_destroy_shared_queue_info(qfle3_adapter * adapter)
{
   int i;
   struct qfle3_fastpath *rxq;
#if (ESX_DDK_VERSION >= 2017)
   QFLE3_DBG(QFLE3_DBG_QUEUE, "freeing %d rqs and %d tqs\n",adapter->num_rxqs_vmk,adapter->uplinkQueueInfo.maxTxQueues);
#else
   QFLE3_DBG(QFLE3_DBG_QUEUE, "freeing %d rqs and %d tqs\n",adapter->uplinkQueueInfo.maxRxQueues,adapter->uplinkQueueInfo.maxTxQueues);
#endif
   for (i = 0; i < QFLE3_NUM_RX_VMK_ETH_QUEUES(adapter); i++) {
      rxq = &adapter->fp[i];
      if (rxq->netpoll) {
         vmk_NetPollDestroy(rxq->netpoll);
         rxq->netpoll = NULL;
         QFLE3_DBG(QFLE3_DBG_QUEUE, "destroy netpoll for fp index %d\n",i);
      }
   }

   for (i = QFLE3_NUM_RX_VMK_ETH_QUEUES(adapter); i < QFLE3_NUM_RX_ETH_QUEUES(adapter); i++) {
      rxq = &adapter->fp[QFLE3_RX_Q_NUM_TO_FP_NUM(adapter,i)];
      if (rxq->netpoll) {
         vmk_NetPollDestroy(rxq->netpoll);
         QFLE3_DBG(QFLE3_DBG_QUEUE, "destroy netpoll for RSS %d\n", rxq->qid);
         rxq->netpoll = NULL;
      }
   }

   for (i = 0; i < adapter->uplinkQueueInfo.maxTxQueues; i++) {
      
      struct qfle3_fastpath *txq = QFLE3_GET_FP_FROM_TQID(adapter,i);
      
      if (txq->netpoll) {
         vmk_NetPollDestroy(txq->sqd->poll);
         QFLE3_DBG(QFLE3_DBG_QUEUE, "destroy netpoll for TX Queue %d\n", txq->qid);
         txq->netpoll = NULL;
      }
   }
   if (adapter->uplinkQueueInfo.activeQueues) {
       vmk_BitVectorFree(qfle3_mod_info.heapID, adapter->uplinkQueueInfo.activeQueues);
       adapter->uplinkQueueInfo.activeQueues = NULL;
   }
   for (i = QFLE3_NUM_ETH_QUEUES(adapter); i < QFLE3_NUM_ETH_QUEUES(adapter)+ QFLE3_NUM_CNIC_QUEUES(adapter); i++) {
      struct qfle3_fastpath *cnicq = QFLE3_GET_FP_FROM_QID(adapter,i);

      if (cnicq->netpoll) {
         vmk_NetPollDestroy(cnicq->netpoll); 
         
         QFLE3_DBG(QFLE3_DBG_QUEUE, "destroy netpoll for cnic Queue %d\n", cnicq->qid);
         cnicq->netpoll = NULL;
      }
   }
}


vmk_Bool
qfle3_tq_stopped(qfle3_adapter * adapter, vmk_uint32 id)
{
   vmk_UplinkSharedQueueData *sqd;

   sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
   sqd += id;
   vmk_CPUMemFenceRead();
   return sqd->state == VMK_UPLINK_QUEUE_STATE_STOPPED;
}

void
qfle3_free_queue_mem(struct qfle3_fastpath *fp, int type)
{
   qfle3_adapter *adapter = fp->adapter;
   vmk_uint8 cos;
   /*
    * allocate DMA memory for each fastpath structure
    */

      /*******************/
      /*
       * FP STATUS BLOCK
       */
      /*******************/
   if (!fp->adapter)
      return;

   if(!IS_FCOE_FP(fp) && !IS_FWD_FP(fp) && !IS_OOO_FP(fp)) {
      if (fp->status_block.e2_sb) {
	 qfle3_free_dma_mapping(adapter,
				adapter->dmaEngineCoherent,
				fp->status_block.e2_sb,
				fp->status_block_ioa,
				sizeof(union qfle3_host_hc_status_block));
	 fp->status_block.e2_sb = NULL;
         fp->status_block_ioa = 0;

//         QFLE3_ERROR("Freeing status block for fp index %d\n", fp->qid);
      }
      }

   for_each_cos_in_tx_queue(fp, cos) {
      /******************/
      /*
       * FP TX BD CHAIN
       */
      /******************/
      if (fp->txdata_ptr[cos]->tx_chain) {
	 qfle3_free_dma_mapping(adapter,
				adapter->dmaEngineCoherent,
                             fp->txdata_ptr[cos]->tx_chain,
                             fp->txdata_ptr[cos]->tx_chain_ioa, (BCM_PAGE_SIZE * TX_BD_NUM_PAGES));
         fp->txdata_ptr[cos]->tx_chain = NULL;
         fp->txdata_ptr[cos]->tx_chain_ioa = 0;

//         QFLE3_ERROR("Freeing tx_chain for fp index %d\n", fp->qid);
   }

      if (fp->txdata_ptr[cos]->txbuf_chain ) {
         qfle3_heap_free(fp->txdata_ptr[cos]->txbuf_chain);
         fp->txdata_ptr[cos]->txbuf_chain = NULL;

//         QFLE3_ERROR("Freeing txbuf_chain for fp index %d\n", fp->qid);
      }
   }

   /******************/
   /*
    * FP RX BD CHAIN
    */
   /******************/

   if (fp->rx_chain) {
      qfle3_free_dma_mapping(adapter,
			     adapter->dmaEngineCoherent,
			     fp->rx_chain,
			     fp->rx_chain_ioa, (BCM_PAGE_SIZE * RX_BD_NUM_PAGES));
      fp->rx_chain = NULL;
      fp->rx_chain_ioa = 0;

//      QFLE3_ERROR("Freeing rx_chain for fp index %d\n", fp->qid);
   }

   if (fp->rxbuf_chain ) {
      qfle3_heap_free(fp->rxbuf_chain);
      fp->rxbuf_chain = NULL;

//      QFLE3_ERROR("Freeing rxbuf_chain for fp index %d\n", fp->qid);
   }


   /*******************/
   /*
    * FP RX RCQ CHAIN
    */
   /*******************/
   if (fp->rcq_chain) {
      qfle3_free_dma_mapping(adapter,
			     adapter->dmaEngineCoherent,
			     fp->rcq_chain,
			     fp->rcq_chain_ioa, (BCM_PAGE_SIZE * RCQ_NUM_PAGES));
      fp->rcq_chain = NULL;
      fp->rcq_chain_ioa = 0;
//      QFLE3_ERROR("Freeing rcq_chain for fp index %d\n", fp->qid);
   }

   /*******************/
   /*
    * FP RX SGE CHAIN
    */
   /*******************/
   if (fp->rx_sge_chain) {
      qfle3_free_dma_mapping(adapter,
			     adapter->dmaEngineCoherent,
			     fp->rx_sge_chain,
			     fp->rx_sge_chain_ioa,
			     (BCM_PAGE_SIZE * RX_SGE_NUM_PAGES));
      fp->rx_sge_chain = NULL;
      fp->rx_sge_chain_ioa = 0;

//      QFLE3_ERROR("Freeing rx_sge_chain for fp index %d\n", fp->qid);
   }
   
//   QFLE3_DBG(QFLE3_DBG_QUEUE, "freeing fp_lock for fp queue %d\n",fp->qid);
   qfle3_destroy_spinlock(fp->fp_lock);
   fp->fp_lock = NULL;
}

void
qfle3_free_txqs_mem(qfle3_adapter * adapter)
{
   vmk_uint32 i, qid;
   struct qfle3_fastpath *fp;
   
   FOREACH_TX_ETH_QUEUE(i,qid,fp)
      qfle3_free_queue_mem(fp, TXQ);
   }
}

void
qfle3_free_rxqs_mem(qfle3_adapter * adapter)
{
   vmk_uint32 i, qid;
   struct qfle3_fastpath *fp;
   
   FOREACH_RX_ETH_QUEUE(i,qid,fp)
      qfle3_free_queue_mem(fp, RXQ);
   }
}
VMK_ReturnStatus
qfle3_alloc_cnic_queue_mem(qfle3_adapter * adapter,struct qfle3_fastpath *fp)
{
   VMK_ReturnStatus status;
   char lockName[32];
   int j;
   vmk_IOA busaddr;
   struct qfle3_fp_txdata *txdata;

   /*************/
   /*
    * FASTPATHS
    */
   /*************/

   /*
    * allocate DMA memory for each fastpath structure
    */
   fp->adapter = adapter;

   vmk_Snprintf(lockName, sizeof(lockName) - 1,
		"cnic%dlck", fp->qid);
   status = qfle3_create_spinlock(lockName,
				  QFLE3_NAME_TO_STRING(adapter->pdev_name),
				  QFLE3_LOCK_RANK_LOW,
				  &fp->fp_lock);

   QFLE3_DBG(QFLE3_DBG_QUEUE, "creating fp_lock for fp queue %d\n",fp->qid);
   if (status != VMK_OK)
      goto failed;

   /*******************/
   /*
    * FP STATUS BLOCK
    */
   /*******************/
   
   if (IS_OOO_FP(fp) || IS_FWD_FP(fp)) {
      fp->status_block.e2_sb = adapter->cnic_sb.e2_sb;
      fp->status_block_ioa = adapter->cnic_sb_mapping;
   }
   
   if (!IS_FCOE_FP(fp))
   {
      QFLE3_FP(adapter, fp->qid, sb_index_values) =
         (__le16 *)fp->status_block.e2_sb->sb.index_values;
      QFLE3_FP(adapter, fp->qid, sb_running_index) =
         (__le16 *)fp->status_block.e2_sb->sb.running_index;
   }

   /*
    * FP RX RCQ CHAIN
    */
    
   fp->rcq_chain =
      qfle3_alloc_dma_mapping(adapter,
			      adapter->dmaEngineCoherent,
			      (BCM_PAGE_SIZE * FCOE_NUM_RX_PAGES_DEF * CQE_BD_REL), &fp->rcq_chain_ioa);

   //debug code
   
   vmk_Memset(fp->rcq_chain,0xff, (BCM_PAGE_SIZE * FCOE_NUM_RX_PAGES_DEF * CQE_BD_REL));

   if (!fp->rcq_chain) {
//      QFLE3_ERROR("Failed to alloc fastpath rcq chain\n");
      status = VMK_NO_MEMORY;
      goto failed;
   }
   
//   QFLE3_ERROR("Allocated fastpath rcq chain %ld pages\n", FCOE_NUM_RX_PAGES_DEF * CQE_BD_REL);
   fp->pollDisabled = VMK_TRUE;

   /*
    * link together the rcq chain pages
    */
   for (j = 1; j <= (FCOE_NUM_RX_PAGES_DEF * CQE_BD_REL); j++) {
      /*
       * index into the rcq chain array to last entry per page
       */
      struct eth_rx_cqe_next_page *rx_cqe_next = (struct eth_rx_cqe_next_page *)
            &fp->rcq_chain[RCQ_TOTAL_PER_PAGE * j - 1];
      /*
       * point to the next page and wrap from last page
       */
      busaddr = (fp->rcq_chain_ioa + (BCM_PAGE_SIZE * (j % (FCOE_NUM_RX_PAGES_DEF * CQE_BD_REL))));
      rx_cqe_next->addr_hi = htole32(U64_HI(busaddr));
      rx_cqe_next->addr_lo = htole32(U64_LO(busaddr));
   }
   /* allocate the bd rings */
   if (IS_FCOE_FP(fp) || IS_FWD_FP(fp)) {

      txdata = fp->txdata_ptr[FIRST_TX_COS_INDEX];
      txdata->tx_chain =
      qfle3_alloc_dma_mapping(adapter,
         adapter->dmaEngineCoherent,
         (BCM_PAGE_SIZE * FCOE_NUM_TX_PAGES_DEF), &txdata->tx_chain_ioa);

      if (!txdata->tx_chain) {
         QFLE3_ERROR("Failed to alloc fastpath tx bd chain\n");
         status = VMK_NO_MEMORY;
         goto failed;
      }
      QFLE3_DBG(QFLE3_DBG_RX,"Allocated fastpath tx bd  chain %u pages\n", FCOE_NUM_TX_PAGES_DEF);

      txdata->txbuf_chain = qfle3_heap_alloc(sizeof(qfle3_txbuf_info) * (TX_BD_TOTAL_PER_PAGE * FCOE_NUM_TX_PAGES_DEF));
      if (txdata->txbuf_chain == NULL) {
	      QFLE3_ERR("Failed to allocate txbuf_chain");
         status = VMK_NO_MEMORY;
         goto failed;
      }
      QFLE3_DBG(QFLE3_DBG_TX,"Allocated fastpath txbuf_chain %ld buffers\n", (TX_BD_TOTAL_PER_PAGE * FCOE_NUM_TX_PAGES_DEF));
      /*
       * link together the tx bd chain pages
       */
      for (j = 1; j <= FCOE_NUM_TX_PAGES_DEF; j++) {
   	 /*
   	  * index into the tx bd chain array to last entry per page
   	  */
         struct eth_tx_next_bd *tx_next_bd =
            &txdata->tx_chain[TX_BD_TOTAL_PER_PAGE * j - 1].next_bd;
         /*
          * point to the next page and wrap from last page
          */
         busaddr = (txdata->tx_chain_ioa + (BCM_PAGE_SIZE * (j % FCOE_NUM_TX_PAGES_DEF)));
         tx_next_bd->addr_hi = htole32(U64_HI(busaddr));
         tx_next_bd->addr_lo = htole32(U64_LO(busaddr));
      }

   }
   
   if (IS_FCOE_FP(fp) || IS_OOO_FP(fp)) {
      fp->rx_chain =
         qfle3_alloc_dma_mapping(adapter,
            adapter->dmaEngineCoherent,
            (BCM_PAGE_SIZE * FCOE_NUM_RX_PAGES_DEF), &fp->rx_chain_ioa);

      if (!fp->rx_chain) {
	      QFLE3_ERROR("Failed to alloc fastpath rx bd chain\n");
         status = VMK_NO_MEMORY;
         goto failed;
      }
      QFLE3_DBG(QFLE3_DBG_RX, "Allocated fastpath rx bd  chain %u pages\n", FCOE_NUM_RX_PAGES_DEF);

      fp->rxbuf_chain = qfle3_heap_alloc(sizeof(qfle3_rxbuf_info) * (RX_BD_TOTAL_PER_PAGE * FCOE_NUM_RX_PAGES_DEF));
      if (fp->rxbuf_chain == NULL) {
	      QFLE3_ERR("Failed to allocate rxbuf_chain");
         status = VMK_NO_MEMORY;
         goto failed;
      }
      QFLE3_DBG(QFLE3_DBG_RX,"Allocated fastpath rxbuf_chain  for %ld buffers\n", (RX_BD_TOTAL_PER_PAGE * FCOE_NUM_RX_PAGES_DEF));

      /*
       * link together the rx bd chain pages
       */
      for (j = 1; j <= FCOE_NUM_RX_PAGES_DEF; j++) {
         /*
          * index into the rx bd chain array to last entry per page
          */
         struct eth_rx_bd *rx_bd = &fp->rx_chain[RX_BD_TOTAL_PER_PAGE * j - 2];
         /*
          * point to the next page and wrap from last page
          */
         busaddr = (fp->rx_chain_ioa + (BCM_PAGE_SIZE * (j % FCOE_NUM_RX_PAGES_DEF)));
         rx_bd->addr_hi = htole32(U64_HI(busaddr));
         rx_bd->addr_lo = htole32(U64_LO(busaddr));
      }



      /*
       * FP RX SGE CHAIN
       */
      fp->rx_sge_chain =
         qfle3_alloc_dma_mapping(adapter,
				 adapter->dmaEngineCoherent,
				 (BCM_PAGE_SIZE * RX_SGE_NUM_PAGES),
				 &fp->rx_sge_chain_ioa);

      if (!fp->rx_sge_chain) {
         QFLE3_ERROR("Failed to alloc fastpath rx sge chain\n");
         status = VMK_NO_MEMORY;
         goto failed;
      }
//   QFLE3_DBG(QFLE3_DBG_RX,"Allocated fastpath rx sge chain %u pages\n", RX_SGE_NUM_PAGES);
      /*
       * link together the sge chain pages
       */
      for (j = 1; j <= RX_SGE_NUM_PAGES; j++) {
         /*
          * index into the rcq chain array to last entry per page
          */
         struct eth_rx_sge *rx_sge = &fp->rx_sge_chain[RX_SGE_TOTAL_PER_PAGE * j - 2];
         /*
          * point to the next page and wrap from last page
          */
         busaddr = (fp->rx_sge_chain_ioa + (BCM_PAGE_SIZE * (j % RX_SGE_NUM_PAGES)));
         rx_sge->addr_hi = htole32(U64_HI(busaddr));
         rx_sge->addr_lo = htole32(U64_LO(busaddr));
      }

      qfle3_init_sge_ring_bit_mask(fp);
   }

   return VMK_OK;

failed:
   return status;
}
void
qfle3_free_cnic_queue_mem(struct qfle3_fastpath *fp)
{
   qfle3_adapter *adapter = fp->adapter;
   vmk_uint8 cos;
   /*
    * allocate DMA memory for each fastpath structure
    */

      /*******************/
      /*
       * FP STATUS BLOCK
       */
      /*******************/
   if (!fp->adapter)
      return;


   for_each_cos_in_tx_queue(fp, cos) {
      /******************/
      /*
       * FP TX BD CHAIN
       */
      /******************/
      if (fp->txdata_ptr[cos]->tx_chain) {
	 qfle3_free_dma_mapping(adapter,
				adapter->dmaEngineCoherent,
                             fp->txdata_ptr[cos]->tx_chain,
                             fp->txdata_ptr[cos]->tx_chain_ioa, (BCM_PAGE_SIZE * FCOE_NUM_TX_PAGES_DEF));
         fp->txdata_ptr[cos]->tx_chain = NULL;
         fp->txdata_ptr[cos]->tx_chain_ioa = 0;

//         QFLE3_ERROR("Freeing tx_chain for fp index %d\n", fp->qid);
   }

      if (fp->txdata_ptr[cos]->txbuf_chain ) {
         qfle3_heap_free(fp->txdata_ptr[cos]->txbuf_chain);
         fp->txdata_ptr[cos]->txbuf_chain = NULL;

//         QFLE3_ERROR("Freeing txbuf_chain for fp index %d\n", fp->qid);
      }
   }

   /******************/
   /*
    * FP RX BD CHAIN
    */
   /******************/

   if (fp->rx_chain) {
      qfle3_free_dma_mapping(adapter,
			     adapter->dmaEngineCoherent,
			     fp->rx_chain,
			     fp->rx_chain_ioa, (BCM_PAGE_SIZE * FCOE_NUM_RX_PAGES_DEF));
      fp->rx_chain = NULL;
      fp->rx_chain_ioa = 0;

//      QFLE3_ERROR("Freeing rx_chain for fp index %d\n", fp->qid);
   }

   if (fp->rxbuf_chain ) {
      qfle3_heap_free(fp->rxbuf_chain);
      fp->rxbuf_chain = NULL;

//      QFLE3_ERROR("Freeing rxbuf_chain for fp index %d\n", fp->qid);
   }


   /*******************/
   /*
    * FP RX RCQ CHAIN
    */
   /*******************/
   if (fp->rcq_chain) {
      qfle3_free_dma_mapping(adapter,
			     adapter->dmaEngineCoherent,
			     fp->rcq_chain,
			     fp->rcq_chain_ioa, (BCM_PAGE_SIZE * (FCOE_NUM_RX_PAGES_DEF * CQE_BD_REL)));
      fp->rcq_chain = NULL;
      fp->rcq_chain_ioa = 0;
//      QFLE3_ERROR("Freeing rcq_chain for fp index %d\n", fp->qid);
   }

   /*******************/
   /*
    * FP RX SGE CHAIN
    */
   /*******************/
   if (fp->rx_sge_chain) {
      qfle3_free_dma_mapping(adapter,
			     adapter->dmaEngineCoherent,
			     fp->rx_sge_chain,
			     fp->rx_sge_chain_ioa,
			     (BCM_PAGE_SIZE * RX_SGE_NUM_PAGES));
      fp->rx_sge_chain = NULL;
      fp->rx_sge_chain_ioa = 0;

//      QFLE3_ERROR("Freeing rx_sge_chain for fp index %d\n", fp->qid);
   }

//   QFLE3_DBG(QFLE3_DBG_QUEUE, "freeing fp_lock for fp queue %d\n",fp->qid);
   qfle3_destroy_spinlock(fp->fp_lock);
   fp->fp_lock = NULL;
}


VMK_ReturnStatus
qfle3_alloc_queue_mem(qfle3_adapter * adapter,
		      int req_type, struct qfle3_fastpath *fp)
{
   VMK_ReturnStatus status;
   char lockName[32];
   int j;
   vmk_IOA busaddr;
   vmk_uint8 cos;
   struct qfle3_fp_txdata *txdata;

   /*************/
   /*
    * FASTPATHS
    */
   /*************/

   /*
    * allocate DMA memory for each fastpath structure
    */
   fp->adapter = adapter;

   vmk_Snprintf(lockName, sizeof(lockName) - 1,
		(req_type == ALLOC_TXQ)? "txq%dlck" :"rxq%dlck", fp->qid);
   status = qfle3_create_spinlock(lockName,
				  QFLE3_NAME_TO_STRING(adapter->pdev_name),
				  QFLE3_LOCK_RANK_LOW,
				  &fp->fp_lock);
   QFLE3_DBG(QFLE3_DBG_QUEUE, "creating fp_lock for fp queue %d\n",fp->qid);

   
   if (status != VMK_OK)
      goto failed;

   /*******************/
   /*
    * FP STATUS BLOCK
    */
   /*******************/
#ifdef QFLE3_CNIC
   if (IS_OOO_FP(fp) || IS_FWD_FP(fp)) {
      fp->status_block.e2_sb = adapter->cnic_sb.e2_sb;
      fp->status_block_ioa = adapter->cnic_sb_mapping;
   } else if (!IS_FCOE_FP(fp))
#endif
   {
      fp->status_block.e2_sb =
         qfle3_alloc_dma_mapping(adapter,
   			      adapter->dmaEngineCoherent,
   			      sizeof(struct host_hc_status_block_e2),
   			      &fp->status_block_ioa);

      if (!fp->status_block.e2_sb) {
         QFLE3_ERROR("Failed to alloc fastpath status block\n");
         status = VMK_NO_MEMORY;
         goto failed;
      }
   }
#ifdef QFLE3_CNIC
   if (!IS_FCOE_FP(fp))
#endif
   {
      QFLE3_FP(adapter, fp->qid, sb_index_values) =
         (__le16 *)fp->status_block.e2_sb->sb.index_values;
      QFLE3_FP(adapter, fp->qid, sb_running_index) =
         (__le16 *)fp->status_block.e2_sb->sb.running_index;
   }
   /*
    * FP RX RCQ CHAIN
    */
   fp->rcq_chain =
      qfle3_alloc_dma_mapping(adapter,
			      adapter->dmaEngineCoherent,
			      (BCM_PAGE_SIZE * RCQ_NUM_PAGES), &fp->rcq_chain_ioa);

   //debug code
   
   if (IS_FCOE_FP(fp))
      vmk_Memset(fp->rcq_chain,0xff, (BCM_PAGE_SIZE * RCQ_NUM_PAGES));

   if (!fp->rcq_chain) {
      QFLE3_ERROR("Failed to alloc fastpath rcq chain\n");
      status = VMK_NO_MEMORY;
      goto failed;
   }
   
//   QFLE3_ERROR("Allocated fastpath rcq chain %ld pages\n", RCQ_NUM_PAGES);
   fp->pollDisabled = VMK_TRUE;

   /*
    * link together the rcq chain pages
    */
   for (j = 1; j <= RCQ_NUM_PAGES; j++) {
      /*
       * index into the rcq chain array to last entry per page
       */
      struct eth_rx_cqe_next_page *rx_cqe_next = (struct eth_rx_cqe_next_page *)
	 &fp->rcq_chain[RCQ_TOTAL_PER_PAGE * j - 1];
      /*
       * point to the next page and wrap from last page
       */
      busaddr = (fp->rcq_chain_ioa + (BCM_PAGE_SIZE * (j % RCQ_NUM_PAGES)));
      rx_cqe_next->addr_hi = htole32(U64_HI(busaddr));
      rx_cqe_next->addr_lo = htole32(U64_LO(busaddr));
   }
   /* allocate the bd rings */

   if (req_type == ALLOC_TXQ) {
      for_each_cos_in_tx_queue(fp, cos){
         txdata = fp->txdata_ptr[cos];
         txdata->tx_chain = qfle3_alloc_dma_mapping(adapter,
				 adapter->dmaEngineCoherent,
        	      (BCM_PAGE_SIZE * TX_BD_NUM_PAGES), &txdata->tx_chain_ioa);

         if (!txdata->tx_chain) {
   	      QFLE3_ERROR("Failed to alloc fastpath tx bd chain %x\n", status);
         status = VMK_NO_MEMORY;
         goto failed;
      }
         QFLE3_DBG(QFLE3_DBG_SP,"Allocated fastpath tx bd  chain %u pages\n", TX_BD_NUM_PAGES);

         txdata->txbuf_chain = qfle3_heap_alloc(sizeof(qfle3_txbuf_info) * TX_BD_TOTAL);
         if (txdata->txbuf_chain == NULL) {
	      QFLE3_ERR("Failed to allocate txbuf_chain");
         status = VMK_NO_MEMORY;
         goto failed;
      }
         QFLE3_DBG(QFLE3_DBG_SP,"Allocated fastpath txbuf_chain %ld buffers\n", TX_BD_TOTAL);
      /*
       * link together the tx bd chain pages
       */
      for (j = 1; j <= TX_BD_NUM_PAGES; j++) {
	 /*
	  * index into the tx bd chain array to last entry per page
	  */
	 struct eth_tx_next_bd *tx_next_bd =
               &txdata->tx_chain[TX_BD_TOTAL_PER_PAGE * j - 1].next_bd;
	 /*
	  * point to the next page and wrap from last page
	  */
            busaddr = (txdata->tx_chain_ioa + (BCM_PAGE_SIZE * (j % TX_BD_NUM_PAGES)));
	 tx_next_bd->addr_hi = htole32(U64_HI(busaddr));
	 tx_next_bd->addr_lo = htole32(U64_LO(busaddr));
      }
      }
   }else {
      fp->rx_chain =
	 qfle3_alloc_dma_mapping(adapter,
				 adapter->dmaEngineCoherent,
				 (BCM_PAGE_SIZE * RX_BD_NUM_PAGES), &fp->rx_chain_ioa);

      if (!fp->rx_chain) {
	      QFLE3_ERROR("Failed to alloc fastpath rx bd chain\n");
         status = VMK_NO_MEMORY;
         goto failed;
      }
      QFLE3_DBG(QFLE3_DBG_SP, "Allocated fastpath rx bd  chain %u pages\n", RX_BD_NUM_PAGES);

      fp->rxbuf_chain = qfle3_heap_alloc(sizeof(qfle3_rxbuf_info) * RX_BD_TOTAL);
      if (fp->rxbuf_chain == NULL) {
	      QFLE3_ERR("Failed to allocate rxbuf_chain");
         status = VMK_NO_MEMORY;
         goto failed;
      }
      QFLE3_DBG(QFLE3_DBG_SP,"Allocated fastpath rxbuf_chain  for %ld buffers\n", RX_BD_TOTAL);

      /*
       * link together the rx bd chain pages
       */
      for (j = 1; j <= RX_BD_NUM_PAGES; j++) {
	 /*
	  * index into the rx bd chain array to last entry per page
	  */
	 struct eth_rx_bd *rx_bd = &fp->rx_chain[RX_BD_TOTAL_PER_PAGE * j - 2];
	 /*
	  * point to the next page and wrap from last page
	  */
	 busaddr = (fp->rx_chain_ioa + (BCM_PAGE_SIZE * (j % RX_BD_NUM_PAGES)));
	 rx_bd->addr_hi = htole32(U64_HI(busaddr));
	 rx_bd->addr_lo = htole32(U64_LO(busaddr));
      }



      /*
       * FP RX SGE CHAIN
       */
      fp->rx_sge_chain =
	 qfle3_alloc_dma_mapping(adapter,
				 adapter->dmaEngineCoherent,
				 (BCM_PAGE_SIZE * RX_SGE_NUM_PAGES),
				 &fp->rx_sge_chain_ioa);

      if (!fp->rx_sge_chain) {
         QFLE3_ERROR("Failed to alloc fastpath rx sge chain\n");
         status = VMK_NO_MEMORY;
         goto failed;
      }
      QFLE3_DBG(QFLE3_DBG_SP,"Allocated fastpath rx sge chain %u pages\n", RX_SGE_NUM_PAGES);
      /*
       * link together the sge chain pages
       */
      for (j = 1; j <= RX_SGE_NUM_PAGES; j++) {
	 /*
	  * index into the rcq chain array to last entry per page
	  */
	 struct eth_rx_sge *rx_sge = &fp->rx_sge_chain[RX_SGE_TOTAL_PER_PAGE * j - 2];
	 /*
	  * point to the next page and wrap from last page
	  */
	 busaddr = (fp->rx_sge_chain_ioa + (BCM_PAGE_SIZE * (j % RX_SGE_NUM_PAGES)));
	 rx_sge->addr_hi = htole32(U64_HI(busaddr));
	 rx_sge->addr_lo = htole32(U64_LO(busaddr));
      }

      qfle3_init_sge_ring_bit_mask(fp);

   }
   return VMK_OK;

  failed:
   return status;
}


VMK_ReturnStatus
qfle3_alloc_txqs_mem(qfle3_adapter * adapter)
{
   vmk_uint32 i, qid;
   VMK_ReturnStatus status = VMK_OK;
   struct qfle3_fastpath *fp;
   vmk_uint8 cos;

   FOREACH_TX_ETH_QUEUE(i,qid,fp)
      fp->qid = QFLE3_TX_Q_NUM_TO_FP_NUM(adapter, i);
      fp->adapter = adapter;
      if (adapter->sw_fcoe)
         fp->max_cos = adapter->max_cos;
      else
         fp->max_cos = 1;
      for_each_cos_in_tx_queue(fp, cos){
         fp->txdata_ptr[cos] = &adapter->qfle3_txq[FP_COS_TO_TXQ(fp,cos,adapter)];
         fp->txdata_ptr[cos]->cos = cos;
         fp->txdata_ptr[cos]->txq_index = FP_COS_TO_TXQ(fp,cos,adapter);
         fp->txdata_ptr[cos]->parent_fp = fp;
      }
      status = qfle3_alloc_queue_mem(adapter, ALLOC_TXQ, fp);
      if (status != VMK_OK) {
         qfle3_free_txqs_mem(adapter);
         return status;
      }
   }
   return status;
}

VMK_ReturnStatus
qfle3_alloc_rxqs_mem(qfle3_adapter * adapter)
{
   vmk_uint32 i, qid;
   VMK_ReturnStatus status = VMK_OK;
   struct qfle3_fastpath *fp;
   vmk_uint8 cos;
   FOREACH_RX_ETH_QUEUE(i,qid,fp)
      fp->qid = QFLE3_RX_Q_NUM_TO_FP_NUM(adapter, i);
      fp->adapter = adapter;
      fp->max_cos = 1;
      for_each_cos_in_tx_queue(fp, cos){
         fp->txdata_ptr[cos] = &adapter->qfle3_txq[FP_COS_TO_TXQ(fp,cos,adapter)];
         fp->txdata_ptr[cos]->cos = cos;
         fp->txdata_ptr[cos]->txq_index = FP_COS_TO_TXQ(fp,cos,adapter);
         fp->txdata_ptr[cos]->parent_fp = fp;
      }
      qfle3_set_fp_rx_buf_size(adapter, fp);
      status = qfle3_alloc_queue_mem(adapter, ALLOC_RXQ, fp);
      
      QFLE3_DBG(QFLE3_DBG_QUEUE, "allocated buffer for rxq %d rxbuf_chain %s\n", (int)(fp- adapter->fp), fp->rxbuf_chain?"allocated":"not allocated");
      if (status != VMK_OK) {
         qfle3_free_rxqs_mem(adapter);
         return status;
      }
   }
   return status;
}

static void
qfle3_tq_cleanup(qfle3_adapter * adapter, struct qfle3_fastpath *tq)
{
   int i;
   vmk_uint8 cos;
   struct qfle3_fp_txdata *txdata;

   for_each_cos_in_tx_queue(tq, cos){
      txdata = tq->txdata_ptr[cos];
   for (i = 0; i < TX_BD_TOTAL; i++) {
      qfle3_txbuf_info *tbi;

         tbi = &txdata->txbuf_chain[i];
      qfle3_unmap_txbuf(adapter, tbi, i, tq->qid);
      if (tbi->pkt) {
	 QFLE3_DBG(QFLE3_DBG_QUEUE, "!txbuf_chain[%d] pkt %p, \n", i, tbi->pkt);
	 vmk_PktRelease(tbi->pkt);
	 tbi->pkt = NULL;
      }
   }
}
}

void
qfle3_tq_cleanup_all(qfle3_adapter * adapter)
{
   int i, fp_num;
   for (i = 0; i < QFLE3_NUM_TX_ETH_QUEUES(adapter); i++) {
      fp_num = QFLE3_TX_Q_NUM_TO_FP_NUM(adapter, i);
      qfle3_tq_cleanup(adapter, QFLE3_GET_FP_FROM_QID(adapter, fp_num));
   }
}
#if (ESX_DDK_VERSION >= 2017)

VMK_ReturnStatus
qfle3_rss_get_engines_info(vmk_AddrCookie drv_data,                             //IN
                           vmk_UplinkQueueRSSEnginesInfo   *enginesInfo)        //OUT
{
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;

   if (!QFLE3_RSS_SUPPORTED(adapter)) {
      QFLE3_ERR("RSS_OP_GET_PARAMS : No RSS support.\n");
      return VMK_NOT_SUPPORTED;
   }

   enginesInfo->numRSSEngines = adapter->num_rss_engines;
   enginesInfo->rssEnginesCap =  VMK_RSS_ENGINE_CAP_NONE;
   if((adapter->num_rss_engines > 1) && (adapter->num_rssqs_def > 1)){
        enginesInfo->maxRSSSecQueues = adapter->num_rssqs_def - 1 + (adapter->num_rss_engines -1) * (adapter->num_rssqs_nd - 1);
        enginesInfo->rssEnginesType = VMK_RSS_ENGINE_TYPE_NETQUEUE |  VMK_RSS_ENGINE_TYPE_DEFAULT_QUEUE;
   } else {
        enginesInfo->maxRSSSecQueues = adapter->num_rss_engines * (adapter->num_rssqs_nd - 1);
        enginesInfo->rssEnginesType = VMK_RSS_ENGINE_TYPE_NETQUEUE;
   }
   QFLE3_DBG(QFLE3_DBG_QUEUE,"numRSSEngines: %d, maxRSSSecQueues: %d, rssEnginesType: %d\n",
                        enginesInfo->numRSSEngines, enginesInfo->maxRSSSecQueues, enginesInfo->rssEnginesType);
   return VMK_OK;

}

VMK_ReturnStatus
qfle3_rss_get_engines_supported_config(vmk_AddrCookie drv_data,                         //IN
                                       vmk_UplinkQueueID qID,                           //IN
                                       vmk_UplinkQueueRSSEngineConfig  *engineConfig)   //OUT
{
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;
   struct qfle3_fastpath *rxq;
   vmk_uint32 fp_id = vmk_UplinkQueueIDVal(qID);
   vmk_uint32 j;

   rxq = &adapter->fp[fp_id];
   QFLE3_DBG(QFLE3_DBG_QUEUE,"VMkernel requesting to get RSS Supported config for RX Queue %d\n", rxq->qid);

   if(rxq->num_rss <=1){
       QFLE3_INFO("Rss not enabled on this Queue\n");
       return VMK_NOT_SUPPORTED;
   }

     if (rxq->fp_state == 0 ) {
         QFLE3_INFO("RSS queue not started yet please check after some time \n");
         return VMK_FAILURE;
   }

   if (Qfle3IsDefaultRxQueue(adapter, rxq)){
      engineConfig->numRSSSecQueues = rxq->num_rss  - 1;
      engineConfig->rssHashKeySize = sizeof(adapter->defq_rss_key_tbl);
      engineConfig->rssIndTableSize = T_ETH_INDIRECTION_TABLE_SIZE;
      engineConfig->rssHashFunction = VMK_RSS_HASH_FUNC_TOEPLITZ_32BIT;
      engineConfig->rssEngineFeature = VMK_RSS_ENGINE_FEAT_INDIRECTION_TABLE;
   }
   else {
      engineConfig->numRSSSecQueues = rxq->num_rss - 1;
      j = rxq->rss_engine_id - (QFLE3_FUNC(adapter) * 2 * QFLE3_DEVICE_MAX_RSS_ENGINE);
      if(j>QFLE3_DEVICE_MAX_RSS_ENGINE){
           QFLE3_INFO("RSS Config %d beyond supported %d \n", j, QFLE3_DEVICE_MAX_RSS_ENGINE);
           return VMK_FAILURE;
      }
      QFLE3_DBG(QFLE3_DBG_QUEUE,"RSS Engine ID is %d and j is %d\n", rxq->rss_engine_id,j);
      engineConfig->rssHashKeySize = sizeof(adapter->netq_rss_key_tbl[j]);
      engineConfig->rssIndTableSize = T_ETH_INDIRECTION_TABLE_SIZE;
      engineConfig->rssHashFunction = VMK_RSS_HASH_FUNC_TOEPLITZ_32BIT;
      engineConfig->rssEngineFeature = VMK_RSS_ENGINE_FEAT_INDIRECTION_TABLE;
   }
   return VMK_OK;
}

VMK_ReturnStatus
qfle3_rss_get_engines_cur_config(vmk_AddrCookie drv_data,                               //IN
                                       vmk_UplinkQueueID qID,                           //IN
                                       vmk_UplinkQueueRSSEngineConfig  *engineConfig)   //OUT
{
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;
   struct qfle3_fastpath *rxq;
   vmk_uint32 fp_id = vmk_UplinkQueueIDVal(qID);
   vmk_uint32 j;

   rxq = &adapter->fp[fp_id];
   QFLE3_DBG(QFLE3_DBG_QUEUE,"VMkernel requesting to get RSS Supported config for RX Queue %d\n", rxq->qid);

   if(rxq->num_rss <=1){
       QFLE3_INFO("Rss not enabled on this Queue\n");
       return VMK_NOT_SUPPORTED;
   }

   if (rxq->fp_state == 0 ) {
         QFLE3_INFO("RSS queue not started  yet please check after some time \n");
         return VMK_FAILURE;
   }

   if (Qfle3IsDefaultRxQueue(adapter, rxq)){
      engineConfig->numRSSSecQueues = rxq->num_rss - 1;
      engineConfig->rssHashKeySize = sizeof(adapter->defq_rss_key_tbl);
      engineConfig->rssIndTableSize = T_ETH_INDIRECTION_TABLE_SIZE;
      engineConfig->rssHashFunction = VMK_RSS_HASH_FUNC_TOEPLITZ_32BIT;
      engineConfig->rssEngineFeature = VMK_RSS_ENGINE_FEAT_INDIRECTION_TABLE;
   }
   else {
      engineConfig->numRSSSecQueues = rxq->num_rss - 1;
      j = rxq->rss_engine_id - (QFLE3_FUNC(adapter) * 2 * QFLE3_DEVICE_MAX_RSS_ENGINE);
      QFLE3_DBG(QFLE3_DBG_QUEUE,"RSS Engine ID is %d and j is %d\n", rxq->rss_engine_id,j);
      if(j>QFLE3_DEVICE_MAX_RSS_ENGINE){
           QFLE3_INFO("RSS Config %d beyond supported %d \n", j, QFLE3_DEVICE_MAX_RSS_ENGINE);
           return VMK_FAILURE;
      }
      engineConfig->rssHashKeySize = sizeof(adapter->netq_rss_key_tbl[j]);
      engineConfig->rssIndTableSize = T_ETH_INDIRECTION_TABLE_SIZE;
      engineConfig->rssHashFunction = VMK_RSS_HASH_FUNC_TOEPLITZ_32BIT;
      engineConfig->rssEngineFeature = VMK_RSS_ENGINE_FEAT_INDIRECTION_TABLE;
   }
   QFLE3_DBG(QFLE3_DBG_QUEUE,"numRSSSecQueues: %d\n", engineConfig->numRSSSecQueues);
   return VMK_OK;
}
VMK_ReturnStatus
qfle3_rss_set_engine_config(vmk_AddrCookie drv_data,                                //IN
                                       vmk_UplinkQueueID qID,                       //IN
                                       vmk_UplinkQueueRSSEngineConfigOp configOp,   //IN
                                       vmk_AddrCookie opArgs)                       //IN
{
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;
   struct qfle3_fastpath *rxq, *rq;
   vmk_uint32 fp_id = vmk_UplinkQueueIDVal(qID);
   vmk_uint16 num_sec_queues;
   vmk_uint16 l;
   vmk_uint32 j;

   if(configOp & VMK_RSS_ENGINE_CONFIG_OP_NONE){
        QFLE3_INFO("Do Nothing\n");
   }

   if(configOp & VMK_RSS_ENGINE_CONFIG_OP_INIT_SEC_QUEUES){
   /**
    * Initialize secondary queues in the RSS engine.
    * \note In case of \ref VMK_RSS_ENGINE_TYPE_NETQUEUE type of
    *       RSS engine, this operation will be performed
    *       after allocating primary RSS queue.
    * \note Argument for this operation is
    *       vmk_UplinkQueueRSSSecQueuesInitArgs.
    */
        vmk_UplinkQueueRSSSecQueuesInitArgs  *sec_init_args = (vmk_UplinkQueueRSSSecQueuesInitArgs  *)opArgs.ptr;
        rxq = &adapter->fp[fp_id];
        QFLE3_DBG(QFLE3_DBG_QUEUE,"Executing VMK_RSS_ENGINE_CONFIG_OP_INIT_SEC_QUEUES for RX Queue:%d "
				  "Number of Sec Queues: %d\n", rxq->qid, sec_init_args->numSecQueues);

        if (rxq->fp_state == 0) {
        QFLE3_INFO("RSS queue not started  yet please check after some time \n");
        return VMK_FAILURE;
        }

        num_sec_queues = sec_init_args->numSecQueues;
        for(l=1; l<=num_sec_queues && l<rxq->num_rss; l++){
                QFLE3_DBG(QFLE3_DBG_QUEUE,"Sec Queue ID: %d Adapter Queue ID: %d\n", l, rxq->sec_quque_ids[l]);
                rq = &adapter->fp[rxq->sec_quque_ids[l]];
                
                vmk_UplinkQueueMkRxQueueID(&sec_init_args->secQueues[l-1].qid, rxq->sec_quque_ids[l], rxq->sec_quque_ids[l]);
                sec_init_args->secQueues[l-1].poll = rq->netpoll;
                sec_init_args->secQueues[l-1].pairedHwQid = 0;   //This should be fine since driver
                                                          //Doesn't claim VMK_RSS_ENGINE_FEAT_PAIR in
                                                          //queueGetRSSEngineSupportedConfig callback
                QFLE3_DBG(QFLE3_DBG_QUEUE,"SecQueueQid = %d\n", vmk_UplinkQueueIDVal(sec_init_args->secQueues[l-1].qid));
        }
   }
   if(configOp & VMK_RSS_ENGINE_CONFIG_OP_SET_HASH_KEY){
   /**
    * Set hash key in the RSS engine.
    * \note In case of \ref VMK_RSS_ENGINE_TYPE_NETQUEUE type of
    *       RSS engine, this operation will be performed
    *       after allocating primary RSS queue.
    * \note Argument for this operation is
    *       vmk_UplinkQueueRSSHashKey.
    */
        vmk_UplinkQueueRSSHashKey *rss_hash_key = (vmk_UplinkQueueRSSHashKey *)opArgs.ptr;
        vmk_uint32 *key = NULL;

        rxq = &adapter->fp[fp_id];
        QFLE3_DBG(QFLE3_DBG_QUEUE,"Execute VMK_RSS_ENGINE_CONFIG_OP_SET_HASH_KEY for RX Queue: %d\n", rxq->qid);

        if (rxq->fp_state == 0) {
                QFLE3_INFO("RSS queue not started  yet please check after some time \n");
                return VMK_FAILURE;
        }


        if (Qfle3IsDefaultRxQueue(adapter, rxq)){
                if (VMK_UNLIKELY(rss_hash_key->keySize != sizeof(adapter->defq_rss_key_tbl))){
                        QFLE3_ERR("VMK_RSS_ENGINE_CONFIG_OP_SET_HASH_KEY : key_size %d != %d\n",
                                rss_hash_key->keySize, (int)sizeof(adapter->defq_rss_key_tbl));
                        return VMK_FAILURE;
                }
                /* Save away RSS keys for later (size in bytes) */
                vmk_Memcpy(adapter->defq_rss_key_tbl, rss_hash_key->key, rss_hash_key->keySize);
                key = (vmk_uint32*)rss_hash_key->key;
                QFLE3_DBG(QFLE3_DBG_QUEUE,
                        "VMK_RSS_ENGINE_CONFIG_OP_SET_HASH_KEY: "
                        "key: 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x - \n",
                          *(key), *(key + 1), *(key + 2), *(key + 3), *(key + 4), *(key + 5), *(key + 6), *(key + 7), *(key + 8), *(key + 9));
                QFLE3_DBG(QFLE3_DBG_QUEUE, "VMK_RSS_ENGINE_CONFIG_OP_SET_HASH_KEY: key_size:%d \n", rss_hash_key->keySize);
        }else{
                j = rxq->rss_engine_id - ( QFLE3_FUNC(adapter) * 2 * QFLE3_DEVICE_MAX_RSS_ENGINE );
                QFLE3_DBG(QFLE3_DBG_QUEUE,"RSS Engine ID is %d and j is %d\n", rxq->rss_engine_id,j);
                if(j > QFLE3_DEVICE_MAX_RSS_ENGINE){
                        QFLE3_ERR("Netqueue RSS Table ID %d is beyond supported \n", j);
                        return VMK_FAILURE;
                }

                if (VMK_UNLIKELY(rss_hash_key->keySize != sizeof(adapter->netq_rss_key_tbl[j]))){
                         QFLE3_ERR("VMK_RSS_ENGINE_CONFIG_OP_SET_HASH_KEY : key_size %d != %d\n",
                                rss_hash_key->keySize,
                                (int)sizeof(adapter->netq_rss_key_tbl[j]));
                        return VMK_FAILURE;
                }

                /* Save away RSS keys for later (size in bytes) */
                vmk_Memcpy(adapter->netq_rss_key_tbl[j], rss_hash_key->key, rss_hash_key->keySize);
                key = (vmk_uint32*)rss_hash_key->key;
                QFLE3_DBG(QFLE3_DBG_QUEUE,
                          "VMK_RSS_ENGINE_CONFIG_OP_SET_HASH_KEY: "
                          "key: 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x - \n",
                          *(key), *(key + 1), *(key + 2), *(key + 3), *(key + 4), *(key + 5), *(key + 6), *(key + 7), *(key + 8), *(key + 9));
                QFLE3_DBG(QFLE3_DBG_QUEUE, "VMK_RSS_ENGINE_CONFIG_OP_SET_HASH_KEY: key_size:%d \n", rss_hash_key->keySize);
        }
   }
   return VMK_OK;
}

VMK_ReturnStatus
qfle3_rss_set_engine_ind_table(vmk_AddrCookie                drv_data,    //IN
                               vmk_UplinkQueueID             qID,           //IN
                               vmk_UplinkQueueRSSIndTable   *rssIndTable)   //IN
{
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;
   struct qfle3_fastpath *rxq;
   vmk_uint32 fp_id;
   vmk_uint32 j,i;
   VMK_ReturnStatus rc = VMK_OK;

   fp_id = vmk_UplinkQueueIDVal(qID);
   rxq = &adapter->fp[fp_id];
   QFLE3_DBG(QFLE3_DBG_QUEUE,"VMkernel requesting to Set RSS Ind Table for RX Queue %d\n", rxq->qid);

   if (rxq->num_rss <=1) {
      QFLE3_ERR( "No RSS support.\n");
      return VMK_NOT_SUPPORTED;
   }


   if (rxq->fp_state == 0) {
        QFLE3_INFO("RSS queue not started  yet please check after some time \n");
        return VMK_FAILURE;
        }

   if (VMK_UNLIKELY(rssIndTable->tableSize > T_ETH_INDIRECTION_TABLE_SIZE)){
        QFLE3_ERR( "ind tbl size %d > %d \n",
                        rssIndTable->tableSize, T_ETH_INDIRECTION_TABLE_SIZE);
        return VMK_FAILURE;
   }


   if (Qfle3IsDefaultRxQueue(adapter, rxq)){
        for (i = 0; i < rssIndTable->tableSize; i++){
                adapter->defq_rss_raw_ind_tbl[i] = (vmk_uint8)rssIndTable->table[i % rssIndTable->tableSize];
                if (adapter->defq_rss_raw_ind_tbl[i] >= adapter->num_rssqs_def){
                        QFLE3_ERR( "Invalid rss ind tbl entry[%d]:%d >= %d\n",
                                i, adapter->defq_rss_raw_ind_tbl[i], adapter->num_rssqs_def);
                        return VMK_FAILURE;
                }
        }
        if (rxq->fp_state == QFLE3_SM_Q_STARTED) {
                rc = qfle3_config_rss(adapter,
                                adapter->defq_rss_raw_ind_tbl, adapter->defq_rss_key_tbl);
                if (rc != VMK_OK) {
                        return VMK_FAILURE;
                }
        }

   } else {
        j = rxq->rss_engine_id - ( QFLE3_FUNC(adapter) * 2 * QFLE3_DEVICE_MAX_RSS_ENGINE );
        QFLE3_DBG(QFLE3_DBG_QUEUE,"RSS Engine ID is %d and j is %d\n", rxq->rss_engine_id,j);
        if(j > QFLE3_DEVICE_MAX_RSS_ENGINE){
                QFLE3_ERR("Netqueue RSS Table ID %d is beyond supported \n", j);
                return VMK_FAILURE;
        }

        /* Overwrite the RAW RSS pool indirection table provided by host*/
        for (i = 0; i < rssIndTable->tableSize; i++){
                adapter->netq_rss_raw_ind_tbl[j][i] =
                        (vmk_uint8)rssIndTable->table[i % rssIndTable->tableSize];

                if (adapter->netq_rss_raw_ind_tbl[j][i] >= adapter->num_rssqs_nd){
                        QFLE3_ERR( "Invalid rss ind tbl entry[%d]:%d >= %d\n",
                                i, adapter->netq_rss_raw_ind_tbl[j][i], adapter->num_rssqs_nd);
                        return VMK_FAILURE;
                }
        }
        if (rxq->fp_state == QFLE3_SM_Q_STARTED) {
                rc = qfle3_config_rss(adapter,
                          adapter->netq_rss_raw_ind_tbl[j], adapter->netq_rss_key_tbl[j]);
                if (rc != VMK_OK) {
                        return VMK_FAILURE;
                }
        }
   }
   QFLE3_DBG(QFLE3_DBG_QUEUE,
                "ind_table_size:%d\n",rssIndTable->tableSize);
   for (i = 0; i < T_ETH_INDIRECTION_TABLE_SIZE; i += 8) {
      QFLE3_DBG(QFLE3_DBG_QUEUE,
                "rssIndTable[%d to %d]: %d %d %d %d %d %d %d %d\n",
                i, i + 7,
                rssIndTable->table[i], rssIndTable->table[i+1], rssIndTable->table[i+2], rssIndTable->table[i+3],
                rssIndTable->table[i+4], rssIndTable->table[i+5], rssIndTable->table[i+6], rssIndTable->table[i+7]);
   }
   return VMK_OK;
}

VMK_ReturnStatus
qfle3_rss_set_engine_ind_table_idx(vmk_AddrCookie        drv_data,   //IN
                                   vmk_UplinkQueueID     qID,        //IN
                                   vmk_uint16            tableIndex, //IN
                                   vmk_uint8             queueIndex) //IN
{
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;
   struct qfle3_fastpath *rxq;
   vmk_uint32 fp_id;
   vmk_uint32 j;
   VMK_ReturnStatus rc;


   fp_id = vmk_UplinkQueueIDVal(qID);
   rxq = &adapter->fp[fp_id];

   if (rxq->fp_state == 0) {
        QFLE3_INFO("RSS queue not started  yet please check after some time \n");
        return VMK_FAILURE;
   }


   if (rxq->num_rss <=1) {
      QFLE3_ERR( "No RSS support.\n");
      return VMK_NOT_SUPPORTED;
   }

   if(VMK_UNLIKELY(tableIndex > T_ETH_INDIRECTION_TABLE_SIZE)){
      QFLE3_ERR( "Vmkernel Table Index %d > %d\n",
                tableIndex,  T_ETH_INDIRECTION_TABLE_SIZE);
      return VMK_FAILURE;
   }

   if(queueIndex >= rxq->num_rss){
      QFLE3_ERR( "Invalid RSS Secondary Queue ID %d >= %d\n",
                queueIndex, rxq->num_rss);
      return VMK_FAILURE;
   }
   if (Qfle3IsDefaultRxQueue(adapter, rxq)){
        if ( adapter->defq_rss_raw_ind_tbl[tableIndex] != queueIndex){
           QFLE3_DBG(QFLE3_DBG_QUEUE,"Existing rssIndTable[%d] is %d Requested is %d \n", tableIndex, adapter->defq_rss_raw_ind_tbl[tableIndex], queueIndex);
           adapter->defq_rss_raw_ind_tbl[tableIndex] = queueIndex;
           rc = qfle3_config_rss(adapter,
                          adapter->defq_rss_raw_ind_tbl, adapter->defq_rss_key_tbl);
           if (rc != VMK_OK) {
                return VMK_FAILURE;
           }
           QFLE3_DBG(QFLE3_DBG_QUEUE,"Current rssIndTable[%d] is %d\n", tableIndex, adapter->defq_rss_raw_ind_tbl[tableIndex]);
      }
   } else {
        j = rxq->rss_engine_id - ( QFLE3_FUNC(adapter) * 2 * QFLE3_DEVICE_MAX_RSS_ENGINE );
        if(j > QFLE3_DEVICE_MAX_RSS_ENGINE){
                 QFLE3_ERR("Netqueue RSS Table ID %d is beyond supported \n", j);
                 return VMK_FAILURE;
        }

        if(adapter->netq_rss_raw_ind_tbl[j][tableIndex] != queueIndex){
           QFLE3_DBG(QFLE3_DBG_QUEUE,"Existing rssIndTable[%d] is %d j is %d requested is %d\n", 
                                 tableIndex, adapter->netq_rss_raw_ind_tbl[j][tableIndex],j, queueIndex);
           adapter->netq_rss_raw_ind_tbl[j][tableIndex] = queueIndex;
           rc = qfle3_config_rss(adapter,
                          adapter->netq_rss_raw_ind_tbl[j], adapter->netq_rss_key_tbl[j]);
           if (rc != VMK_OK) {
                return VMK_FAILURE;
           }
          QFLE3_DBG(QFLE3_DBG_QUEUE,"Current rssIndTable[%d] is %d j is %d\n", tableIndex, adapter->netq_rss_raw_ind_tbl[j][tableIndex],j);
      }
   }
   return VMK_OK;
}

VMK_ReturnStatus
qfle3_rss_get_engine_ind_table(vmk_AddrCookie drv_data,                         //IN
                                vmk_UplinkQueueID qID,                          //IN
                                vmk_UplinkQueueRSSIndTable *rssIndTable)        //OUT
{
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;
   struct qfle3_fastpath *rxq;
   vmk_uint32 fp_id;
   vmk_uint32 j,i;

   fp_id = vmk_UplinkQueueIDVal(qID);
   rxq = &adapter->fp[fp_id];
   QFLE3_DBG(QFLE3_DBG_QUEUE,"VMkernel requesting to Get RSS Ind Table for RX Queue %d\n", rxq->qid);

   if (rxq->num_rss <=1) {
      QFLE3_ERR( "No RSS support.\n");
      return VMK_NOT_SUPPORTED;
   }

   if (rxq->fp_state == 0) {
        QFLE3_INFO("RSS queue not started  yet please check after some time \n");
        return VMK_FAILURE;
   }


   if (Qfle3IsDefaultRxQueue(adapter, rxq)){
        if(rssIndTable->tableSize != T_ETH_INDIRECTION_TABLE_SIZE) {
                QFLE3_ERR( "indirection table size != %d\n", T_ETH_INDIRECTION_TABLE_SIZE);
                return VMK_FAILURE;
        }

        for(i=0; i < rssIndTable->tableSize; i++) {
                rssIndTable->table[i] = adapter->defq_rss_raw_ind_tbl[i];
        }
   } else {

        j = rxq->rss_engine_id - ( QFLE3_FUNC(adapter) * 2 * QFLE3_DEVICE_MAX_RSS_ENGINE );
        QFLE3_DBG(QFLE3_DBG_QUEUE,"RSS Engine ID is %d and j is %d\n", rxq->rss_engine_id,j);
        if(j > QFLE3_DEVICE_MAX_RSS_ENGINE){
                QFLE3_ERR("Netqueue RSS Table ID %d is beyond supported \n", j);
                return VMK_FAILURE;
        }

        if(rssIndTable->tableSize != T_ETH_INDIRECTION_TABLE_SIZE) {
                QFLE3_ERR( "indirection table size != %d\n", T_ETH_INDIRECTION_TABLE_SIZE);
                return VMK_FAILURE;
        }

        for(i=0; i < rssIndTable->tableSize; i++) {
                rssIndTable->table[i] = adapter->netq_rss_raw_ind_tbl[j][i];
        }
   }
   QFLE3_DBG(QFLE3_DBG_QUEUE,"ind_table_size:%d\n", rssIndTable->tableSize);
   for (i = 0; i < T_ETH_INDIRECTION_TABLE_SIZE; i += 8) {
      QFLE3_DBG(QFLE3_DBG_QUEUE,
                "ind_table[%d to %d]: %d %d %d %d %d %d %d %d\n",
                i, i + 7,
                rssIndTable->table[i], rssIndTable->table[i+1], rssIndTable->table[i+2], rssIndTable->table[i+3],
                rssIndTable->table[i+4], rssIndTable->table[i+5], rssIndTable->table[i+6], rssIndTable->table[i+7]);
   }
   return VMK_OK;
}

#else
VMK_ReturnStatus
qfle3_rss_get_params(vmk_AddrCookie drv_data,
		     vmk_UplinkQueueRSSParams *params)
{
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;

   if (!QFLE3_RSS_SUPPORTED(adapter)) {
      QFLE3_ERR("RSS_OP_GET_PARAMS : No RSS support.\n");
      return VMK_FAILURE;
   }

   params->numRSSPools = 1;
   params->numRSSQueuesPerPool = adapter->num_rssqs_nd;

   params->rssHashKeySize = sizeof(adapter->netq_rss_key_tbl);
   params->rssIndTableSize = T_ETH_INDIRECTION_TABLE_SIZE;

   QFLE3_DBG(QFLE3_DBG_QUEUE,
	     "RSS_OP_GET_PARAMS: (rss_pools:%d, qs_per_pool:%d, "
	     "rss_hash_key_sz:%d, rss_ind_table_sz:%d)\n",
	     params->numRSSPools,
	     params->numRSSQueuesPerPool,
	     params->rssHashKeySize,
	     params->rssIndTableSize);

   return VMK_OK;
}

VMK_ReturnStatus
qfle3_rss_state_init(vmk_AddrCookie drv_data,
		     vmk_UplinkQueueRSSHashKey *rss_key,
		     vmk_UplinkQueueRSSIndTable *ind_table)
{
   vmk_int32 i;
   vmk_uint32 *key = NULL;
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;

   if (!adapter->num_rssqs_nd) {
      QFLE3_ERR( "RSS_OP_INIT_STATE : No RSS support.\n");
      return VMK_FAILURE;
   }

   if (VMK_UNLIKELY(ind_table->tableSize >
		    T_ETH_INDIRECTION_TABLE_SIZE ||
		    rss_key->keySize != sizeof(adapter->netq_rss_key_tbl))) {
      QFLE3_ERR( "RSS_OP_INIT_STATE : ind tbl size %d > %d || key_size %d != %d\n",
		 ind_table->tableSize,
		 T_ETH_INDIRECTION_TABLE_SIZE,
		 rss_key->keySize,
		 (int)sizeof(adapter->netq_rss_key_tbl));
      return VMK_FAILURE;
   }

   /* Overwrite the RAW RSS pool indirection table provided by host*/
   for (i = 0; i < ind_table->tableSize; i++) {
      adapter->netq_rss_raw_ind_tbl[i] =
	 (vmk_uint8)ind_table->table[i % ind_table->tableSize];

      if (adapter->netq_rss_raw_ind_tbl[i] >= adapter->num_rssqs_nd) {
	 QFLE3_ERR( "RSS_OP_INIT_STATE : Invalid rss ind tbl entry[%d]:%d >= %d\n",
		    i, adapter->netq_rss_raw_ind_tbl[i],
		    adapter->num_rssqs_nd);
	 return VMK_FAILURE;
      }
   }

   /* Save away RSS keys for later (size in bytes) */
   vmk_Memcpy(adapter->netq_rss_key_tbl, rss_key->key,  rss_key->keySize);

   key = (vmk_uint32*) rss_key->key;
   QFLE3_DBG(QFLE3_DBG_QUEUE,
	     "RSS_OP_INIT_STATE: "
	     "(key: 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x 0x%04x - "
	     "key_size:%d, ind_table_size:%d)\n",
	     *(key), *(key + 1), *(key + 2), *(key + 3),
	     *(key + 4), *(key + 5), *(key + 6), *(key + 7),
	     *(key + 8), *(key + 9),
	     rss_key->keySize, ind_table->tableSize);

   for (i = 0; i < T_ETH_INDIRECTION_TABLE_SIZE; i += 8) {
      QFLE3_DBG(QFLE3_DBG_QUEUE,
		"ind_table[%d to %d]: %d %d %d %d %d %d %d %d\n",
		i, i + 7,
		ind_table->table[i], ind_table->table[i+1], ind_table->table[i+2], ind_table->table[i+3],
		ind_table->table[i+4], ind_table->table[i+5], ind_table->table[i+6], ind_table->table[i+7]);
   }

   return VMK_OK;
}


VMK_ReturnStatus
qfle3_rss_ind_tbl_update(vmk_AddrCookie drv_data,
			 vmk_UplinkQueueRSSIndTable *ind_table)
{
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;
   int i;
   VMK_ReturnStatus rc = VMK_OK;

   if (!QFLE3_RSS_SUPPORTED(adapter)) {
      QFLE3_DBG(QFLE3_DBG_QUEUE,
		"RSS_OP_UPDATE_IND_TABLE : No RSS support or no RSS queue allocated.\n");
      return VMK_FAILURE;
   }

   if (VMK_UNLIKELY(ind_table->tableSize !=
		    T_ETH_INDIRECTION_TABLE_SIZE)) {
      QFLE3_ERR( "RSS_OP_UPDATE_IND_TABLE : ind tbl size %d != %d\n",
		 ind_table->tableSize,
		 T_ETH_INDIRECTION_TABLE_SIZE);
      return VMK_FAILURE;
   }

   /* Overwrite the RAW RSS pool indirection table with the table provided by host*/
   for (i = 0; i < ind_table->tableSize; i++) {
      adapter->netq_rss_raw_ind_tbl[i] = (vmk_uint8)ind_table->table[i];

      if (adapter->netq_rss_raw_ind_tbl[i] >= adapter->num_rssqs_nd) {
	 QFLE3_ERR("RSS_OP_UPDATE_IND_TABLE : Invalid rss ind tbl entry[%d]:%d >= %d\n",
		   i, adapter->netq_rss_raw_ind_tbl[i],
		   adapter->num_rssqs_nd);
	 return VMK_FAILURE;
      }
   }

   rc = qfle3_config_rss(adapter,
			 adapter->netq_rss_raw_ind_tbl, adapter->netq_rss_key_tbl);
   if (rc != VMK_OK) {
      return VMK_FAILURE;
   }

   QFLE3_DBG(QFLE3_DBG_QUEUE,
	     "RSS_OP_UPDATE_IND_TABLE: ind_table_size:%d\n", ind_table->tableSize);

   for (i = 0; i < T_ETH_INDIRECTION_TABLE_SIZE; i += 8) {
      QFLE3_DBG(QFLE3_DBG_QUEUE,
		"ind_table[%d to %d]: %d %d %d %d %d %d %d %d\n",
		i, i + 7,
		ind_table->table[i], ind_table->table[i+1], ind_table->table[i+2], ind_table->table[i+3],
		ind_table->table[i+4], ind_table->table[i+5], ind_table->table[i+6], ind_table->table[i+7]);
   }
   return VMK_OK;
}

VMK_ReturnStatus
qfle3_rss_get_ind_tbl(vmk_AddrCookie drv_data,
		      vmk_UplinkQueueRSSIndTable *ind_table)
{
   int i;
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;

   if (adapter->num_rssqs_nd == 1) {
      QFLE3_ERR("RSS_OP_GET_IND_TABLE : No RSS support.\n");
      return VMK_FAILURE;
   }

   if (ind_table->tableSize != T_ETH_INDIRECTION_TABLE_SIZE) {
      QFLE3_ERR( "RSS_OP_GET_IND_TABLE : indirection table size != %d\n",
		 T_ETH_INDIRECTION_TABLE_SIZE);
      return VMK_FAILURE;
   }

   for (i = 0; i < ind_table->tableSize; i++) {
      ind_table->table[i] =  adapter->netq_rss_raw_ind_tbl[i];
   }

   QFLE3_DBG(QFLE3_DBG_QUEUE,
	     "RSS_OP_GET_IND_TABLE: ind_table_size:%d\n", ind_table->tableSize);

   for (i = 0; i < T_ETH_INDIRECTION_TABLE_SIZE; i += 8) {
      QFLE3_DBG(QFLE3_DBG_QUEUE,
		"ind_table[%d to %d]: %d %d %d %d %d %d %d %d\n",
		i, i + 7,
		ind_table->table[i], ind_table->table[i+1], ind_table->table[i+2], ind_table->table[i+3],
		ind_table->table[i+4], ind_table->table[i+5], ind_table->table[i+6], ind_table->table[i+7]);
   }

   return VMK_OK;
}
#endif
