# Driver definition for native vmxnet3 driver.
#
# When developing a driver for release through the async program:
#  * set "vendor" to the name of your company
#  * set "vendor email" to the contact e-mail provided by your company
#  * increment the version number if the source has come from VMware
#  * remove "version_bump" if present
#
# When bringing an async driver inbox at VMware:
#  * leave "version" as is from the async release
#  * set "version_bump" to 1
#  * set "vendor" to 'Vmware, Inc.'
#  * set "vendorEmail" to the VMware contact e-mail address
#
# If updating the driver at VMware:
#  * increment "version bump" or contact the IHV for a new version number
#
# If updating the driver at an async vendor:
#  * increment the version number (do not touch version bump)

#
# identification section
#
from SCons.Script import *
import os

try:
   from vmware.vmkapi import ESXVmkapiCurrentVersion
except ImportError:
   # 5.5 DDK
   from vmware import ESXVmkapiCurrentVersion
from inspect import getsourcefile
from os.path import dirname, join
ddk_ver_mapping = {
   "v2_2_0_0"  :  2013,
   "v2_3_0_0"  :  2015,
   "v2_4_0_0"  :  2016,
   "v2_5_0_0"  :  2017,

   "5.5.0"     :  2013,
   "6.0.0"     :  2015,
   "6.1.0"     :  2016,
   "6.5.0"     :  2016,
   "6.6.0"     :  2017,
}
ddk_ver = ddk_ver_mapping[ESXVmkapiCurrentVersion()]
driver_name = "qfle3"
driver_dump_name = "qfle3_dump"
driver_ver = "1.0.84.0"

qfle3_identification = {
   "name"            : driver_name,
   "module type"     : "device driver",
   "binary compat"   : "yes",
   "summary"         : "Network driver for QLogic E3 Controller",
   "description"     : "Native QLogic E3 network driver for VMware ESXi",
   "version"         : driver_ver,
   "license"         : "QLogic_Proprietary",
   "vendor"          : "QLogic",
   "vendor_code"     : "QLC",
   "vendor_email"    : "support@qlogic.com",
}
#
# Build the Driver Module
#
module_def = {
   "identification"  : qfle3_identification,

   "source files"    : [
                        "qfle3_mod.c",
			"qfle3_drv.c",
			"qfle3_utils.c",
			"qfle3_mq.c",
			"qfle3_uplink.c",
			"qfle3_dcbx.c",
			"qfle3_stats.c",
			"qfle3_ecore_link.c",
			"qfle3_ecore_sp.c",
			"qfle3_init_values_e2.c",
			"qfle3_mgmt.c",
			"qfle3_mgmt_api.c",
			"qfle3_cnic.c",
                       ],
   "cc warnings"     : [ ],
   "cc flags"        : ["-D__LITTLE_ENDIAN",
                        "-DQFLE3_SRIOV",
			"-DQFLE3_NPAR",
		        "-DQFLE3_CNIC",
		    "-DQFLE3_CREATE_CNIC_QS",
		    "-DECORE_L5",
		       ],
   "cc defs"         : { 'QFLE3_DRIVER_NAME=\\"%s\\"' % (driver_name)    : None,
                         'QFLE3_DUMP_FILE_NAME=\\"%s\\"' % (driver_dump_name)  : None,
                         'QFLE3_DRIVER_VERSION=\\"%s\\"' % (driver_ver)  : None,
                         "ESX_DDK_VERSION=%d"            % (ddk_ver) : None,
			},
}
qfle3_module = defineKernelModule(module_def)
#
# Build the Driver's Device Definition
#
device_def = {
   "identification"  : qfle3_identification,
   "device spec"     : "qfle3_devices.py",
}
qfle3_device_def = defineDeviceSpec(device_def)

#
# Build the VIB
#
qfle3_vib_def = {
   "identification"  : qfle3_identification,
   "payload"         : [ qfle3_module,
                         qfle3_device_def,
                       ],
   "vib properties"  : {
      "provides"                : [ ],
      "depends"                 : [ ],
      "conflicts"               : [ ],
      "replaces"                : [ ],
      "maintenance-mode"        : True,
      "live-install-allowed"    : False,
      "live-remove-allowed"     : False,
      "stateless-ready"         : True,
      "acceptance-level"        : 'certified',
   }
}
qfle3_vib = defineModuleVib(qfle3_vib_def)

#
# Build the Offline Bundle
#
qfle3_bulletin_def = {
   "identification" : qfle3_identification,
   "vib"            : qfle3_vib,

   "bulletin" : {
      # These elements show the default values for the corresponding items in bulletin.xml file
      # Uncomment a line if you need to use a different value
      #'severity'    : 'general',
      #'category'    : 'Enhancement',
      #'releaseType' : 'extension',
      #'urgency'     : 'Important',

      'kbUrl'       : 'http://kb.vmware.com/kb/example.html',

      # 1. At least one target platform needs to be specified with 'productLineID'
      # 2. The product version number may be specified explicitly, like 7.8.9,
      # or, when it's None or skipped, be a default one for the devkit
      # 3. 'locale' element is optional
      'platforms'   : [ {'productLineID':'ESXi'},
      #                 {'productLineID':'ESXi', 'version':"7.8.9", 'locale':''}
                      ]
   }
}
qfle3_bundle =  defineOfflineBundle(qfle3_bulletin_def)
