/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

/* SV: XXX: include all qfle3 headers. */
#include "qcnic_vmk.h"
#include "qcnic_if.h"
#include "qfle3.h"
#include "qfle3_legacy.h"

//#include "bnx2_compat0.h"
#include "57xx_iscsi_constants.h"
#include "57xx_iscsi_hsi.h"
#include "bnx2fc_constants.h"
#include "qcnic.h"
#include "qfle3_57710_int_offsets.h"
#include "qfle3_57711_int_offsets.h"
#include "qfle3_57712_int_offsets.h"
#include "qcnic_defs.h"
//#include "cnic_register.h"

//SV: XXX: Include qfle3 namespace for exported symbol access.
//__VMK_NAMESPACE_REQUIRED("qfle3_if", "1");

struct cnic_driver_info_t cnic_driver_info = {
	.heap_id = VMK_INVALID_HEAP_ID,
	.heap_id_dma = VMK_INVALID_HEAP_ID,
	.cnic_vmkLog = VMK_INVALID_LOG_HANDLE,
	.lock_domain = VMK_LOCKDOMAIN_INVALID,
	.timer_queue = VMK_INVALID_TIMER_QUEUE,
	.delayed_tq = VMK_INVALID_TIMER_QUEUE,
	.drv_lock = VMK_LOCK_INVALID,

	/*
	 * Empty vmk_MgmtApiSignature on which we can attach
	 * key-value pairs.
	 */
	.kvSig = {
		.version = VMK_REVISION_FROM_NUMBERS(1,0,0,0),
		.name.string = "QCNIC",
		.vendor.string = "qlogic",
		.numCallbacks = 0,
		.callbacks = NULL,
	},

	/* DEBUG: XXX: trace memory operations. */
	.mem = 0,
	.count = 0,
};

static void cnic_msix_rx(void *, vmk_IntrCookie);

static void cnic_ack_igu_sb(struct cnic_dev *, vmk_uint8, vmk_uint8,
	vmk_uint16, vmk_uint8, vmk_uint8);
static inline void cnic_ack_qfle3_int(struct cnic_dev *, vmk_uint8, vmk_uint8,
	vmk_uint16, vmk_uint8, vmk_uint8);

struct cnic_dev *cnic_cm_select_dev(vmk_IscsiNetHandle,
	vmk_SocketIPAddress *, int);

struct msix_handlers cnic_msix_handlers[1] = {
	{cnic_msix_rx},		/* CNIC_RX_IDX */
};

#if defined(CONFIG_VLAN_8021Q) || defined(CONFIG_VLAN_8021Q_MODULE)
#define BCM_VLAN 1
#endif

unsigned long cnic_debug = 0;
VMK_MODPARAM(cnic_debug, long, "Set driver debug msglevel.");

unsigned long cnic_dump_kwqe_en = 1;
VMK_MODPARAM(cnic_dump_kwqe_en, long,
	"parameter to enable (1)/disable (0) kwqe "
	"(single work-queue element message) logging.");

static int cnic_alloc_kcq(struct cnic_dev *dev, struct kcq_info *info,
		int use_pg_tbl);

VMK_ReturnStatus cnic_register_cb_ops(struct cnic_dev *,
	struct cnic_ll2_cb_ops *, vmk_uint64 cookie);
VMK_ReturnStatus cnic_unregister_cb_ops(struct cnic_dev *);
VMK_ReturnStatus cnic_ll2_send(struct cnic_dev *, vmk_PktHandle *);

static struct cnic_ll2_ops ll2_ops = {
	.ll2_send = cnic_ll2_send,
	.register_cb_ops = cnic_register_cb_ops,
	.unregister_cb_ops = cnic_unregister_cb_ops,
};

static int cnic_service_bnx2(void *, void *);
static int cnic_service_qfle3(void *, void *);
static int cnic_ctl(void *, struct cnic_ctl_info *);
int cnic_link_update(void *, vmk_uint8, vmk_uint16);
int cnic_ll2_recv(void *, vmk_PktHandle *);

#if (CNIC_ISCSI_OOO_SUPPORT)
static void cnic_alloc_bnx2_ooo_resc(struct cnic_dev *dev);
static void cnic_alloc_qfle3_ooo_resc(struct cnic_dev *dev);
static void cnic_start_bnx2_ooo_hw(struct cnic_dev *dev);
static void cnic_start_qfle3_ooo_hw(struct cnic_dev *dev);
static void cnic_stop_bnx2_ooo_hw(struct cnic_dev *dev);
static void cnic_stop_qfle3_ooo_hw(struct cnic_dev *dev);
static void cnic_handle_bnx2_ooo_rx_event(struct cnic_dev *dev);
static void cnic_handle_qfle3_ooo_rx_event(struct cnic_dev *dev);
static void cnic_handle_bnx2_ooo_tx_event(struct cnic_dev *dev);
static void cnic_handle_qfle3_ooo_tx_event(struct cnic_dev *dev);
static void cnic_bnx2_ooo_iscsi_conn_update(struct cnic_dev *dev,
		struct kwqe *kwqe);
static void cnic_bnx2_ooo_iscsi_destroy(struct cnic_dev *dev,
		struct kwqe *kwqe);
static void cnic_qfle3_ooo_iscsi_conn_update(struct cnic_dev *dev,
		struct kwqe *kwqe);
static void cnic_free_ooo_resc(struct cnic_dev *dev);
static void cnic_conn_ooo_init(struct cnic_local *cp, vmk_uint32 l5_cid);
static void cnic_flush_ooo(struct cnic_dev *dev, vmk_uint32 l5_cid);
#endif

int cnic_init_pci(struct cnic_dev *, void *, void *);
void cnic_release_pci(struct cnic_dev *);
static void cnic_delete_task(void *);

static struct cnicOps cnic_bnx2_ops = {
	.cnicIRQHandler	= cnic_service_bnx2,
	.cnicNotify		= cnic_ctl,
};

static struct cnicOps cnic_qfle3_ops = {
	.cnicIRQHandler	= cnic_service_qfle3,
	.cnicNotify		= cnic_ctl,
	.cnicLinkUpdate	= cnic_link_update,
	.cnicLL2Rx		= cnic_ll2_recv,
};

static inline void cnic_hold(struct cnic_dev *dev)
{
	vmk_AtomicInc64(&dev->ref_count);
}

static inline void cnic_put(struct cnic_dev *dev)
{
	vmk_AtomicDec64(&dev->ref_count);
}

static inline void csk_hold(struct cnic_sock *csk)
{
	vmk_AtomicInc64(&csk->ref_count);
}

static inline void csk_put(struct cnic_sock *csk)
{
	vmk_AtomicDec64(&csk->ref_count);
}

static void cnic_print_ramrod_info(struct cnic_dev *dev, vmk_uint32 cmd,
	vmk_uint32 cid, struct kwqe_16 *kwqe)
{
	char kwq_op[16];

	if (!cnic_dump_kwqe_en)
		return;

	switch(cmd) {
		/* Common RAMROD's */
		case RAMROD_CMD_ID_ETH_CLIENT_SETUP:
			vmk_Sprintf(kwq_op, "%s", "CLIENT_SETUP");
			break;
		case RAMROD_CMD_ID_COMMON_CFC_DEL:
			vmk_Sprintf(kwq_op, "%s", "CFC_DEL");
			break;
			/* iSCSI RAMROD's */
		case L5CM_RAMROD_CMD_ID_TCP_CONNECT:
			vmk_Sprintf(kwq_op, "%s", "TCP_CONNECT");
			break;
		case ISCSI_RAMROD_CMD_ID_UPDATE_CONN:
			vmk_Sprintf(kwq_op, "%s", "UPDATE_CONN");
			break;
		case L5CM_RAMROD_CMD_ID_CLOSE:
			vmk_Sprintf(kwq_op, "%s", "TCP_CLOSE");
			break;
		case L5CM_RAMROD_CMD_ID_SEARCHER_DELETE:
			vmk_Sprintf(kwq_op, "%s", "SEARCHER_DEL");
			break;
		case L5CM_RAMROD_CMD_ID_TERMINATE_OFFLOAD:
			vmk_Sprintf(kwq_op, "%s", "TERMINATE");
			break;
		case L5CM_RAMROD_CMD_ID_ABORT:
			vmk_Sprintf(kwq_op, "%s", "TCP_ABORT");
			break;
			/* FCoE RAMROD's */
		case FCOE_RAMROD_CMD_ID_INIT_FUNC:
			vmk_Sprintf(kwq_op, "%s", "FCOE_INIT");
			break;
		case FCOE_RAMROD_CMD_ID_OFFLOAD_CONN:
			vmk_Sprintf(kwq_op, "%s", "FCOE_OFLD_CONN");
			break;
		case FCOE_RAMROD_CMD_ID_ENABLE_CONN:
			vmk_Sprintf(kwq_op, "%s", "FCOE_EN_CONN");
			break;
		case FCOE_RAMROD_CMD_ID_DISABLE_CONN:
			vmk_Sprintf(kwq_op, "%s", "FCOE_DIS_CONN");
			break;
		case FCOE_RAMROD_CMD_ID_TERMINATE_CONN:
			vmk_Sprintf(kwq_op, "%s", "FCOE_TERM_CONN");
			break;
		case FCOE_RAMROD_CMD_ID_DESTROY_FUNC:
			vmk_Sprintf(kwq_op, "%s", "FCOE_DESTROY");
			break;
		default:
			vmk_Sprintf(kwq_op, "%s:%x", "Unknown", cmd);
	}

	CNIC_INFO(dev, CNIC_MSG_DRV, "CID=0x%x, %s RAMROD :: 0x%x, 0x%x"
		"0x%x, 0x%x\n", cid, kwq_op,
		kwqe->kwqe_info0, kwqe->kwqe_info1,
		kwqe->kwqe_info2, kwqe->kwqe_info3);
}


static void cnic_dump_kcq_entry(struct cnic_dev *dev, struct kcqe *kcqe,
	int index, char *add_info)
{
	if (add_info)
		CNIC_INFO(dev, CNIC_MSG_DRV, "KCQE[%x] = 0x%0x, 0x%x, 0x%x, 0x%x, "
			"0x%x, 0x%x, 0x%x, OP=0x%x ---> [%s]", index,
			kcqe->kcqe_info0, kcqe->kcqe_info1, kcqe->kcqe_info2,
			kcqe->kcqe_info3, kcqe->kcqe_info4, kcqe->kcqe_info5,
			kcqe->kcqe_info6, kcqe->kcqe_op_flag, add_info);
	else
		CNIC_INFO(dev, CNIC_MSG_DRV, "KCQE[%x] = 0x%0x, 0x%x, 0x%x, 0x%x, "
			"0x%x, 0x%x, 0x%x, OP=0x%x", index,
			kcqe->kcqe_info0, kcqe->kcqe_info1, kcqe->kcqe_info2,
			kcqe->kcqe_info3, kcqe->kcqe_info4, kcqe->kcqe_info5,
			kcqe->kcqe_info6, kcqe->kcqe_op_flag);
}

/* This function will dump KCQEs pointed by page. */
static void cnic_dump_kcq_page(struct cnic_dev *dev, int kcq_page,
	vmk_uint16 sw_prod, vmk_uint16 hw_prod)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct kcq_info *info = &cp->kcq1;
	struct kcqe *kcqe;
	int i;

	kcqe = &info->kcq[kcq_page][0];

	for (i = 0; i < MAX_KCQE_CNT; i++, kcqe++) {
		if (KCQ_PG(sw_prod) == kcq_page && KCQ_IDX(sw_prod) == i)
			/* SW PROD INDEX */
			cnic_dump_kcq_entry(dev, kcqe, i, "SW_PROD");
		else if (KCQ_PG(hw_prod) == kcq_page && KCQ_IDX(hw_prod) == i)
			/* HW PROD INDEX */
			cnic_dump_kcq_entry(dev, kcqe, i, "HW_PROD");
		else
			cnic_dump_kcq_entry(dev, kcqe, i, NULL);
	}
}

/*
 * This function will print 3 pages worth of KCQEs,
 * currently active, previous and next page.
 */
void cnic_dump_kcq(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct kcq_info *info = &cp->kcq1;
	int current_page;
	int prev_page;
	int next_page;
	vmk_uint16 sw_prod = 0;
	vmk_uint16 hw_prod;

	sw_prod = info->sw_prod_idx;
	sw_prod &= MAX_KCQ_IDX;
	hw_prod = *info->hw_prod_idx_ptr;
	hw_prod = info->hw_idx(hw_prod);

	CNIC_INFO(dev, CNIC_MSG_DRV, "sw_prod_idx = 0x%x, hw_prod_idx = 0x%x",
		info->sw_prod_idx, hw_prod);

	current_page = KCQ_PG(sw_prod);
	if (current_page == 0)
		prev_page = KCQ_PAGE_CNT - 1;
	else
		prev_page = current_page - 1;

	if (current_page == KCQ_PAGE_CNT - 1)
		next_page = 0;
	else
		next_page = current_page + 1;

	CNIC_INFO(dev, CNIC_MSG_DRV, "dumping PREVIOUS PAGE (%d)", prev_page);
	cnic_dump_kcq_page(dev, prev_page, sw_prod, hw_prod);

	CNIC_INFO(dev, CNIC_MSG_DRV, "dumping CURRENT PAGE (%d)", current_page);
	cnic_dump_kcq_page(dev, current_page, sw_prod, hw_prod);

	CNIC_INFO(dev, CNIC_MSG_DRV, "dumping NEXT PAGE (%d)", next_page);
	cnic_dump_kcq_page(dev, next_page, sw_prod, hw_prod);
}

static struct cnic_dev *cnic_from_netdev(vmk_Device netdev)
{
	struct cnic_dev *cdev;

	vmk_SemaLock(&cnic_driver_info.cnic_lock);
	ql_vmk_list_each_entry(cdev, &cnic_driver_info.cnic_dev_list,
		list, struct cnic_dev) {
		if (netdev == cdev->netdev) {
			cnic_hold(cdev);
			vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
			return cdev;
		}
	}
	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

	return NULL;
}

static inline void ulp_get(struct cnic_ulp_ops *ulp_ops)
{
	vmk_AtomicInc64(&ulp_ops->ref_count);
}

static inline void ulp_put(struct cnic_ulp_ops *ulp_ops)
{
	vmk_AtomicDec64(&ulp_ops->ref_count);
}

static void cnic_ctx_wr(struct cnic_dev *dev, vmk_uint32 cid_addr,
	vmk_uint32 off, vmk_uint32 val)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct drv_ctl_info info;
	struct drv_ctl_io *io = &info.data.io;

	vmk_Memset(&info, 0, sizeof(struct drv_ctl_info));
	info.cmd = DRV_CTL_CTX_WR_CMD;
	io->cid_addr = cid_addr;
	io->offset = off;
	io->data = val;
	ethdev->cnicDriverCtl(dev->netdev, &info);
}

static void cnic_ctx_tbl_wr(struct cnic_dev *dev, vmk_uint32 off, vmk_IOA addr)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct drv_ctl_info info;
	struct drv_ctl_io *io = &info.data.io;

	vmk_Memset(&info, 0, sizeof(struct drv_ctl_info));
	info.cmd = DRV_CTL_CTXTBL_WR_CMD;
	io->offset = off;
	io->dma_addr = addr;
	ethdev->cnicDriverCtl(dev->netdev, &info);
}

static void cnic_npar_ring_ctl(struct cnic_dev *dev, int start)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct drv_ctl_info info;

	vmk_Memset(&info, 0, sizeof(struct drv_ctl_info));
	if (start)
		info.cmd = DRV_CTL_START_NPAR_CMD;
	else
		info.cmd = DRV_CTL_STOP_NPAR_CMD;

	ethdev->cnicDriverCtl(dev->netdev, &info);
}

static void cnic_reg_wr_ind(struct cnic_dev *dev, vmk_uint32 off, vmk_uint32 val)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct drv_ctl_info info;
	struct drv_ctl_io *io = &info.data.io;

	vmk_Memset(&info, 0, sizeof(struct drv_ctl_info));
	info.cmd = DRV_CTL_IO_WR_CMD;
	io->offset = off;
	io->data = val;
	ethdev->cnicDriverCtl(dev->netdev, &info);
}

static vmk_uint32 cnic_reg_rd_ind(struct cnic_dev *dev, vmk_uint32 off)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct drv_ctl_info info;
	struct drv_ctl_io *io = &info.data.io;

	vmk_Memset(&info, 0, sizeof(struct drv_ctl_info));
	info.cmd = DRV_CTL_IO_RD_CMD;
	io->offset = off;
	ethdev->cnicDriverCtl(dev->netdev, &info);
	return io->data;
}

static void cnic_ulp_ctl(struct cnic_dev *dev, int ulp_type, vmk_Bool reg, int state)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct drv_ctl_info info;
	struct fcoe_capabilities *fcoe_cap =
		&info.data.register_data.fcoe_features;

	vmk_Memset(&info, 0, sizeof(struct drv_ctl_info));
	if (reg) {
		info.cmd = DRV_CTL_ULP_REGISTER_CMD;
		if (ulp_type == CNIC_ULP_FCOE && dev->fcoe_cap)
			vmk_Memcpy(fcoe_cap, dev->fcoe_cap, sizeof(*fcoe_cap));
	} else
		info.cmd = DRV_CTL_ULP_UNREGISTER_CMD;

	info.data.ulp_type = ulp_type;
	info.drv_state = state;
	ethdev->cnicDriverCtl(dev->netdev, &info);
}

static int cnic_in_use(struct cnic_sock *csk)
{
	return vmk_BitVectorTest(csk->flags, SK_F_INUSE);
}

static void cnic_spq_completion(struct cnic_dev *dev, int cmd, vmk_uint32 count)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct drv_ctl_info info;

	vmk_Memset(&info, 0, sizeof(struct drv_ctl_info));
	info.cmd = cmd;
	info.data.credit.credit_count = count;
	ethdev->cnicDriverCtl(dev->netdev, &info);
}

#if (CNIC_ISCSI_OOO_SUPPORT)
static int cnic_get_ooo_cqe(struct cnic_dev *dev, struct cnic_ooo_cqe* cqe)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct drv_ctl_info info;
	struct drv_ctl_ooo_cqe *ooo_cqe = &info.data.ooo_cqe;

	vmk_Memset(&info, 0, sizeof(struct drv_ctl_info));
	info.cmd = DRV_CTL_GET_OOO_CQE;
	ooo_cqe->cqe = cqe;
	return ethdev->cnicDriverCtl(dev->netdev, &info);
}

static int cnic_send_ooo_pkt(vmk_PktHandle *skb, struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct drv_ctl_info info;
	struct drv_ctl_ooo_pkt *ooo_pd = &info.data.pkt_desc;

	vmk_Memset(&info, 0, sizeof(struct drv_ctl_info));
	info.cmd = DRV_CTL_SEND_OOO_PKT;
	ooo_pd->skb = skb;

	return ethdev->cnicDriverCtl(dev->netdev, &info);
}

static int cnic_reuse_ooo_pkt(vmk_PktHandle *skb, struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct drv_ctl_info info;
	struct drv_ctl_ooo_pkt *ooo_pd = &info.data.pkt_desc;

	if (!skb) {
		CNIC_ERR(dev, "skb is NULL in reuse!");
		return VMK_BAD_PARAM;
	}

	vmk_Memset(&info, 0, sizeof(struct drv_ctl_info));
	info.cmd = DRV_CTL_REUSE_OOO_PKT;
	ooo_pd->skb = skb;
	return ethdev->cnicDriverCtl(dev->netdev, &info);
}

static int cnic_comp_ooo_tx_pkts(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct drv_ctl_info info;

	vmk_Memset(&info, 0, sizeof(struct drv_ctl_info));
	info.cmd = DRV_CTL_COMP_OOO_TX_PKTS;
	return ethdev->cnicDriverCtl(dev->netdev, &info);
}
#endif

static int cnic_get_l5_cid(struct cnic_local *cp, vmk_uint32 cid, vmk_uint32 *l5_cid)
{
	vmk_uint32 i;

	if (!cp->ctx_tbl)
		return VMK_BAD_PARAM;

	for (i = 0; i < cp->max_cid_space; i++) {
		if (cp->ctx_tbl[i].cid == cid) {
			*l5_cid = i;
			return 0;
		}
	}
	return VMK_BAD_PARAM;
}

static int cnic_offld_prep(struct cnic_sock *csk)
{
	if (vmk_BitVectorAtomicTestAndSet(csk->flags, SK_F_OFFLD_SCHED))
		return 0;

	if (!vmk_BitVectorTest(csk->flags, SK_F_CONNECT_START)) {
		vmk_BitVectorClear(csk->flags, SK_F_OFFLD_SCHED);
		return 0;
	}

	return 1;
}

static int cnic_close_prep(struct cnic_sock *csk)
{
	vmk_BitVectorClear(csk->flags, SK_F_CONNECT_START);
	vmk_CPUMemFenceReadWrite();

	if (vmk_BitVectorAtomicTestAndClear(csk->flags, SK_F_OFFLD_COMPLETE)) {
		while (vmk_BitVectorAtomicTestAndClear(csk->flags, SK_F_OFFLD_SCHED))
			vmk_WorldSleep(1*VMK_USEC_PER_MSEC);

		return 1;
	}

	return 0;
}

static int cnic_abort_prep(struct cnic_sock *csk)
{
	int i = 0;

	vmk_BitVectorClear(csk->flags, SK_F_CONNECT_START);
	vmk_CPUMemFenceReadWrite();

	while (vmk_BitVectorAtomicTestAndClear(csk->flags, SK_F_OFFLD_SCHED)) {
		if (++i > 10000) {
			CNIC_WARN(csk->dev, "stuck on CID=0x%x, aborting",
				csk->cid);
			break;
		}
		vmk_WorldSleep(1*VMK_USEC_PER_MSEC);
	}

	if (vmk_BitVectorAtomicTestAndClear(csk->flags, SK_F_OFFLD_COMPLETE)) {
		csk->state = L4_KCQE_OPCODE_VALUE_RESET_COMP;
		return 1;
	}

	return 0;
}

/* SV: No longer in use. */
#if 0
int cnic_register_driver(int ulp_type, struct cnic_ulp_ops *ulp_ops)
{
	struct cnic_dev *dev;

	if (ulp_ops->version != CNIC_ULP_OPS_VER) {
		DRV_ERR("ulp %x not compatible with cnic, "
			"expecting: 0x%x got: 0x%x", ulp_type,
			CNIC_ULP_OPS_VER, ulp_ops->version);
		return VMK_BAD_PARAM;
	}

	if (ulp_type < 0 || ulp_type >= MAX_CNIC_ULP_TYPE) {
		DRV_ERR("Bad type [%d].", ulp_type);
		return VMK_BAD_PARAM;
	}

	vmk_SemaLock(&cnic_driver_info.cnic_lock);
	if (cnic_ulp_tbl[ulp_type]) {
		DRV_ERR("Type [%d] has already been registered.", ulp_type);
		vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
		return VMK_BUSY;
	}

	ql_vmk_list_each_entry(dev, &cnic_driver_info.cnic_dev_list,
		list, struct cnic_dev) {
		struct cnic_local *cp = dev->cnic_priv;

		vmk_BitVectorClear(cp->ulp_flags[ulp_type], ULP_F_INIT);
	}

	vmk_AtomicWrite64(&ulp_ops->ref_count, 0);
	cnic_ulp_tbl[ulp_type] = ulp_ops;
	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

	/* Prevent race conditions with netdev_event */
	vmk_SemaLock(&cnic_driver_info.cnic_lock);
	ql_vmk_list_each_entry(dev, &cnic_driver_info.cnic_dev_list,
		list, struct cnic_dev) {
		struct cnic_local *cp = dev->cnic_priv;

		if (!vmk_BitVectorAtomicTestAndClear(cp->ulp_flags[ulp_type],
				ULP_F_INIT)) {
			vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
			ulp_ops->cnic_init(dev);
			vmk_SemaLock(&cnic_driver_info.cnic_lock);
		}
	}
	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

	return 0;
}

int cnic_unregister_driver(int ulp_type)
{
	struct cnic_dev *dev;
	struct cnic_ulp_ops *ulp_ops;
	int i = 0;

	if (ulp_type < 0 || ulp_type >= MAX_CNIC_ULP_TYPE) {
		DRV_ERR("Bad type [%d].", ulp_type);
		return VMK_BAD_PARAM;
	}

	vmk_SemaLock(&cnic_driver_info.cnic_lock);
	ulp_ops = cnic_ulp_tbl[ulp_type];
	if (!ulp_ops) {
		DRV_ERR("Type [%d] has not been registered.", ulp_type);
		goto out_unlock;
	}

	ql_vmk_list_each_entry(dev, &cnic_driver_info.cnic_dev_list,
		list, struct cnic_dev) {
		struct cnic_local *cp = dev->cnic_priv;

		if (cp->ulp_ops[ulp_type]) {
			CNIC_ERR(dev, "Type [%d] still has devices registered.", ulp_type);
			goto out_unlock;
		}
	}

	cnic_ulp_tbl[ulp_type] = NULL;
	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

	while ((vmk_AtomicRead64(&ulp_ops->ref_count) != 0) && (i < 20)) {
		vmk_WorldSleep(100*VMK_USEC_PER_MSEC);
		i++;
	}

	if (vmk_AtomicRead64(&ulp_ops->ref_count) != 0)
		CNIC_ERR(dev, "Failed waiting for ref count to go to zero.");

	return 0;

out_unlock:
	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

	return VMK_BAD_PARAM;
}
#endif

static int cnic_register_netdev(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	VMK_ReturnStatus status;

	if (!ethdev)
		return VMK_INVALID_ADAPTER;

	if (ethdev->driverState & CNIC_DRV_STATE_REGD)
		return 0;

	CNIC_INFO(dev, CNIC_MSG_INIT, "before ethdev->cnicRegister.");
	status = ethdev->cnicRegister(dev->netdev, cp->cnic_ops, dev);
	if (status != VMK_OK)
		CNIC_ERR(dev, "cnic register failed.");

	dev->max_iscsi_conn = ethdev->max_iscsi_conn;
	if (ethdev->driverState & CNIC_DRV_STATE_NO_ISCSI)
		dev->max_iscsi_conn = 0;

	CNIC_INFO(dev, CNIC_MSG_DRV, "status=0x%x", status);

	return status;
}

static void cnic_unregister_netdev(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;

	if (!ethdev)
		return;

	if (ethdev->driverState & CNIC_DRV_STATE_REGD)
		ethdev->cnicUnregister(dev->netdev);
}

static int cnic_start_hw(struct cnic_dev *);
static void cnic_stop_hw(struct cnic_dev *);

VMK_ReturnStatus cnic_register_cb_ops(struct cnic_dev *cdev,
	struct cnic_ll2_cb_ops *ll2_cb_ops, vmk_uint64 cookie)
{
	if (!cdev) {
		DRV_ERR("Error: cnic-dev is NULL!!");
		return VMK_FAILURE;
	}

	/* Register LL2 callback. */
	vmk_SpinlockLock(cdev->ll2_cb_info.ll2_cb_lck);
	cdev->ll2_cb_info.ll2_cb_ops = ll2_cb_ops;
	cdev->ll2_cb_info.cookie = cookie;
	vmk_SpinlockUnlock(cdev->ll2_cb_info.ll2_cb_lck);

	return VMK_OK;
}

VMK_ReturnStatus cnic_unregister_cb_ops(struct cnic_dev *cdev)
{
	if (!cdev) {
		DRV_ERR("Error: cnic-dev is NULL!!");
		return VMK_FAILURE;
	}

	/* UnRegister LL2 callback. */
	vmk_SpinlockLock(cdev->ll2_cb_info.ll2_cb_lck);
	cdev->ll2_cb_info.ll2_cb_ops = NULL;
	cdev->ll2_cb_info.cookie = 0;
	vmk_SpinlockUnlock(cdev->ll2_cb_info.ll2_cb_lck);

	return VMK_OK;
}


VMK_ReturnStatus cnic_ll2_send(struct cnic_dev *dev, vmk_PktHandle *pkt)
{
	struct qfle3_adapter *adapter = NULL;
	struct cnic_local *cp = dev->cnic_priv;
	VMK_ReturnStatus status = VMK_OK;

	vmk_DeviceGetRegisteringDriverData(dev->ldev, (vmk_AddrCookie *) &adapter);

	CNIC_INFO(dev, CNIC_MSG_NW, "before tx-pkt");

	/* LL2-TX: Submit a control-path pkt to nic's FP queue (FCoE Tx). */
	if (cp->nic_ops && cp->nic_ops->cnicLL2Tx &&
			vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP))
		status = cp->nic_ops->cnicLL2Tx(adapter, pkt);
	else
		status = VMK_NOT_FOUND;

	CNIC_INFO(dev, CNIC_MSG_NW, "after tx-pkt. status=%s",
		vmk_StatusToString(status));

	if (VMK_UNLIKELY(status != VMK_OK))
		CNIC_ERR(dev, "Error: in submitting ll2 pkt. %s",
			vmk_StatusToString(status));

	return status;
}

static int cnic_register_device(struct cnic_dev *dev, int ulp_type,
	void *ulp_ctx, struct cnic_ulp_ops *ulp_ops)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (ulp_type < 0 || ulp_type >= MAX_CNIC_ULP_TYPE) {
		CNIC_ERR(dev, "Bad type [%d].", ulp_type);
		return VMK_BAD_PARAM;
	}

	/* Associate cnic_ulp_ops. First verify interface version. */
	if (ulp_ops->version != CNIC_ULP_OPS_VER) {
		CNIC_ERR(dev, "ulp %x not compatible with cnic, "
			"expecting: 0x%x got: 0x%x", ulp_type,
			CNIC_ULP_OPS_VER, ulp_ops->version);
		return VMK_BAD_PARAM;
	}

	vmk_SemaLock(&cnic_driver_info.cnic_lock);
	if (dev->cnic_ulp_tbl[ulp_type]) {
		CNIC_ERR(dev, "Type [%d] has already been registered.", ulp_type);
		vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
		return VMK_BUSY;
	}

	vmk_BitVectorClear(cp->ulp_flags[ulp_type], ULP_F_INIT);

	vmk_AtomicWrite64(&ulp_ops->ref_count, 0);
	dev->cnic_ulp_tbl[ulp_type] = ulp_ops;
	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);


	vmk_SemaLock(&cnic_driver_info.cnic_lock);
	if (dev->cnic_ulp_tbl[ulp_type] == NULL) {
		CNIC_ERR(dev, "Driver with type [%d] has not been registered.",
			ulp_type);
		vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
		return VMK_STATUS_PENDING;
	}
	if (cp->ulp_ops[ulp_type]) {
		CNIC_ERR(dev, "Type [%d] has already been registered to this device.",
			ulp_type);
		vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
		return VMK_BUSY;
	}

	dev->uplinkLinkState = vmk_UplinkLinkStatusGet(adapter->uplink);
	vmk_NameCopy(&dev->uplinkName, &adapter->uplinkName);

	vmk_BitVectorClear(cp->ulp_flags[ulp_type], ULP_F_START);
	cp->ulp_handle[ulp_type] = ulp_ctx;
	ulp_ops = dev->cnic_ulp_tbl[ulp_type];
	cp->ulp_ops[ulp_type] = ulp_ops;
	cnic_hold(dev);
	if (!dev->use_count) {
		if (!vmk_BitVectorTest(dev->flags, CNIC_F_IF_GOING_DOWN) &&
				(vmk_UplinkStateGet(adapter->uplink) ==
				 VMK_UPLINK_STATE_ENABLED)) {
			vmk_BitVectorSet(dev->flags, CNIC_F_IF_UP);
		}
	}
	dev->use_count++;


	if (vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP))
		if (!vmk_BitVectorAtomicTestAndSet(cp->ulp_flags[ulp_type],
				ULP_F_START))
			ulp_ops->cnic_start(cp->ulp_handle[ulp_type]);

	/* Populate ll2 ops for this cnic-device. */
	dev->ll2_ops.ll2_send = cnic_ll2_send;
	dev->ll2_ops.register_cb_ops = cnic_register_cb_ops;
	dev->ll2_ops.unregister_cb_ops = cnic_unregister_cb_ops;

	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

	cnic_ulp_ctl(dev, ulp_type, VMK_TRUE, DRV_ACTIVE);

	return 0;
}

static int cnic_unregister_device(struct cnic_dev *dev, int ulp_type)
{
	struct cnic_local *cp = dev->cnic_priv;
	int i = 0;

	if (ulp_type < 0 || ulp_type >= MAX_CNIC_ULP_TYPE) {
		CNIC_ERR(dev, "Bad type [%d].", ulp_type);
		return VMK_BAD_PARAM;
	}

	vmk_SemaLock(&cnic_driver_info.cnic_lock);
	if (cp->ulp_ops[ulp_type]) {
		dev->use_count--;
		cp->ulp_ops[ulp_type] = NULL;
		cnic_put(dev);
		dev->cnic_ulp_tbl[ulp_type] = NULL;
	} else {
		CNIC_ERR(dev, "device not registered to this ulp type [%d].", ulp_type);
		vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
		return VMK_BAD_PARAM;
	}
	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

	if (ulp_type == CNIC_ULP_FCOE)
		dev->fcoe_cap = NULL;

	while (vmk_BitVectorTest(cp->ulp_flags[ulp_type], ULP_F_CALL_PENDING)
			&& i < 20) {
		vmk_WorldSleep(100*VMK_USEC_PER_MSEC);
		i++;
	}
	if (vmk_BitVectorTest(cp->ulp_flags[ulp_type], ULP_F_CALL_PENDING))
		CNIC_WARN(dev, "Failed waiting for ULP up call to complete.");

	if(vmk_BitVectorTest(cp->ulp_flags[ulp_type], ULP_F_INIT))
		cnic_ulp_ctl(dev, ulp_type, VMK_FALSE, DRV_UNLOADED);
	else
		cnic_ulp_ctl(dev, ulp_type, VMK_FALSE, DRV_INACTIVE);

	return 0;
}

static void printchar(char ch)
{
	if (ch>=32)
		DUMP_CHAR("%c", ch);
	else
		DUMP_CHAR(".");
}

void qcnic_dump_buffer(void *buffer, vmk_uint32 size)
{
	vmk_uint32 i;
	vmk_uint8 *data = (vmk_uint8*)buffer;
	//vmk_LogMessage("        0  1  2  3  4  5  6  7 -  8  9  A  B  C  D  E  F\n");
	//vmk_LogMessage("---------------------------------------------------------\n");
	for (i = 0; i < size; i++, data++) {
		if (i % 0x10 == 0) {
			DUMP_CHAR("%04X:  %02X", i, *data);
		} else if (i % 0x10 == 0x08) {
			DUMP_CHAR(" - %02X", *data);
		} else if (i % 0x10 == 0xF) {
			DUMP_CHAR(" %02X:  ", *data);
			printchar(*(data-15));
			printchar(*(data-14));
			printchar(*(data-13));
			printchar(*(data-12));
			printchar(*(data-11));
			printchar(*(data-10));
			printchar(*(data-9));
			printchar(*(data-8));
			printchar(*(data-7));
			printchar(*(data-6));
			printchar(*(data-5));
			printchar(*(data-4));
			printchar(*(data-3));
			printchar(*(data-2));
			printchar(*(data-1));
			printchar(*data);
			DUMP_CHAR("\n");
		} else {
			DUMP_CHAR(" %02X", *data);
		}
	}

	if ((i != 0) && (i % 0x10)) {
		DUMP_CHAR("\n");
	}
	DUMP_CHAR("\n");
}

struct eth_spe *
qfle3i_sp_get_next(struct qfle3_adapter *adapter)
{
	struct eth_spe *next_spe = adapter->spq_prod_bd;

	if (adapter->spq_prod_bd == adapter->spq_last_bd) {
		adapter->spq_prod_bd = adapter->spq;
		adapter->spq_prod_idx = 0;
	} else {
		adapter->spq_prod_bd++;
		adapter->spq_prod_idx++;
	}
	return next_spe;
}

/****************************************************************************
* ILT management
****************************************************************************/
struct ilt_line {
    ecore_dma_addr_t page_mapping;
    void *page;
    vmk_uint32 size;
};  
        
struct ilt_client_info {
    vmk_uint32 page_size;
    vmk_uint16 start;
    vmk_uint16 end;
    vmk_uint16 client_num;
    vmk_uint16 flags;
#define ILT_CLIENT_SKIP_INIT    0x1
#define ILT_CLIENT_SKIP_MEM 0x2
};
    
struct ecore_ilt {
    vmk_uint32 start_line;
    struct ilt_line     *lines;
    struct ilt_client_info  clients[4];
#define ILT_CLIENT_CDU  0
#define ILT_CLIENT_QM   1
#define ILT_CLIENT_SRC  2
#define ILT_CLIENT_TM   3
};  

static void cnic_dump_queues(struct cnic_dev *cdev)
{
	struct qfle3_adapter *adapter = NULL;
	struct eth_spe *tmp_spq = NULL;
	struct eth_spe *spq = NULL;
	union event_ring_elem *elem = NULL;
	VMK_ReturnStatus status = VMK_OK;
	int i = 0, rc = 0;
	struct ilt_line *line = NULL;

	if (cdev == NULL)
		return;

	status = vmk_DeviceGetAttachedDriverData(cdev->netdev, (vmk_AddrCookie *) &adapter);
	if (status != VMK_OK) {
        vmk_WarningMessage("qcnic: "
        "vmk_DeviceGetAttachedDriverData failed. status=%s qfle3_adapter:%p\n",
        vmk_StatusToString(status), adapter);
		return;
	}
	CNIC_WARN(cdev, "qfle3_adapter:%p\n", adapter);
	CNIC_WARN(cdev, "Dumping Timer Database\n");

	struct ecore_ilt *ilt = QFLE3_ILT(adapter);
	struct ilt_client_info *ilt_cli = &ilt->clients[ILT_CLIENT_TM];

	for (rc = 0, i = ilt_cli->start; i <= ilt_cli->end; i++) {
		line = &ilt->lines[i];
		qcnic_dump_buffer(line->page, ilt_cli->page_size);
	}

	tmp_spq = adapter->spq;
	if(tmp_spq) {
		CNIC_WARN(cdev, "*** SPQ START ***\n");
		qcnic_dump_buffer(tmp_spq, QFLE3_PAGE_SIZE);
		CNIC_WARN(cdev, "*** SPQ END ***\n");
	}

	CNIC_WARN(cdev, "tmp_spq %p spq %p last bd %p MAX_SP_DESC_CNT 0x%lx}\n",
				tmp_spq, adapter->spq, adapter->spq_last_bd, MAX_SP_DESC_CNT);
	i = 0;
	if(!tmp_spq) {
		CNIC_WARN(cdev, "*** SPQ is NULL ***\n");
		goto EQ;
	}
//   while(1) {
//       qcnic_dump_buffer((void*)&tmp_spq->hdr, sizeof(struct spe_hdr));
//       qcnic_dump_buffer(&tmp_spq->data, sizeof(struct regpair));
//       tmp_spq = qfle3i_sp_get_next(adapter);
//       i++;
//       if((tmp_spq == spq) ||(i == MAX_SP_DESC_CNT))
//           break;
//   }
EQ:
	CNIC_WARN(cdev, "\n*** EQ START %lx mappings %x %x***\n",
			EQ_DESC_MAX_PAGE, U64_HI(adapter->eq_mapping),
			U64_LO(adapter->eq_mapping));
	elem = adapter->eq_ring;
	if (elem) {
		qcnic_dump_buffer(elem, QFLE3_PAGE_SIZE);
		CNIC_WARN(cdev, "\n*** EQ END ***\n");
	}
}

static VMK_ReturnStatus cnic_get_sfp_diagnostic(struct cnic_dev *dev,
			struct sfp_diagnostic *sfp_diag)
{
	struct cnic_local *cp = dev->cnic_priv;
	return cp->ethdev->drv_get_sfp_diagnostic(dev->netdev, sfp_diag);
}

static VMK_ReturnStatus cnic_get_link_error(struct cnic_dev *dev,
			struct link_error_status *lesb)
{
	struct cnic_local *cp = dev->cnic_priv;
	return cp->ethdev->drv_get_link_error(dev->netdev, lesb);
}

static int cnic_init_id_tbl(struct cnic_id_tbl *id_tbl, vmk_uint32 size,
	vmk_uint32 start_id)
{
	id_tbl->start = start_id;
	id_tbl->max = size;
	id_tbl->next = 0;
	id_tbl->count = 0;
	ql_vmk_spin_lock_init(&id_tbl->lock, LOCK_RANK_HIGHEST);

	id_tbl->table = vmk_BitVectorAlloc(cnic_driver_info.heap_id, size);
	if (VMK_UNLIKELY(!id_tbl->table)) {
		DRV_ERR("Error: Failed to allocate id-tbl. size=0x%x", size);
		return VMK_NO_MEMORY;
	}

	return 0;
}

static vmk_uint32 cnic_free_id_tbl(struct cnic_id_tbl *id_tbl)
{
	if (id_tbl->table) {
		vmk_BitVectorFree(cnic_driver_info.heap_id, id_tbl->table);
		id_tbl->table = NULL;

		vmk_SpinlockDestroy(id_tbl->lock);
	}

	return 0;
}

/* Returns -1 if not successful */
static vmk_uint32 cnic_alloc_new_id(struct cnic_id_tbl *id_tbl)
{
	vmk_uint32 id;
	vmk_Bool status;

	DRV_INFO(CNIC_MSG_DRV, "Enter");

	vmk_SpinlockLock(id_tbl->lock);

	status = vmk_BitVectorNextBit(id_tbl->table, id_tbl->next, VMK_FALSE, &id);
	if ((status == VMK_FALSE) || (id >= id_tbl->max)) {
		id = -1;
		if (id_tbl->next != 0) {
			status = vmk_BitVectorNextBit(id_tbl->table, 0, VMK_FALSE, &id);
			if (id >= id_tbl->next)
				status = VMK_FALSE;
		}
	}

	if ((status == VMK_TRUE) && (id < id_tbl->max)) {
		vmk_BitVectorSet(id_tbl->table, id);
		id_tbl->next = (id + 1) & (id_tbl->max - 1);
		id += id_tbl->start;
		id_tbl->count++;
	} else
		id = -1;

	vmk_SpinlockUnlock(id_tbl->lock);

	DRV_INFO(CNIC_MSG_DRV, "Exit. id=0x%x", id);

	return id;
}

void cnic_free_id(struct cnic_id_tbl *id_tbl, vmk_uint32 id)
{
	if (id == -1)
		return;

	id -= id_tbl->start;
	if (id >= id_tbl->max)
		return;

	vmk_SpinlockLock(id_tbl->lock);
	vmk_BitVectorClear(id_tbl->table, id);
	id_tbl->count--;
	vmk_SpinlockUnlock(id_tbl->lock);
}

static void cnic_free_dma(struct cnic_dev *dev, struct cnic_dma *dma)
{
	int i;

	if (!dma->pg_arr)
		return;

	CNIC_INFO(dev, CNIC_MSG_MEM, "Freeing pgtbl=%p pgtbl-map=0x%lx size=0x%x",
		dma->pgtbl, dma->pgtbl_map, dma->pgtbl_size);

	for (i = 0; i < dma->num_pages; i++) {
		if (dma->pg_arr[i]) {
			CNIC_INFO(dev, CNIC_MSG_MEM,
				"Freeing pg_arr[%d]=%p pg_map_arr=0x%lx",
				i, dma->pg_arr[i], dma->pg_map_arr[i]);
			ql_vmk_dma_free(dev, dma->pg_arr[i], BNX2_PAGE_SIZE);
			dma->pg_arr[i] = NULL;
		}
	}

	if (dma->pgtbl) {
		ql_vmk_dma_free(dev, dma->pgtbl, dma->pgtbl_size);
		dma->pgtbl = NULL;
	}

	ql_vmk_free(dma->pg_arr);
	dma->pg_arr = NULL;

	if (dma->pg_map_arr) {
		ql_vmk_free(dma->pg_map_arr);
		dma->pg_map_arr = NULL;
	}

	dma->num_pages = 0;
}

static void cnic_setup_page_tbl(struct cnic_dev *dev, struct cnic_dma *dma)
{
	int i;
	vmk_uint32 *page_table = (vmk_uint32 *) dma->pgtbl;

	for (i = 0; i < dma->num_pages; i++) {
		/* Each entry needs to be in big endian format. */
		*page_table = QL_CPU_TO_LE32((vmk_uint64) dma->pg_map_arr[i] >> 32);
		page_table++;
		*page_table = QL_CPU_TO_LE32(dma->pg_map_arr[i] & 0xffffffff);
		page_table++;
	}
}

static void cnic_setup_page_tbl_le(struct cnic_dev *dev, struct cnic_dma *dma)
{
	int i;
	vmk_uint32 *page_table = (vmk_uint32 *) dma->pgtbl;

	for (i = 0; i < dma->num_pages; i++) {
		/* Each entry needs to be in little endian format. */
		*page_table = QL_CPU_TO_LE32(dma->pg_map_arr[i] & 0xffffffff);
		page_table++;
		*page_table = QL_CPU_TO_LE32((vmk_uint64) dma->pg_map_arr[i] >> 32);
		page_table++;
	}
}

static int cnic_alloc_dma(struct cnic_dev *dev, struct cnic_dma *dma,
			  int pages, int use_pg_tbl)
{
	int i, dma_arr_size, pg_arr_size;
	struct cnic_local *cp = dev->cnic_priv;

	pg_arr_size = pages * sizeof(void *);
	dma->pg_arr = ql_vmk_alloc(pg_arr_size);
	if (dma->pg_arr == NULL)
		return VMK_NO_MEMORY;

	dma_arr_size = pages * sizeof(vmk_IOA);
	dma->pg_map_arr = ql_vmk_alloc(dma_arr_size);
	if (dma->pg_map_arr == NULL) {
		ql_vmk_free(dma->pg_arr);
		return VMK_NO_MEMORY;
	}

	dma->num_pages = pages;

	CNIC_INFO(dev, CNIC_MSG_MEM,
		"before dma pg_arr, use_pg_tbl=0x%x pages=0x%x BNX2_PAGE_SIZE=0x%x",
		use_pg_tbl, pages, BNX2_PAGE_SIZE);

	for (i = 0; i < pages; i++) {
		dma->pg_arr[i] = ql_vmk_dma_alloc(dev, &dma->pg_map_arr[i],
				BNX2_PAGE_SIZE);
		if (dma->pg_arr[i] == NULL)
			goto error;
	}
	if (!use_pg_tbl)
		return 0;

	/* XXX: we already page align in ql_vmk_dma_alloc, so no need here. */
	//dma->pgtbl_size = ((pages * 8) + BNX2_PAGE_SIZE - 1) &
	//	~(BNX2_PAGE_SIZE - 1);
	dma->pgtbl_size = (pages * 8);
	CNIC_INFO(dev, CNIC_MSG_MEM, "before dma pgtbl, dma->pgtbl_size=0x%x",
			dma->pgtbl_size);
	dma->pgtbl = ql_vmk_dma_alloc(dev, &dma->pgtbl_map, dma->pgtbl_size);
	if (dma->pgtbl == NULL)
		goto error;

	CNIC_INFO(dev, CNIC_MSG_MEM, "before setup_pgtbl");
	cp->setup_pgtbl(dev, dma);
	CNIC_INFO(dev, CNIC_MSG_MEM, "after setup_pgtbl");

	return 0;

error:
	cnic_free_dma(dev, dma);
	return VMK_NO_MEMORY;
}

static void cnic_free_context(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	int i;

	if (cp->ctx_arr) {
		for (i = 0; i < cp->ctx_blks; i++) {
			if (cp->ctx_arr[i].ctx) {
				CNIC_INFO(dev, CNIC_MSG_MEM,
					"Freeing ctx_arr[%d].ctx=%p map=0x%lx",
					i, cp->ctx_arr[i].ctx, cp->ctx_arr[i].mapping);
				ql_vmk_dma_free(dev, cp->ctx_arr[i].ctx, cp->ctx_blk_size);
				cp->ctx_arr[i].ctx = NULL;
			}
		}

		ql_vmk_free(cp->ctx_arr);
		cp->ctx_arr = NULL;
		cp->ctx_blks = 0;
	}
}

static void cnic_free_resc(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	int i;
#if (CNIC_ISCSI_OOO_SUPPORT)
	cnic_free_ooo_resc(dev);
#endif
	cnic_free_context(dev);
	cnic_free_dma(dev, &cp->gbl_buf_info);

	/* commenting below as this allocation is for bnx2 only */
//	cnic_free_dma(dev, &cp->kwq_info);
	cnic_free_dma(dev, &cp->kcq2.dma);
	cnic_free_dma(dev, &cp->kcq1.dma);
	cnic_free_dma(dev, &cp->kwq_16_data_info);

	for (i = 0; i < cp->max_cid_space; i++) {
		vmk_BitVectorFree(cnic_driver_info.heap_id,
							cp->ctx_tbl[i].ctx_flags);
	}

	if (cp->ctx_tbl) {
		ql_vmk_free(cp->ctx_tbl);
		cp->ctx_tbl = NULL;
	}

	if (cp->iscsi_tbl) {
		ql_vmk_free(cp->iscsi_tbl);
		cp->iscsi_tbl = NULL;
	}

	CNIC_INFO(dev, CNIC_MSG_DEBUGFS, "Exiting");
//	cnic_free_id_tbl(&cp->fcoe_cid_tbl);
//	cnic_free_id_tbl(&cp->cid_tbl);
}

static int cnic_alloc_context(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;

	if (BNX2_CHIP(cp) == BNX2_CHIP_5709) {
		int i, k, arr_size;

		cp->ctx_blk_size = BNX2_PAGE_SIZE;
		cp->cids_per_blk = BNX2_PAGE_SIZE / 128;
		arr_size = BNX2_MAX_CID / cp->cids_per_blk *
			   sizeof(struct cnic_ctx);
		cp->ctx_arr = ql_vmk_alloc(arr_size);
		if (VMK_UNLIKELY(cp->ctx_arr == NULL)) {
			CNIC_ERR(dev, "Failed to allocate context array.");
			return VMK_NO_MEMORY;
		}

		k = 0;
		for (i = 0; i < 2; i++) {
			vmk_uint32 j, reg, off, lo, hi;

			if (i == 0)
				off = BNX2_PG_CTX_MAP;
			else
				off = BNX2_ISCSI_CTX_MAP;

			reg = cnic_reg_rd_ind(dev, off);
			lo = reg >> 16;
			hi = reg & 0xffff;
			for (j = lo; j < hi; j += cp->cids_per_blk, k++)
				cp->ctx_arr[k].cid = j;
		}

		cp->ctx_blks = k;
		if (VMK_UNLIKELY(cp->ctx_blks >= (BNX2_MAX_CID / cp->cids_per_blk))) {
			CNIC_ERR(dev, "cp->ctx_blks (%d) >= (%d) "
				"(BNX2_MAX_CID(%d) / cp->cids_per_blk(%d))",
				cp->ctx_blks, (BNX2_MAX_CID / cp->cids_per_blk),
				BNX2_MAX_CID, cp->cids_per_blk);
			cp->ctx_blks = 0;
			return VMK_NO_MEMORY;
		}

		for (i = 0; i < cp->ctx_blks; i++) {
			cp->ctx_arr[i].ctx = ql_vmk_dma_alloc(dev,
					&cp->ctx_arr[i].mapping, BNX2_PAGE_SIZE);
			if (VMK_UNLIKELY(cp->ctx_arr[i].ctx == NULL)) {
				CNIC_ERR(dev, "Failed to allocate context array.");
				return VMK_NO_MEMORY;
			}
		}
	}

	return 0;
}

static vmk_uint16 cnic_bnx2_next_idx(vmk_uint16 idx)
{
	return idx + 1;
}

static vmk_uint16 cnic_bnx2_hw_idx(vmk_uint16 idx)
{
	return idx;
}

static vmk_uint16 cnic_qfle3_next_idx(vmk_uint16 idx)
{
	idx++;
	if ((idx & MAX_KCQE_CNT) == MAX_KCQE_CNT)
		idx++;

	return idx;
}

static vmk_uint16 cnic_qfle3_hw_idx(vmk_uint16 idx)
{
	if ((idx & MAX_KCQE_CNT) == MAX_KCQE_CNT)
		idx++;
	return idx;
}

static int cnic_alloc_kcq(struct cnic_dev *dev, struct kcq_info *info,
	int use_pg_tbl)
{
	int err, i;
	struct kcqe **kcq;

	err = cnic_alloc_dma(dev, &info->dma, KCQ_PAGE_CNT, use_pg_tbl);
	if (VMK_UNLIKELY(err))
		return err;

	kcq = (struct kcqe **) info->dma.pg_arr;
	info->kcq = kcq;

	info->next_idx = cnic_bnx2_next_idx;
	info->hw_idx = cnic_bnx2_hw_idx;
	if (use_pg_tbl)
		return 0;

	info->next_idx = cnic_qfle3_next_idx;
	info->hw_idx = cnic_qfle3_hw_idx;

	for (i = 0; i < KCQ_PAGE_CNT; i++) {
		struct qfle3_bd_chain_next *next =
			(struct qfle3_bd_chain_next *) &kcq[i][MAX_KCQE_CNT];
		int j = i + 1;

		if (j >= KCQ_PAGE_CNT)
			j = 0;
		next->addr_hi = (vmk_uint64) info->dma.pg_map_arr[j] >> 32;
		next->addr_lo = info->dma.pg_map_arr[j] & 0xffffffff;
	}

	return 0;
}

static int cnic_alloc_bnx2_resc(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	int i, ret;

	cp->max_cid_space = MAX_ISCSI_TBL_SZ;

	cp->iscsi_tbl = ql_vmk_alloc(sizeof(struct cnic_iscsi) * MAX_ISCSI_TBL_SZ);
	if (VMK_UNLIKELY(!cp->iscsi_tbl)) {
		CNIC_ERR(dev, "failed to allocate iscsi_tbl.");
		ret = VMK_NO_MEMORY;
		goto error;
	}

	cp->ctx_tbl = ql_vmk_alloc(sizeof(struct cnic_context) * cp->max_cid_space);
	if (VMK_UNLIKELY(!cp->ctx_tbl)) {
		CNIC_ERR(dev, "failed to allocate ctx_arr.");
		ret = VMK_NO_MEMORY;
		goto error;
	}

	for (i = 0; i < MAX_ISCSI_TBL_SZ; i++) {
		cp->ctx_tbl[i].proto.iscsi = &cp->iscsi_tbl[i];
		cp->ctx_tbl[i].ulp_proto_id = CNIC_ULP_ISCSI;
	}

	for (i = 0; i < cp->max_cid_space; i++) {
		cp->ctx_tbl[i].ctx_flags = vmk_BitVectorAlloc(cnic_driver_info.heap_id,
				(sizeof(long) * VMK_BITS_PER_BYTE));
		vmk_BitVectorZap(cp->ctx_tbl[i].ctx_flags);
	}

	ret = cnic_alloc_dma(dev, &cp->kwq_info, KWQ_PAGE_CNT, 1);
	if (VMK_UNLIKELY(ret)) {
		CNIC_ERR(dev, "failed to allocate kwq.");
		goto error;
	}
	cp->kwq = (struct kwqe **) cp->kwq_info.pg_arr;

	ret = cnic_alloc_kcq(dev, &cp->kcq1, 1);
	if (VMK_UNLIKELY(ret)) {
		CNIC_ERR(dev, "failed to allocate kcq.");
		goto error;
	}

	ret = cnic_alloc_context(dev);
	if (VMK_UNLIKELY(ret)) {
		CNIC_ERR(dev, "failed to allocate context memory.");
		goto error;
	}

#if (CNIC_ISCSI_OOO_SUPPORT)
	cnic_alloc_bnx2_ooo_resc(dev);
#endif
	return 0;

error:
	cnic_free_resc(dev);
	return ret;
}

static int cnic_alloc_qfle3_context(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	int ctx_blk_size = cp->ethdev->ctx_blk_size;
	int total_mem, blks, i;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	total_mem = QFLE3_CONTEXT_MEM_SIZE * cp->max_cid_space;
	blks = total_mem / ctx_blk_size;
	if (total_mem % ctx_blk_size)
		blks++;

	if (VMK_UNLIKELY(blks > cp->ethdev->ctx_tbl_len)) {
		CNIC_ERR(dev, "blks(%d) > cp->ethdev->ctx_tbl_len(%d)",
			blks, cp->ethdev->ctx_tbl_len);
		return VMK_NO_MEMORY;
	}

	cp->ctx_arr = ql_vmk_alloc(blks * sizeof(struct cnic_ctx));
	if (VMK_UNLIKELY(cp->ctx_arr == NULL)) {
		CNIC_ERR(dev, "failed to allocate ctx_arr.");
		return VMK_NO_MEMORY;
	}

	cp->ctx_blks = blks;
	cp->ctx_blk_size = ctx_blk_size;
	if (!CHIP_IS_E1(adapter))
		cp->ctx_align = 0;
	else
		cp->ctx_align = ctx_blk_size;

	cp->cids_per_blk = ctx_blk_size / QFLE3_CONTEXT_MEM_SIZE;

	for (i = 0; i < blks; i++) {
		cp->ctx_arr[i].ctx = ql_vmk_dma_alloc(dev, &cp->ctx_arr[i].mapping,
				cp->ctx_blk_size);
		if (VMK_UNLIKELY(cp->ctx_arr[i].ctx == NULL)) {
			CNIC_ERR(dev, "Could not allocate PCI memory for CTX array.");
			return VMK_NO_MEMORY;
		}

		if (cp->ctx_align && cp->ctx_blk_size == ctx_blk_size) {
			if (cp->ctx_arr[i].mapping & (cp->ctx_align - 1)) {
				cnic_free_context(dev);
				cp->ctx_blk_size += cp->ctx_align;
				i = -1;
				continue;
			}
		}
	}

	return 0;
}

static int cnic_alloc_qfle3_resc(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	vmk_uint32 start_cid = ethdev->starting_cid;
	int i, j, n, ret, pages;
	struct cnic_dma *kwq_16_dma = &cp->kwq_16_data_info;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (CHIP_IS_E1(adapter))
		cp->iro_arr = e1_iro_arr;
	else if (CHIP_IS_E1H(adapter))
		cp->iro_arr = e1h_iro_arr;
	else if (QFLE3_CHIP_IS_E2_PLUS(adapter))
		cp->iro_arr = e2_iro_arr;

	cp->max_cid_space = MAX_ISCSI_TBL_SZ;
	cp->iscsi_start_cid = start_cid;
	cp->fcoe_start_cid = start_cid + MAX_ISCSI_TBL_SZ;

	if (QFLE3_CHIP_IS_E2_PLUS(adapter)) {
		cp->max_cid_space += dev->max_fcoe_conn;
		cp->fcoe_init_cid = ethdev->fcoe_init_cid;
		if (!cp->fcoe_init_cid)
			cp->fcoe_init_cid = 0x10;
	}

	if (start_cid < QFLE3_ISCSI_START_CID) {
		vmk_uint32 delta = QFLE3_ISCSI_START_CID - start_cid;

		cp->iscsi_start_cid = QFLE3_ISCSI_START_CID;
		cp->fcoe_start_cid += delta;
		cp->max_cid_space += delta;
	}

	//SV: XXX: safe code
	CNIC_INFO(dev, CNIC_MSG_MEM, "max_cid_space=0x%x iscsi_start_cid=0x%x "
		"fcoe_start_cid=0x%x fcoe_init_cid=0x%x start_cid=0x%x",
		cp->max_cid_space, cp->iscsi_start_cid, cp->fcoe_start_cid,
		cp->fcoe_init_cid, start_cid);

	cp->iscsi_tbl = ql_vmk_alloc(sizeof(struct cnic_iscsi) * MAX_ISCSI_TBL_SZ);
	if (VMK_UNLIKELY(!cp->iscsi_tbl)) {
		CNIC_ERR(dev, "failed to allocate iscsi_tbl.");
		goto error;
	}

	cp->ctx_tbl = ql_vmk_alloc(sizeof(struct cnic_context) * cp->max_cid_space);
	if (VMK_UNLIKELY(!cp->ctx_tbl)) {
		CNIC_ERR(dev, "failed to allocate ctx_tbl.");
		goto error;
	}

	for (i = 0; i < MAX_ISCSI_TBL_SZ; i++) {
		cp->ctx_tbl[i].proto.iscsi = &cp->iscsi_tbl[i];
		cp->ctx_tbl[i].ulp_proto_id = CNIC_ULP_ISCSI;
	}

	for (i = MAX_ISCSI_TBL_SZ; i < cp->max_cid_space; i++)
		cp->ctx_tbl[i].ulp_proto_id = CNIC_ULP_FCOE;

	/* Calculate nof pages. */
	pages = (((cp->max_cid_space * CNIC_KWQ16_DATA_SIZE) + VMK_PAGE_SIZE - 1) /
			VMK_PAGE_SIZE);

	//SV: XXX: safe code
	CNIC_INFO(dev, CNIC_MSG_MEM, "pages=0x%x", pages);

	ret = cnic_alloc_dma(dev, kwq_16_dma, pages, 0);
	if (VMK_UNLIKELY(ret)) {
		CNIC_ERR(dev, "failed to allocate kwq.");
		return VMK_NO_MEMORY;
	}

	n = VMK_PAGE_SIZE / CNIC_KWQ16_DATA_SIZE;
	for (i = 0, j = 0; i < cp->max_cid_space; i++) {
		long off = CNIC_KWQ16_DATA_SIZE * (i % n);

		cp->ctx_tbl[i].kwqe_data = kwq_16_dma->pg_arr[j] + off;
		cp->ctx_tbl[i].kwqe_data_mapping = kwq_16_dma->pg_map_arr[j] +
						   off;

		cp->ctx_tbl[i].ctx_flags = vmk_BitVectorAlloc(cnic_driver_info.heap_id,
				(sizeof(long) * VMK_BITS_PER_BYTE));
		vmk_BitVectorZap(cp->ctx_tbl[i].ctx_flags);

		if ((i % n) == (n - 1))
			j++;
	}

	ret = cnic_alloc_kcq(dev, &cp->kcq1, 0);
	if (VMK_UNLIKELY(ret)) {
		CNIC_ERR(dev, "failed to allocate kcq1: ret=0x%x", ret);
		goto error;
	}

	if (QFLE3_CHIP_IS_E2_PLUS(adapter)) {
		ret = cnic_alloc_kcq(dev, &cp->kcq2, 1);

		if (VMK_UNLIKELY(ret)) {
			CNIC_ERR(dev, "failed to allocate kcq2: ret=0x%x", ret);
			goto error;
		}
	}

	pages = ((QFLE3_ISCSI_GLB_BUF_SIZE + VMK_PAGE_SIZE - 1) / VMK_PAGE_SIZE);
	ret = cnic_alloc_dma(dev, &cp->gbl_buf_info, pages, 0);
	if (VMK_UNLIKELY(ret)) {
		CNIC_ERR(dev, "failed to allocate gbl_buf_info: ret=0x%x", ret);
		goto error;
	}

	CNIC_INFO(dev, CNIC_MSG_MEM, "before cnic_alloc_qfle3_context");
	ret = cnic_alloc_qfle3_context(dev);
	if (VMK_UNLIKELY(ret)) {
		CNIC_ERR(dev, "Failed to allocate qfle3 context.");
		goto error;
	}

	CNIC_INFO(dev, CNIC_MSG_MEM, "before cnic_alloc_qfle3_ooo_resc");
#if (CNIC_ISCSI_OOO_SUPPORT)
	cnic_alloc_qfle3_ooo_resc(dev);
#endif
	CNIC_INFO(dev, CNIC_MSG_MEM, "Exiting.");
	return 0;

error:
	cnic_free_resc(dev);
	return VMK_NO_MEMORY;
}

static inline vmk_uint32 cnic_kwq_avail(struct cnic_local *cp)
{
	return cp->max_kwq_idx -
		((cp->kwq_prod_idx - cp->kwq_con_idx) & cp->max_kwq_idx);
}

static int cnic_submit_bnx2_kwqes(struct cnic_dev *dev, struct kwqe *wqes[],
				  vmk_uint32 num_wqes)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct kwqe *prod_qe;
	vmk_uint16 prod, sw_prod, i;
#if (CNIC_ISCSI_OOO_SUPPORT)
	vmk_uint32 opcode;
	struct kwqe *kwqe;
#endif

	if (!vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP))
		return VMK_STATUS_PENDING;		/* bnx2 is down */

	vmk_SpinlockLock(cp->cnic_ulp_lock);
	if (num_wqes > cnic_kwq_avail(cp) &&
	    !vmk_BitVectorTest(cp->cnic_local_flags, CNIC_LCL_FL_KWQ_INIT)) {
		vmk_SpinlockUnlock(cp->cnic_ulp_lock);
		return VMK_STATUS_PENDING;
	}

	vmk_BitVectorClear(cp->cnic_local_flags, CNIC_LCL_FL_KWQ_INIT);

	prod = cp->kwq_prod_idx;
	sw_prod = prod & MAX_KWQ_IDX;
	for (i = 0; i < num_wqes; i++) {
		prod_qe = &cp->kwq[KWQ_PG(sw_prod)][KWQ_IDX(sw_prod)];
#if (CNIC_ISCSI_OOO_SUPPORT)
		kwqe = wqes[i];
		opcode = KWQE_OPCODE(kwqe->kwqe_op_flag);
		switch (opcode) {
		case ISCSI_KWQE_OPCODE_UPDATE_CONN:
			cnic_bnx2_ooo_iscsi_conn_update(dev, kwqe);
			break;
		case ISCSI_KWQE_OPCODE_DESTROY_CONN:
			cnic_bnx2_ooo_iscsi_destroy(dev, kwqe);
			break;
		default:
			break;
		}
#endif
		vmk_Memcpy(prod_qe, wqes[i], sizeof(struct kwqe));
		prod++;
		sw_prod = prod & MAX_KWQ_IDX;
	}
	cp->kwq_prod_idx = prod;

	CNIC_WR16(dev, cp->kwq_io_addr, cp->kwq_prod_idx);

	vmk_SpinlockUnlock(cp->cnic_ulp_lock);
	return 0;
}

static void *cnic_get_kwqe_16_data(struct cnic_local *cp, vmk_uint32 l5_cid,
				   union l5cm_specific_data *l5_data)
{
	struct cnic_context *ctx = &cp->ctx_tbl[l5_cid];
	vmk_IOA map;

	map = ctx->kwqe_data_mapping;
	l5_data->phy_address.lo = (vmk_uint64) map & 0xffffffff;
	l5_data->phy_address.hi = (vmk_uint64) map >> 32;
	DRV_INFO(CNIC_MSG_DRV, "map-lo=0x%x map-hi=0x%x",
		l5_data->phy_address.lo, l5_data->phy_address.hi);
	return ctx->kwqe_data;
}

static int cnic_submit_kwqe_16(struct cnic_dev *dev, vmk_uint32 cmd, vmk_uint32 cid,
				vmk_uint32 type, union l5cm_specific_data *l5_data)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	struct l5cm_spe kwqe;
	struct kwqe_16 *kwq[1];
	vmk_uint16 type_16;
	int nof_work_sent = 0, nof_work_processed = 0;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	kwqe.hdr.conn_and_cmd_data =
		QL_CPU_TO_LE32(((cmd << SPE_HDR_CMD_ID_SHIFT) |
			     HW_CID(adapter, cid)));
	type_16 = (type << SPE_HDR_CONN_TYPE_SHIFT) & SPE_HDR_CONN_TYPE;
	type_16 |= (adapter->pf_id << SPE_HDR_FUNCTION_ID_SHIFT) &
		   SPE_HDR_FUNCTION_ID;

	kwqe.hdr.type = QL_CPU_TO_LE16(type_16);
	kwqe.hdr.reserved1 = 0;
	kwqe.data.phy_address.lo = QL_CPU_TO_LE32(l5_data->phy_address.lo);
	kwqe.data.phy_address.hi = QL_CPU_TO_LE32(l5_data->phy_address.hi);

	kwq[0] = (struct kwqe_16 *) &kwqe;
	nof_work_sent = 1;

	vmk_SpinlockLock(cp->cnic_ulp_lock);
	nof_work_processed =
		cp->ethdev->cnicSlowPathCmd(dev->netdev, kwq, nof_work_sent);
	vmk_SpinlockUnlock(cp->cnic_ulp_lock);

	cnic_print_ramrod_info(dev, cmd, cid, kwq[0]);

	/* nof_work_processed = success: nof work processed, failure: VMK status */
	if (nof_work_processed == nof_work_sent)
		return VMK_OK;

	/* XXX: Note: In error conditions return value is VMK error status. */
	return nof_work_processed;
}

static void cnic_reply_qfle3_kcqes(struct cnic_dev *dev, int ulp_type,
   struct kcqe *cqes[], vmk_uint32 num_cqes)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_ulp_ops *ulp_ops;

	CNIC_INFO(dev, CNIC_MSG_DRV, "ulp_type=0x%x num_cqes=0x%x", ulp_type, num_cqes);
	//SV: XXX: Need better way to handle "ulp_xxx" element access.
	//Also as cdev now is per function, so should this locking.
	//SV: XXX: Use rcu_read_lock?
	//vmk_SemaLock(&cnic_driver_info.cnic_lock);
	ulp_ops = cp->ulp_ops[ulp_type];
	if (VMK_LIKELY(ulp_ops)) {
		ulp_ops->indicate_kcqes(cp->ulp_handle[ulp_type],
					  cqes, num_cqes);
	}
	//vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
}

static void cnic_qfle3_set_tcp_options(struct cnic_dev *dev, int time_stamps,
	int en_tcp_dack)
{
	struct qfle3_adapter *adapter = NULL;
	vmk_uint8 xstorm_flags = XSTORM_L5CM_TCP_FLAGS_WND_SCL_EN;
	vmk_uint16 tstorm_flags = 0;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (time_stamps) {
		xstorm_flags |= XSTORM_L5CM_TCP_FLAGS_TS_ENABLED;
		tstorm_flags |= TSTORM_L5CM_TCP_FLAGS_TS_ENABLED;
	}
	if (en_tcp_dack)
		tstorm_flags |= TSTORM_L5CM_TCP_FLAGS_DELAYED_ACK_EN;

	CNIC_WR8(dev, BAR_XSTRORM_INTMEM +
		XSTORM_ISCSI_TCP_VARS_FLAGS_OFFSET(adapter->pf_id), xstorm_flags);

	CNIC_WR16(dev, BAR_TSTRORM_INTMEM +
		TSTORM_ISCSI_TCP_VARS_FLAGS_OFFSET(adapter->pf_id), tstorm_flags);
}

static int cnic_qfle3_iscsi_init1(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	struct iscsi_kwqe_init1 *req1 = (struct iscsi_kwqe_init1 *) kwqe;
	int hq_bds, pages;
	vmk_uint32 pfid;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);
	pfid = adapter->pf_id;

	cp->num_iscsi_tasks = req1->num_tasks_per_conn;
	cp->num_ccells = req1->num_ccells_per_conn;
	cp->task_array_size = QFLE3_ISCSI_TASK_CONTEXT_SIZE *
			      cp->num_iscsi_tasks;
	cp->r2tq_size = cp->num_iscsi_tasks * QFLE3_ISCSI_MAX_PENDING_R2TS *
			QFLE3_ISCSI_R2TQE_SIZE;
	cp->hq_size = cp->num_ccells * QFLE3_ISCSI_HQ_BD_SIZE;
	pages = ((cp->hq_size + VMK_PAGE_SIZE - 1) / VMK_PAGE_SIZE);
	hq_bds = pages * (VMK_PAGE_SIZE / QFLE3_ISCSI_HQ_BD_SIZE);
	cp->num_cqs = req1->num_cqs;

	if (!dev->max_iscsi_conn)
		return 0;

	/* init Tstorm RAM */
	CNIC_WR16(dev, BAR_TSTRORM_INTMEM + TSTORM_ISCSI_RQ_SIZE_OFFSET(pfid),
		  req1->rq_num_wqes);
	CNIC_WR16(dev, BAR_TSTRORM_INTMEM + TSTORM_ISCSI_PAGE_SIZE_OFFSET(pfid),
		  VMK_PAGE_SIZE);
	CNIC_WR8(dev, BAR_TSTRORM_INTMEM +
		 TSTORM_ISCSI_PAGE_SIZE_LOG_OFFSET(pfid), VMK_PAGE_SHIFT);
	CNIC_WR16(dev, BAR_TSTRORM_INTMEM +
		  TSTORM_ISCSI_NUM_OF_TASKS_OFFSET(pfid),
		  req1->num_tasks_per_conn);

	/* init Ustorm RAM */
	CNIC_WR16(dev, BAR_USTRORM_INTMEM +
		  USTORM_ISCSI_RQ_BUFFER_SIZE_OFFSET(pfid),
		  req1->rq_buffer_size);
	CNIC_WR16(dev, BAR_USTRORM_INTMEM + USTORM_ISCSI_PAGE_SIZE_OFFSET(pfid),
		  VMK_PAGE_SIZE);
	CNIC_WR8(dev, BAR_USTRORM_INTMEM +
		 USTORM_ISCSI_PAGE_SIZE_LOG_OFFSET(pfid), VMK_PAGE_SHIFT);
	CNIC_WR16(dev, BAR_USTRORM_INTMEM +
		  USTORM_ISCSI_NUM_OF_TASKS_OFFSET(pfid),
		  req1->num_tasks_per_conn);
	CNIC_WR16(dev, BAR_USTRORM_INTMEM + USTORM_ISCSI_RQ_SIZE_OFFSET(pfid),
		  req1->rq_num_wqes);
	CNIC_WR16(dev, BAR_USTRORM_INTMEM + USTORM_ISCSI_CQ_SIZE_OFFSET(pfid),
		  req1->cq_num_wqes);
	CNIC_WR16(dev, BAR_USTRORM_INTMEM + USTORM_ISCSI_R2TQ_SIZE_OFFSET(pfid),
		  cp->num_iscsi_tasks * QFLE3_ISCSI_MAX_PENDING_R2TS);

	/* init Xstorm RAM */
	CNIC_WR16(dev, BAR_XSTRORM_INTMEM + XSTORM_ISCSI_PAGE_SIZE_OFFSET(pfid),
		  VMK_PAGE_SIZE);
	CNIC_WR8(dev, BAR_XSTRORM_INTMEM +
		 XSTORM_ISCSI_PAGE_SIZE_LOG_OFFSET(pfid), VMK_PAGE_SHIFT);
	CNIC_WR16(dev, BAR_XSTRORM_INTMEM +
		  XSTORM_ISCSI_NUM_OF_TASKS_OFFSET(pfid),
		  req1->num_tasks_per_conn);
	CNIC_WR16(dev, BAR_XSTRORM_INTMEM + XSTORM_ISCSI_HQ_SIZE_OFFSET(pfid),
		  hq_bds);
	CNIC_WR16(dev, BAR_XSTRORM_INTMEM + XSTORM_ISCSI_SQ_SIZE_OFFSET(pfid),
		  req1->num_tasks_per_conn);
	CNIC_WR16(dev, BAR_XSTRORM_INTMEM + XSTORM_ISCSI_R2TQ_SIZE_OFFSET(pfid),
		  cp->num_iscsi_tasks * QFLE3_ISCSI_MAX_PENDING_R2TS);

	/* init Cstorm RAM */
	CNIC_WR16(dev, BAR_CSTRORM_INTMEM + CSTORM_ISCSI_PAGE_SIZE_OFFSET(pfid),
		  VMK_PAGE_SIZE);
	CNIC_WR8(dev, BAR_CSTRORM_INTMEM +
		 CSTORM_ISCSI_PAGE_SIZE_LOG_OFFSET(pfid), VMK_PAGE_SHIFT);
	CNIC_WR16(dev, BAR_CSTRORM_INTMEM +
		  CSTORM_ISCSI_NUM_OF_TASKS_OFFSET(pfid),
		  req1->num_tasks_per_conn);
	CNIC_WR16(dev, BAR_CSTRORM_INTMEM + CSTORM_ISCSI_CQ_SIZE_OFFSET(pfid),
		  req1->cq_num_wqes);
	CNIC_WR16(dev, BAR_CSTRORM_INTMEM + CSTORM_ISCSI_HQ_SIZE_OFFSET(pfid),
		  hq_bds);

	cnic_qfle3_set_tcp_options(dev,
		req1->flags & ISCSI_KWQE_INIT1_TIME_STAMPS_ENABLE,
		req1->flags & ISCSI_KWQE_INIT1_DELAYED_ACK_ENABLE);

	return 0;
}

static int cnic_qfle3_iscsi_init2(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct iscsi_kwqe_init2 *req2 = (struct iscsi_kwqe_init2 *) kwqe;
	struct qfle3_adapter *adapter = NULL;
	vmk_uint32 pfid;
	struct iscsi_kcqe kcqe;
	struct kcqe *cqes[1];

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);
	pfid = adapter->pf_id;

	vmk_Memset(&kcqe, 0, sizeof(kcqe));
	if (!dev->max_iscsi_conn) {
		kcqe.completion_status =
			ISCSI_KCQE_COMPLETION_STATUS_ISCSI_NOT_SUPPORTED;
		goto done;
	}

	CNIC_WR(dev, BAR_TSTRORM_INTMEM +
		TSTORM_ISCSI_ERROR_BITMAP_OFFSET(pfid), req2->error_bit_map[0]);
	CNIC_WR(dev, BAR_TSTRORM_INTMEM +
		TSTORM_ISCSI_ERROR_BITMAP_OFFSET(pfid) + 4,
		req2->error_bit_map[1]);

	CNIC_WR16(dev, BAR_USTRORM_INTMEM +
		  USTORM_ISCSI_CQ_SQN_SIZE_OFFSET(pfid), req2->max_cq_sqn);
	CNIC_WR(dev, BAR_USTRORM_INTMEM +
		USTORM_ISCSI_ERROR_BITMAP_OFFSET(pfid), req2->error_bit_map[0]);
	CNIC_WR(dev, BAR_USTRORM_INTMEM +
		USTORM_ISCSI_ERROR_BITMAP_OFFSET(pfid) + 4,
		req2->error_bit_map[1]);

	CNIC_WR16(dev, BAR_CSTRORM_INTMEM +
		  CSTORM_ISCSI_CQ_SQN_SIZE_OFFSET(pfid), req2->max_cq_sqn);

	kcqe.completion_status = ISCSI_KCQE_COMPLETION_STATUS_SUCCESS;

done:
	kcqe.op_code = ISCSI_KCQE_OPCODE_INIT;
	cqes[0] = (struct kcqe *) &kcqe;
	cnic_reply_qfle3_kcqes(dev, CNIC_ULP_ISCSI, cqes, 1);

	return 0;
}

static void cnic_free_qfle3_conn_resc(struct cnic_dev *dev, vmk_uint32 l5_cid)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_context *ctx = &cp->ctx_tbl[l5_cid];
	struct cnic_eth_dev *ethdev = cp->ethdev;

	if (ctx->ulp_proto_id == CNIC_ULP_ISCSI) {
		struct cnic_iscsi *iscsi = ctx->proto.iscsi;

		cnic_free_dma(dev, &iscsi->hq_info);
		cnic_free_dma(dev, &iscsi->r2tq_info);
		cnic_free_dma(dev, &iscsi->task_array_info);
		cnic_free_id(&cp->cid_tbl, ctx->cid);
	} else {
		cnic_free_id(&cp->fcoe_cid_tbl, ctx->cid);
	}

	ctx->cid = 0;
	ethdev->nof_ofld_conns = cp->fcoe_cid_tbl.count + cp->cid_tbl.count;
}

static int cnic_alloc_qfle3_conn_resc(struct cnic_dev *dev, vmk_uint32 l5_cid)
{
	vmk_uint32 cid;
	int ret, pages;
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_context *ctx = &cp->ctx_tbl[l5_cid];
	struct cnic_iscsi *iscsi = ctx->proto.iscsi;

	pages = ((cp->task_array_size + VMK_PAGE_SIZE - 1) / VMK_PAGE_SIZE);

	ret = cnic_alloc_dma(dev, &iscsi->task_array_info, pages, 1);
	if (VMK_UNLIKELY(ret)) {
		CNIC_ERR(dev, "Failed to allocate iSCSI task array: ret=0x%x", ret);
		goto error;
	}

	pages = ((cp->r2tq_size + VMK_PAGE_SIZE - 1) / VMK_PAGE_SIZE);
	ret = cnic_alloc_dma(dev, &iscsi->r2tq_info, pages, 1);
	if (VMK_UNLIKELY(ret)) {
		CNIC_ERR(dev, "Failed to allocate r2tq: ret=0x%x", ret);
		goto error;
	}

	pages = ((cp->hq_size + VMK_PAGE_SIZE - 1) / VMK_PAGE_SIZE);
	ret = cnic_alloc_dma(dev, &iscsi->hq_info, pages, 1);
	if (VMK_UNLIKELY(ret)) {
		CNIC_ERR(dev, "Failed to allocate hq: ret=0x%x", ret);
		goto error;
	}

	return 0;

error:
	cnic_free_qfle3_conn_resc(dev, l5_cid);
	return ret;
}

static void *cnic_get_qfle3_ctx(struct cnic_dev *dev, vmk_uint32 cid, int init,
		struct regpair *ctx_addr)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	int blk = (cid - ethdev->starting_cid) / cp->cids_per_blk;
	int off = (cid - ethdev->starting_cid) % cp->cids_per_blk;
	unsigned long align_off = 0;
	vmk_IOA ctx_map;
	void *ctx;

	if (cp->ctx_align) {
		unsigned long mask = cp->ctx_align - 1;

		if (cp->ctx_arr[blk].mapping & mask)
			align_off = cp->ctx_align -
				    (cp->ctx_arr[blk].mapping & mask);
	}
	ctx_map = cp->ctx_arr[blk].mapping + align_off +
		(off * QFLE3_CONTEXT_MEM_SIZE);
	ctx = cp->ctx_arr[blk].ctx + align_off +
	       (off * QFLE3_CONTEXT_MEM_SIZE);
	if (init)
		vmk_Memset(ctx, 0, QFLE3_CONTEXT_MEM_SIZE);

	ctx_addr->lo = ctx_map & 0xffffffff;
	ctx_addr->hi = (vmk_uint64) ctx_map >> 32;
	return ctx;
}

static int cnic_setup_qfle3_ctx(struct cnic_dev *dev, struct kwqe *wqes[],
	vmk_uint32 num)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	struct iscsi_kwqe_conn_offload1 *req1 =
			(struct iscsi_kwqe_conn_offload1 *) wqes[0];
	struct iscsi_kwqe_conn_offload2 *req2 =
			(struct iscsi_kwqe_conn_offload2 *) wqes[1];
	struct iscsi_kwqe_conn_offload3 *req3;
	struct cnic_context *ctx = &cp->ctx_tbl[req1->iscsi_conn_id];
	struct cnic_iscsi *iscsi = ctx->proto.iscsi;
	vmk_uint32 cid = ctx->cid;
	vmk_uint32 hw_cid;
	struct iscsi_context *ictx;
	struct regpair context_addr;
	int i, j, n = 2, n_max;
	vmk_uint8 port;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);
	hw_cid = HW_CID(adapter, cid);
	port = QFLE3_PORT(adapter);

	CNIC_INFO(dev, CNIC_MSG_DRV, "cid=0x%x hw_cid=0x%x port=0x%x port_mode=0x%x",
		cid, hw_cid, port, adapter->hw_info.port_mode);

	if (!req2->num_additional_wqes)
		return VMK_BAD_PARAM;

	n_max = req2->num_additional_wqes + 2;

	ictx = cnic_get_qfle3_ctx(dev, cid, 1, &context_addr);
	if (VMK_UNLIKELY(ictx == NULL)) {
		CNIC_ERR(dev, "Could not get context: cid: 0x%x", cid);
		return VMK_NO_MEMORY;
	}

	req3 = (struct iscsi_kwqe_conn_offload3 *) wqes[n++];

	ictx->xstorm_ag_context.hq_prod = 1;

	ictx->xstorm_st_context.iscsi.first_burst_length =
		ISCSI_DEF_FIRST_BURST_LEN;
	ictx->xstorm_st_context.iscsi.max_send_pdu_length =
		ISCSI_DEF_MAX_RECV_SEG_LEN;
	ictx->xstorm_st_context.iscsi.sq_pbl_base.lo =
		req1->sq_page_table_addr_lo;
	ictx->xstorm_st_context.iscsi.sq_pbl_base.hi =
		req1->sq_page_table_addr_hi;
	ictx->xstorm_st_context.iscsi.sq_curr_pbe.lo = req2->sq_first_pte.hi;
	ictx->xstorm_st_context.iscsi.sq_curr_pbe.hi = req2->sq_first_pte.lo;
	ictx->xstorm_st_context.iscsi.hq_pbl_base.lo =
		iscsi->hq_info.pgtbl_map & 0xffffffff;
	ictx->xstorm_st_context.iscsi.hq_pbl_base.hi =
		(vmk_uint64) iscsi->hq_info.pgtbl_map >> 32;
	ictx->xstorm_st_context.iscsi.hq_curr_pbe_base.lo =
		iscsi->hq_info.pgtbl[0];
	ictx->xstorm_st_context.iscsi.hq_curr_pbe_base.hi =
		iscsi->hq_info.pgtbl[1];
	ictx->xstorm_st_context.iscsi.r2tq_pbl_base.lo =
		iscsi->r2tq_info.pgtbl_map & 0xffffffff;
	ictx->xstorm_st_context.iscsi.r2tq_pbl_base.hi =
		(vmk_uint64) iscsi->r2tq_info.pgtbl_map >> 32;
	ictx->xstorm_st_context.iscsi.r2tq_curr_pbe_base.lo =
		iscsi->r2tq_info.pgtbl[0];
	ictx->xstorm_st_context.iscsi.r2tq_curr_pbe_base.hi =
		iscsi->r2tq_info.pgtbl[1];
	ictx->xstorm_st_context.iscsi.task_pbl_base.lo =
		iscsi->task_array_info.pgtbl_map & 0xffffffff;
	ictx->xstorm_st_context.iscsi.task_pbl_base.hi =
		(vmk_uint64) iscsi->task_array_info.pgtbl_map >> 32;
	ictx->xstorm_st_context.iscsi.task_pbl_cache_idx =
		QFLE3_ISCSI_PBL_NOT_CACHED;
	ictx->xstorm_st_context.iscsi.flags.flags |=
		XSTORM_ISCSI_CONTEXT_FLAGS_B_IMMEDIATE_DATA;
	ictx->xstorm_st_context.iscsi.flags.flags |=
		XSTORM_ISCSI_CONTEXT_FLAGS_B_INITIAL_R2T;
	ictx->xstorm_st_context.common.ethernet.reserved_vlan_type =
		VMK_ETH_TYPE_VLAN;
	if (QFLE3_CHIP_IS_E2_PLUS(adapter) &&
			adapter->hw_info.port_mode == CHIP_2_PORT_MODE) {
		port = 0;
	}

	CNIC_INFO(dev, CNIC_MSG_DRV, "sq: pbl-hi=0x%x pbl-lo=0x%x\n"
		"curr_pbe.lo=0x%x curr_pbe.hi=0x%x",
		ictx->xstorm_st_context.iscsi.sq_pbl_base.hi,
		ictx->xstorm_st_context.iscsi.sq_pbl_base.lo,
		ictx->xstorm_st_context.iscsi.sq_curr_pbe.lo,
		ictx->xstorm_st_context.iscsi.sq_curr_pbe.hi);
	CNIC_INFO(dev, CNIC_MSG_DRV, "hq: pbl_base.lo=0x%x pbl_base.hi=0x%x\n"
		"curr_pbe_base.lo=0x%x curr_pbe_base.hi=0x%x",
		ictx->xstorm_st_context.iscsi.hq_pbl_base.lo,
		ictx->xstorm_st_context.iscsi.hq_pbl_base.hi,
		ictx->xstorm_st_context.iscsi.hq_curr_pbe_base.lo,
		ictx->xstorm_st_context.iscsi.hq_curr_pbe_base.hi);
	CNIC_INFO(dev, CNIC_MSG_DRV, "r2tq: pbl_base.lo=0x%x pbl_base.hi=0x%x\n"
		"curr_pbe_base.lo=0x%x curr_pbe_base.hi=0x%x",
		ictx->xstorm_st_context.iscsi.r2tq_pbl_base.lo,
		ictx->xstorm_st_context.iscsi.r2tq_pbl_base.hi,
		ictx->xstorm_st_context.iscsi.r2tq_curr_pbe_base.lo,
		ictx->xstorm_st_context.iscsi.r2tq_curr_pbe_base.hi);
	CNIC_INFO(dev, CNIC_MSG_DRV, "task: pbl_base.lo=0x%x pbl_base.hi=0x%x\n"
		"port=0x%x", ictx->xstorm_st_context.iscsi.task_pbl_base.lo,
		ictx->xstorm_st_context.iscsi.task_pbl_base.hi,
		port);

	ictx->tstorm_st_context.iscsi.hdr_bytes_2_fetch = ISCSI_HEADER_SIZE;
	/* TSTORM requires the base address of RQ DB & not PTE */
	ictx->tstorm_st_context.iscsi.rq_db_phy_addr.lo =
		req2->rq_page_table_addr_lo & (~VMK_PAGE_MASK);
	ictx->tstorm_st_context.iscsi.rq_db_phy_addr.hi =
		req2->rq_page_table_addr_hi;
	ictx->tstorm_st_context.iscsi.iscsi_conn_id = req1->iscsi_conn_id;
	ictx->tstorm_st_context.tcp.cwnd = 0x5A8;
	ictx->tstorm_st_context.tcp.flags2 |=
		TSTORM_TCP_ST_CONTEXT_SECTION_DA_EN;
	ictx->tstorm_st_context.tcp.ooo_support_mode =
		TCP_TSTORM_OOO_DROP_AND_PROC_ACK;

	CNIC_INFO(dev, CNIC_MSG_DRV, "rq: addr.lo=0x%x addr.hi=0x%x\n"
		"iscsi_conn_id=0x%x", ictx->tstorm_st_context.iscsi.rq_db_phy_addr.lo,
		ictx->tstorm_st_context.iscsi.rq_db_phy_addr.hi, req1->iscsi_conn_id);

	ictx->timers_context.flags |= TIMERS_BLOCK_CONTEXT_CONN_VALID_FLG;

	ictx->ustorm_st_context.ring.rq.pbl_base.lo =
		req2->rq_page_table_addr_lo;
	ictx->ustorm_st_context.ring.rq.pbl_base.hi =
		req2->rq_page_table_addr_hi;
	ictx->ustorm_st_context.ring.rq.curr_pbe.lo = req3->qp_first_pte[0].hi;
	ictx->ustorm_st_context.ring.rq.curr_pbe.hi = req3->qp_first_pte[0].lo;
	ictx->ustorm_st_context.ring.r2tq.pbl_base.lo =
		iscsi->r2tq_info.pgtbl_map & 0xffffffff;
	ictx->ustorm_st_context.ring.r2tq.pbl_base.hi =
		(vmk_uint64) iscsi->r2tq_info.pgtbl_map >> 32;
	ictx->ustorm_st_context.ring.r2tq.curr_pbe.lo =
		iscsi->r2tq_info.pgtbl[0];
	ictx->ustorm_st_context.ring.r2tq.curr_pbe.hi =
		iscsi->r2tq_info.pgtbl[1];
	ictx->ustorm_st_context.ring.cq_pbl_base.lo =
		req1->cq_page_table_addr_lo;
	ictx->ustorm_st_context.ring.cq_pbl_base.hi =
		req1->cq_page_table_addr_hi;
	ictx->ustorm_st_context.ring.cq[0].cq_sn = ISCSI_INITIAL_SN;
	ictx->ustorm_st_context.ring.cq[0].curr_pbe.lo = req2->cq_first_pte.hi;
	ictx->ustorm_st_context.ring.cq[0].curr_pbe.hi = req2->cq_first_pte.lo;
	ictx->ustorm_st_context.task_pbe_cache_index =
		QFLE3_ISCSI_PBL_NOT_CACHED;
	ictx->ustorm_st_context.task_pdu_cache_index =
		QFLE3_ISCSI_PDU_HEADER_NOT_CACHED;

	CNIC_INFO(dev, CNIC_MSG_DRV, "rq: pbl_base.lo=0x%x pbl_base.hi=0x%x\n"
		"curr_pbe.lo=0x%x curr_pbe.hi=0x%x",
		ictx->ustorm_st_context.ring.rq.pbl_base.lo,
		ictx->ustorm_st_context.ring.rq.pbl_base.hi,
		ictx->ustorm_st_context.ring.rq.curr_pbe.lo,
		ictx->ustorm_st_context.ring.rq.curr_pbe.hi);
	CNIC_INFO(dev, CNIC_MSG_DRV, "r2tq: pbl_base.lo=0x%x pbl_base.hi=0x%x\n"
		"curr_pbe.lo=0x%x curr_pbe.hi=0x%x",
		ictx->ustorm_st_context.ring.r2tq.pbl_base.lo,
		ictx->ustorm_st_context.ring.r2tq.pbl_base.hi,
		ictx->ustorm_st_context.ring.r2tq.curr_pbe.lo,
		ictx->ustorm_st_context.ring.r2tq.curr_pbe.hi);
	CNIC_INFO(dev, CNIC_MSG_DRV, "cq: pbl_base.lo=0x%x pbl_base.hi=0x%x\n"
		"curr_pbe.lo=0x%x curr_pbe.hi=0x%x",
		ictx->ustorm_st_context.ring.cq_pbl_base.lo,
		ictx->ustorm_st_context.ring.cq_pbl_base.hi,
		ictx->ustorm_st_context.ring.cq[0].curr_pbe.lo,
		ictx->ustorm_st_context.ring.cq[0].curr_pbe.hi);

	for (i = 1, j = 1; i < cp->num_cqs; i++, j++) {
		if (j == 3) {
			if (n >= n_max)
				break;
			req3 = (struct iscsi_kwqe_conn_offload3 *) wqes[n++];
			j = 0;
		}
		ictx->ustorm_st_context.ring.cq[i].cq_sn = ISCSI_INITIAL_SN;
		ictx->ustorm_st_context.ring.cq[i].curr_pbe.lo =
			req3->qp_first_pte[j].hi;
		ictx->ustorm_st_context.ring.cq[i].curr_pbe.hi =
			req3->qp_first_pte[j].lo;
	}

	ictx->ustorm_st_context.task_pbl_base.lo =
		iscsi->task_array_info.pgtbl_map & 0xffffffff;
	ictx->ustorm_st_context.task_pbl_base.hi =
		(vmk_uint64) iscsi->task_array_info.pgtbl_map >> 32;
	ictx->ustorm_st_context.tce_phy_addr.lo =
		iscsi->task_array_info.pgtbl[0];
	ictx->ustorm_st_context.tce_phy_addr.hi =
		iscsi->task_array_info.pgtbl[1];
	ictx->ustorm_st_context.iscsi_conn_id = req1->iscsi_conn_id;
	ictx->ustorm_st_context.num_cqs = cp->num_cqs;
	ictx->ustorm_st_context.negotiated_rx |= ISCSI_DEF_MAX_RECV_SEG_LEN;
	ictx->ustorm_st_context.negotiated_rx_and_flags |=
		ISCSI_DEF_MAX_BURST_LEN;
	ictx->ustorm_st_context.negotiated_rx |=
		ISCSI_DEFAULT_MAX_OUTSTANDING_R2T <<
		USTORM_ISCSI_ST_CONTEXT_MAX_OUTSTANDING_R2TS_SHIFT;

	ictx->cstorm_st_context.hq_pbl_base.lo =
		iscsi->hq_info.pgtbl_map & 0xffffffff;
	ictx->cstorm_st_context.hq_pbl_base.hi =
		(vmk_uint64) iscsi->hq_info.pgtbl_map >> 32;
	ictx->cstorm_st_context.hq_curr_pbe.lo = iscsi->hq_info.pgtbl[0];
	ictx->cstorm_st_context.hq_curr_pbe.hi = iscsi->hq_info.pgtbl[1];
	ictx->cstorm_st_context.task_pbl_base.lo =
		iscsi->task_array_info.pgtbl_map & 0xffffffff;
	ictx->cstorm_st_context.task_pbl_base.hi =
		(vmk_uint64) iscsi->task_array_info.pgtbl_map >> 32;
	/*
	 * CSTORM and USTORM initialization is different,
	 * CSTORM requires CQ DB base & not PTE address.
	 */
	ictx->cstorm_st_context.cq_db_base.lo =
		req1->cq_page_table_addr_lo & (~VMK_PAGE_MASK);
	ictx->cstorm_st_context.cq_db_base.hi = req1->cq_page_table_addr_hi;
	ictx->cstorm_st_context.iscsi_conn_id = req1->iscsi_conn_id;
	ictx->cstorm_st_context.cq_proc_en_bit_map = (1 << cp->num_cqs) - 1;
	for (i = 0; i < cp->num_cqs; i++) {
		ictx->cstorm_st_context.cq_c_prod_sqn_arr.sqn[i] =
			ISCSI_INITIAL_SN;
		ictx->cstorm_st_context.cq_c_sqn_2_notify_arr.sqn[i] =
			ISCSI_INITIAL_SN;
	}

	ictx->xstorm_ag_context.cdu_reserved =
		CDU_RSRVD_VALUE_TYPE_A(hw_cid, CDU_REGION_NUMBER_XCM_AG,
				ISCSI_CONNECTION_TYPE);
	ictx->ustorm_ag_context.cdu_usage =
		CDU_RSRVD_VALUE_TYPE_A(hw_cid, CDU_REGION_NUMBER_UCM_AG,
				ISCSI_CONNECTION_TYPE);

	return VMK_OK;
}

static int cnic_qfle3_iscsi_ofld1(struct cnic_dev *dev, struct kwqe *wqes[],
	vmk_uint32 num, int *work)
{
	struct qfle3_adapter *adapter = NULL;
	struct iscsi_kwqe_conn_offload1 *req1;
	struct iscsi_kwqe_conn_offload2 *req2;
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_context *ctx;
	struct iscsi_kcqe kcqe;
	struct kcqe *cqes[1];
	vmk_uint32 l5_cid;
	int ret = VMK_OK;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (num < 2) {
		*work = num;
		return VMK_BAD_PARAM;
	}

	req1 = (struct iscsi_kwqe_conn_offload1 *) wqes[0];
	req2 = (struct iscsi_kwqe_conn_offload2 *) wqes[1];
	if ((num - 2) < req2->num_additional_wqes) {
		*work = num;
		return VMK_BAD_PARAM;
	}
	*work = 2 + req2->num_additional_wqes;

	l5_cid = req1->iscsi_conn_id;
	if (l5_cid >= MAX_ISCSI_TBL_SZ)
		return VMK_BAD_PARAM;

	vmk_Memset(&kcqe, 0, sizeof(kcqe));
	kcqe.op_code = ISCSI_KCQE_OPCODE_OFFLOAD_CONN;
	kcqe.iscsi_conn_id = l5_cid;
	kcqe.completion_status = ISCSI_KCQE_COMPLETION_STATUS_CTX_ALLOC_FAILURE;

	ctx = &cp->ctx_tbl[l5_cid];
	if (vmk_BitVectorTest(ctx->ctx_flags, CTX_FL_OFFLD_START)) {
		kcqe.completion_status =
			ISCSI_KCQE_COMPLETION_STATUS_CID_BUSY;
		goto done;
	}

	/* Note: XXX: readInc returns value before inc, so compare accordingly. */
	if ((vmk_AtomicReadInc64(&cp->iscsi_conn) + 1) > dev->max_iscsi_conn) {
		vmk_AtomicDec64(&cp->iscsi_conn);
		CNIC_ERR(dev, "Error: iscsi_conn=0x%lx max_iscsi_conn=0x%x",
			vmk_AtomicRead64(&cp->iscsi_conn), dev->max_iscsi_conn);
		goto done;
	}

	ret = cnic_setup_qfle3_ctx(dev, wqes, num);
	if (ret >= VMK_FAILURE) {
		cnic_free_qfle3_conn_resc(dev, l5_cid);
		vmk_AtomicDec64(&cp->iscsi_conn);
		goto done;
	}

	kcqe.completion_status = ISCSI_KCQE_COMPLETION_STATUS_SUCCESS;
	kcqe.iscsi_conn_context_id = HW_CID(adapter, cp->ctx_tbl[l5_cid].cid);

	CNIC_INFO(dev, CNIC_MSG_DRV, "iscsi_conn_context_id=0x%x l5_cid=0x%x\n"
		"iscsi_conn=0x%lx",
		kcqe.iscsi_conn_context_id, l5_cid, vmk_AtomicRead64(&cp->iscsi_conn));

done:
	cqes[0] = (struct kcqe *) &kcqe;
	cnic_reply_qfle3_kcqes(dev, CNIC_ULP_ISCSI, cqes, 1);

	return ret;
}


static int cnic_qfle3_iscsi_update(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iscsi_kwqe_conn_update *req =
		(struct iscsi_kwqe_conn_update *) kwqe;
	void *data;
	union l5cm_specific_data l5_data;
	vmk_uint32 l5_cid, cid = QFLE3_SW_CID(req->context_id);
	int ret;

	if (VMK_UNLIKELY(cnic_get_l5_cid(cp, cid, &l5_cid) != 0)) {
		CNIC_ERR(dev, "Could not get l5 CID: cid=0x%x", cid);
		return VMK_BAD_PARAM;
	}

	data = cnic_get_kwqe_16_data(cp, l5_cid, &l5_data);
	if (!data)
		return VMK_NO_MEMORY;

#if (CNIC_ISCSI_OOO_SUPPORT)
	cnic_qfle3_ooo_iscsi_conn_update(dev, kwqe);
#endif
	vmk_Memcpy(data, kwqe, sizeof(struct kwqe));

	ret = cnic_submit_kwqe_16(dev, ISCSI_RAMROD_CMD_ID_UPDATE_CONN,
			req->context_id, ISCSI_CONNECTION_TYPE, &l5_data);
	return ret;
}

static int cnic_qfle3_destroy_ramrod(struct cnic_dev *dev, vmk_uint32 l5_cid)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	struct cnic_context *ctx = &cp->ctx_tbl[l5_cid];
	union l5cm_specific_data l5_data;
	int ret;
	vmk_uint32 hw_cid, type;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (dev->fw_recov_in_progress) {
		CNIC_WARN(dev, "FW recovery in progress, l5_cid=0x%x", l5_cid);
		return 0;
	}
	CNIC_INFO(dev, CNIC_MSG_SESS, "cfc del l5_cid=0x%x", l5_cid);

	ql_vmk_init_completion(&ctx->waitq);
	ctx->wait_cond = 0;
	vmk_Memset(&l5_data, 0, sizeof(l5_data));
	hw_cid = HW_CID(adapter, ctx->cid);
	type = NONE_CONNECTION_TYPE;

	ctx->waitq.cmpl_done = 0;
	ret = cnic_submit_kwqe_16(dev, RAMROD_CMD_ID_COMMON_CFC_DEL,
				  hw_cid, type, &l5_data);

	if (ret == 0) {
		CNIC_INFO(dev, CNIC_MSG_SESS, "Calling ql_vmk_wait_for_completion.");
		VMK_ReturnStatus status =
			ql_vmk_wait_for_completion(&ctx->waitq, QFLE3_RAMROD_TMO);
		VMK_ASSERT(status != VMK_TIMEOUT, "conn-destroy tmo.");

		if (VMK_UNLIKELY(vmk_BitVectorTest(ctx->ctx_flags, CTX_FL_CID_ERROR))) {
			vmk_SpinlockDestroy(ctx->waitq.cmpl_lock);
			return VMK_BUSY;
		}
	}
	vmk_SpinlockDestroy(ctx->waitq.cmpl_lock);

	return 0;
}

static int cnic_qfle3_iscsi_destroy(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iscsi_kwqe_conn_destroy *req =
		(struct iscsi_kwqe_conn_destroy *) kwqe;
	vmk_uint32 l5_cid = req->reserved0;
	struct cnic_context *ctx = &cp->ctx_tbl[l5_cid];
	int ret = 0;
	struct iscsi_kcqe kcqe;
	struct kcqe *cqes[1];

	if (!vmk_BitVectorTest(ctx->ctx_flags, CTX_FL_OFFLD_START)){
		cnic_free_qfle3_conn_resc(dev, l5_cid);
		goto destroy_reply;
	}

	if (vmk_GetTimerCycles() <
			(ctx->timestamp + (2 * vmk_TimerCyclesPerSecond()))) {
		unsigned long delta = (ctx->timestamp +
			(2 * vmk_TimerCyclesPerSecond())) - vmk_GetTimerCycles();

		if (delta > (2 * vmk_TimerCyclesPerSecond()))
			delta = 0;

		CNIC_INFO(dev, CNIC_MSG_DRV, "ctx=%p delta=0x%lx",
			ctx, vmk_TimerUnsignedTCToMS(delta));

		vmk_BitVectorSet(ctx->ctx_flags, CTX_FL_DELETE_WAIT);
		ql_vmk_queue_delayed_work(cnic_driver_info.delayed_tq,
			cnic_driver_info.delayed_wq, &cp->delete_task, cnic_delete_task,
			cp, vmk_TimerUnsignedTCToMS(delta));
		goto destroy_reply;
	}

	ret = cnic_qfle3_destroy_ramrod(dev, l5_cid);
	cnic_free_qfle3_conn_resc(dev, l5_cid);

	if (!ret) {
		vmk_AtomicDec64(&cp->iscsi_conn);
		vmk_BitVectorClear(ctx->ctx_flags, CTX_FL_OFFLD_START);
	}

destroy_reply:
	vmk_Memset(&kcqe, 0, sizeof(kcqe));
	kcqe.op_code = ISCSI_KCQE_OPCODE_DESTROY_CONN;
	kcqe.iscsi_conn_id = l5_cid;
	kcqe.completion_status = ISCSI_KCQE_COMPLETION_STATUS_SUCCESS;
	kcqe.iscsi_conn_context_id = req->context_id;

	cqes[0] = (struct kcqe *) &kcqe;
	cnic_reply_qfle3_kcqes(dev, CNIC_ULP_ISCSI, cqes, 1);

	return 0;
}

static void cnic_init_storm_conn_bufs(struct cnic_dev *dev,
				      struct l4_kwq_connect_req1 *kwqe1,
				      struct l4_kwq_connect_req3 *kwqe3,
				      struct l5cm_active_conn_buffer *conn_buf)
{
	struct l5cm_conn_addr_params *conn_addr = &conn_buf->conn_addr_buf;
	struct l5cm_xstorm_conn_buffer *xstorm_buf =
		&conn_buf->xstorm_conn_buffer;
	struct l5cm_tstorm_conn_buffer *tstorm_buf =
		&conn_buf->tstorm_conn_buffer;
	struct regpair context_addr;
	vmk_uint32 cid = QFLE3_SW_CID(kwqe1->cid);
	struct vmk_SocketIPv6AddressAddr src_ip, dst_ip;
	int i;
	vmk_uint8 proto = VMK_IP_PROTO_TCP;
	vmk_uint32 *addrp;

	addrp = (vmk_uint32 *) &conn_addr->local_ip_addr;
	for (i = 0; i < 4; i++, addrp++)
		src_ip.__u6_addr.__u6_addr32[i] = vmk_CPUToBE32(*addrp);

	addrp = (vmk_uint32 *) &conn_addr->remote_ip_addr;
	for (i = 0; i < 4; i++, addrp++)
		dst_ip.__u6_addr.__u6_addr32[i] = vmk_CPUToBE32(*addrp);

	cnic_get_qfle3_ctx(dev, cid, 0, &context_addr);

	xstorm_buf->context_addr.hi = context_addr.hi;
	xstorm_buf->context_addr.lo = context_addr.lo;
	xstorm_buf->mss = 0xffff;
	xstorm_buf->rcv_buf = kwqe3->rcv_buf;
	if (kwqe1->tcp_flags & L4_KWQ_CONNECT_REQ1_NAGLE_ENABLE)
		xstorm_buf->params |= L5CM_XSTORM_CONN_BUFFER_NAGLE_ENABLE;
	//SV: XXX: verify replacement of swab16 with vmk_CPUToBE16
	xstorm_buf->pseudo_header_checksum =
		vmk_CPUToBE16(~vmk_NetCsumIPv6Pseudo(src_ip.__u6_addr.__u6_addr8,
				dst_ip.__u6_addr.__u6_addr8, &proto, 0, 0));

	if (kwqe3->ka_timeout) {
		tstorm_buf->ka_enable = 1;
		tstorm_buf->ka_timeout = kwqe3->ka_timeout;
		tstorm_buf->ka_interval = kwqe3->ka_interval;
		tstorm_buf->ka_max_probe_count = kwqe3->ka_max_probe_count;
	}
	tstorm_buf->max_rt_time = 0xffffffff;
}

static void cnic_init_qfle3_mac(struct cnic_dev *dev, vmk_uint8 *mac)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	vmk_uint32 pfid;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);
	pfid = adapter->pf_id;

	vmk_Memcpy(adapter->iscsi_mac, mac, VMK_ETH_ADDR_LENGTH);

	if (cp->ethdev->mf_mode == MULTI_FUNCTION_SI ||
			cp->ethdev->mf_mode == MULTI_FUNCTION_AFEX)
		cnic_npar_ring_ctl(dev, 1);

	CNIC_WR8(dev, BAR_XSTRORM_INTMEM +
		 XSTORM_ISCSI_LOCAL_MAC_ADDR0_OFFSET(pfid), mac[0]);
	CNIC_WR8(dev, BAR_XSTRORM_INTMEM +
		 XSTORM_ISCSI_LOCAL_MAC_ADDR1_OFFSET(pfid), mac[1]);
	CNIC_WR8(dev, BAR_XSTRORM_INTMEM +
		 XSTORM_ISCSI_LOCAL_MAC_ADDR2_OFFSET(pfid), mac[2]);
	CNIC_WR8(dev, BAR_XSTRORM_INTMEM +
		 XSTORM_ISCSI_LOCAL_MAC_ADDR3_OFFSET(pfid), mac[3]);
	CNIC_WR8(dev, BAR_XSTRORM_INTMEM +
		 XSTORM_ISCSI_LOCAL_MAC_ADDR4_OFFSET(pfid), mac[4]);
	CNIC_WR8(dev, BAR_XSTRORM_INTMEM +
		 XSTORM_ISCSI_LOCAL_MAC_ADDR5_OFFSET(pfid), mac[5]);

	CNIC_WR8(dev, BAR_TSTRORM_INTMEM +
		 TSTORM_ISCSI_TCP_VARS_LSB_LOCAL_MAC_ADDR_OFFSET(pfid), mac[5]);
	CNIC_WR8(dev, BAR_TSTRORM_INTMEM +
		 TSTORM_ISCSI_TCP_VARS_LSB_LOCAL_MAC_ADDR_OFFSET(pfid) + 1,
		 mac[4]);
	CNIC_WR8(dev, BAR_TSTRORM_INTMEM +
		 TSTORM_ISCSI_TCP_VARS_MID_LOCAL_MAC_ADDR_OFFSET(pfid), mac[3]);
	CNIC_WR8(dev, BAR_TSTRORM_INTMEM +
		 TSTORM_ISCSI_TCP_VARS_MID_LOCAL_MAC_ADDR_OFFSET(pfid) + 1,
		 mac[2]);
	CNIC_WR8(dev, BAR_TSTRORM_INTMEM +
		 TSTORM_ISCSI_TCP_VARS_MSB_LOCAL_MAC_ADDR_OFFSET(pfid), mac[1]);
	CNIC_WR8(dev, BAR_TSTRORM_INTMEM +
		 TSTORM_ISCSI_TCP_VARS_MSB_LOCAL_MAC_ADDR_OFFSET(pfid) + 1,
		 mac[0]);
}

static int cnic_qfle3_connect(struct cnic_dev *dev, struct kwqe *wqes[],
	vmk_uint32 num, int *work)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	struct l4_kwq_connect_req1 *kwqe1 =
		(struct l4_kwq_connect_req1 *) wqes[0];
	struct l4_kwq_connect_req3 *kwqe3;
	struct l5cm_active_conn_buffer *conn_buf;
	struct l5cm_conn_addr_params *conn_addr;
	union l5cm_specific_data l5_data;
	vmk_uint32 l5_cid = kwqe1->pg_cid;
	struct cnic_sock *csk = &cp->csk_tbl[l5_cid];
	struct cnic_context *ctx = &cp->ctx_tbl[l5_cid];
	int ret;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (num < 2) {
		*work = num;
		return VMK_BAD_PARAM;
	}

	if (kwqe1->conn_flags & L4_KWQ_CONNECT_REQ1_IP_V6)
		*work = 3;
	else
		*work = 2;

	if (num < *work) {
		*work = num;
		return VMK_BAD_PARAM;
	}

	if (sizeof(*conn_buf) > CNIC_KWQ16_DATA_SIZE) {
		CNIC_ERR(dev, "conn_buf size too big: sizeof(*conn_buf)(%ld) > "
			"CNIC_KWQ16_DATA_SIZE(%d)",
			sizeof(*conn_buf), CNIC_KWQ16_DATA_SIZE);
		return VMK_NO_MEMORY;
	}
	conn_buf = cnic_get_kwqe_16_data(cp, l5_cid, &l5_data);
	if (!conn_buf)
		return VMK_NO_MEMORY;

	vmk_Memset(conn_buf, 0, sizeof(*conn_buf));

	conn_addr = &conn_buf->conn_addr_buf;
	conn_addr->remote_addr_0 = csk->ha[0];
	conn_addr->remote_addr_1 = csk->ha[1];
	conn_addr->remote_addr_2 = csk->ha[2];
	conn_addr->remote_addr_3 = csk->ha[3];
	conn_addr->remote_addr_4 = csk->ha[4];
	conn_addr->remote_addr_5 = csk->ha[5];

	/* Use the stored MAC address */
	cnic_init_qfle3_mac(dev, cp->srcMACAddr);

	if (kwqe1->conn_flags & L4_KWQ_CONNECT_REQ1_IP_V6) {
		struct l4_kwq_connect_req2 *kwqe2 =
			(struct l4_kwq_connect_req2 *) wqes[1];

		conn_addr->local_ip_addr.ip_addr_hi_hi = kwqe2->src_ip_v6_4;
		conn_addr->local_ip_addr.ip_addr_hi_lo = kwqe2->src_ip_v6_3;
		conn_addr->local_ip_addr.ip_addr_lo_hi = kwqe2->src_ip_v6_2;

		conn_addr->remote_ip_addr.ip_addr_hi_hi = kwqe2->dst_ip_v6_4;
		conn_addr->remote_ip_addr.ip_addr_hi_lo = kwqe2->dst_ip_v6_3;
		conn_addr->remote_ip_addr.ip_addr_lo_hi = kwqe2->dst_ip_v6_2;
		conn_addr->params |= L5CM_CONN_ADDR_PARAMS_IP_VERSION;
	}
	kwqe3 = (struct l4_kwq_connect_req3 *) wqes[*work - 1];

	conn_addr->local_ip_addr.ip_addr_lo_lo = kwqe1->src_ip;
	conn_addr->remote_ip_addr.ip_addr_lo_lo = kwqe1->dst_ip;
	conn_addr->local_tcp_port = kwqe1->src_port;
	conn_addr->remote_tcp_port = kwqe1->dst_port;

	conn_addr->pmtu = kwqe3->pmtu;
	cnic_init_storm_conn_bufs(dev, kwqe1, kwqe3, conn_buf);

	CNIC_INFO(dev, CNIC_MSG_DRV, "remote_addr_0=0x%x %x %x %x %x %x\n"
		"local srcip=0x%x dstip=0x%x src-port=0x%x dst-port=0x%x",
		conn_addr->remote_addr_0, conn_addr->remote_addr_1,
		conn_addr->remote_addr_2, conn_addr->remote_addr_3,
		conn_addr->remote_addr_4, conn_addr->remote_addr_5,
		conn_addr->local_ip_addr.ip_addr_lo_lo,
		conn_addr->remote_ip_addr.ip_addr_lo_lo, conn_addr->local_tcp_port,
		conn_addr->remote_tcp_port);

	CNIC_WR16(dev, BAR_XSTRORM_INTMEM +
		XSTORM_ISCSI_LOCAL_VLAN_OFFSET(adapter->pf_id), csk->vlan_id);

	CNIC_INFO(dev, CNIC_MSG_DRV, "pfid=0x%x vlan_id=0x%x ramrod=0x%x, cid=0x%x",
		adapter->pf_id, csk->vlan_id, L5CM_RAMROD_CMD_ID_TCP_CONNECT,
		kwqe1->cid);

	ret = cnic_submit_kwqe_16(dev, L5CM_RAMROD_CMD_ID_TCP_CONNECT,
			kwqe1->cid, ISCSI_CONNECTION_TYPE, &l5_data);

	if (!ret)
		vmk_BitVectorSet(ctx->ctx_flags, CTX_FL_OFFLD_START);

	return ret;
}

static int cnic_qfle3_close(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct l4_kwq_close_req *req = (struct l4_kwq_close_req *) kwqe;
	union l5cm_specific_data l5_data;
	int ret;

	vmk_Memset(&l5_data, 0, sizeof(l5_data));
	/*
	 * Close TCP connection (i.e. send FIN/RST, wait for peer response as
	 * appropriate.
	 */
	ret = cnic_submit_kwqe_16(dev, L5CM_RAMROD_CMD_ID_CLOSE,
			req->cid, ISCSI_CONNECTION_TYPE, &l5_data);

	return ret;
}

static int cnic_qfle3_reset(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct l4_kwq_reset_req *req = (struct l4_kwq_reset_req *) kwqe;
	union l5cm_specific_data l5_data;
	int ret;

	vmk_Memset(&l5_data, 0, sizeof(l5_data));
	ret = cnic_submit_kwqe_16(dev, L5CM_RAMROD_CMD_ID_ABORT,
			req->cid, ISCSI_CONNECTION_TYPE, &l5_data);

	return ret;
}

static int cnic_qfle3_offload_pg(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct l4_kwq_offload_pg *req = (struct l4_kwq_offload_pg *) kwqe;
	struct l4_kcq kcqe;
	struct kcqe *cqes[1];

	vmk_Memset(&kcqe, 0, sizeof(kcqe));
	kcqe.pg_host_opaque = req->host_opaque;
	kcqe.pg_cid = req->host_opaque;
	kcqe.op_code = L4_KCQE_OPCODE_VALUE_OFFLOAD_PG;
	cqes[0] = (struct kcqe *) &kcqe;

	CNIC_INFO(dev, CNIC_MSG_DRV, "cqe opcode=0x%x cid=0x%x",
		kcqe.op_code, kcqe.pg_cid);
	cnic_reply_qfle3_kcqes(dev, CNIC_ULP_L4, cqes, 1);

	return 0;
}

static int cnic_qfle3_update_pg(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct l4_kwq_update_pg *req = (struct l4_kwq_update_pg *) kwqe;
	struct l4_kcq kcqe;
	struct kcqe *cqes[1];

	vmk_Memset(&kcqe, 0, sizeof(kcqe));
	kcqe.pg_host_opaque = req->pg_host_opaque;
	kcqe.pg_cid = req->pg_cid;
	kcqe.op_code = L4_KCQE_OPCODE_VALUE_UPDATE_PG;
	cqes[0] = (struct kcqe *) &kcqe;
	cnic_reply_qfle3_kcqes(dev, CNIC_ULP_L4, cqes, 1);

	return 0;
}

static int cnic_qfle3_fcoe_stat(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct qfle3_adapter *adapter = NULL;
	struct fcoe_kwqe_stat *req;
	struct fcoe_stat_ramrod_params *fcoe_stat;
	union l5cm_specific_data l5_data;
	struct cnic_local *cp = dev->cnic_priv;
	int ret = VMK_OK;
	vmk_uint32 cid;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	req = (struct fcoe_kwqe_stat *) kwqe;
	cid = HW_CID(adapter, cp->fcoe_init_cid);

	fcoe_stat = cnic_get_kwqe_16_data(cp, QFLE3_FCOE_L5_CID_BASE, &l5_data);
	if (!fcoe_stat)
		return VMK_NO_MEMORY;

	vmk_Memset(fcoe_stat, 0, sizeof(*fcoe_stat));
	vmk_Memcpy(&fcoe_stat->stat_kwqe, req, sizeof(*req));

	ret = cnic_submit_kwqe_16(dev, FCOE_RAMROD_CMD_ID_STAT_FUNC, cid,
			FCOE_CONNECTION_TYPE, &l5_data);

	return ret;
}

static int cnic_qfle3_fcoe_init1(struct cnic_dev *dev, struct kwqe *wqes[],
	vmk_uint32 num, int *work)
{
	int ret = VMK_OK;
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	vmk_uint32 cid;
	struct fcoe_init_ramrod_params *fcoe_init;
	struct fcoe_kwqe_init1 *req1;
	struct fcoe_kwqe_init2 *req2;
	struct fcoe_kwqe_init3 *req3;
	union l5cm_specific_data l5_data;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (num < 3) {
		*work = num;
		return VMK_BAD_PARAM;
	}

	req1 = (struct fcoe_kwqe_init1 *) wqes[0];
	req2 = (struct fcoe_kwqe_init2 *) wqes[1];
	req3 = (struct fcoe_kwqe_init3 *) wqes[2];
	if (req2->hdr.op_code != FCOE_KWQE_OPCODE_INIT2) {
		*work = 1;
		return VMK_BAD_PARAM;
	}
	if (req3->hdr.op_code != FCOE_KWQE_OPCODE_INIT3) {
		*work = 2;
		return VMK_BAD_PARAM;
	}

	if (sizeof(*fcoe_init) > CNIC_KWQ16_DATA_SIZE) {
		CNIC_ERR(dev, "fcoe_init size too big.");
		return VMK_NO_MEMORY;
	}
	fcoe_init = cnic_get_kwqe_16_data(cp, QFLE3_FCOE_L5_CID_BASE, &l5_data);
	if (!fcoe_init)
		return VMK_NO_MEMORY;

	if (IS_MF_UFP(adapter))
		req1->flags |= FCOE_KWQE_INIT1_CLASSIFY_FAILED_ALLOWED;

	vmk_Memset(fcoe_init, 0, sizeof(*fcoe_init));
	vmk_Memcpy(&fcoe_init->init_kwqe1, req1, sizeof(*req1));
	vmk_Memcpy(&fcoe_init->init_kwqe2, req2, sizeof(*req2));
	vmk_Memcpy(&fcoe_init->init_kwqe3, req3, sizeof(*req3));
	fcoe_init->eq_pbl_base.lo = cp->kcq2.dma.pgtbl_map & 0xffffffff;
	fcoe_init->eq_pbl_base.hi = (vmk_uint64) cp->kcq2.dma.pgtbl_map >> 32;
	fcoe_init->eq_pbl_size = cp->kcq2.dma.num_pages;
	/* SV: XXX: For now, single SB for all queues like old E3 driver. */
	fcoe_init->sb_num = cp->cfp[CNIC_RX_IDX].status_blk_num;
	fcoe_init->eq_prod = MAX_KCQ_IDX;
	fcoe_init->sb_id = HC_INDEX_FCOE_EQ_CONS;
	cp->kcq2.sw_prod_idx = 0;

	cid = HW_CID(adapter, cp->fcoe_init_cid);
	CNIC_INFO(dev, CNIC_MSG_DRV, "bdbg: submitting INIT RAMROD.\n"
		"fcoe-init-cid=0x%x cid=0x%x", cp->fcoe_init_cid, cid);
	ret = cnic_submit_kwqe_16(dev, FCOE_RAMROD_CMD_ID_INIT_FUNC, cid,
			FCOE_CONNECTION_TYPE, &l5_data);
	*work = 3;

	return ret;
}

static int cnic_qfle3_fcoe_ofld1(struct cnic_dev *dev, struct kwqe *wqes[],
	vmk_uint32 num, int *work)
{
	int ret = VMK_FAILURE;
	vmk_uint32 cid = -1, l5_cid;
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	struct fcoe_kwqe_conn_offload1 *req1;
	struct fcoe_kwqe_conn_offload2 *req2;
	struct fcoe_kwqe_conn_offload3 *req3;
	struct fcoe_kwqe_conn_offload4 *req4;
	struct fcoe_conn_offload_ramrod_params *fcoe_offload;
	struct cnic_context *ctx;
	struct fcoe_context *fctx;
	struct regpair ctx_addr;
	union l5cm_specific_data l5_data;
	struct fcoe_kcqe kcqe;
	struct kcqe *cqes[1];

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (num < 4) {
		*work = num;
		return VMK_BAD_PARAM;
	}

	req1 = (struct fcoe_kwqe_conn_offload1 *) wqes[0];
	req2 = (struct fcoe_kwqe_conn_offload2 *) wqes[1];
	req3 = (struct fcoe_kwqe_conn_offload3 *) wqes[2];
	req4 = (struct fcoe_kwqe_conn_offload4 *) wqes[3];

	*work = 4;

	l5_cid = req1->fcoe_conn_id;
	if (l5_cid >= dev->max_fcoe_conn)
		goto err_reply;

	l5_cid += QFLE3_FCOE_L5_CID_BASE;

	ctx = &cp->ctx_tbl[l5_cid];
	if (vmk_BitVectorTest(ctx->ctx_flags, CTX_FL_OFFLD_START))
		goto err_reply;

	cid = ctx->cid;

	fctx = cnic_get_qfle3_ctx(dev, cid, 1, &ctx_addr);
	if (fctx) {
		vmk_uint32 hw_cid = HW_CID(adapter, cid);
		vmk_uint32 val;

		val = CDU_RSRVD_VALUE_TYPE_A(hw_cid, CDU_REGION_NUMBER_XCM_AG,
					     FCOE_CONNECTION_TYPE);
		fctx->xstorm_ag_context.cdu_reserved = val;
		val = CDU_RSRVD_VALUE_TYPE_A(hw_cid, CDU_REGION_NUMBER_UCM_AG,
					     FCOE_CONNECTION_TYPE);
		fctx->ustorm_ag_context.cdu_usage = val;
	}
	if (sizeof(*fcoe_offload) > CNIC_KWQ16_DATA_SIZE) {
		CNIC_ERR(dev, "fcoe_offload size too big.");
		goto err_reply;
	}
	fcoe_offload = cnic_get_kwqe_16_data(cp, l5_cid, &l5_data);
	if (!fcoe_offload)
		goto err_reply;

	vmk_Memset(fcoe_offload, 0, sizeof(*fcoe_offload));
	vmk_Memcpy(&fcoe_offload->offload_kwqe1, req1, sizeof(*req1));
	vmk_Memcpy(&fcoe_offload->offload_kwqe2, req2, sizeof(*req2));
	vmk_Memcpy(&fcoe_offload->offload_kwqe3, req3, sizeof(*req3));
	vmk_Memcpy(&fcoe_offload->offload_kwqe4, req4, sizeof(*req4));

	cid = HW_CID(adapter, cid);
	ret = cnic_submit_kwqe_16(dev, FCOE_RAMROD_CMD_ID_OFFLOAD_CONN, cid,
				  FCOE_CONNECTION_TYPE, &l5_data);
	if (ret == VMK_OK)
		vmk_BitVectorSet(ctx->ctx_flags, CTX_FL_OFFLD_START);

	return ret;

err_reply:
	if (cid != -1)
		cnic_free_qfle3_conn_resc(dev, l5_cid);

	vmk_Memset(&kcqe, 0, sizeof(kcqe));
	kcqe.op_code = FCOE_KCQE_OPCODE_OFFLOAD_CONN;
	kcqe.fcoe_conn_id = req1->fcoe_conn_id;
	kcqe.completion_status = FCOE_KCQE_COMPLETION_STATUS_CTX_ALLOC_FAILURE;

	cqes[0] = (struct kcqe *) &kcqe;
	cnic_reply_qfle3_kcqes(dev, CNIC_ULP_FCOE, cqes, 1);

	return ret;
}

static int cnic_qfle3_fcoe_enable(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct fcoe_kwqe_conn_enable_disable *req;
	struct fcoe_conn_enable_disable_ramrod_params *fcoe_enable;
	union l5cm_specific_data l5_data;
	int ret = VMK_OK;
	vmk_uint32 cid, l5_cid;
	struct cnic_local *cp = dev->cnic_priv;

	req = (struct fcoe_kwqe_conn_enable_disable *) kwqe;
	cid = req->context_id;
	l5_cid = req->conn_id + QFLE3_FCOE_L5_CID_BASE;

	if (sizeof(*fcoe_enable) > CNIC_KWQ16_DATA_SIZE) {
		CNIC_ERR(dev, "fcoe_enable size too big.");
		return VMK_NO_MEMORY;
	}

	fcoe_enable = cnic_get_kwqe_16_data(cp, l5_cid, &l5_data);
	if (!fcoe_enable)
		return VMK_NO_MEMORY;

	vmk_Memset(fcoe_enable, 0, sizeof(*fcoe_enable));
	vmk_Memcpy(&fcoe_enable->enable_disable_kwqe, req, sizeof(*req));
	ret = cnic_submit_kwqe_16(dev, FCOE_RAMROD_CMD_ID_ENABLE_CONN, cid,
				  FCOE_CONNECTION_TYPE, &l5_data);

	return ret;
}

static int cnic_qfle3_fcoe_disable(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct fcoe_kwqe_conn_enable_disable *req;
	struct fcoe_conn_enable_disable_ramrod_params *fcoe_disable;
	union l5cm_specific_data l5_data;
	int ret = VMK_OK;
	vmk_uint32 cid, l5_cid;
	struct cnic_local *cp = dev->cnic_priv;

	req = (struct fcoe_kwqe_conn_enable_disable *) kwqe;
	cid = req->context_id;
	l5_cid = req->conn_id;
	if (l5_cid >= dev->max_fcoe_conn)
		return VMK_BAD_PARAM;

	l5_cid += QFLE3_FCOE_L5_CID_BASE;

	if (sizeof(*fcoe_disable) > CNIC_KWQ16_DATA_SIZE) {
		CNIC_ERR(dev, "fcoe_disable size too big.");
		return VMK_NO_MEMORY;
	}
	fcoe_disable = cnic_get_kwqe_16_data(cp, l5_cid, &l5_data);
	if (!fcoe_disable)
		return VMK_NO_MEMORY;

	vmk_Memset(fcoe_disable, 0, sizeof(*fcoe_disable));
	vmk_Memcpy(&fcoe_disable->enable_disable_kwqe, req, sizeof(*req));
	ret = cnic_submit_kwqe_16(dev, FCOE_RAMROD_CMD_ID_DISABLE_CONN, cid,
			FCOE_CONNECTION_TYPE, &l5_data);

	return ret;
}

static int cnic_qfle3_fcoe_destroy(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct fcoe_kwqe_conn_destroy *req;
	union l5cm_specific_data l5_data;
	int ret = VMK_OK;
	vmk_uint32 cid, l5_cid;
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_context *ctx;
	struct fcoe_kcqe kcqe;
	struct kcqe *cqes[1];

	req = (struct fcoe_kwqe_conn_destroy *) kwqe;
	cid = req->context_id;
	l5_cid = req->conn_id;
	if (l5_cid >= dev->max_fcoe_conn)
		return VMK_BAD_PARAM;

	l5_cid += QFLE3_FCOE_L5_CID_BASE;

	ctx = &cp->ctx_tbl[l5_cid];

	if (dev->fw_recov_in_progress) {
		CNIC_WARN(dev, "FW recovery in progress, l5_cid=0x%x", l5_cid);
		kcqe.completion_status = 0;
		goto skip_term_ramrod;
	}

	ql_vmk_init_completion(&ctx->waitq);
	ctx->wait_cond = 0;

	vmk_Memset(&kcqe, 0, sizeof(kcqe));
	kcqe.completion_status = FCOE_KCQE_COMPLETION_STATUS_ERROR;
	vmk_Memset(&l5_data, 0, sizeof(l5_data));

	ctx->waitq.cmpl_done = 0;

	ret = cnic_submit_kwqe_16(dev, FCOE_RAMROD_CMD_ID_TERMINATE_CONN, cid,
			FCOE_CONNECTION_TYPE, &l5_data);
	if (ret == 0) {
		CNIC_INFO(dev, CNIC_MSG_SESS, "Calling ql_vmk_wait_for_completion.");
		VMK_ReturnStatus status =
			ql_vmk_wait_for_completion(&ctx->waitq, QFLE3_RAMROD_TMO);
		VMK_ASSERT(status != VMK_TIMEOUT, "FCoE Session destroy tmo.");

		if (ctx->wait_cond)
			kcqe.completion_status = 0;
	}
	vmk_SpinlockDestroy(ctx->waitq.cmpl_lock);

skip_term_ramrod:
	vmk_BitVectorSet(ctx->ctx_flags, CTX_FL_DELETE_WAIT);
	ql_vmk_queue_delayed_work(cnic_driver_info.delayed_tq,
		cnic_driver_info.delayed_wq, &cp->delete_task, cnic_delete_task,
		cp, 2000);

	kcqe.op_code = FCOE_KCQE_OPCODE_DESTROY_CONN;
	kcqe.fcoe_conn_id = req->conn_id;
	kcqe.fcoe_conn_context_id = cid;

	cqes[0] = (struct kcqe *) &kcqe;
	cnic_reply_qfle3_kcqes(dev, CNIC_ULP_FCOE, cqes, 1);

	return ret;
}

static void cnic_qfle3_delete_wait(struct cnic_dev *dev, vmk_uint32 start_cid)
{
	struct cnic_local *cp = dev->cnic_priv;
	vmk_uint32 i;

	for (i = start_cid; i < cp->max_cid_space; i++) {
		struct cnic_context *ctx = &cp->ctx_tbl[i];
		int j;

		/* ctx_flags valid only if cid in use. */
		if (ctx->ctx_flags == NULL)
			continue;

		while (vmk_BitVectorTest(ctx->ctx_flags, CTX_FL_DELETE_WAIT))
			vmk_WorldSleep(10*VMK_USEC_PER_MSEC);

		for (j = 0; j < 200; j++) {
			if (!vmk_BitVectorTest(ctx->ctx_flags, CTX_FL_OFFLD_START))
				break;
			vmk_WorldSleep(20*VMK_USEC_PER_MSEC);
		}

		if (vmk_BitVectorTest(ctx->ctx_flags, CTX_FL_OFFLD_START))
			CNIC_ERR(dev, "CID=0x%x not deleted.", ctx->cid);
	}
}

static int cnic_qfle3_fcoe_fw_destroy(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct fcoe_kwqe_destroy *req;
	union l5cm_specific_data l5_data;
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	int ret = VMK_OK;
	vmk_uint32 cid;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	cnic_qfle3_delete_wait(dev, MAX_ISCSI_TBL_SZ);

	req = (struct fcoe_kwqe_destroy *) kwqe;
	cid = HW_CID(adapter, cp->fcoe_init_cid);

	vmk_Memset(&l5_data, 0, sizeof(l5_data));
	ret = cnic_submit_kwqe_16(dev, FCOE_RAMROD_CMD_ID_DESTROY_FUNC, cid,
			FCOE_CONNECTION_TYPE, &l5_data);

	return ret;
}

static void cnic_qfle3_kwqe_err(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct kcqe kcqe;
	struct kcqe *cqes[1];
	vmk_uint32 cid;
	vmk_uint32 opcode = KWQE_OPCODE(kwqe->kwqe_op_flag);
	vmk_uint32 layer_code = kwqe->kwqe_op_flag & KWQE_LAYER_MASK;
	int ulp_type;

	cid = kwqe->kwqe_info0;
	vmk_Memset(&kcqe, 0, sizeof(kcqe));

	if (layer_code == KWQE_FLAGS_LAYER_MASK_L5_ISCSI) {
		ulp_type = CNIC_ULP_ISCSI;
		if (opcode == ISCSI_KWQE_OPCODE_UPDATE_CONN)
			cid = kwqe->kwqe_info1;

		kcqe.kcqe_op_flag = (opcode + 0x10) << KCQE_FLAGS_OPCODE_SHIFT;
		kcqe.kcqe_op_flag |= KCQE_FLAGS_LAYER_MASK_L5_ISCSI;
		kcqe.kcqe_info1 = FCOE_KCQE_COMPLETION_STATUS_PARITY_ERROR;
		kcqe.kcqe_info2 = cid;
		cnic_get_l5_cid(cp, QFLE3_SW_CID(cid), &kcqe.kcqe_info0);

	} else if (layer_code == KWQE_FLAGS_LAYER_MASK_L4) {
		struct l4_kcq *l4kcqe = (struct l4_kcq *) &kcqe;
		vmk_uint32 kcqe_op;

		ulp_type = CNIC_ULP_L4;
		if (opcode == L4_KWQE_OPCODE_VALUE_CONNECT1)
			kcqe_op = L4_KCQE_OPCODE_VALUE_CONNECT_COMPLETE;
		else if (opcode == L4_KWQE_OPCODE_VALUE_RESET)
			kcqe_op = L4_KCQE_OPCODE_VALUE_RESET_COMP;
		else if (opcode == L4_KWQE_OPCODE_VALUE_CLOSE)
			kcqe_op = L4_KCQE_OPCODE_VALUE_CLOSE_COMP;
		else
			return;

		kcqe.kcqe_op_flag = (kcqe_op << KCQE_FLAGS_OPCODE_SHIFT) |
				    KCQE_FLAGS_LAYER_MASK_L4;
		l4kcqe->status = L4_KCQE_COMPLETION_STATUS_PARITY_ERROR;
		l4kcqe->cid = cid;
		cnic_get_l5_cid(cp, QFLE3_SW_CID(cid), &l4kcqe->conn_id);
	} else {
		return;
	}

	cqes[0] = (struct kcqe *) &kcqe;
	cnic_reply_qfle3_kcqes(dev, ulp_type, cqes, 1);
}

static int cnic_submit_qfle3_iscsi_kwqes(struct cnic_dev *dev,
					 struct kwqe *wqes[], vmk_uint32 num_wqes)
{
	int i, work, ret;
	vmk_uint32 opcode;
	struct kwqe *kwqe;

	if (!vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP))
		return VMK_STATUS_PENDING;		/* bnx2 is down */

	for (i = 0; i < num_wqes; ) {
		kwqe = wqes[i];
		opcode = KWQE_OPCODE(kwqe->kwqe_op_flag);
		work = 1;

		CNIC_INFO(dev, CNIC_MSG_DRV, "opcode=0x%x", opcode);

		switch (opcode) {
			case ISCSI_KWQE_OPCODE_INIT1:
				ret = cnic_qfle3_iscsi_init1(dev, kwqe);
				break;
			case ISCSI_KWQE_OPCODE_INIT2:
				ret = cnic_qfle3_iscsi_init2(dev, kwqe);
				break;
			case ISCSI_KWQE_OPCODE_OFFLOAD_CONN1:
				ret = cnic_qfle3_iscsi_ofld1(dev, &wqes[i], num_wqes - i,
						&work);
				break;
			case ISCSI_KWQE_OPCODE_UPDATE_CONN:
				ret = cnic_qfle3_iscsi_update(dev, kwqe);
				break;
			case ISCSI_KWQE_OPCODE_DESTROY_CONN:
				ret = cnic_qfle3_iscsi_destroy(dev, kwqe);
				break;
			case L4_KWQE_OPCODE_VALUE_CONNECT1:
				ret = cnic_qfle3_connect(dev, &wqes[i], num_wqes - i, &work);
				break;
			case L4_KWQE_OPCODE_VALUE_CLOSE:
				ret = cnic_qfle3_close(dev, kwqe);
				break;
			case L4_KWQE_OPCODE_VALUE_RESET:
				ret = cnic_qfle3_reset(dev, kwqe);
				break;
			case L4_KWQE_OPCODE_VALUE_OFFLOAD_PG:
				ret = cnic_qfle3_offload_pg(dev, kwqe);
				break;
			case L4_KWQE_OPCODE_VALUE_UPDATE_PG:
				ret = cnic_qfle3_update_pg(dev, kwqe);
				break;
			case L4_KWQE_OPCODE_VALUE_UPLOAD_PG:
				ret = 0;
				break;
			default:
				ret = 0;
				CNIC_ERR(dev, "Unknown type of KWQE(0x%x)", opcode);
				break;
		}

		if (ret >= VMK_FAILURE) {
			CNIC_WARN(dev, "KWQE(0x%x) failed. ret=0x%x", opcode, ret);

			if (ret == VMK_IO_ERROR || ret == VMK_STATUS_PENDING)
				cnic_qfle3_kwqe_err(dev, kwqe);
		}

		i += work;
	}

	return VMK_OK;
}

static int cnic_submit_qfle3_fcoe_kwqes(struct cnic_dev *dev,
					struct kwqe *wqes[], vmk_uint32 num_wqes)
{
	struct qfle3_adapter *adapter = NULL;
	int i, work, ret = VMK_OK;
	vmk_uint32 opcode;
	struct kwqe *kwqe;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (!vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP))
		return VMK_STATUS_PENDING;		/* bnx2 is down */

	if (!QFLE3_CHIP_IS_E2_PLUS(adapter))
		return VMK_BAD_PARAM;

	for (i = 0; i < num_wqes; ) {
		kwqe = wqes[i];
		opcode = KWQE_OPCODE(kwqe->kwqe_op_flag);
		work = 1;

		switch (opcode) {
			case FCOE_KWQE_OPCODE_INIT1:
				ret = cnic_qfle3_fcoe_init1(dev, &wqes[i],
						num_wqes - i, &work);
				break;
			case FCOE_KWQE_OPCODE_OFFLOAD_CONN1:
				ret = cnic_qfle3_fcoe_ofld1(dev, &wqes[i],
						num_wqes - i, &work);
				break;
			case FCOE_KWQE_OPCODE_ENABLE_CONN:
				ret = cnic_qfle3_fcoe_enable(dev, kwqe);
				break;
			case FCOE_KWQE_OPCODE_DISABLE_CONN:
				ret = cnic_qfle3_fcoe_disable(dev, kwqe);
				break;
			case FCOE_KWQE_OPCODE_DESTROY_CONN:
				ret = cnic_qfle3_fcoe_destroy(dev, kwqe);
				break;
			case FCOE_KWQE_OPCODE_DESTROY:
				ret = cnic_qfle3_fcoe_fw_destroy(dev, kwqe);
				break;
			case FCOE_KWQE_OPCODE_STAT:
				ret = cnic_qfle3_fcoe_stat(dev, kwqe);
				break;
			default:
				ret = 0;
				CNIC_ERR(dev, "Unknown type of KWQE(0x%x)", opcode);
				break;
		}

		if (ret >= VMK_FAILURE)
			CNIC_ERR(dev, "KWQE(0x%x) failed: ret=0x%x", opcode, ret);
		i += work;
	}

	return 0;
}

static int cnic_submit_qfle3_kwqes(struct cnic_dev *dev, struct kwqe *wqes[],
				   vmk_uint32 num_wqes)
{
	int ret = VMK_BAD_PARAM;
	vmk_uint32 layer_code;

	if (!vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP))
		return VMK_STATUS_PENDING;		/* qfle3 is down */

	if (!num_wqes)
		return 0;

	layer_code = wqes[0]->kwqe_op_flag & KWQE_LAYER_MASK;
	CNIC_INFO(dev, CNIC_MSG_DRV, "layer-code=0x%x", layer_code);

	switch (layer_code) {
	case KWQE_FLAGS_LAYER_MASK_L5_ISCSI:
	case KWQE_FLAGS_LAYER_MASK_L4:
	case KWQE_FLAGS_LAYER_MASK_L2:
		ret = cnic_submit_qfle3_iscsi_kwqes(dev, wqes, num_wqes);
		break;

	case KWQE_FLAGS_LAYER_MASK_L5_FCOE:
		ret = cnic_submit_qfle3_fcoe_kwqes(dev, wqes, num_wqes);
		break;
	}

	return ret;
}

static inline vmk_uint32 cnic_get_kcqe_layer_mask(vmk_uint32 opflag)
{
	if (VMK_UNLIKELY(KCQE_OPCODE(opflag) == FCOE_RAMROD_CMD_ID_TERMINATE_CONN))
		return KCQE_FLAGS_LAYER_MASK_L4;

	return opflag & KCQE_FLAGS_LAYER_MASK;
}

static void service_kcqes(struct cnic_dev *dev, int num_cqes)
{
	struct cnic_local *cp = dev->cnic_priv;
	int i, j, comp = 0;

	i = 0;
	j = 1;
	while (num_cqes) {
		struct cnic_ulp_ops *ulp_ops;
		int ulp_type;
		vmk_uint32 kcqe_op_flag = cp->completed_kcq[i]->kcqe_op_flag;
		vmk_uint32 kcqe_layer = cnic_get_kcqe_layer_mask(kcqe_op_flag);

		if (VMK_UNLIKELY(kcqe_op_flag & KCQE_RAMROD_COMPLETION))
			comp++;

		while (j < num_cqes) {
			vmk_uint32 next_op = cp->completed_kcq[i + j]->kcqe_op_flag;

			if (cnic_get_kcqe_layer_mask(next_op) != kcqe_layer)
				break;

			if (VMK_UNLIKELY(next_op & KCQE_RAMROD_COMPLETION))
				comp++;
			j++;
		}

		if (kcqe_layer == KCQE_FLAGS_LAYER_MASK_L5_RDMA) {
			ulp_type = CNIC_ULP_RDMA;
			dev->isr_rdma_count++;
		} else if (kcqe_layer == KCQE_FLAGS_LAYER_MASK_L5_ISCSI) {
			ulp_type = CNIC_ULP_ISCSI;
			dev->isr_iscsi_count++;
		} else if (kcqe_layer == KCQE_FLAGS_LAYER_MASK_L5_FCOE) {
			ulp_type = CNIC_ULP_FCOE;
			dev->isr_fcoe_count++;
		} else if (kcqe_layer == KCQE_FLAGS_LAYER_MASK_L4) {
			ulp_type = CNIC_ULP_L4;
			dev->isr_l4_count++;
		} else if (kcqe_layer == KCQE_FLAGS_LAYER_MASK_L2) {
			dev->isr_l2_count++;
			goto end;
		} else {
			CNIC_ERR(dev, "Unknown type of KCQE(0x%x)", kcqe_op_flag);
			dev->isr_unknown_count++;
			goto end;
		}

		CNIC_INFO(dev, CNIC_MSG_EVENT, "indicate-kcqe: ulp_type=0x%x \n"
			"kcqe_layer=0x%x num_cqes=0x%x j=0x%x kcqe_op_flag=0x%x i=0x%x",
			ulp_type, kcqe_layer, num_cqes, j, kcqe_op_flag, i);

		//SV: XXX: Use rcu_read_lock?
		//vmk_SemaLock(&cnic_driver_info.cnic_lock);
		ulp_ops = cp->ulp_ops[ulp_type];
		if (VMK_LIKELY(ulp_ops)) {
			ulp_ops->indicate_kcqes(cp->ulp_handle[ulp_type],
					cp->completed_kcq + i, j);
		} else {
			CNIC_ERR(dev, "ulp_ops not provided!! num_cqes=0x%x skipped.",
				num_cqes);
		}
		//vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
end:
		num_cqes -= j;
		i += j;
		j = 1;
	}

	if (VMK_UNLIKELY(comp))
		cnic_spq_completion(dev, DRV_CTL_RET_L5_SPQ_CREDIT_CMD, comp);

	CNIC_INFO(dev, CNIC_MSG_DEBUGFS, "Exit");
}



static int cnic_get_kcqes(struct cnic_dev *dev, struct kcq_info *info)
{
	struct cnic_local *cp = dev->cnic_priv;
	vmk_uint16 i, ri, hw_prod, last;
	struct kcqe *kcqe;
	int kcqe_cnt = 0, last_cnt = 0;
	vmk_uint32 kcq_diff;

	i = ri = last = info->sw_prod_idx;
	ri &= MAX_KCQ_IDX;
	hw_prod = *info->hw_prod_idx_ptr;
	hw_prod = info->hw_idx(hw_prod);

	if (VMK_UNLIKELY(hw_prod < last))
		kcq_diff = (65536 + hw_prod) - last;
	else
		kcq_diff = hw_prod - last;

	if (VMK_UNLIKELY(kcq_diff > MAX_KCQ_IDX))
		CNIC_ERR(dev, "kcq abs(hw_prod(%d) - sw_prod(%d)) > MAX_KCQ_IDX(%lu)",
			hw_prod, last, (unsigned long) MAX_KCQ_IDX);

	while ((i != hw_prod) && (kcqe_cnt < MAX_COMPLETED_KCQE)) {
		kcqe = &info->kcq[KCQ_PG(ri)][KCQ_IDX(ri)];
		cp->completed_kcq[kcqe_cnt++] = kcqe;
		i = info->next_idx(i);
		ri = i & MAX_KCQ_IDX;
		if (VMK_LIKELY(!(kcqe->kcqe_op_flag & KCQE_FLAGS_NEXT))) {
			last_cnt = kcqe_cnt;
			last = i;
		}
	}

	info->sw_prod_idx = last;
	return last_cnt;
}

static vmk_uint32 cnic_service_bnx2_queues(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	vmk_uint32 status_idx = (vmk_uint16) *cp->kcq1.status_idx_ptr;
	int kcqe_cnt;

	/* status block index must be read before reading other fields */
	vmk_CPUMemFenceRead();
	cp->kwq_con_idx = *cp->kwq_con_idx_ptr;

	while ((kcqe_cnt = cnic_get_kcqes(dev, &cp->kcq1))) {

		service_kcqes(dev, kcqe_cnt);

		/* Tell compiler that status_blk fields can change. */
		vmk_CPUMemFenceReadWrite();
		status_idx = (vmk_uint16) *cp->kcq1.status_idx_ptr;
		/* status block index must be read first */
		vmk_CPUMemFenceRead();
		cp->kwq_con_idx = *cp->kwq_con_idx_ptr;
	}

	CNIC_WR16(dev, cp->kcq1.io_addr, cp->kcq1.sw_prod_idx);

#if (CNIC_ISCSI_OOO_SUPPORT)
	cnic_handle_bnx2_ooo_rx_event(dev);
	cnic_handle_bnx2_ooo_tx_event(dev);
#endif
	return status_idx;
}

static int cnic_service_bnx2(void *data, void *status_blk)
{
	struct cnic_dev *dev = data;

	if (VMK_UNLIKELY(!vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP))) {
		struct status_block *sblk = status_blk;

		return sblk->status_idx;
	}

	return cnic_service_bnx2_queues(dev);
}

static void cnic_service_bnx2_msix(unsigned long data)
{
	struct cnic_dev *dev = (struct cnic_dev *) data;
	struct cnic_local *cp = dev->cnic_priv;

	cp->last_status_idx = cnic_service_bnx2_queues(dev);

	CNIC_WR(dev, BNX2_PCICFG_INT_ACK_CMD, cp->int_num |
		BNX2_PCICFG_INT_ACK_CMD_INDEX_VALID | cp->last_status_idx);
}

static void cnic_doirq(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;

	dev->isr_count++;
	if (VMK_LIKELY(vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP))) {
		vmk_uint16 prod = cp->kcq1.sw_prod_idx & MAX_KCQ_IDX;

		//SV: XXX:
		//prefetch(cp->status_blk.gen);
		//prefetch(&cp->kcq1.kcq[KCQ_PG(prod)][KCQ_IDX(prod)]);

		//CNIC_INFO(dev, CNIC_MSG_DRV, "sched irq-task.");
		//ql_vmk_tasklet_schedule(&cp->cnic_irq_task);
		//CNIC_INFO(dev, CNIC_MSG_DRV, "sched irq-task done.");

		/* XXX: Process completion directly in interrupt context. */
		cp->cnic_service_bh((unsigned long)dev);
	}
}

VMK_ReturnStatus cnic_ack_msix_common(void *handlerData,
	vmk_IntrCookie cookie)
{
	return VMK_OK;
}

VMK_ReturnStatus cnic_ack_msix(void *handlerData,
	vmk_IntrCookie cookie)
{
	struct cnic_dev *dev = (struct cnic_dev *)handlerData;
	struct cnic_local *cp = NULL;

	if (!dev) {
		DRV_ERR("Error: Not this device. dev=NULL");
		return VMK_NOT_THIS_DEVICE;
	}

	cp = dev->cnic_priv;

	CNIC_INFO(dev, CNIC_MSG_INTR, "Enter. dev=%p cp=%p", dev, cp);
	if (cp->ack_int)
		cp->ack_int(dev);
	CNIC_INFO(dev, CNIC_MSG_INTR, "Exit.");

	return VMK_OK;
}

void cnic_ack_sb(struct cnic_dev *dev, vmk_uint8 igu_sb_id, vmk_uint8 storm,
	vmk_uint16 index, vmk_uint8 op, vmk_uint8 update)
{
	struct qfle3_adapter *adapter = NULL;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (adapter->hw_info.int_block == INT_BLOCK_HC)
		cnic_ack_qfle3_int(dev, igu_sb_id, storm, index, op, update);
	else {
		vmk_uint8 segment;

		if (CHIP_INT_MODE_IS_BC(adapter))
			segment = storm;
		else if (igu_sb_id != adapter->igu_dsb_id)
			segment = IGU_SEG_ACCESS_DEF;
		else if (storm == ATTENTION_ID)
			segment = IGU_SEG_ACCESS_ATTN;
		else
			segment = IGU_SEG_ACCESS_DEF;
		cnic_ack_igu_sb(dev, igu_sb_id, segment, index, op, update);
	}
}


static void cnic_msix_rx(void *dev_instance, vmk_IntrCookie cookie)
{
	struct cnic_dev *dev = (struct cnic_dev *)dev_instance;
	struct cnic_local *cp = dev->cnic_priv;

	CNIC_INFO(dev, CNIC_MSG_INTR, "Enter.");
	cnic_doirq(dev);
	CNIC_INFO(dev, CNIC_MSG_INTR, "Exit.");
}

static inline void cnic_ack_qfle3_int(struct cnic_dev *dev, vmk_uint8 id,
	vmk_uint8 storm, vmk_uint16 index, vmk_uint8 op, vmk_uint8 update)
{
	struct qfle3_adapter *adapter = NULL;
	vmk_uint32 hc_addr;
	struct igu_ack_register igu_ack;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);
	hc_addr = (HC_REG_COMMAND_REG + QFLE3_PORT(adapter) * 32 +
		       COMMAND_REG_INT_ACK);

	igu_ack.status_block_index = index;
	igu_ack.sb_id_and_flags =
			((id << IGU_ACK_REGISTER_STATUS_BLOCK_ID_SHIFT) |
			 (storm << IGU_ACK_REGISTER_STORM_ID_SHIFT) |
			 (update << IGU_ACK_REGISTER_UPDATE_INDEX_SHIFT) |
			 (op << IGU_ACK_REGISTER_INTERRUPT_MODE_SHIFT));

	CNIC_WR(dev, hc_addr, (*(vmk_uint32 *)&igu_ack));

	/* Make sure that ACK is written. */
	vmk_CPUMemFenceReadWrite();
}

static void cnic_ack_igu_sb(struct cnic_dev *dev, vmk_uint8 igu_sb_id,
	vmk_uint8 segment, vmk_uint16 index, vmk_uint8 op, vmk_uint8 update)
{
	struct igu_regular cmd_data;
	vmk_uint32 igu_addr = BAR_IGU_INTMEM + (IGU_CMD_INT_ACK_BASE + igu_sb_id) * 8;

	cmd_data.sb_id_and_flags =
		(index 		<< IGU_REGULAR_SB_INDEX_SHIFT		)|
		(segment	<< IGU_REGULAR_SEGMENT_ACCESS_SHIFT	)|
		(update		<< IGU_REGULAR_BUPDATE_SHIFT		)|
		(op		<< IGU_REGULAR_ENABLE_INT_SHIFT		);


	CNIC_WR(dev, igu_addr, cmd_data.sb_id_and_flags);

	/* Make sure that ACK is written. */
	vmk_CPUMemFenceReadWrite();
}

static void cnic_ack_qfle3_msix(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;

	cnic_ack_qfle3_int(dev, cp->cfp[CNIC_RX_IDX].qfle3_igu_sb_id,
		CSTORM_ID, 0, IGU_INT_DISABLE, 0);
}

static void cnic_ack_qfle3_e2_msix(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;

	cnic_ack_igu_sb(dev, cp->cfp[CNIC_RX_IDX].qfle3_igu_sb_id,
		IGU_SEG_ACCESS_DEF, 0, IGU_INT_DISABLE, 0);
}

static vmk_uint32 cnic_service_qfle3_kcq(struct cnic_dev *dev, struct kcq_info *info)
{
	vmk_uint32 last_status = *info->status_idx_ptr;
	int kcqe_cnt;

	/* status block index must be read before reading the KCQ */
	vmk_CPUMemFenceRead();
	while ((kcqe_cnt = cnic_get_kcqes(dev, info))) {

		service_kcqes(dev, kcqe_cnt);

		/* Tell compiler that sblk fields can change. */
		vmk_CPUMemFenceReadWrite();

		last_status = *info->status_idx_ptr;
		/* status block index must be read before reading the KCQ */
		vmk_CPUMemFenceRead();
	}
	return last_status;
}


static void cnic_service_qfle3_bh(unsigned long data)
{
	struct cnic_dev *dev = (struct cnic_dev *) data;
	struct qfle3_adapter *adapter = NULL;
	struct cnic_local *cp = dev->cnic_priv;
	vmk_uint32 status_idx, new_status_idx;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (VMK_UNLIKELY(!vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP)))
		return;

	while (1) {
		status_idx = cnic_service_qfle3_kcq(dev, &cp->kcq1);

		CNIC_WR16(dev, cp->kcq1.io_addr,
			  cp->kcq1.sw_prod_idx + MAX_KCQ_IDX);

#if (CNIC_ISCSI_OOO_SUPPORT)
		cnic_handle_qfle3_ooo_tx_event(dev);
		cnic_handle_qfle3_ooo_rx_event(dev);
#endif

		if (!QFLE3_CHIP_IS_E2_PLUS(adapter)) {
			cnic_ack_qfle3_int(dev, cp->cfp[CNIC_RX_IDX].qfle3_igu_sb_id,
				USTORM_ID, status_idx, IGU_INT_ENABLE, 1);
			break;
		}
		new_status_idx = cnic_service_qfle3_kcq(dev, &cp->kcq2);

		if (new_status_idx != status_idx)
			continue;

		CNIC_WR16(dev, cp->kcq2.io_addr, cp->kcq2.sw_prod_idx +
			  MAX_KCQ_IDX);

		cnic_ack_igu_sb(dev, cp->cfp[CNIC_RX_IDX].qfle3_igu_sb_id,
			IGU_SEG_ACCESS_DEF, status_idx, IGU_INT_ENABLE, 1);
		break;
	}
}

/* SP cmd responses on default-sb to be routed by nic here. */
static int cnic_service_qfle3(void *data, void *status_blk)
{
	struct cnic_dev *dev = data;
	struct cnic_local *cp = dev->cnic_priv;

	if (!(cp->ethdev->driverState & CNIC_DRV_STATE_USING_MSIX)) {
		CNIC_INFO(dev, CNIC_MSG_INTR, "Enter cnic_doirq.");
		cnic_doirq(dev);
		CNIC_INFO(dev, CNIC_MSG_INTR, "Exit cnic_doirq.");
	}

	return 0;
}

static void cnic_ulp_stop(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	int if_type;

	for (if_type = 0; if_type < MAX_CNIC_ULP_TYPE; if_type++) {
		struct cnic_ulp_ops *ulp_ops;

		vmk_SemaLock(&cnic_driver_info.cnic_lock);
		ulp_ops = cp->ulp_ops[if_type];
		if (!ulp_ops) {
			vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
			continue;
		}
		vmk_BitVectorSet(cp->ulp_flags[if_type], ULP_F_CALL_PENDING);

		//SV: XXX: Check if lock needed after invoking cnic_stop?
		vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

		if (vmk_BitVectorAtomicTestAndClear(cp->ulp_flags[if_type], ULP_F_START))
			ulp_ops->cnic_stop(cp->ulp_handle[if_type]);

		vmk_BitVectorClear(cp->ulp_flags[if_type], ULP_F_CALL_PENDING);
	}
}

static void cnic_ulp_start(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	int if_type;

	for (if_type = 0; if_type < MAX_CNIC_ULP_TYPE; if_type++) {
		struct cnic_ulp_ops *ulp_ops;

		if (if_type == CNIC_ULP_ISCSI && !dev->max_iscsi_conn)
			continue;

		vmk_SemaLock(&cnic_driver_info.cnic_lock);
		ulp_ops = cp->ulp_ops[if_type];
		if (!ulp_ops || !ulp_ops->cnic_start) {
			vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
			continue;
		}
		vmk_BitVectorSet(cp->ulp_flags[if_type], ULP_F_CALL_PENDING);

		//SV: XXX: Check if lock needed after invoking cnic_start?
		vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

		if (!vmk_BitVectorAtomicTestAndSet(cp->ulp_flags[if_type], ULP_F_START))
			ulp_ops->cnic_start(cp->ulp_handle[if_type]);

		vmk_BitVectorClear(cp->ulp_flags[if_type], ULP_F_CALL_PENDING);
	}
}
static void cnic_ulp_recovery_notify(struct cnic_dev *dev, int event)
{
	struct cnic_local *cp = dev->cnic_priv;
	int if_type;

	for (if_type = 0; if_type < MAX_CNIC_ULP_TYPE; if_type++) {
		struct cnic_ulp_ops *ulp_ops;

		vmk_SemaLock(&cnic_driver_info.cnic_lock);
		ulp_ops = cp->ulp_ops[if_type];
		if (!ulp_ops) {
			vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
			continue;
		}
		vmk_BitVectorSet(cp->ulp_flags[if_type], ULP_F_CALL_PENDING);
		vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

		CNIC_ERR(dev, "Calling fw_recovery_notify for if_type:%d\n", if_type);
		if (ulp_ops->cnic_fw_recovery_notify)
			ulp_ops->cnic_fw_recovery_notify(cp->ulp_handle[if_type], event);

		vmk_BitVectorClear(cp->ulp_flags[if_type], ULP_F_CALL_PENDING);
	}
}

static int cnic_copy_ulp_stats(struct cnic_dev *dev, int ulp_type)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_ulp_ops *ulp_ops;
	int rc;

	vmk_SemaLock(&cnic_driver_info.cnic_lock);
	ulp_ops = cp->ulp_ops[ulp_type];
	if (ulp_ops && ulp_ops->cnic_get_stats)
		rc = ulp_ops->cnic_get_stats(cp->ulp_handle[ulp_type]);
	else
		rc = VMK_INVALID_ADAPTER;
	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
	return rc;
}

static int cnic_ctl(void *data, struct cnic_ctl_info *info)
{
	struct cnic_dev *dev = data;
	int ulp_type = CNIC_ULP_ISCSI;

	CNIC_INFO(dev, CNIC_MSG_CTL, "ctl cmd=0x%x", info->cmd);

	switch (info->cmd) {
		case CNIC_CTL_STOP_CMD:
			CNIC_ERR(dev, "Enter ctl cmd=0x%x", info->cmd);
			cnic_hold(dev);

			if (vmk_BitVectorTest(dev->flags, CNIC_F_IF_UP)) {
				vmk_BitVectorClear(dev->flags, CNIC_F_IF_UP);
				cnic_ulp_stop(dev);
				cnic_stop_hw(dev);
				//if (dev->fw_recov_in_progress)
				//	cnic_unregister_netdev(dev);
			}

			cnic_put(dev);
			CNIC_ERR(dev, "Exit ctl cmd=0x%x", info->cmd);
			break;
		case CNIC_CTL_START_CMD:
			CNIC_ERR(dev, "Enter ctl cmd=0x%x", info->cmd);
			cnic_hold(dev);

			vmk_BitVectorSet(dev->flags, CNIC_F_IF_UP);
			//if (dev->fw_recov_in_progress)
			//		cnic_register_netdev(dev);
			if (!cnic_start_hw(dev))
				cnic_ulp_start(dev);

			cnic_put(dev);
			CNIC_ERR(dev, "Exit ctl cmd=0x%x", info->cmd);
			break;
		case CNIC_CTL_NIC_RECOVERY_CMD:
			CNIC_ERR(dev, "Enter ctl cmd=0x%x", info->cmd);
			cnic_hold(dev);

			dev->fw_recov_in_progress = 1;
			if (vmk_BitVectorTest(dev->flags, CNIC_F_IF_UP)) {
				cnic_ulp_recovery_notify(dev, info->cmd);
			}

			cnic_put(dev);
			CNIC_ERR(dev, "Exit for ctl cmd:0x%x\n", info->cmd);
			break;
		case CNIC_CTL_NIC_RECOVERY_DONE_CMD:
			CNIC_ERR(dev, "Enter ctl cmd=0x%x", info->cmd);
			cnic_hold(dev);

			dev->fw_recov_in_progress = 0;
			if (vmk_BitVectorTest(dev->flags, CNIC_F_IF_UP)) {
				cnic_ulp_recovery_notify(dev, info->cmd);
			}

			cnic_put(dev);
			CNIC_ERR(dev, "Exit for ctl cmd:0x%x\n", info->cmd);
			break;
		case CNIC_CTL_COMPLETION_CMD:
			{
				struct cnic_ctl_completion *comp = &info->data.comp;
				vmk_uint32 cid = QFLE3_SW_CID(comp->cid);
				vmk_uint32 l5_cid;
				struct cnic_local *cp = dev->cnic_priv;

				if (!vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP))
					break;

				if (cnic_get_l5_cid(cp, cid, &l5_cid) == 0) {
					struct cnic_context *ctx = &cp->ctx_tbl[l5_cid];

					CNIC_INFO(dev, CNIC_MSG_CTL,
						"ctl: cmd-compl: cid=0x%x l5_cid=0x%x",
						cid, l5_cid);

					if (VMK_UNLIKELY(comp->error)) {
						vmk_BitVectorSet(ctx->ctx_flags, CTX_FL_CID_ERROR);
						CNIC_ERR(dev, "CID=0x%x CFC delete comp error 0x%x",
							cid, comp->error);
					}

					ctx->wait_cond = 1;
					CNIC_INFO(dev, CNIC_MSG_CTL,
						"Trying to wake up completion event=%p",
						&ctx->waitq.cmpl_event);
					ql_vmk_world_wakeup(&ctx->waitq);
				}
				break;
			}
		case CNIC_CTL_FCOE_STATS_GET_CMD:
			ulp_type = CNIC_ULP_FCOE;
			/* fall through */
		case CNIC_CTL_ISCSI_STATS_GET_CMD:
			cnic_hold(dev);
			cnic_copy_ulp_stats(dev, ulp_type);
			cnic_put(dev);
			break;

		default:
			CNIC_INFO(dev, CNIC_MSG_CTL, "Unknown ctl cmd=0x%x", info->cmd);
			return VMK_BAD_PARAM;
	}

	return 0;
}

/* SV: XXX: Borrowed from non-native. To be removed later. */
#define NETDEV_UP			0x0001
#define NETDEV_DOWN			0x0002
#define NETDEV_GOING_DOWN	0x0009

int cnic_link_update(void *data, vmk_uint8 link_state, vmk_uint16 speed)
{
	struct cnic_dev *dev = data;
	struct cnic_local *cp = dev->cnic_priv;
	/* SV: XXX: Since storage drivers use old netdev event mapping. */
	unsigned long event = ((link_state == 1) ? NETDEV_UP : NETDEV_GOING_DOWN);
	int if_type = 0;

	CNIC_INFO(dev, CNIC_MSG_DRV, "Link state=0x%x speed=0x%x",
		link_state, speed);

	/* Update shadow linkState. */
	dev->uplinkLinkState.speed = speed;
	dev->uplinkLinkState.state = ((link_state == 1) ?
			VMK_LINK_STATE_UP : VMK_LINK_STATE_DOWN);

	for (if_type = 0; if_type < MAX_CNIC_ULP_TYPE; if_type++) {
		struct cnic_ulp_ops *ulp_ops = NULL;
		void *ctx = NULL;

		vmk_SemaLock(&cnic_driver_info.cnic_lock);
		ulp_ops = cp->ulp_ops[if_type];
		if (!ulp_ops || !ulp_ops->indicate_netevent) {
			vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
			continue;
		}

		CNIC_INFO(dev, CNIC_MSG_DRV, "Sending netevent: if-type=0x%x", if_type);
		ctx = cp->ulp_handle[if_type];
		ulp_ops->indicate_netevent(ctx, event);
		vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
	}

	return VMK_OK;
}


int cnic_ll2_recv(void *data, vmk_PktHandle *pkt)
{
	struct cnic_dev *dev = data;

	CNIC_INFO(dev, CNIC_MSG_NW, "before rx-pkt: %p", pkt);

	if (!vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP))
		return VMK_FAILURE;

	if(vmk_SpinlockLock(dev->fcoe_rx_list_lck) == VMK_DEATH_PENDING)
		return VMK_FAILURE;

	/* Queue pkt to fcoe-rx-list. */
	vmk_PktListAppendPkt(dev->fcoe_rx_list, pkt);

	vmk_SpinlockUnlock(dev->fcoe_rx_list_lck);

	/* SV: XXX: Ideally we should invoke fcoe-rx-dpc for "budget" nof pkts. */
	ql_vmk_tasklet_schedule(&dev->fcoe_rx_dpc);

	CNIC_INFO(dev, CNIC_MSG_NW, "after rx-pkt sched");

	return VMK_OK;
}

static int cnic_queue_work(struct cnic_local *cp, vmk_uint32 work_type, void *data)
{
	struct cnic_work_node *node;
	int bytes = sizeof(vmk_uint32 *);

	vmk_SpinlockLock(cp->wr_lock);

	node = &cp->cnic_work_ring[cp->cnic_wr_prod];
	node->work_type = work_type;
	if (work_type == WORK_TYPE_KCQE)
		bytes = sizeof(struct kcqe);
	if (work_type == WORK_TYPE_REDIRECT)
		bytes = sizeof(struct cnic_redirect_entry);
	vmk_Memcpy(&node->work_data, data, bytes);
	cp->cnic_wr_prod++;
	cp->cnic_wr_prod &= WORK_RING_SIZE_MASK;

	vmk_SpinlockUnlock(cp->wr_lock);
	return 0;
}

static int cnic_cm_offload_pg(struct cnic_sock *csk)
{
	struct cnic_dev *dev = csk->dev;
	struct l4_kwq_offload_pg *l4kwqe;
	struct kwqe *wqes[1];
	struct cnic_local *cp = dev->cnic_priv;

	l4kwqe = (struct l4_kwq_offload_pg *) &csk->kwqe1;
	vmk_Memset(l4kwqe, 0, sizeof(*l4kwqe));
	wqes[0] = (struct kwqe *) l4kwqe;

	l4kwqe->op_code = L4_KWQE_OPCODE_VALUE_OFFLOAD_PG;
	l4kwqe->flags =
		L4_LAYER_CODE << L4_KWQ_OFFLOAD_PG_LAYER_CODE_SHIFT;
	l4kwqe->l2hdr_nbytes = ETH_HLEN;
	l4kwqe->da0 = csk->ha[0];
	l4kwqe->da1 = csk->ha[1];
	l4kwqe->da2 = csk->ha[2];
	l4kwqe->da3 = csk->ha[3];
	l4kwqe->da4 = csk->ha[4];
	l4kwqe->da5 = csk->ha[5];

	l4kwqe->sa0 = cp->srcMACAddr[0];
	l4kwqe->sa1 = cp->srcMACAddr[1];
	l4kwqe->sa2 = cp->srcMACAddr[2];
	l4kwqe->sa3 = cp->srcMACAddr[3];
	l4kwqe->sa4 = cp->srcMACAddr[4];
	l4kwqe->sa5 = cp->srcMACAddr[5];

	l4kwqe->etype = VMK_ETH_TYPE_IPV4;

	l4kwqe->ipid_start = DEF_IPID_START;
	l4kwqe->host_opaque = csk->l5_cid;

	if (csk->vlan_id) {
		l4kwqe->pg_flags |= L4_KWQ_OFFLOAD_PG_VLAN_TAGGING;
		l4kwqe->vlan_tag = csk->vlan_id;
		l4kwqe->l2hdr_nbytes += 4;
	}

	CNIC_INFO(dev, CNIC_MSG_DRV, "opcode=0x%x flags=0x%x \nl2hdr_nbytes=0x%x "
		"da=0x%x %x %x %x %x %x sa=0x%x %x %x %x %x %x etype=0x%x \n"
		"ipid_start=0x%x host_opaque=0x%x vlan_id=0x%x pg_flags=0x%x "
		"vlan_tag=0x%x", l4kwqe->op_code, l4kwqe->flags, l4kwqe->l2hdr_nbytes,
		l4kwqe->da0, l4kwqe->da1, l4kwqe->da2, l4kwqe->da3, l4kwqe->da4,
		l4kwqe->da5, l4kwqe->sa0, l4kwqe->sa1, l4kwqe->sa2, l4kwqe->sa3,
		l4kwqe->sa4, l4kwqe->sa5, l4kwqe->etype, l4kwqe->ipid_start,
		l4kwqe->host_opaque, csk->vlan_id, l4kwqe->pg_flags, l4kwqe->vlan_tag);

	return dev->submit_kwqes(dev, wqes, 1);
}

static int cnic_cm_upload_pg(struct cnic_sock *csk)
{
	struct cnic_dev *dev = csk->dev;
	struct l4_kwq_upload *l4kwqe;
	struct kwqe *wqes[1];

	l4kwqe = (struct l4_kwq_upload *) &csk->kwqe1;
	vmk_Memset(l4kwqe, 0, sizeof(*l4kwqe));
	wqes[0] = (struct kwqe *) l4kwqe;

	l4kwqe->opcode = L4_KWQE_OPCODE_VALUE_UPLOAD_PG;
	l4kwqe->flags =
		L4_LAYER_CODE << L4_KWQ_UPLOAD_LAYER_CODE_SHIFT;
	l4kwqe->cid = csk->pg_cid;

	return dev->submit_kwqes(dev, wqes, 1);
}

static int cnic_cm_conn_req(struct cnic_sock *csk)
{
	struct cnic_dev *dev = csk->dev;
	struct l4_kwq_connect_req1 *l4kwqe1;
	struct l4_kwq_connect_req2 *l4kwqe2;
	struct l4_kwq_connect_req3 *l4kwqe3;
	struct kwqe *wqes[3];
	vmk_uint8 tcp_flags = 0;
	int num_wqes = 2;

	l4kwqe1 = (struct l4_kwq_connect_req1 *) &csk->kwqe1;
	l4kwqe2 = (struct l4_kwq_connect_req2 *) &csk->kwqe2;
	l4kwqe3 = (struct l4_kwq_connect_req3 *) &csk->kwqe3;
	vmk_Memset(l4kwqe1, 0, sizeof(*l4kwqe1));
	vmk_Memset(l4kwqe2, 0, sizeof(*l4kwqe2));
	vmk_Memset(l4kwqe3, 0, sizeof(*l4kwqe3));

	l4kwqe3->op_code = L4_KWQE_OPCODE_VALUE_CONNECT3;
	l4kwqe3->flags =
		L4_LAYER_CODE << L4_KWQ_CONNECT_REQ3_LAYER_CODE_SHIFT;
	l4kwqe3->ka_timeout = csk->ka_timeout;
	l4kwqe3->ka_interval = csk->ka_interval;
	l4kwqe3->ka_max_probe_count = csk->ka_max_probe_count;
	l4kwqe3->tos = csk->tos;
	l4kwqe3->ttl = csk->ttl;
	l4kwqe3->snd_seq_scale = csk->snd_seq_scale;
	l4kwqe3->pmtu = csk->pmtu;
	l4kwqe3->rcv_buf = csk->rcv_buf;
	l4kwqe3->snd_buf = csk->snd_buf;
	l4kwqe3->seed = csk->seed;

	CNIC_INFO(dev, CNIC_MSG_DRV,
		"l4kwqe3: op_code=0x%x flags=0x%x ka_timeout=0x%x \n"
		"ka_interval=0x%x ka_max_probe_count=0x%x tos=0x%x ttl=0x%x\n"
		"snd_seq_scale=0x%x pmtu=0x%x rcv_buf=0x%x snd_buf=0x%x seed=0x%x",
		l4kwqe3->op_code, l4kwqe3->flags, l4kwqe3->ka_timeout,
		l4kwqe3->ka_interval, l4kwqe3->ka_max_probe_count, l4kwqe3->tos,
		l4kwqe3->ttl, l4kwqe3->snd_seq_scale, l4kwqe3->pmtu, l4kwqe3->rcv_buf,
		l4kwqe3->snd_buf, l4kwqe3->seed);

	wqes[0] = (struct kwqe *) l4kwqe1;
	if (vmk_BitVectorTest(csk->flags, SK_F_IPV6)) {
		wqes[1] = (struct kwqe *) l4kwqe2;
		wqes[2] = (struct kwqe *) l4kwqe3;
		num_wqes = 3;

		l4kwqe1->conn_flags = L4_KWQ_CONNECT_REQ1_IP_V6;
		l4kwqe2->op_code = L4_KWQE_OPCODE_VALUE_CONNECT2;
		l4kwqe2->flags =
			L4_KWQ_CONNECT_REQ2_LINKED_WITH_NEXT |
			L4_LAYER_CODE << L4_KWQ_CONNECT_REQ2_LAYER_CODE_SHIFT;
		l4kwqe2->src_ip_v6_2 = vmk_BE32ToCPU(csk->src_ip[1]);
		l4kwqe2->src_ip_v6_3 = vmk_BE32ToCPU(csk->src_ip[2]);
		l4kwqe2->src_ip_v6_4 = vmk_BE32ToCPU(csk->src_ip[3]);
		l4kwqe2->dst_ip_v6_2 = vmk_BE32ToCPU(csk->dst_ip[1]);
		l4kwqe2->dst_ip_v6_3 = vmk_BE32ToCPU(csk->dst_ip[2]);
		l4kwqe2->dst_ip_v6_4 = vmk_BE32ToCPU(csk->dst_ip[3]);
		l4kwqe3->mss = l4kwqe3->pmtu - sizeof(struct vmk_IPv6Hdr) -
			       sizeof(struct vmk_TCPHdr);
	} else {
		wqes[1] = (struct kwqe *) l4kwqe3;
		l4kwqe3->mss = l4kwqe3->pmtu - sizeof(struct vmk_IPv4Hdr) -
			       sizeof(struct vmk_TCPHdr);
	}

	l4kwqe1->op_code = L4_KWQE_OPCODE_VALUE_CONNECT1;
	l4kwqe1->flags =
		(L4_LAYER_CODE << L4_KWQ_CONNECT_REQ1_LAYER_CODE_SHIFT) |
		 L4_KWQ_CONNECT_REQ3_LINKED_WITH_NEXT;
	l4kwqe1->cid = csk->cid;
	l4kwqe1->pg_cid = csk->pg_cid;
	l4kwqe1->src_ip = vmk_BE32ToCPU(csk->src_ip[0]);
	l4kwqe1->dst_ip = vmk_BE32ToCPU(csk->dst_ip[0]);
	l4kwqe1->src_port = vmk_BE16ToCPU(csk->src_port);
	l4kwqe1->dst_port = vmk_BE16ToCPU(csk->dst_port);
	if (csk->tcp_flags & SK_TCP_NO_DELAY_ACK)
		tcp_flags |= L4_KWQ_CONNECT_REQ1_NO_DELAY_ACK;
	if (csk->tcp_flags & SK_TCP_KEEP_ALIVE)
		tcp_flags |= L4_KWQ_CONNECT_REQ1_KEEP_ALIVE;
	if (csk->tcp_flags & SK_TCP_NAGLE)
		tcp_flags |= L4_KWQ_CONNECT_REQ1_NAGLE_ENABLE;
	if (csk->tcp_flags & SK_TCP_TIMESTAMP)
		tcp_flags |= L4_KWQ_CONNECT_REQ1_TIME_STAMP;
	if (csk->tcp_flags & SK_TCP_SACK)
		tcp_flags |= L4_KWQ_CONNECT_REQ1_SACK;
	if (csk->tcp_flags & SK_TCP_SEG_SCALING)
		tcp_flags |= L4_KWQ_CONNECT_REQ1_SEG_SCALING;

	l4kwqe1->tcp_flags = tcp_flags;

	CNIC_INFO(dev, CNIC_MSG_DRV, "l4kwqe1: op_code=0x%x flags=0x%x\n"
		"cid=0x%x pg_cid=0x%x src_ip=0x%x dst_ip=0x%x src_port=0x%x\n"
		"dst_port=0x%x tcp_flags=0x%x", l4kwqe1->op_code, l4kwqe1->flags,
		l4kwqe1->cid, l4kwqe1->pg_cid, l4kwqe1->src_ip, l4kwqe1->dst_ip,
		l4kwqe1->src_port, l4kwqe1->dst_port, l4kwqe1->tcp_flags);
	return dev->submit_kwqes(dev, wqes, num_wqes);
}

static int cnic_cm_close_req(struct cnic_sock *csk)
{
	struct cnic_dev *dev = csk->dev;
	struct l4_kwq_close_req *l4kwqe;
	struct kwqe *wqes[1];

	l4kwqe = (struct l4_kwq_close_req *) &csk->kwqe2;
	vmk_Memset(l4kwqe, 0, sizeof(*l4kwqe));
	wqes[0] = (struct kwqe *) l4kwqe;

	l4kwqe->op_code = L4_KWQE_OPCODE_VALUE_CLOSE;
	l4kwqe->flags = L4_LAYER_CODE << L4_KWQ_CLOSE_REQ_LAYER_CODE_SHIFT;
	l4kwqe->cid = csk->cid;

	return dev->submit_kwqes(dev, wqes, 1);
}

static int cnic_cm_abort_req(struct cnic_sock *csk)
{
	struct cnic_dev *dev = csk->dev;
	struct l4_kwq_reset_req *l4kwqe;
	struct kwqe *wqes[1];

	l4kwqe = (struct l4_kwq_reset_req *) &csk->kwqe2;
	vmk_Memset(l4kwqe, 0, sizeof(*l4kwqe));
	wqes[0] = (struct kwqe *) l4kwqe;

	l4kwqe->op_code = L4_KWQE_OPCODE_VALUE_RESET;
	l4kwqe->flags = L4_LAYER_CODE << L4_KWQ_RESET_REQ_LAYER_CODE_SHIFT;
	l4kwqe->cid = csk->cid;

	return dev->submit_kwqes(dev, wqes, 1);
}

static int cnic_cm_get_uplinkName(struct cnic_dev *dev, vmk_Name *uplinkName)
{
	struct qfle3_adapter *adapter = NULL;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (adapter == NULL)
		return VMK_NOT_INITIALIZED;

	/*
	 * Save uplinkName for future reference.
	 * XXX: This assumes to be called after uplink device has been
	 * associated with adapter.
	 */
	vmk_NameCopy(&dev->uplinkName, &adapter->uplinkName);
	vmk_NameCopy(uplinkName, &adapter->uplinkName);

	return VMK_OK;
}

static int cnic_cm_acquire(struct cnic_dev *dev, vmk_uint8 ptype,
	vmk_uint32 *icid)
{
	int rc = VMK_OK;
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_context *ctx = NULL;
	vmk_uint32 ctx_tbl_idx = 0, l5_cid = 0, cid = 0;
	struct cnic_eth_dev *ethdev = cp->ethdev;

	CNIC_INFO(dev, CNIC_MSG_DRV, "Enter. ptype=0x%x", ptype);

	if (ptype == CNIC_ULP_FCOE) {
		cid = cnic_alloc_new_id(&cp->fcoe_cid_tbl);
		if (cid == -1) {
			CNIC_ERR(dev, "fcoe_cid_tbl: failed to allocate new id.");
			return VMK_NO_MEMORY;
		}
		l5_cid = (cid - cp->fcoe_start_cid);
		ctx_tbl_idx = l5_cid + QFLE3_FCOE_L5_CID_BASE;
	} else if (ptype == CNIC_ULP_ISCSI) {
		cid = cnic_alloc_new_id(&cp->cid_tbl);
		if (cid == -1) {
			CNIC_ERR(dev, "iscsi_cid_tbl: failed to allocate new id.");
			return VMK_NO_MEMORY;
		}
		l5_cid = ctx_tbl_idx = (cid - cp->iscsi_start_cid);
	}

	ctx = &cp->ctx_tbl[ctx_tbl_idx];
	ctx->cid = cid;
	if (vmk_BitVectorTest(ctx->ctx_flags, CTX_FL_OFFLD_START)) {
		rc = VMK_FAILURE;
		goto error;
	}

	if (ptype == CNIC_ULP_ISCSI) {
		rc = cnic_alloc_qfle3_conn_resc(dev, l5_cid);
		if (rc)
			goto error;
	}

	if (icid) {
		*icid = l5_cid;
		ethdev->nof_ofld_conns = cp->fcoe_cid_tbl.count + cp->cid_tbl.count;
	} else {
		CNIC_ERR(dev, "Invalid icid provided.");
		rc = VMK_FAILURE;
		goto error;
	}

	CNIC_INFO(dev, CNIC_MSG_DRV, "Exit. SUCCESS");

	return rc;

error:
	if (ptype == CNIC_ULP_FCOE)
		cnic_free_id(&cp->fcoe_cid_tbl, ctx->cid);
	else if (ptype == CNIC_ULP_ISCSI)
		cnic_free_id(&cp->cid_tbl, ctx->cid);
	CNIC_INFO(dev, CNIC_MSG_DRV, "Exit. rc=0x%x", rc);

	return rc;
}

static int cnic_cm_create(struct cnic_dev *dev, int ulp_type, vmk_uint32 cid,
	vmk_uint32 l5_cid, struct cnic_sock **csk, void *context)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_sock *csk1 = NULL;

	if (l5_cid >= MAX_CM_SK_TBL_SZ)
		return VMK_BAD_PARAM;

	if (cp->ctx_tbl) {
		struct cnic_context *ctx = &cp->ctx_tbl[l5_cid];

		if (vmk_BitVectorTest(ctx->ctx_flags, CTX_FL_OFFLD_START))
			return VMK_STATUS_PENDING;
	}

	csk1 = &cp->csk_tbl[l5_cid];
	if (vmk_AtomicRead64(&csk1->ref_count))
		return VMK_STATUS_PENDING;

	if (vmk_BitVectorAtomicTestAndSet(csk1->flags, SK_F_INUSE))
		return VMK_BUSY;

	csk1->dev = dev;
	csk1->cid = cid;
	csk1->l5_cid = l5_cid;
	csk1->ulp_type = ulp_type;
	csk1->context = context;

	csk1->ka_timeout = DEF_KA_TIMEOUT;
	csk1->ka_interval = DEF_KA_INTERVAL;
	csk1->ka_max_probe_count = DEF_KA_MAX_PROBE_COUNT;
	csk1->tos = DEF_TOS;
	csk1->ttl = DEF_TTL;
	csk1->snd_seq_scale = DEF_SND_SEQ_SCALE;
	csk1->rcv_buf = DEF_RCV_BUF;
	csk1->snd_buf = DEF_SND_BUF;
	csk1->seed = DEF_SEED;

#if (CNIC_ISCSI_OOO_SUPPORT)
	cnic_conn_ooo_init(cp, l5_cid);
#endif
	*csk = csk1;

	return 0;
}

static void cnic_cm_cleanup(struct cnic_sock *csk)
{
	if (csk->src_port) {
		struct cnic_dev *dev = csk->dev;
		struct cnic_local *cp = dev->cnic_priv;

		cnic_free_id(&cp->csk_port_tbl, vmk_BE16ToCPU(csk->src_port));
		csk->src_port = 0;
	}
}

static void cnic_close_conn(struct cnic_sock *csk)
{
	if (vmk_BitVectorTest(csk->flags, SK_F_PG_OFFLD_COMPLETE)) {
		cnic_cm_upload_pg(csk);
		vmk_BitVectorClear(csk->flags, SK_F_PG_OFFLD_COMPLETE);
	}
	cnic_cm_cleanup(csk);
}

static int cnic_cm_destroy(struct cnic_sock *csk)
{
	if (!cnic_in_use(csk))
		return VMK_BAD_PARAM;

#if (CNIC_ISCSI_OOO_SUPPORT)
	cnic_flush_ooo(csk->dev, csk->l5_cid);
#endif
	csk_hold(csk);
	vmk_BitVectorClear(csk->flags, SK_F_INUSE);
	vmk_CPUMemFenceReadWrite();
	while (vmk_AtomicRead64(&csk->ref_count) != 1)
		vmk_WorldSleep(1*VMK_USEC_PER_MSEC);
	cnic_cm_cleanup(csk);

	vmk_BitVectorZap(csk->flags);
	csk_put(csk);
	return 0;
}

//SV: XXX:
#if 0
static inline vmk_uint16 cnic_get_vlan(vmk_Device *dev,
	struct vmk_Device **vlan_dev)
{
	*vlan_dev = dev;
	return 0;
}
#endif

struct cnic_dev *cnic_cm_select_dev(vmk_IscsiNetHandle iscsiNetHandle,
	vmk_SocketIPAddress *dst_addr, int ulp_type)
{
	char devName[NET_IF_NAME_SZ];
	struct cnic_dev *dev;
	int found = 0;
	struct qfle3_adapter *adapter = NULL;
	VMK_ReturnStatus status;

	status = vmk_IscsiTransportGetUplink(iscsiNetHandle, devName);
	if (status != VMK_OK) {
		DRV_ERR("Unable to get iscsi-transport uplink. status=%s",
			vmk_StatusToString(status));
		return NULL;
	}

	vmk_SemaLock(&cnic_driver_info.cnic_lock);
	ql_vmk_list_each_entry(dev, &cnic_driver_info.cnic_dev_list,
		list, struct cnic_dev) {
		struct cnic_local *cp = dev->cnic_priv;

		vmk_DeviceGetAttachedDriverData(dev->netdev,
			(vmk_AddrCookie *) &adapter);

		CNIC_INFO(dev, CNIC_MSG_DRV, "adapter=%p devName=%s name=%s",
			adapter, devName, vmk_NameToString(&dev->uplinkName));

		if (adapter && vmk_BitVectorTest(dev->flags, CNIC_F_IF_UP)) {
			if (!vmk_Strcmp(vmk_NameToString(&dev->uplinkName), devName)) {
				found = 1;
				DRV_INFO(CNIC_MSG_DRV, "cp=%p dev=%p dst_addr=%p",
					cp, dev, dst_addr);

				/* Retrieve the source MAC */
				if ((status = vmk_IscsiTransportGetSrcMAC(
								iscsiNetHandle,
								cp->srcMACAddr)) != VMK_OK) {
					CNIC_ERR(dev, "Get SRC MAC failed %d", status);
					found = 0;
					break;
				} else
					vmk_Memcpy(dev->vmk_mac, cp->srcMACAddr,
						VMK_ETH_ADDR_LENGTH);

				/* Next hop (arp) resolve */
				if ((status = vmk_IscsiTransportGetNextHopMAC(
					iscsiNetHandle,
					cp->nextHopMACAddr)) != VMK_OK) {
					CNIC_ERR(dev, "Get next hop MAC failed %d", status);
					found = 0;
					break;
				}
				/* Retrieve the source IP Address */
				if ((status = vmk_IscsiTransportGetSrcIP(
					iscsiNetHandle,
					&cp->srcFamily,
					cp->srcIPAddr)) != VMK_OK) {
					CNIC_ERR(dev, "Get SRC IP failed %d", status);
					found = 0;
					break;
				} else {
					if (cp->srcFamily == VMK_SOCKET_AF_INET) {
						vmk_Memcpy(cp->src_IP4Addr, cp->srcIPAddr,
							IPv4_ADDRLEN);
						vmk_Memcpy(&dev->ipv4_addr, cp->srcIPAddr,
							IPv4_ADDRLEN);
					} else if (cp->srcFamily == AF_INET6) {
						vmk_Memcpy(&dev->ipv6_addr, cp->srcIPAddr,
							IPv6_ADDRLEN);
					}
					vmk_Memcpy(&dev->srcfamily, &cp->srcFamily,
						sizeof(cp->srcFamily));
				}

				/* Retrieve the source subnet id */
				if ((status = vmk_IscsiTransportGetSrcSubnet(
						iscsiNetHandle, &cp->srcFamily,
						cp->src_subnet_mask)) != VMK_OK) {
					CNIC_ERR(dev, "Get SRC subnet_mask failed %d", status);
					found = 0;
					break;
				} else {
					if (cp->srcFamily == VMK_SOCKET_AF_INET) {
						vmk_Memcpy(&dev->subnet_mask, cp->src_subnet_mask,
							IPv4_ADDRLEN);
					} else if (cp->srcFamily == AF_INET6) {
						vmk_Memcpy(&dev->subnet_mask_ipv6, cp->src_subnet_mask,
							IPv6_ADDRLEN);
					}
				}

				/* Path MTU */
				if ((status = vmk_IscsiTransportGetPmtu(
					iscsiNetHandle, &cp->pmtu)) != VMK_OK) {
					CNIC_ERR(dev, "Get PMTU failed %d", status);
					cp->pmtu = 1500;
				}
				if (cp->pmtu > adapter->mtu) {
					dev->pmtu_fails++;
					cp->pmtu = adapter->mtu;
				}
				/* VLAN Tag */
				if ((status = vmk_IscsiTransportGetVlan(
					iscsiNetHandle,
					&cp->vlan_id)) != VMK_OK) {
					CNIC_ERR(dev, "Get vlan ID failed %d", status);
					cp->vlan_id = 0;
				} else
					dev->vlan_id = cp->vlan_id;

				/* Allocate TCP ports, only once per device */
				if (cp->csk_port_tbl.table) {
					CNIC_INFO(dev, CNIC_MSG_DRV,
						"Port-table already allocated.");
					break;
				}

				cp->cnic_local_port_nr = MAX_CM_SK_TBL_SZ;
				cp->cnic_local_port_min = 1;
				status = vmk_IscsiTransportGetPortReservation(
						iscsiNetHandle,
						&cp->cnic_local_port_min,
						&cp->cnic_local_port_nr);
				if (status == VMK_OK) {
					vmk_uint32 port_id;

					CNIC_INFO(dev, CNIC_MSG_DRV, "local_port_nr=0x%x \n"
						"local_port_min=0x%x", cp->cnic_local_port_nr,
						cp->cnic_local_port_min);

					cnic_init_id_tbl(&cp->csk_port_tbl,
						cp->cnic_local_port_nr,
						cp->cnic_local_port_min);
				} else {
					found = 0;
					CNIC_ERR(dev, "TCP port alloc failed %d", status);
				}
				break;
			}
		}
	}
	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

	if (!found)
		dev = NULL;

	DRV_INFO(CNIC_MSG_DRV, "Exit. dev=%p", dev);

	return dev;
}

static int cnic_cm_get_fc_npiv_tbl(struct cnic_dev *dev,
	struct qcnic_fc_npiv_tbl *npiv_tbl)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	int status = VMK_OK;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (!vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP))
		return VMK_STATUS_PENDING;     /* qfle3 is down! */

	if (!QFLE3_CHIP_IS_E2_PLUS(adapter))
		return VMK_BAD_PARAM;

	status = cp->ethdev->cnicGetFcNpivTbl(dev->netdev, npiv_tbl);

	return status;
}

static int cnic_get_route(struct cnic_sock *csk, struct cnic_sockaddr *saddr)
{
	struct cnic_dev *dev = csk->dev;
	struct cnic_local *cp = dev->cnic_priv;
	int is_v6, err;
	int local_port;

	CNIC_INFO(dev, CNIC_MSG_DRV, "Enter.");

	if (saddr->local.v6.sin6_family == AF_INET6 &&
	    saddr->remote.v6.sin6_family == AF_INET6)
		is_v6 = 1;
	else if (saddr->local.v4.sin_family == VMK_SOCKET_AF_INET &&
		 saddr->remote.v4.sin_family == VMK_SOCKET_AF_INET)
		is_v6 = 0;
	else {
		CNIC_ERR(dev, "Error: sock-family not correct. sin_family=0x%x",
			saddr->local.v4.sin_family);
		return VMK_BAD_PARAM;
	}

	/* Use stored next hop MAC address */
	if (!cp->nextHopMACAddr[0] && !cp->nextHopMACAddr[1] &&
		!cp->nextHopMACAddr[2]) {
		CNIC_ERR(dev, "Zero next hop address, aborting.");
		return VMK_BAD_PARAM;
	}

	if (!cnic_offld_prep(csk)) {
		CNIC_ERR(dev, "Error: cnic_offld_prep.");
		return VMK_BAD_PARAM;
	}

	vmk_BitVectorClear(csk->flags, SK_F_IPV6);
	vmk_Memcpy(csk->ha, cp->nextHopMACAddr, VMK_ETH_ADDR_LENGTH);
	local_port = cnic_alloc_new_id(&cp->csk_port_tbl);
	if (local_port == -1) {
		vmk_BitVectorClear(csk->flags, SK_F_OFFLD_SCHED);
		return VMK_NO_MEMORY;
	}
	csk->pmtu = cp->pmtu;
	csk->vlan_id = cp->vlan_id;

	if (is_v6) {
		vmk_BitVectorSet(csk->flags, SK_F_IPV6);
		vmk_Memcpy((vmk_uint8 *)csk->src_ip, cp->srcIPAddr, 16);
		vmk_Memcpy(
			(vmk_uint8 *)csk->dst_ip,
			saddr->remote.v6.sin6_addr.__u6_addr.__u6_addr8, 16);
		csk->src_port = vmk_CPUToBE16(local_port);
		csk->dst_port = saddr->remote.v6.sin6_port;
		dev->is_ipv6 = 1;
	} else {
		csk->src_ip[0] = *((vmk_uint32 *)cp->srcIPAddr);
		csk->dst_ip[0] = saddr->remote.v4.sin_addr.s_addr;
		csk->src_port = vmk_CPUToBE16(local_port);
		csk->dst_port = saddr->remote.v4.sin_port;
	}

	CNIC_INFO(dev, CNIC_MSG_DRV, "is_v6=0x%x src_port=0x%x \n"
		"dst_port=0x%x",
		is_v6, csk->src_port, csk->dst_port);

	err = cnic_cm_offload_pg(csk);

	CNIC_INFO(dev, CNIC_MSG_DRV, "Exit. err=0x%x", err);

	return err;
}

static void cnic_init_csk_state(struct cnic_sock *csk)
{
	csk->state = 0;
	vmk_BitVectorClear(csk->flags, SK_F_OFFLD_SCHED);
	vmk_BitVectorClear(csk->flags, SK_F_CLOSING);
}

static int cnic_cm_connect(struct cnic_sock *csk, struct cnic_sockaddr *saddr)
{
	int err = 0;

	DRV_INFO(CNIC_MSG_DRV, "Enter.");

	if (!cnic_in_use(csk)) {
		DRV_ERR("Error: cnic_in_use not set.");
		return VMK_NOT_INITIALIZED;
	}

	if (vmk_BitVectorAtomicTestAndSet(csk->flags, SK_F_CONNECT_START))
		return VMK_EALREADY;

	cnic_init_csk_state(csk);

	err = cnic_get_route(csk, saddr);
	if (err)
		goto err_out;


err_out:
	vmk_BitVectorClear(csk->flags, SK_F_CONNECT_START);
	DRV_INFO(CNIC_MSG_DRV, "Exit. err=0x%x", err);
	return err;
}

static int cnic_cm_abort(struct cnic_sock *csk)
{
	struct cnic_local *cp = csk->dev->cnic_priv;
	vmk_uint32 opcode = L4_KCQE_OPCODE_VALUE_RESET_COMP;

	if (!cnic_in_use(csk))
		return VMK_BAD_PARAM;

	if (cnic_abort_prep(csk))
		return cnic_cm_abort_req(csk);

	/* Getting here means that we haven't started connect, or
	 * connect was not successful, or it has been reset by the target.
	 */

	cp->close_conn(csk, opcode);
	if (csk->state != opcode) {
		/* Wait for remote reset sequence to complete */
		while (vmk_BitVectorTest(csk->flags, SK_F_PG_OFFLD_COMPLETE))
			vmk_WorldSleep(1*VMK_USEC_PER_MSEC);

		return VMK_EALREADY;
	}
	return 0;
}

static int cnic_cm_close(struct cnic_sock *csk)
{
	if (!cnic_in_use(csk))
		return VMK_BAD_PARAM;

	if (cnic_close_prep(csk)) {
		csk->state = L4_KCQE_OPCODE_VALUE_CLOSE_COMP;
		return cnic_cm_close_req(csk);
	} else {
		/* Wait for remote reset sequence to complete */
		while (vmk_BitVectorTest(csk->flags, SK_F_PG_OFFLD_COMPLETE))
			vmk_WorldSleep(1*VMK_USEC_PER_MSEC);

		return VMK_EALREADY;
	}
	return 0;
}

static void cnic_cm_upcall(struct cnic_local *cp, struct cnic_sock *csk,
			   vmk_uint8 opcode)
{
	struct cnic_ulp_ops *ulp_ops;
	int ulp_type = csk->ulp_type;

	vmk_SemaLock(&cnic_driver_info.cnic_lock);
	ulp_ops = cp->ulp_ops[ulp_type];
	CNIC_INFO(cp->dev, CNIC_MSG_DRV, "ulp_type=0x%x, opcode=0x%x", ulp_type, opcode);
	if (ulp_ops) {
		if (opcode == L4_KCQE_OPCODE_VALUE_CONNECT_COMPLETE)
			ulp_ops->cm_connect_complete(csk);
		else if (opcode == L4_KCQE_OPCODE_VALUE_CLOSE_COMP)
			ulp_ops->cm_close_complete(csk);
		else if (opcode == L4_KCQE_OPCODE_VALUE_RESET_RECEIVED)
			ulp_ops->cm_remote_abort(csk);
		else if (opcode == L4_KCQE_OPCODE_VALUE_RESET_COMP)
			ulp_ops->cm_abort_complete(csk);
		else if (opcode == L4_KCQE_OPCODE_VALUE_CLOSE_RECEIVED)
			ulp_ops->cm_remote_close(csk);
	}
	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
}

static void cnic_cm_process_neigh(struct cnic_dev *dev,
	struct neighbour *neigh)
{
}

static void cnic_cm_process_redirect(struct cnic_dev *dev,
	struct cnic_redirect_entry *redir)
{
}

static void cnic_cm_process_offld_pg(struct cnic_dev *dev, struct l4_kcq *kcqe)
{
	struct cnic_local *cp = dev->cnic_priv;
	vmk_uint32 l5_cid = kcqe->pg_host_opaque;
	vmk_uint8 opcode = kcqe->op_code;
	struct cnic_sock *csk = &cp->csk_tbl[l5_cid];

	CNIC_INFO(dev, CNIC_MSG_DRV, "Enter. opcode=0x%x l5_cid=0x%x",
		opcode, l5_cid);

	csk_hold(csk);
	if (!cnic_in_use(csk))
		goto done;

	if (opcode == L4_KCQE_OPCODE_VALUE_UPDATE_PG) {
		vmk_BitVectorClear(csk->flags, SK_F_OFFLD_SCHED);
		goto done;
	}
	/* Possible PG kcqe status:  SUCCESS, OFFLOADED_PG, or CTX_ALLOC_FAIL */
	if (kcqe->status == L4_KCQE_COMPLETION_STATUS_CTX_ALLOC_FAIL) {
		vmk_BitVectorClear(csk->flags, SK_F_OFFLD_SCHED);
		cnic_cm_upcall(cp, csk,
			       L4_KCQE_OPCODE_VALUE_CONNECT_COMPLETE);
		goto done;
	}

	csk->pg_cid = kcqe->pg_cid;
	vmk_BitVectorSet(csk->flags, SK_F_PG_OFFLD_COMPLETE);
	cnic_cm_conn_req(csk);

done:
	csk_put(csk);
}

static void cnic_process_fcoe_term_conn(struct cnic_dev *dev, struct kcqe *kcqe)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct fcoe_kcqe *fc_kcqe = (struct fcoe_kcqe *) kcqe;
	vmk_uint32 l5_cid = fc_kcqe->fcoe_conn_id + QFLE3_FCOE_L5_CID_BASE;
	struct cnic_context *ctx = &cp->ctx_tbl[l5_cid];

	ctx->timestamp = vmk_GetTimerCycles();
	ctx->wait_cond = 1;

	CNIC_INFO(dev, CNIC_MSG_EVENT,
		"Trying to wake up completion event=%p",
		&ctx->waitq.cmpl_event);
	ql_vmk_world_wakeup(&ctx->waitq);
}

static void cnic_cm_process_kcqe(struct cnic_dev *dev, struct kcqe *kcqe)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct l4_kcq *l4kcqe = (struct l4_kcq *) kcqe;
	vmk_uint8 opcode = l4kcqe->op_code;
	vmk_uint32 l5_cid;
	struct cnic_sock *csk;

	CNIC_INFO(dev, CNIC_MSG_DRV, "opcode=0x%x conn_id=0x%x \n"
		"cid=0x%x status=0x%x",
		opcode, l4kcqe->conn_id, l4kcqe->cid, l4kcqe->status);
	switch (opcode) {
		case FCOE_RAMROD_CMD_ID_TERMINATE_CONN:
			cnic_process_fcoe_term_conn(dev, kcqe);
			return;
		case L4_KCQE_OPCODE_VALUE_OFFLOAD_PG:
		case L4_KCQE_OPCODE_VALUE_UPDATE_PG:
			cnic_cm_process_offld_pg(dev, l4kcqe);
			return;
#if (CNIC_ISCSI_OOO_SUPPORT)
		case L4_KCQE_OPCODE_VALUE_OOO_FLUSH:
			cnic_flush_ooo(dev, l4kcqe->cid);
			return;

		case L4_KCQE_OPCODE_VALUE_OOO_EVENT_NOTIFICATION:
			vmk_BitVectorSet(cp->iooo_mgmr.flags, IOOO_START_HANDLER);
			return;
#endif
	}

	l5_cid = l4kcqe->conn_id;
	/* Hack */
	if (opcode & 0x80)
		l5_cid = l4kcqe->cid;
	if (l5_cid >= MAX_CM_SK_TBL_SZ) {
		CNIC_ERR(dev, "l5_cid=0x%x >= MAX_CM_SK_TBL_SZ=0x%x",
			l5_cid, MAX_CM_SK_TBL_SZ);
		return;
	}

	csk = &cp->csk_tbl[l5_cid];
	csk_hold(csk);

	if (!cnic_in_use(csk)) {
		csk_put(csk);
		CNIC_ERR(dev, "Error: cnic_in_use not set!");
		return;
	}

	switch (opcode) {
		case L5CM_RAMROD_CMD_ID_TCP_CONNECT:
			if (l4kcqe->status != 0) {
				vmk_BitVectorClear(csk->flags, SK_F_OFFLD_SCHED);
				cnic_cm_upcall(cp, csk, L4_KCQE_OPCODE_VALUE_CONNECT_COMPLETE);
			}
			break;
		case L4_KCQE_OPCODE_VALUE_CONNECT_COMPLETE:
			if (l4kcqe->status == 0) {
				vmk_BitVectorSet(csk->flags, SK_F_OFFLD_COMPLETE);
			} else {
				CNIC_ERR(dev, "Connect completion failed: "
					"status: 0x%x cid: 0x%x l5_cid: 0x%x",
					l4kcqe->status, l4kcqe->cid, l5_cid);
				if (l4kcqe->status == L4_KCQE_COMPLETION_STATUS_PARITY_ERROR)
					vmk_BitVectorSet(csk->flags, SK_F_HW_ERR);
			}

			vmk_CPUMemFenceReadWrite();
			vmk_BitVectorClear(csk->flags, SK_F_OFFLD_SCHED);
			cnic_cm_upcall(cp, csk, opcode);
			break;
		case L5CM_RAMROD_CMD_ID_CLOSE:
			{
				struct iscsi_kcqe *l5kcqe = (struct iscsi_kcqe *)kcqe;

				if (l4kcqe->status != 0 || l5kcqe->completion_status != 0) {
					CNIC_WARN(dev, "RAMROD CLOSE compl with status 0x%x "
						"completion status 0x%x",
						l4kcqe->status, l5kcqe->completion_status);
					opcode = L4_KCQE_OPCODE_VALUE_CLOSE_COMP;
					/* Fall through */
				} else {
					break;
				}
			}
		case L4_KCQE_OPCODE_VALUE_RESET_RECEIVED:
		case L4_KCQE_OPCODE_VALUE_CLOSE_COMP:
		case L4_KCQE_OPCODE_VALUE_RESET_COMP:
		case L5CM_RAMROD_CMD_ID_SEARCHER_DELETE:
		case L5CM_RAMROD_CMD_ID_TERMINATE_OFFLOAD:
			if (l4kcqe->status == L4_KCQE_COMPLETION_STATUS_PARITY_ERROR)
				vmk_BitVectorSet(csk->flags, SK_F_HW_ERR);

			cp->close_conn(csk, opcode);
			break;

		case L4_KCQE_OPCODE_VALUE_CLOSE_RECEIVED:
			/* after we already sent CLOSE_REQ */
			if (vmk_BitVectorTest(dev->flags, CNIC_F_QFLE3_CLASS) &&
					!vmk_BitVectorTest(csk->flags, SK_F_OFFLD_COMPLETE) &&
					csk->state == L4_KCQE_OPCODE_VALUE_CLOSE_COMP)
				cp->close_conn(csk, L4_KCQE_OPCODE_VALUE_RESET_COMP);
			else
				cnic_cm_upcall(cp, csk, opcode);
			break;
	}
	csk_put(csk);
}

static void cnic_cm_indicate_kcqe(void *data, struct kcqe *kcqe[],
	vmk_uint32 num_cqe)
{
	struct cnic_dev *dev = data;
	int i;
	struct cnic_local *cp = dev->cnic_priv;

	CNIC_INFO(dev, CNIC_MSG_DRV, "queue cnic-task: num_cqe=0x%x", num_cqe);
	for (i = 0; i < num_cqe; i++)
		cnic_queue_work(cp, WORK_TYPE_KCQE, kcqe[i]);

	ql_vmk_tasklet_schedule(&cp->cnic_task);
}

static void cnic_cm_indicate_event(void *data, unsigned long event)
{
}

static void cnic_cm_dummy(void *data)
{
}

static struct cnic_ulp_ops cm_ulp_ops = {
	.cnic_start		= cnic_cm_dummy,
	.cnic_stop		= cnic_cm_dummy,
	.indicate_kcqes		= cnic_cm_indicate_kcqe,
	.indicate_netevent	= cnic_cm_indicate_event,
	.indicate_inetevent	= cnic_cm_indicate_event,
};

static void cnic_task(unsigned long data)
{
	struct cnic_local *cp = (struct cnic_local *) data;
	struct cnic_dev *dev = cp->dev;
	vmk_uint32 cons = cp->cnic_wr_cons;
	vmk_uint32 prod = cp->cnic_wr_prod;

	CNIC_INFO(dev, CNIC_MSG_DRV, "cons=0x%x prod=0x%x", cons, prod);
	while (cons != prod) {
		struct cnic_work_node *node;

		node = &cp->cnic_work_ring[cons];
		CNIC_INFO(dev, CNIC_MSG_DRV, "work_type=0x%x", node->work_type);
		if (node->work_type == WORK_TYPE_KCQE)
			cnic_cm_process_kcqe(dev, &node->work_data.kcqe);
		else if (node->work_type == WORK_TYPE_NEIGH_UPDATE)
			cnic_cm_process_neigh(dev, node->work_data.neigh);
		else if (node->work_type == WORK_TYPE_REDIRECT)
			cnic_cm_process_redirect(dev,
				&node->work_data.cnic_redir);
		cons++;
		cons &= WORK_RING_SIZE_MASK;
	}
	cp->cnic_wr_cons = cons;
}

static void cnic_free_dev(struct cnic_dev *cdev)
{
	int i = 0;
	struct cnic_local *cp = cdev->cnic_priv;

	while ((vmk_AtomicRead64(&cdev->ref_count) != 0) && i < 10) {
		vmk_WorldSleep(100*VMK_USEC_PER_MSEC);
		i++;
	}
	if (vmk_AtomicRead64(&cdev->ref_count) != 0)
		CNIC_ERR(cdev, "Failed waiting for ref count to goto zero.");


	if (cdev->flags) {
		vmk_BitVectorFree(cnic_driver_info.heap_id, cdev->flags);
		cdev->flags = NULL;
	}
	if (cp->cnic_local_flags) {
		vmk_BitVectorFree(cnic_driver_info.heap_id, cp->cnic_local_flags);
		cp->cnic_local_flags = NULL;
	}
	for (i = 0; i < MAX_CNIC_ULP_TYPE; i++) {
		if (cp->ulp_flags[i]) {
			vmk_BitVectorFree(cnic_driver_info.heap_id, cp->ulp_flags[i]);
			cp->ulp_flags[i] = NULL;
		}
	}
	vmk_SpinlockDestroy(cp->cnic_ulp_lock);


	CNIC_INFO(cdev, CNIC_MSG_DRV,
		"Remove CNIC device: %s", vmk_NameToString(&cdev->name));
	ql_vmk_free(cdev);
}

static void cnic_cm_free_mem(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	int i = 0;

	if (cp->csk_tbl) {
		for (i = 0; i < MAX_CM_SK_TBL_SZ; i++) {
			if (cp->csk_tbl[i].flags) {
				vmk_BitVectorFree(cnic_driver_info.heap_id,
						cp->csk_tbl[i].flags);
				cp->csk_tbl[i].flags = NULL;
			}
		}

		ql_vmk_free(cp->csk_tbl);
		cp->csk_tbl = NULL;
	}

	//SV: XXX: cp->next_tcp_port = cnic_free_id_tbl(&cp->csk_port_tbl);
	cnic_free_id_tbl(&cp->csk_port_tbl);
}

static int cnic_cm_alloc_mem(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	int i = 0;

	cp->csk_tbl = ql_vmk_alloc(sizeof(struct cnic_sock) * MAX_CM_SK_TBL_SZ);
	if (!cp->csk_tbl)
		return VMK_NO_MEMORY;

	for (i = 0; i < MAX_CM_SK_TBL_SZ; i++) {
		cp->csk_tbl[i].flags = vmk_BitVectorAlloc(cnic_driver_info.heap_id,
			(sizeof(long) * VMK_BITS_PER_BYTE));
		vmk_BitVectorZap(cp->csk_tbl[i].flags);
	}

	return 0;
}

static int cnic_ready_to_close(struct cnic_sock *csk, vmk_uint32 opcode)
{
	if (vmk_BitVectorAtomicTestAndClear(csk->flags, SK_F_OFFLD_COMPLETE)) {
		/* Unsolicited RESET_COMP or RESET_RECEIVED */
		opcode = L4_KCQE_OPCODE_VALUE_RESET_RECEIVED;
		csk->state = opcode;
	}

	/* 1. If event opcode matches the expected event in csk->state
	 * 2. If the expected event is CLOSE_COMP or RESET_COMP, we accept any
	 *    event
	 * 3. If the expected event is 0, meaning the connection was never
	 *    never established, we accept the opcode from cm_abort.
	 */
	if (opcode == csk->state || csk->state == 0 ||
	    csk->state == L4_KCQE_OPCODE_VALUE_CLOSE_COMP ||
	    csk->state == L4_KCQE_OPCODE_VALUE_RESET_COMP) {
		if (!vmk_BitVectorAtomicTestAndClear(csk->flags, SK_F_CLOSING)) {
			if (csk->state == 0)
				csk->state = opcode;
			return 1;
		}
	}
	return 0;
}

static void cnic_close_bnx2_conn(struct cnic_sock *csk, vmk_uint32 opcode)
{
	struct cnic_dev *dev = csk->dev;
	struct cnic_local *cp = dev->cnic_priv;

	vmk_BitVectorClear(csk->flags, SK_F_CONNECT_START);
	cnic_close_conn(csk);
	cnic_cm_upcall(cp, csk, opcode);
}

static void cnic_cm_stop_bnx2_hw(struct cnic_dev *dev)
{
}

static int cnic_cm_init_bnx2_hw(struct cnic_dev *dev)
{
	vmk_uint32 seed;

//SV: XXX: #if (LINUX_VERSION_CODE >= 0x020612)
//	get_random_bytes(&seed, 4);
//#else
	seed = 0x12345678;
//#endif
	cnic_ctx_wr(dev, 45, 0, seed);
	return 0;
}

static void cnic_close_qfle3_conn(struct cnic_sock *csk, vmk_uint32 opcode)
{
	struct cnic_dev *dev = csk->dev;
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_context *ctx = &cp->ctx_tbl[csk->l5_cid];
	union l5cm_specific_data l5_data;
	vmk_uint32 cmd = 0;
	int close_complete = 0;

	switch (opcode) {
	case L4_KCQE_OPCODE_VALUE_RESET_RECEIVED:
	case L4_KCQE_OPCODE_VALUE_CLOSE_COMP:
	case L4_KCQE_OPCODE_VALUE_RESET_COMP:
		if (cnic_ready_to_close(csk, opcode)) {
			if (vmk_BitVectorTest(csk->flags, SK_F_HW_ERR))
				close_complete = 1;
			else if (vmk_BitVectorTest(csk->flags, SK_F_PG_OFFLD_COMPLETE))
				cmd = L5CM_RAMROD_CMD_ID_SEARCHER_DELETE;
			else
				close_complete = 1;
		}
		break;
	case L5CM_RAMROD_CMD_ID_SEARCHER_DELETE:
		cmd = L5CM_RAMROD_CMD_ID_TERMINATE_OFFLOAD;
		break;
	case L5CM_RAMROD_CMD_ID_TERMINATE_OFFLOAD:
		close_complete = 1;
		break;
	}
	if (cmd) {
		vmk_Memset(&l5_data, 0, sizeof(l5_data));

		cnic_submit_kwqe_16(dev, cmd, csk->cid, ISCSI_CONNECTION_TYPE,
				&l5_data);
	} else if (close_complete) {
		ctx->timestamp = vmk_GetTimerCycles();
		cnic_close_conn(csk);
		cnic_cm_upcall(cp, csk, csk->state);
	}
}

static void cnic_cm_stop_qfle3_hw(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;

	if (!cp->ctx_tbl)
		return;

	cnic_qfle3_delete_wait(dev, 0);

	ql_vmk_cancel_delayed_work(&cp->delete_task);
	ql_vmk_flush_singlethread_workqueue(cnic_driver_info.delayed_wq);

	if (vmk_AtomicRead64(&cp->iscsi_conn) != 0)
		CNIC_WARN(dev, "0x%lx iSCSI connections not destroyed.",
			vmk_AtomicRead64(&cp->iscsi_conn));
}

static int cnic_cm_init_qfle3_hw(struct cnic_dev *dev)
{
	struct qfle3_adapter *adapter = NULL;
	vmk_uint32 pfid;
	vmk_uint32 port;
	vmk_Device netdev = dev->netdev;

	vmk_DeviceGetAttachedDriverData(netdev, (vmk_AddrCookie *) &adapter);
	pfid = adapter->pf_id;
	port = QFLE3_PORT(adapter);

	cnic_init_qfle3_mac(dev, adapter->hwMacAddr);
	cnic_qfle3_set_tcp_options(dev, 0, 1);

	CNIC_WR16(dev, BAR_XSTRORM_INTMEM +
		  XSTORM_ISCSI_LOCAL_VLAN_OFFSET(pfid), 0);

	CNIC_WR(dev, BAR_XSTRORM_INTMEM +
		XSTORM_TCP_GLOBAL_DEL_ACK_COUNTER_ENABLED_OFFSET(port), 1);
	CNIC_WR(dev, BAR_XSTRORM_INTMEM +
		XSTORM_TCP_GLOBAL_DEL_ACK_COUNTER_MAX_COUNT_OFFSET(port),
		DEF_MAX_DA_COUNT);

	CNIC_WR8(dev, BAR_XSTRORM_INTMEM +
		 XSTORM_ISCSI_TCP_VARS_TTL_OFFSET(pfid), DEF_TTL);
	CNIC_WR8(dev, BAR_XSTRORM_INTMEM +
		 XSTORM_ISCSI_TCP_VARS_TOS_OFFSET(pfid), DEF_TOS);
	CNIC_WR8(dev, BAR_XSTRORM_INTMEM +
		 XSTORM_ISCSI_TCP_VARS_ADV_WND_SCL_OFFSET(pfid), 0);
	CNIC_WR(dev, BAR_XSTRORM_INTMEM +
		XSTORM_TCP_TX_SWS_TIMER_VAL_OFFSET(pfid), DEF_SWS_TIMER);

	CNIC_WR(dev, BAR_TSTRORM_INTMEM + TSTORM_TCP_MAX_CWND_OFFSET(pfid),
		DEF_MAX_CWND);

	return 0;
}

static void cnic_delete_task(void *data)
{
	struct cnic_local *cp;
	struct cnic_dev *dev;
	vmk_uint32 i;
	int need_resched = 0;

	cp = (struct cnic_local *) data;
	dev = cp->dev;

	for (i = 0; i < cp->max_cid_space; i++) {
		struct cnic_context *ctx = &cp->ctx_tbl[i];
		int err;

		if (!vmk_BitVectorTest(ctx->ctx_flags, CTX_FL_OFFLD_START) ||
		    !vmk_BitVectorTest(ctx->ctx_flags, CTX_FL_DELETE_WAIT))
			continue;

		if (vmk_GetTimerCycles() <
				(ctx->timestamp + (2 * vmk_TimerCyclesPerSecond()))) {
			need_resched = 1;
			continue;
		}

		if (!vmk_BitVectorAtomicTestAndClear(ctx->ctx_flags,
					CTX_FL_DELETE_WAIT))
			continue;

		err = cnic_qfle3_destroy_ramrod(dev, i);

		cnic_free_qfle3_conn_resc(dev, i);
		if (!err) {
			CNIC_INFO(dev, CNIC_MSG_DRV, "ctx destroy successful.");

			if (ctx->ulp_proto_id == CNIC_ULP_ISCSI)
				vmk_AtomicDec64(&cp->iscsi_conn);

			vmk_BitVectorClear(ctx->ctx_flags, CTX_FL_OFFLD_START);
		}
	}

	if (need_resched)
		ql_vmk_queue_delayed_work(cnic_driver_info.delayed_tq,
			cnic_driver_info.delayed_wq, &cp->delete_task, cnic_delete_task,
			cp, 10);
}

static int cnic_cm_open(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	int err;

	CNIC_INFO(dev, CNIC_MSG_DRV, "Entered");

	err = cnic_cm_alloc_mem(dev);
	if (err)
		return err;

	err = cp->start_cm(dev);

	if (err)
		goto err_out;

	ql_vmk_spin_lock_init(&cp->wr_lock, LOCK_RANK_HIGHEST);

	ql_vmk_tasklet_init(&cp->cnic_task, cnic_task, (void *)cp);

	dev->cm_get_uplinkName = cnic_cm_get_uplinkName;
	dev->cm_acquire = cnic_cm_acquire;
	dev->cm_create = cnic_cm_create;
	dev->cm_destroy = cnic_cm_destroy;
	dev->cm_connect = cnic_cm_connect;
	dev->cm_abort = cnic_cm_abort;
	dev->cm_close = cnic_cm_close;
	dev->cm_select_dev = cnic_cm_select_dev;
	dev->cm_get_fc_npiv_tbl = cnic_cm_get_fc_npiv_tbl;

	cp->ulp_handle[CNIC_ULP_L4] = dev;
	//SV: XXX: Check if locking needed?
	cp->ulp_ops[CNIC_ULP_L4] = &cm_ulp_ops;

	CNIC_INFO(dev, CNIC_MSG_DRV, "exiting - success.");
	return 0;

err_out:
	cnic_cm_free_mem(dev);
	CNIC_INFO(dev, CNIC_MSG_DRV, "exiting - failure.");

	return err;
}

static int cnic_cm_shutdown(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	int i;

	ql_vmk_tasklet_disable(&cp->cnic_task);

	if (!cp->csk_tbl)
		return 0;

	for (i = 0; i < MAX_CM_SK_TBL_SZ; i++) {
		struct cnic_sock *csk = &cp->csk_tbl[i];

		vmk_BitVectorClear(csk->flags, SK_F_INUSE);
		cnic_cm_cleanup(csk);
	}
	cnic_cm_free_mem(dev);

	vmk_SpinlockDestroy(cp->wr_lock);

	return 0;
}

static void cnic_init_context(struct cnic_dev *dev, vmk_uint32 cid)
{
	vmk_uint32 cid_addr;
	int i;

	cid_addr = GET_CID_ADDR(cid);

	for (i = 0; i < CTX_SIZE; i += 4)
		cnic_ctx_wr(dev, cid_addr, i, 0);
}

static int cnic_setup_5709_context(struct cnic_dev *dev, int valid)
{
	struct cnic_local *cp = dev->cnic_priv;
	int ret = 0, i;
	vmk_uint32 valid_bit = valid ? BNX2_CTX_HOST_PAGE_TBL_DATA0_VALID : 0;

	if (BNX2_CHIP(cp) != BNX2_CHIP_5709)
		return 0;

	for (i = 0; i < cp->ctx_blks; i++) {
		int j;
		vmk_uint32 idx = cp->ctx_arr[i].cid / cp->cids_per_blk;
		vmk_uint32 val;

		vmk_Memset(cp->ctx_arr[i].ctx, 0, BNX2_PAGE_SIZE);

		CNIC_WR(dev, BNX2_CTX_HOST_PAGE_TBL_DATA0,
			(cp->ctx_arr[i].mapping & 0xffffffff) | valid_bit);
		CNIC_WR(dev, BNX2_CTX_HOST_PAGE_TBL_DATA1,
			(vmk_uint64) cp->ctx_arr[i].mapping >> 32);
		CNIC_WR(dev, BNX2_CTX_HOST_PAGE_TBL_CTRL, idx |
			BNX2_CTX_HOST_PAGE_TBL_CTRL_WRITE_REQ);
		for (j = 0; j < 10; j++) {

			val = CNIC_RD(dev, BNX2_CTX_HOST_PAGE_TBL_CTRL);
			if (!(val & BNX2_CTX_HOST_PAGE_TBL_CTRL_WRITE_REQ))
				break;
			vmk_DelayUsecs(5);
		}
		if (val & BNX2_CTX_HOST_PAGE_TBL_CTRL_WRITE_REQ) {
			ret = VMK_BUSY;
			break;
		}
	}
	return ret;
}

static void cnic_free_irq(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	int i = 0;

	if (ethdev->driverState & CNIC_DRV_STATE_USING_MSIX) {
		cp->disable_int_sync(dev);
		//ql_vmk_tasklet_disable(&cp->cnic_irq_task);
		if (vmk_BitVectorTest(cp->cnic_local_flags, CNIC_LCL_FL_IRQ_REQD)) {
			ql_vmk_free_irq(dev, ethdev->irq_arr[0].cnic_intr_cookie,
					(void *)dev);
		}
		vmk_BitVectorClear(cp->cnic_local_flags, CNIC_LCL_FL_IRQ_REQD);
	}
}

static int cnic_request_irq(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct qfle3_adapter *adapter = NULL;
	VMK_ReturnStatus err;
	vmk_Name name;
	int i = 0;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (vmk_BitVectorTest(cp->cnic_local_flags, CNIC_LCL_FL_IRQ_REQD))
		return 0;

	CNIC_INFO(dev, CNIC_MSG_DRV, "Request cnic irq: 0x%x",
		ethdev->irq_arr[0].cnic_intr_cookie);

	vmk_NameFormat(&name, "%s-cnic",
		(char *)vmk_NameToString(&adapter->pdev_name));
	err = ql_vmk_request_irq(dev, ethdev->irq_arr[0].cnic_intr_cookie,
			(void *)(cnic_msix_handlers[0].intr_handler),
			vmk_NameToString(&name), dev, cnic_ack_msix);

	CNIC_INFO(dev, CNIC_MSG_INIT, "irq name: %s", vmk_NameToString(&name));

	if (err != VMK_OK) {
		CNIC_ERR(dev, "Failed to request irq. status=[%s].",
			vmk_StatusToString(err));
		//ql_vmk_tasklet_disable(&cp->cnic_irq_task);
	} else
		vmk_BitVectorSet(cp->cnic_local_flags, CNIC_LCL_FL_IRQ_REQD);

	/* SV: Since we do not setup queues, we need to enable cnic-irq ourselves. */
	cnic_ack_igu_sb(dev, ethdev->irq_arr[0].status_blk_num2,
		IGU_SEG_ACCESS_DEF, 0, IGU_INT_ENABLE, 0);

	return err;
}

static int cnic_init_bnx2_irq(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;

	if (ethdev->driverState & CNIC_DRV_STATE_USING_MSIX) {
		int err, i = 0;
		int sblk_num = cp->cfp[CNIC_RX_IDX].status_blk_num;
		vmk_uint32 base = ((sblk_num - 1) * BNX2_HC_SB_CONFIG_SIZE) +
			   BNX2_HC_SB_CONFIG_1;

		CNIC_WR(dev, base, BNX2_HC_SB_CONFIG_1_ONE_SHOT);

		CNIC_WR(dev, base + BNX2_HC_COMP_PROD_TRIP_OFF, (2 << 16) | 8);
		CNIC_WR(dev, base + BNX2_HC_COM_TICKS_OFF, (64 << 16) | 220);
		CNIC_WR(dev, base + BNX2_HC_CMD_TICKS_OFF, (64 << 16) | 220);

		cp->last_status_idx = cp->cfp[CNIC_RX_IDX].status_blk.bnx2->status_idx;

		cp->cnic_service_bh = cnic_service_bnx2_msix;

		//ql_vmk_tasklet_init(&cp->cnic_irq_task, cnic_service_bnx2_msix,
		//	(void *)dev);
		err = cnic_request_irq(dev);
		if (err)
			return err;

		while (cp->cfp[CNIC_RX_IDX].status_blk.bnx2->status_completion_producer_index &&
		       i < 10) {
			CNIC_WR(dev, BNX2_HC_COALESCE_NOW,
				1 << (11 + sblk_num));
			vmk_DelayUsecs(10);
			i++;
			vmk_CPUMemFenceReadWrite();
		}
		if (cp->cfp[CNIC_RX_IDX].status_blk.bnx2->status_completion_producer_index) {

			cnic_free_irq(dev);
			goto failed;
		}

	} else {
		struct status_block *sblk = cp->cfp[CNIC_RX_IDX].status_blk.gen;
		vmk_uint32 hc_cmd = CNIC_RD(dev, BNX2_HC_COMMAND);
		int i = 0;

		while (sblk->status_completion_producer_index && i < 10) {
			CNIC_WR(dev, BNX2_HC_COMMAND,
				hc_cmd | BNX2_HC_COMMAND_COAL_NOW_WO_INT);
			vmk_DelayUsecs(10);
			i++;
			vmk_CPUMemFenceReadWrite();
		}
		if (sblk->status_completion_producer_index)
			goto failed;

	}
	return 0;

failed:
	CNIC_ERR(dev, "KCQ index not resetting to 0.");
	return VMK_BUSY;
}

static void cnic_enable_bnx2_int(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;

	if (!(ethdev->driverState & CNIC_DRV_STATE_USING_MSIX))
		return;

	CNIC_WR(dev, BNX2_PCICFG_INT_ACK_CMD, cp->int_num |
		BNX2_PCICFG_INT_ACK_CMD_INDEX_VALID | cp->last_status_idx);
}

static void cnic_disable_bnx2_int_sync(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;

	if (!(ethdev->driverState & CNIC_DRV_STATE_USING_MSIX))
		return;

	CNIC_WR(dev, BNX2_PCICFG_INT_ACK_CMD, cp->int_num |
		BNX2_PCICFG_INT_ACK_CMD_MASK_INT);
	CNIC_RD(dev, BNX2_PCICFG_INT_ACK_CMD);
	//SV: XXX: alternative? synchronize_irq(ethdev->irq_arr[0].vector);
}

#if (CNIC_ISCSI_OOO_SUPPORT)
static void cnic_init_bnx2_tx_ring_start(struct cnic_dev *dev, vmk_uint32 cid,
	vmk_IOA ring_map)
{
	struct cnic_local *cp = dev->cnic_priv;
	vmk_uint32 cid_addr, val, offset0, offset1, offset2, offset3;

	cid_addr = GET_CID_ADDR(cid);
	if (BNX2_CHIP(cp) == BNX2_CHIP_5709) {
		int i;
		vmk_uint32 cid_addr2 = GET_CID_ADDR(cid + 4) + 0x40;

		for (i = 0; i < PHY_CTX_SIZE; i += 4)
			cnic_ctx_wr(dev, cid_addr2, i, 0);

		offset0 = BNX2_L2CTX_TYPE_XI;
		offset1 = BNX2_L2CTX_CMD_TYPE_XI;
		offset2 = BNX2_L2CTX_TBDR_BHADDR_HI_XI;
		offset3 = BNX2_L2CTX_TBDR_BHADDR_LO_XI;
	} else {
		cnic_init_context(dev, cid);
		cnic_init_context(dev, cid + 1);

		offset0 = BNX2_L2CTX_TYPE;
		offset1 = BNX2_L2CTX_CMD_TYPE;
		offset2 = BNX2_L2CTX_TBDR_BHADDR_HI;
		offset3 = BNX2_L2CTX_TBDR_BHADDR_LO;
	}
	val = BNX2_L2CTX_TYPE_TYPE_L2 | BNX2_L2CTX_TYPE_SIZE_L2;
	cnic_ctx_wr(dev, cid_addr, offset0, val);

	val = BNX2_L2CTX_CMD_TYPE_TYPE_L2 | (8 << 16);
	cnic_ctx_wr(dev, cid_addr, offset1, val);

	val = (vmk_uint64) ring_map >> 32;
	cnic_ctx_wr(dev, cid_addr, offset2, val);

	val = (vmk_uint64) ring_map & 0xffffffff;
	cnic_ctx_wr(dev, cid_addr, offset3, val);
}

static void cnic_init_bnx2_rx_ring_start(struct cnic_dev *dev, vmk_uint32 cid,
					 vmk_uint16 *sb_idx, vmk_IOA ring_map,
					 vmk_uint32 sb_id)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	vmk_uint32 cid_addr, val, coal_reg, coal_val;
	int i;

	cnic_init_context(dev, cid);
	coal_reg = BNX2_HC_COMMAND;
	coal_val = CNIC_RD(dev, coal_reg);
	if (ethdev->driverState & CNIC_DRV_STATE_USING_MSIX) {
		coal_reg = BNX2_HC_COALESCE_NOW;
		coal_val = 1 << (11 + sb_id);
	}
	i = 0;
	while (*sb_idx != 0 && i < 10) {
		CNIC_WR(dev, coal_reg, coal_val);
		vmk_DelayUsecs(10);
		i++;
		vmk_CPUMemFenceReadWrite();
	}

	cid_addr = GET_CID_ADDR(cid);
	val = BNX2_L2CTX_CTX_TYPE_CTX_BD_CHN_TYPE_VALUE |
	      BNX2_L2CTX_CTX_TYPE_SIZE_L2 | (0x02 << 8);
	cnic_ctx_wr(dev, cid_addr, BNX2_L2CTX_CTX_TYPE, val);

	if (sb_id == 0)
		val = 2 << BNX2_L2CTX_L2_STATUSB_NUM_SHIFT;
	else
		val = BNX2_L2CTX_L2_STATUSB_NUM(sb_id);
	cnic_ctx_wr(dev, cid_addr, BNX2_L2CTX_HOST_BDIDX, val);

	val = (vmk_uint64) ring_map >> 32;
	cnic_ctx_wr(dev, cid_addr, BNX2_L2CTX_NX_BDHADDR_HI, val);

	val = (vmk_uint64) ring_map & 0xffffffff;
	cnic_ctx_wr(dev, cid_addr, BNX2_L2CTX_NX_BDHADDR_LO, val);
}

static void cnic_set_bnx2_rxbd(struct bnx2_rx_bd *rxbd, vmk_uint32 len, vmk_IOA map)
{
	rxbd->rx_bd_len = len;
	rxbd->rx_bd_flags = RX_BD_FLAGS_START | RX_BD_FLAGS_END;
	rxbd->rx_bd_haddr_hi = (vmk_uint64) map >> 32;
	rxbd->rx_bd_haddr_lo = (vmk_uint64) map & 0xffffffff;
}
#endif

static int cnic_start_bnx2_hw(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct status_block *sblk = cp->cfp[CNIC_RX_IDX].status_blk.gen;
	vmk_uint32 val;
	int err;

	val = CNIC_RD(dev, BNX2_MQ_CONFIG);
	val &= ~BNX2_MQ_CONFIG_KNL_BYP_BLK_SIZE;
	if (BNX2_PAGE_BITS > 12)
		val |= (12 - 8)  << 4;
	else
		val |= (BNX2_PAGE_BITS - 8)  << 4;

	CNIC_WR(dev, BNX2_MQ_CONFIG, val);

	CNIC_WR(dev, BNX2_HC_COMP_PROD_TRIP, (2 << 16) | 8);
	CNIC_WR(dev, BNX2_HC_COM_TICKS, (64 << 16) | 220);
	CNIC_WR(dev, BNX2_HC_CMD_TICKS, (64 << 16) | 220);

	err = cnic_setup_5709_context(dev, 1);
	if (err)
		return err;

	cnic_init_context(dev, KWQ_CID);
	cnic_init_context(dev, KCQ_CID);

	cp->kwq_cid_addr = GET_CID_ADDR(KWQ_CID);
	cp->kwq_io_addr = MB_GET_CID_ADDR(KWQ_CID) + L5_KRNLQ_HOST_QIDX;

	cp->max_kwq_idx = MAX_KWQ_IDX;
	cp->kwq_prod_idx = 0;
	cp->kwq_con_idx = 0;
	vmk_BitVectorSet(cp->cnic_local_flags, CNIC_LCL_FL_KWQ_INIT);

	if (BNX2_CHIP(cp) == BNX2_CHIP_5706 || BNX2_CHIP(cp) == BNX2_CHIP_5708)
		cp->kwq_con_idx_ptr = &sblk->status_rx_quick_consumer_index15;
	else
		cp->kwq_con_idx_ptr = &sblk->status_cmd_consumer_index;

	/* Initialize the kernel work queue context. */
	val = KRNLQ_TYPE_TYPE_KRNLQ | KRNLQ_SIZE_TYPE_SIZE |
	      (BNX2_PAGE_BITS - 8) | KRNLQ_FLAGS_QE_SELF_SEQ;
	cnic_ctx_wr(dev, cp->kwq_cid_addr, L5_KRNLQ_TYPE, val);

	val = (BNX2_PAGE_SIZE / sizeof(struct kwqe) - 1) << 16;
	cnic_ctx_wr(dev, cp->kwq_cid_addr, L5_KRNLQ_QE_SELF_SEQ_MAX, val);

	val = ((BNX2_PAGE_SIZE / sizeof(struct kwqe)) << 16) | KWQ_PAGE_CNT;
	cnic_ctx_wr(dev, cp->kwq_cid_addr, L5_KRNLQ_PGTBL_NPAGES, val);

	val = (vmk_uint32) ((vmk_uint64) cp->kwq_info.pgtbl_map >> 32);
	cnic_ctx_wr(dev, cp->kwq_cid_addr, L5_KRNLQ_PGTBL_HADDR_HI, val);

	val = (vmk_uint32) cp->kwq_info.pgtbl_map;
	cnic_ctx_wr(dev, cp->kwq_cid_addr, L5_KRNLQ_PGTBL_HADDR_LO, val);

	cp->kcq_cid_addr = GET_CID_ADDR(KCQ_CID);
	cp->kcq1.io_addr = MB_GET_CID_ADDR(KCQ_CID) + L5_KRNLQ_HOST_QIDX;

	cp->kcq1.sw_prod_idx = 0;
	cp->kcq1.hw_prod_idx_ptr =
		(vmk_uint16 *) &sblk->status_completion_producer_index;

	cp->kcq1.status_idx_ptr = (vmk_uint16 *) &sblk->status_idx;

	/* Initialize the kernel complete queue context. */
	val = KRNLQ_TYPE_TYPE_KRNLQ | KRNLQ_SIZE_TYPE_SIZE |
	      (BNX2_PAGE_BITS - 8) | KRNLQ_FLAGS_QE_SELF_SEQ;
	cnic_ctx_wr(dev, cp->kcq_cid_addr, L5_KRNLQ_TYPE, val);

	val = (BNX2_PAGE_SIZE / sizeof(struct kcqe) - 1) << 16;
	cnic_ctx_wr(dev, cp->kcq_cid_addr, L5_KRNLQ_QE_SELF_SEQ_MAX, val);

	val = ((BNX2_PAGE_SIZE / sizeof(struct kcqe)) << 16) | KCQ_PAGE_CNT;
	cnic_ctx_wr(dev, cp->kcq_cid_addr, L5_KRNLQ_PGTBL_NPAGES, val);

	val = (vmk_uint32) ((vmk_uint64) cp->kcq1.dma.pgtbl_map >> 32);
	cnic_ctx_wr(dev, cp->kcq_cid_addr, L5_KRNLQ_PGTBL_HADDR_HI, val);

	val = (vmk_uint32) cp->kcq1.dma.pgtbl_map;
	cnic_ctx_wr(dev, cp->kcq_cid_addr, L5_KRNLQ_PGTBL_HADDR_LO, val);

	cp->int_num = 0;
	if (ethdev->driverState & CNIC_DRV_STATE_USING_MSIX) {
		struct status_block_msix *msblk = cp->cfp[CNIC_RX_IDX].status_blk.bnx2;
		vmk_uint32 sb_id = cp->cfp[CNIC_RX_IDX].status_blk_num;
		vmk_uint32 sb = BNX2_L2CTX_L5_STATUSB_NUM(sb_id);

		cp->kcq1.hw_prod_idx_ptr =
			(vmk_uint16 *) &msblk->status_completion_producer_index;
		cp->kcq1.status_idx_ptr = (vmk_uint16 *) &msblk->status_idx;
		cp->kwq_con_idx_ptr = (vmk_uint16 *) &msblk->status_cmd_consumer_index;
		cp->int_num = sb_id << BNX2_PCICFG_INT_ACK_CMD_INT_NUM_SHIFT;
		cnic_ctx_wr(dev, cp->kwq_cid_addr, L5_KRNLQ_HOST_QIDX, sb);
		cnic_ctx_wr(dev, cp->kcq_cid_addr, L5_KRNLQ_HOST_QIDX, sb);
	}

	/* Enable Commnad Scheduler notification when we write to the
	 * host producer index of the kernel contexts. */
	CNIC_WR(dev, BNX2_MQ_KNL_CMD_MASK1, 2);

	/* Enable Command Scheduler notification when we write to either
	 * the Send Queue or Receive Queue producer indexes of the kernel
	 * bypass contexts. */
	CNIC_WR(dev, BNX2_MQ_KNL_BYP_CMD_MASK1, 7);
	CNIC_WR(dev, BNX2_MQ_KNL_BYP_WRITE_MASK1, 7);

	/* Notify COM when the driver post an application buffer. */
	CNIC_WR(dev, BNX2_MQ_KNL_RX_V2P_MASK2, 0x2000);

	/* Set the CP and COM doorbells.  These two processors polls the
	 * doorbell for a non zero value before running.  This must be done
	 * after setting up the kernel queue contexts. */
	val = cnic_reg_rd_ind(dev, BNX2_CP_SCRATCH + 0x20);
	cnic_reg_wr_ind(dev, BNX2_CP_SCRATCH + 0x20, val | 1);

	val = cnic_reg_rd_ind(dev, BNX2_COM_SCRATCH + 0x20);
	cnic_reg_wr_ind(dev, BNX2_COM_SCRATCH + 0x20, val | 1);

	err = cnic_init_bnx2_irq(dev);
	if (err) {
		CNIC_ERR(dev, "cnic_init_irq failed: err:0x%x", err);

		val = cnic_reg_rd_ind(dev, BNX2_CP_SCRATCH + 0x20);
		cnic_reg_wr_ind(dev, BNX2_CP_SCRATCH + 0x20, val & ~0x1);

		val = cnic_reg_rd_ind(dev, BNX2_COM_SCRATCH + 0x20);
		cnic_reg_wr_ind(dev, BNX2_COM_SCRATCH + 0x20, val & ~0x1);

		return err;
	}

	ethdev->driverState |= CNIC_DRV_STATE_HANDLES_IRQ;
#if (CNIC_ISCSI_OOO_SUPPORT)
	cnic_start_bnx2_ooo_hw(dev);
#endif
	return 0;
}

static void cnic_setup_qfle3_context(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	vmk_uint32 start_offset = ethdev->ctx_tbl_offset;
	int i;

	for (i = 0; i < cp->ctx_blks; i++) {
		struct cnic_ctx *ctx = &cp->ctx_arr[i];
		vmk_IOA map = ctx->mapping;

		if (cp->ctx_align) {
			unsigned long mask = cp->ctx_align - 1;

			map = (map + mask) & ~mask;
		}

		cnic_ctx_tbl_wr(dev, start_offset + i, map);
	}
}

static int cnic_init_qfle3_irq(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	int err = 0;

	cp->cnic_service_bh = cnic_service_qfle3_bh;

	//ql_vmk_tasklet_init(&cp->cnic_irq_task, cnic_service_qfle3_bh, (void *)dev);
	if (ethdev->driverState & CNIC_DRV_STATE_USING_MSIX)
		err = cnic_request_irq(dev);

	return err;
}

static inline void cnic_storm_memset_hc_disable(struct cnic_dev *dev,
	vmk_uint16 sb_id, vmk_uint8 sb_index, vmk_uint8 disable)
{
	struct qfle3_adapter *adapter = NULL;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	vmk_uint32 addr = BAR_CSTRORM_INTMEM +
			CSTORM_STATUS_BLOCK_DATA_OFFSET(sb_id) +
			vmk_offsetof(struct hc_status_block_data_e1x, index_data) +
			sizeof(struct hc_index_data)*sb_index +
			vmk_offsetof(struct hc_index_data, flags);
	vmk_uint16 flags = CNIC_RD16(dev, addr);
	/* clear and set */
	flags &= ~HC_INDEX_DATA_HC_ENABLED;
	flags |= (((~disable) << HC_INDEX_DATA_HC_ENABLED_SHIFT) &
		  HC_INDEX_DATA_HC_ENABLED);
	CNIC_WR16(dev, addr, flags);
}

static void cnic_enable_qfle3_int(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	vmk_uint8 sb_id = cp->cfp[CNIC_RX_IDX].status_blk_num;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	CNIC_WR8(dev, BAR_CSTRORM_INTMEM +
			CSTORM_STATUS_BLOCK_DATA_OFFSET(sb_id) +
			vmk_offsetof(struct hc_status_block_data_e1x, index_data) +
			sizeof(struct hc_index_data)*HC_INDEX_ISCSI_EQ_CONS +
			vmk_offsetof(struct hc_index_data, timeout), 64 / 4);
	cnic_storm_memset_hc_disable(dev, sb_id, HC_INDEX_ISCSI_EQ_CONS, 0);
}

static void cnic_disable_qfle3_int_sync(struct cnic_dev *dev)
{
}


static void cnic_init_qfle3_kcq(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	vmk_uint32 pfid;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);
	pfid = adapter->pf_id;

	cp->kcq1.io_addr = BAR_CSTRORM_INTMEM +
			   CSTORM_ISCSI_EQ_PROD_OFFSET(pfid, 0);
	cp->kcq1.sw_prod_idx = 0;

	if (QFLE3_CHIP_IS_E2_PLUS(adapter)) {
		struct host_hc_status_block_e2 *sb =
			cp->cfp[CNIC_RX_IDX].status_blk.gen;

		cp->kcq1.hw_prod_idx_ptr =
			&sb->sb.index_values[HC_INDEX_ISCSI_EQ_CONS];
		cp->kcq1.status_idx_ptr =
			&sb->sb.running_index[SM_RX_ID];
	} else {
		struct host_hc_status_block_e1x *sb =
			cp->cfp[CNIC_RX_IDX].status_blk.gen;

		cp->kcq1.hw_prod_idx_ptr =
			&sb->sb.index_values[HC_INDEX_ISCSI_EQ_CONS];
		cp->kcq1.status_idx_ptr =
			&sb->sb.running_index[SM_RX_ID];
	}

	if (QFLE3_CHIP_IS_E2_PLUS(adapter)) {
		struct host_hc_status_block_e2 *sb =
			cp->cfp[CNIC_RX_IDX].status_blk.gen;

		cp->kcq2.io_addr = BAR_USTRORM_INTMEM +
					USTORM_FCOE_EQ_PROD_OFFSET(pfid);
		cp->kcq2.sw_prod_idx = 0;
		cp->kcq2.hw_prod_idx_ptr =
			&sb->sb.index_values[HC_INDEX_FCOE_EQ_CONS];
		cp->kcq2.status_idx_ptr =
			&sb->sb.running_index[SM_RX_ID];
	}
}

static int cnic_start_qfle3_hw(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	int func = CNIC_FUNC(cp), ret;
	vmk_uint32 pfid;
	vmk_uint16 eq_idx;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	dev->stats_addr = ethdev->addr_drv_info_to_mcp;
	cp->func = adapter->pf_num;

	func = CNIC_FUNC(cp);
	pfid = adapter->pf_id;

	ret = cnic_init_id_tbl(&cp->cid_tbl, MAX_ISCSI_TBL_SZ,
			cp->iscsi_start_cid);

	if (VMK_UNLIKELY(ret)) {
		CNIC_ERR(dev, "Couldn't allocate iSCSI CID table: ret=0x%x", ret);
		return VMK_NO_MEMORY;
	}

	if (QFLE3_CHIP_IS_E2_PLUS(adapter)) {
		CNIC_INFO(dev, CNIC_MSG_DRV,
			"Init fcoe_cid_tbl: max_fcoe_conn=0x%x fcoe_start_cid=0x%x",
			dev->max_fcoe_conn, cp->fcoe_start_cid);
		ret = cnic_init_id_tbl(&cp->fcoe_cid_tbl,
				dev->max_fcoe_conn,
				cp->fcoe_start_cid);

		if (VMK_UNLIKELY(ret)) {
			CNIC_ERR(dev, "Couldn't allocate FCoE CID table: ret=0x%x", ret);
			return VMK_NO_MEMORY;
		}
	}

	cnic_init_qfle3_kcq(dev);

	/* Only 1 EQ */
	CNIC_WR16(dev, cp->kcq1.io_addr, MAX_KCQ_IDX);
	CNIC_WR(dev, BAR_CSTRORM_INTMEM +
		CSTORM_ISCSI_EQ_CONS_OFFSET(pfid, 0), 0);
	CNIC_WR(dev, BAR_CSTRORM_INTMEM +
		CSTORM_ISCSI_EQ_NEXT_PAGE_ADDR_OFFSET(pfid, 0),
		cp->kcq1.dma.pg_map_arr[1] & 0xffffffff);
	CNIC_WR(dev, BAR_CSTRORM_INTMEM +
		CSTORM_ISCSI_EQ_NEXT_PAGE_ADDR_OFFSET(pfid, 0) + 4,
		(vmk_uint64) cp->kcq1.dma.pg_map_arr[1] >> 32);
	CNIC_WR(dev, BAR_CSTRORM_INTMEM +
		CSTORM_ISCSI_EQ_NEXT_EQE_ADDR_OFFSET(pfid, 0),
		cp->kcq1.dma.pg_map_arr[0] & 0xffffffff);
	CNIC_WR(dev, BAR_CSTRORM_INTMEM +
		CSTORM_ISCSI_EQ_NEXT_EQE_ADDR_OFFSET(pfid, 0) + 4,
		(vmk_uint64) cp->kcq1.dma.pg_map_arr[0] >> 32);
	CNIC_WR8(dev, BAR_CSTRORM_INTMEM +
		CSTORM_ISCSI_EQ_NEXT_PAGE_ADDR_VALID_OFFSET(pfid, 0), 1);
	CNIC_WR16(dev, BAR_CSTRORM_INTMEM +
		CSTORM_ISCSI_EQ_SB_NUM_OFFSET(pfid, 0),
		cp->cfp[CNIC_RX_IDX].status_blk_num);
	CNIC_WR8(dev, BAR_CSTRORM_INTMEM +
		CSTORM_ISCSI_EQ_SB_INDEX_OFFSET(pfid, 0),
		HC_INDEX_ISCSI_EQ_CONS);

	CNIC_WR(dev, BAR_USTRORM_INTMEM +
		USTORM_ISCSI_GLOBAL_BUF_PHYS_ADDR_OFFSET(pfid),
		cp->gbl_buf_info.pg_map_arr[0] & 0xffffffff);
	CNIC_WR(dev, BAR_USTRORM_INTMEM +
		USTORM_ISCSI_GLOBAL_BUF_PHYS_ADDR_OFFSET(pfid) + 4,
		(vmk_uint64) cp->gbl_buf_info.pg_map_arr[0] >> 32);

	CNIC_WR(dev, BAR_TSTRORM_INTMEM +
		TSTORM_ISCSI_TCP_LOCAL_ADV_WND_OFFSET(pfid), DEF_RCV_BUF);

	cnic_setup_qfle3_context(dev);

	eq_idx = 0;
	if (VMK_UNLIKELY(eq_idx != 0)) {
		CNIC_ERR(dev, "EQ cons index %x != 0", eq_idx);
		return VMK_BUSY;
	}

#if (CNIC_ISCSI_OOO_SUPPORT)
	cnic_start_qfle3_ooo_hw(dev);
#endif
	ret = cnic_init_qfle3_irq(dev);
	if (VMK_UNLIKELY(ret)) {
		CNIC_ERR(dev, "Failed to request IRQ: ret=0x%x", ret);
		return ret;
	}

	ethdev->driverState |= CNIC_DRV_STATE_HANDLES_IRQ;

	return 0;
}

static int cnic_start_hw(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_eth_dev *ethdev = cp->ethdev;
	struct qfle3_adapter *adapter = NULL;
	int err, i = 0;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	CNIC_INFO(dev, CNIC_MSG_DRV, "Entered.");

	if (vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP))
		return VMK_EALREADY;

	cp->func = dev->pf_id;

	/* SV: TODO: modularize this. */
	//cp->cfp[0].fp = &adapter->fp[adapter->num_ethqs];
	cp->cfp[0].status_blk.gen = ethdev->irq_arr[0].status_blk;
	cp->cfp[0].status_blk_num = ethdev->irq_arr[0].status_blk_num;
	cp->cfp[0].qfle3_igu_sb_id = ethdev->irq_arr[0].status_blk_num2;

	cnic_init_pci(dev, (void *)ethdev->reg_base, (void *)ethdev->db_base);

	CNIC_INFO(dev, CNIC_MSG_DRV, "reg-base=%p regview=%p \n"
		"db_base=%p doorbells=%p",
		ethdev->reg_base, dev->regview, ethdev->db_base, dev->doorbells);

	err = cp->alloc_resc(dev);
	if (VMK_UNLIKELY(err)) {
		CNIC_ERR(dev, "allocate resource failure: err=0x%x", err);
		goto err1;
	}

	err = cp->start_hw(dev);
	if (VMK_UNLIKELY(err)) {
		CNIC_ERR(dev, "start HW failure: err=0x%x", err);
		goto err1;
	}

	vmk_BitVectorSet(cp->cnic_local_flags, CNIC_LCL_FL_HW_START);

	err = cnic_cm_open(dev);
	if (VMK_UNLIKELY(err)) {
		CNIC_ERR(dev, "CM open failure: err=0x%x", err);
		goto err1;
	}

	vmk_BitVectorSet(dev->flags, CNIC_F_CNIC_UP);

	//SV: XXX: safe code:
	cp->enable_int(dev);
	CNIC_INFO(dev, CNIC_MSG_DRV, "exiting.");

	return 0;

err1:
	//SV: XXX: safe code
	//cp->free_resc(dev);
	return err;
}

static void cnic_stop_bnx2_hw(struct cnic_dev *dev)
{
	vmk_uint32 val;

	cnic_disable_bnx2_int_sync(dev);

	val = cnic_reg_rd_ind(dev, BNX2_CP_SCRATCH + 0x20);
	cnic_reg_wr_ind(dev, BNX2_CP_SCRATCH + 0x20, val & ~0x1);

	val = cnic_reg_rd_ind(dev, BNX2_COM_SCRATCH + 0x20);
	cnic_reg_wr_ind(dev, BNX2_COM_SCRATCH + 0x20, val & ~0x1);

	cnic_init_context(dev, KWQ_CID);
	cnic_init_context(dev, KCQ_CID);

	cnic_setup_5709_context(dev, 0);
	cnic_free_irq(dev);

	cnic_free_resc(dev);
}

static void cnic_stop_qfle3_hw(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	vmk_uint32 hc_index = HC_INDEX_ISCSI_EQ_CONS;
	vmk_uint32 sb_id = cp->cfp[CNIC_RX_IDX].status_blk_num;
	vmk_uint32 idx_off, syn_off;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (cp->ethdev->mf_mode == MULTI_FUNCTION_SI ||
	    cp->ethdev->mf_mode == MULTI_FUNCTION_AFEX)
		cnic_npar_ring_ctl(dev, 0);

	if (QFLE3_CHIP_IS_E2_PLUS(adapter))
		cnic_free_id_tbl(&cp->fcoe_cid_tbl);
	cnic_free_id_tbl(&cp->cid_tbl);

	cnic_free_irq(dev);

	if (QFLE3_CHIP_IS_E2_PLUS(adapter)) {
		idx_off = vmk_offsetof(struct hc_status_block_e2, index_values) +
			  (hc_index * sizeof(vmk_uint16));

		syn_off = CSTORM_HC_SYNC_LINE_INDEX_E2_OFFSET(hc_index, sb_id);
	} else {
		idx_off = vmk_offsetof(struct hc_status_block_e1x, index_values) +
			  (hc_index * sizeof(vmk_uint16));

		syn_off = CSTORM_HC_SYNC_LINE_INDEX_E1X_OFFSET(hc_index, sb_id);
	}
	CNIC_WR16(dev, BAR_CSTRORM_INTMEM + syn_off, 0);
	CNIC_WR16(dev, BAR_CSTRORM_INTMEM + CSTORM_STATUS_BLOCK_OFFSET(sb_id) +
		  idx_off, 0);

	*cp->kcq1.hw_prod_idx_ptr = 0;
	CNIC_WR(dev, BAR_CSTRORM_INTMEM +
		CSTORM_ISCSI_EQ_CONS_OFFSET(adapter->pf_id, 0), 0);
	CNIC_WR16(dev, cp->kcq1.io_addr, 0);
//	cnic_free_resc(dev);
}

static void cnic_stop_hw(struct cnic_dev *dev)
{
	if (vmk_BitVectorTest(dev->flags, CNIC_F_CNIC_UP)) {
		struct cnic_local *cp = dev->cnic_priv;

#if (CNIC_ISCSI_OOO_SUPPORT)
		/* Must stop the ooo engine before freeing kwqe resources */
		cp->stop_ooo_hw(dev);
#endif
		cp->disable_int_sync(dev);
		cp->stop_cm(dev);
		cp->ethdev->driverState &= ~CNIC_DRV_STATE_HANDLES_IRQ;
		vmk_BitVectorClear(dev->flags, CNIC_F_CNIC_UP);
		//SV: XXX: Check if locking needed?
		cp->ulp_ops[CNIC_ULP_L4] = NULL;
		cnic_cm_shutdown(dev);
		cp->stop_hw(dev);
		vmk_BitVectorClear(cp->cnic_local_flags, CNIC_LCL_FL_HW_START);
		cp->free_resc(dev);
		cnic_release_pci(dev);
	}
}

static struct cnic_dev *cnic_alloc_dev(vmk_Device ldev,
	vmk_Device ndev)
{
	struct cnic_dev *cdev;
	struct cnic_local *cp;
	struct qfle3_adapter *adapter = NULL;
	int alloc_size, i;

	vmk_DeviceGetAttachedDriverData(ndev, (vmk_AddrCookie *) &adapter);

	alloc_size = sizeof(struct cnic_dev) + sizeof(struct cnic_local);

	cdev = ql_vmk_alloc(alloc_size);
	if (VMK_UNLIKELY(cdev == NULL)) {
		DRV_ERR("cdev allocation failure.");
		return NULL;
	}

	cdev->version = CNIC_DEV_VER;
	cdev->netdev = ndev;
	cdev->ldev = ldev;
	cdev->cnic_priv = (char *)cdev + sizeof(struct cnic_dev);
	cdev->register_device = cnic_register_device;
	cdev->unregister_device = cnic_unregister_device;
	cdev->drv_get_sfp_diagnostic = cnic_get_sfp_diagnostic;
	cdev->drv_get_link_error = cnic_get_link_error;
	cdev->dump_queues = cnic_dump_queues;

	cdev->flags = vmk_BitVectorAlloc(cnic_driver_info.heap_id,
			(sizeof(long) * VMK_BITS_PER_BYTE));
	vmk_BitVectorZap(cdev->flags);

	cp = cdev->cnic_priv;
	cp->dev = cdev;
	cp->cnic_local_flags = vmk_BitVectorAlloc(cnic_driver_info.heap_id,
			(sizeof(long) * VMK_BITS_PER_BYTE));
	vmk_BitVectorZap(cp->cnic_local_flags);
	for (i = 0; i < MAX_CNIC_ULP_TYPE; i++) {
		cp->ulp_flags[i] = vmk_BitVectorAlloc(cnic_driver_info.heap_id,
				(sizeof(long) * VMK_BITS_PER_BYTE));
		vmk_BitVectorZap(cp->ulp_flags[i]);
	}

	ql_vmk_spin_lock_init(&cp->cnic_ulp_lock, LOCK_RANK_HIGHEST);
	DRV_INFO(CNIC_MSG_DRV, "Added CNIC device.");

	return cdev;
}

static struct cnic_dev *init_bnx2_cnic(vmk_Device ldev)
{
	struct cnic_dev *cdev;
	struct cnic_local *cp;
	struct qfle3_adapter *adapter = NULL;
	struct cnic_eth_dev *ethdev = NULL;
	struct nicOps *nic_ops = NULL;
	vmk_Device ndev;


	/* SV: TODO: Get global info from corresponding nic pci function.
	 * 		Get: cnic_eth_dev from nic.
	 * 		XXX: Move this to init_[qfle3/bnx2]_cnic
	 */
	/* Get the nicOps. */
	vmk_DeviceGetRegistrationData(ldev, (vmk_AddrCookie *) &nic_ops);
	/* Get the qfle3_adapter. */
	vmk_DeviceGetRegisteringDriverData(ldev, (vmk_AddrCookie *) &adapter);

	ethdev = nic_ops->cnicProbeNic(adapter);
	if (!ethdev)
		return NULL;

	if (ethdev->apiRevision != CNIC_ETH_DEV_VER) {
		DRV_WARN("qfle3 not compatible with cnic. expecting: 0x%x got: 0x%x",
			CNIC_ETH_DEV_VER, ethdev->apiRevision);
		return NULL;
	}

	ndev = ethdev->device;
	//vmk_DeviceGetAttachedDriverData(ndev, (vmk_AddrCookie *) &adapter);

	/* SV: XXX: correct naming and make global later. */
#ifndef PCI_DEVICE_ID_NX2_5709
#define PCI_DEVICE_ID_NX2_5709          0x1639
#endif

#ifndef PCI_DEVICE_ID_NX2_5709S
#define PCI_DEVICE_ID_NX2_5709S         0x163a
#endif

	if ((ethdev->chip_id >> 16) == PCI_DEVICE_ID_NX2_5709 ||
	    (ethdev->chip_id >> 16) == PCI_DEVICE_ID_NX2_5709S) {
		if ((ethdev->chip_id & CHIP_REV_MASK) < 0x10)
			goto cnic_err; //SV: XXX: Something wrong here!
	}

	cdev = cnic_alloc_dev(ldev, ndev);
	if (VMK_UNLIKELY(cdev == NULL))
		goto cnic_err;

	vmk_BitVectorSet(cdev->flags, CNIC_F_QFLE3_LEGACY_CLASS);
	cdev->submit_kwqes = cnic_submit_bnx2_kwqes;

	cp = cdev->cnic_priv;
	cp->ethdev = ethdev;
	cdev->pdev = adapter->pdev;
	cdev->chip_id = ethdev->chip_id;
	cdev->link_supported_speeds = ethdev->link_supported_speeds;

	cdev->max_iscsi_conn = ethdev->max_iscsi_conn;
	CNIC_INFO(cdev, CNIC_MSG_DRV,
		"Num 1G iSCSI licenses = %d", cdev->max_iscsi_conn);

	cp->cnic_ops = &cnic_bnx2_ops;
	cp->start_hw = cnic_start_bnx2_hw;
	cp->stop_hw = cnic_stop_bnx2_hw;
	cp->setup_pgtbl = cnic_setup_page_tbl;
	cp->alloc_resc = cnic_alloc_bnx2_resc;
	cp->free_resc = cnic_free_resc;
	cp->start_cm = cnic_cm_init_bnx2_hw;
	cp->stop_cm = cnic_cm_stop_bnx2_hw;
	cp->enable_int = cnic_enable_bnx2_int;
	cp->disable_int_sync = cnic_disable_bnx2_int_sync;
	cp->close_conn = cnic_close_bnx2_conn;

#if (CNIC_ISCSI_OOO_SUPPORT)
	cp->stop_ooo_hw = cnic_stop_bnx2_ooo_hw;
#endif
	return cdev;

cnic_err:
	return NULL;
}

static void cnic_print_device_probe_info(struct cnic_dev *dev,
	struct cnic_eth_dev *ethdev)
{
	CNIC_INFO(dev, CNIC_MSG_DRV,
		"version 0x%x\n driverState 0x%x, chip_id 0x%x, "
		"ctx_tbl_offset 0x%x, ctx_tbl_len 0x%x",
		ethdev->apiRevision, ethdev->driverState, ethdev->chip_id,
		ethdev->ctx_tbl_offset, ethdev->ctx_tbl_len);
	CNIC_INFO(dev, CNIC_MSG_DRV,"ctx_blk_size %d,\n "
		"starting_cid 0x%x, max_iscsi_conn %d, max_fcoe_conn %d,\n "
		"fcoe_init_cid 0x%x, iscsi_l2_client_id 0x%x",
		ethdev->ctx_blk_size, ethdev->starting_cid,
		ethdev->max_iscsi_conn, ethdev->max_fcoe_conn,
		ethdev->fcoe_init_cid, ethdev->iscsi_l2_client_id);
}

static struct cnic_dev *init_qfle3_cnic(vmk_Device ldev)
{
	struct cnic_dev *cdev;
	struct cnic_local *cp;
	struct cnic_eth_dev *ethdev = NULL;
	struct qfle3_adapter *adapter = NULL;
	struct nicOps *nic_ops = NULL;
	vmk_Device ndev;

	/* Get the cnicOps. */
	vmk_DeviceGetRegistrationData(ldev, (vmk_AddrCookie *) &nic_ops);
	/* Get the qfle3_adapter. */
	vmk_DeviceGetRegisteringDriverData(ldev, (vmk_AddrCookie *) &adapter);

	ethdev = nic_ops->cnicProbeNic(adapter);
	if (!ethdev)
		return NULL;

	if (ethdev->apiRevision != CNIC_ETH_DEV_VER) {
		DRV_WARN("qfle3 not compatible with cnic "
			"expecting: 0x%x got: 0x%x",
			CNIC_ETH_DEV_VER, ethdev->apiRevision);
		return NULL;
	}

	ndev = ethdev->device;

	//SV: XXX: check alternative: dev_hold(ldev);
	cdev = cnic_alloc_dev(ldev, ndev);
	if (cdev == NULL) {
		//SV: XXX: check alternative: dev_put(ldev);
		DRV_WARN("cdev allocation failure.");
		return NULL;
	}

	cnic_print_device_probe_info(cdev, ethdev);

	vmk_BitVectorSet(cdev->flags, CNIC_F_QFLE3_CLASS);
	cdev->submit_kwqes = cnic_submit_qfle3_kwqes;

	cdev->pf_num = adapter->pf_num;
	cdev->pf_id = adapter->pf_id;

	cp = cdev->cnic_priv;
	cp->ethdev = ethdev;
	cdev->pdev = adapter->pdev;
	cdev->mtu = adapter->mtu;
	cdev->uplinkSharedData = &adapter->uplinkSharedData;
	vmk_NameCopy(&cdev->name, &ethdev->pdev_name);
	cdev->mf_mode     = ethdev->mf_mode;
	//cdev->mf_sub_mode = adapter->mf_sub_mode;
	cdev->e1hov_tag = ethdev->e1hov_tag;
	cdev->cna_vlgrp = ethdev->cna_vlgrp;
	cdev->fcoe_wwpn = (((vmk_uint64)ethdev->fcoe_wwn_port_name_hi << 32) |
			ethdev->fcoe_wwn_port_name_lo);
	cdev->fcoe_wwnn = (((vmk_uint64)ethdev->fcoe_wwn_node_name_hi << 32) |
			ethdev->fcoe_wwn_node_name_lo);
	cdev->chip_id = ethdev->chip_id;
	cdev->link_supported_speeds = ethdev->link_supported_speeds;
	ql_vmk_init_singlethread_delayed_workitem(&cp->delete_task);

	cdev->stats_addr = ethdev->addr_drv_info_to_mcp;

	if (!(ethdev->driverState & CNIC_DRV_STATE_NO_ISCSI)) {
		cdev->max_iscsi_conn = ethdev->max_iscsi_conn;
		CNIC_INFO(cdev, CNIC_MSG_DRV, "Num 10G iSCSI licenses = %d",
			cdev->max_iscsi_conn);
	}
	if (QFLE3_CHIP_IS_E2_PLUS(adapter) &&
	    !(ethdev->driverState & CNIC_DRV_STATE_NO_FCOE))
		cdev->max_fcoe_conn = ethdev->max_fcoe_conn;

	if (cdev->max_fcoe_conn > QFLE3_FCOE_NUM_CONNECTIONS)
		cdev->max_fcoe_conn = QFLE3_FCOE_NUM_CONNECTIONS;

	vmk_Memcpy(cdev->nic_mac, adapter->hwMacAddr, VMK_ETH_ADDR_LENGTH);
	vmk_Memcpy(cdev->iscsi_mac, adapter->iscsi_mac, VMK_ETH_ADDR_LENGTH);
	vmk_Memcpy(cdev->fip_mac, adapter->fip_mac, VMK_ETH_ADDR_LENGTH);

	cp->cnic_ops = &cnic_qfle3_ops;
	cp->nic_ops = nic_ops;
	cp->start_hw = cnic_start_qfle3_hw;
	cp->stop_hw = cnic_stop_qfle3_hw;
	cp->setup_pgtbl = cnic_setup_page_tbl_le;
	cp->alloc_resc = cnic_alloc_qfle3_resc;
	cp->free_resc = cnic_free_resc;
	cp->start_cm = cnic_cm_init_qfle3_hw;
	cp->stop_cm = cnic_cm_stop_qfle3_hw;
	cp->enable_int = cnic_enable_qfle3_int;
	cp->disable_int_sync = cnic_disable_qfle3_int_sync;
	if (QFLE3_CHIP_IS_E2_PLUS(adapter))
		cp->ack_int = cnic_ack_qfle3_e2_msix;
	else
		cp->ack_int = cnic_ack_qfle3_msix;
	cp->close_conn = cnic_close_qfle3_conn;
#if (CNIC_ISCSI_OOO_SUPPORT)
	cp->stop_ooo_hw = cnic_stop_qfle3_ooo_hw;
#endif

	return cdev;
}

vmk_DMAConstraints cnic_dma_constraints = {
	.sgMaxEntries     = 256,
	/* Required scatter/gather element alignment */
	.sgElemAlignment  = 0,
	/* If non-zero, scatter/gather element length multiple */
	.sgElemSizeMult   = 0,
	.sgElemMaxSize    = 0xffffffff,
	/* Max # of blocks per i/o */
	.maxTransfer      = 0xffffffff,
	/* Address mask to get boundary limit for scatter/gather
	 * element */
	.sgElemStraddle   = VMK_ADDRESS_MASK_32BIT+1,
	.addressMask      = VMK_ADDRESS_MASK_64BIT,
};

static VMK_ReturnStatus
cnic_create_dma_engines(struct cnic_dev *cdev)
{
	VMK_ReturnStatus status;
	vmk_DMAEngineProps props;

	/* DMA engine for dma mapping. */
	vmk_NameFormat(&props.name, "cnic-%s", vmk_NameToString(&cdev->name));
	props.module = vmk_ModuleCurrentID;
	props.device = cdev->ldev;
	props.constraints = &cnic_dma_constraints;
	props.bounce = NULL;
	props.flags = VMK_DMA_ENGINE_FLAGS_COHERENT;

	status = vmk_DMAEngineCreate(&props, &cdev->dmaEngineCoherent);
	if (status != VMK_OK) {
		CNIC_ERR(cdev, "Failed to create DMA engine (%x)", status);
		goto fail_engine;
	}

	return VMK_OK;

fail_engine:
	return status;
}

static inline void
cnic_destroy_dma_engines(struct cnic_dev *cdev)
{
	vmk_DMAEngineDestroy(cdev->dmaEngineCoherent);
}

void cnic_fcoe_rx_work(long unsigned int cookie)
{
	struct cnic_dev *cdev = (struct cnic_dev *)cookie;
	vmk_PktHandle *pkt = NULL;

	CNIC_INFO(cdev, CNIC_MSG_NW, "before rx-pkt work.");

	if (vmk_SpinlockLock(cdev->ll2_cb_info.ll2_cb_lck) == VMK_DEATH_PENDING)
		return;
	if (!cdev->ll2_cb_info.ll2_cb_ops ||
		!cdev->ll2_cb_info.ll2_cb_ops->rx_cb) {
		vmk_SpinlockUnlock(cdev->ll2_cb_info.ll2_cb_lck);
		CNIC_ERR(cdev, "Error: ll2-cb-ops not registered!");
		return;
	}
	vmk_SpinlockUnlock(cdev->ll2_cb_info.ll2_cb_lck);

	if(vmk_SpinlockLock(cdev->fcoe_rx_list_lck) == VMK_DEATH_PENDING)
		return;
	while (!(vmk_PktListIsEmpty(cdev->fcoe_rx_list))) {
		pkt = vmk_PktListPopFirstPkt(cdev->fcoe_rx_list);
		vmk_SpinlockUnlock(cdev->fcoe_rx_list_lck);

		CNIC_INFO(cdev, CNIC_MSG_NW, "rx-pkt: %p", pkt);
		/* Submit LL2 pkt to ULP. */
		if (vmk_SpinlockLock(cdev->ll2_cb_info.ll2_cb_lck) == VMK_DEATH_PENDING)
			return;
		if (cdev->ll2_cb_info.ll2_cb_ops &&
			cdev->ll2_cb_info.ll2_cb_ops->rx_cb) {
			cdev->ll2_cb_info.ll2_cb_ops->rx_cb(cdev, pkt);
		} else {
			CNIC_ERR(cdev, "Error: ll2-cb-ops not registered!");
			vmk_SpinlockUnlock(cdev->ll2_cb_info.ll2_cb_lck);

			/* Drop the LL2 pkt. */
			vmk_PktRelease(pkt);

			return;
		}
		vmk_SpinlockUnlock(cdev->ll2_cb_info.ll2_cb_lck);

		if(vmk_SpinlockLock(cdev->fcoe_rx_list_lck) == VMK_DEATH_PENDING)
			return;
	}
	vmk_SpinlockUnlock(cdev->fcoe_rx_list_lck);

	CNIC_INFO(cdev, CNIC_MSG_NW, "after rx-pkt work.");
}

static struct cnic_dev *
is_cnic_dev(vmk_Device ldev, vmk_uint8 is_e3)
{
	struct cnic_dev *cdev = NULL;
	vmk_SpinlockCreateProps lock_props;
	VMK_ReturnStatus status = VMK_OK;

	if (is_e3 == QFLE3_DRIVER)
		cdev = init_qfle3_cnic(ldev);
	else if (is_e3 == QFLE3_LEGACY_DRIVER)
		cdev = init_bnx2_cnic(ldev);

	if (cdev) {
		/* SV: XXX: Need a better place for this:
		 * setup memory for comunicating with hardware. */
		status = cnic_create_dma_engines(cdev);
		if (status != VMK_OK)
			return NULL;

		/* SV: XXX: Find a placeholder for fcoe-rx-list init. */
		/************************************************/
		/* Create fcoe_rx_list lock. */
		lock_props.moduleID = vmk_ModuleCurrentID;
		lock_props.heapID = cnic_driver_info.heap_id;
		lock_props.type = VMK_SPINLOCK;
		lock_props.domain = cnic_driver_info.lock_domain;
		lock_props.rank = CNIC_LOW_LOCK_RANK;
		vmk_NameInitialize(&lock_props.name, "fcoe_rx_list_lck");

		status = vmk_SpinlockCreate(&lock_props, &cdev->fcoe_rx_list_lck);
		if (status != VMK_OK) {
			CNIC_ERR(cdev, "Failed to create fcoe_rx_list lock.");
			return NULL;
		}

		/* Create cb lock. */
		lock_props.moduleID = vmk_ModuleCurrentID;
		lock_props.heapID = cnic_driver_info.heap_id;
		lock_props.type = VMK_SPINLOCK;
		lock_props.domain = cnic_driver_info.lock_domain;
		lock_props.rank = CNIC_HIGH_LOCK_RANK;
		vmk_NameInitialize(&lock_props.name, "ll2_cb_lck");

		status = vmk_SpinlockCreate(&lock_props, &cdev->ll2_cb_info.ll2_cb_lck);
		if (status != VMK_OK) {
			CNIC_ERR(cdev, "Failed to create ll2_cb lock.");
			return NULL;
		}

		/* Initialize fcoe_rx_list. */
		if (vmk_PktListAlloc(&cdev->fcoe_rx_list) == VMK_OK)
			vmk_PktListInit(cdev->fcoe_rx_list);

		/* Initialize fcoe_rx_dpc. */
		ql_vmk_tasklet_init(&cdev->fcoe_rx_dpc, cnic_fcoe_rx_work,
			(void *)cdev);
		/************************************************/

		vmk_SemaLock(&cnic_driver_info.cnic_lock);
		vmk_ListInsert(&cdev->list,
			vmk_ListAtFront(&cnic_driver_info.cnic_dev_list));
		vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
	}

	return cdev;
}

#if 0
//SV: XXX: Check if we still need this.
/*
 * netdev event handler
 */
static int cnic_netdev_event_esx(struct notifier_block *this,
	unsigned long event, void *ptr)
{
	vmk_Device *netdev = ptr;
	struct cnic_dev *dev;
	int if_type;

	dev = cnic_from_netdev(netdev);

	if (dev) {
		struct cnic_local *cp = dev->cnic_priv;

		for (if_type = 0; if_type < MAX_CNIC_ULP_TYPE; if_type++) {
			struct cnic_ulp_ops *ulp_ops;
			void *ctx;

			vmk_SemaLock(&cnic_driver_info.cnic_lock);
			ulp_ops = cp->ulp_ops[if_type];
			if (!ulp_ops || !ulp_ops->indicate_netevent)
				continue;

			ctx = cp->ulp_handle[if_type];
			ulp_ops->indicate_netevent(ctx, event);
			vmk_SemaUnlock(&cnic_driver_info.cnic_lock);
		}

	}
	return NOTIFY_DONE;
}

static struct notifier_block cnic_netdev_notifier_esx = {
	cnic_netdev_event_esx,
	0
};
#endif


#if (CNIC_ISCSI_OOO_SUPPORT)
/* General OOO engine initialization after it is enabled successfully */
static void ooo_init(struct iooo_mgmt *im)
{
	int i;

	/* Defaults */
	im->flags = vmk_BitVectorAlloc(cnic_driver_info.heap_id,
		(sizeof(long) * VMK_BITS_PER_BYTE));
	vmk_BitVectorZap(im->flags);

	/* Rings */
	im->rxr.rx_max_ring = 0;
	im->txr.tx_max_ring = 0;

	/* Packet descriptors */
	for (i = 0; i < MAX_OOO_RX_DESC_CNT; i++)
		im->rxr.rx_pkt_desc[i] = NULL;
	for (i = 0; i < MAX_OOO_TX_DESC_CNT; i++)
		im->txr.tx_pkt_desc[i] = NULL;
	im->blk_prod = MAX_IOOO_BLOCK_SUPPORTED - 1;
	im->blk_cons = 0;

	/* Blocks */
	for (i = 0; i < MAX_IOOO_BLOCK_SUPPORTED; i++) {
		im->blk_alloc[i] = i;
		im->blk[i].id = i;
		im->blk[i].pkt_cnt = 0;
		vmk_ListInit(&im->blk[i].pd_head.list);
	}

	/* Pending queues */
	vmk_ListInit(&im->txr.tx_pend_pd_head.list);
	im->txr.tx_pend_pd_cnt = 0;

	/* Statistics */
	im->txr.tx_total_pkt_sent = 0;

	ql_vmk_spin_lock_init(&im->lock, LOCK_RANK_HIGHEST);
}

static int ooo_free_buf_single(struct cnic_dev *dev,
			       struct iooo_pkt_desc *pd)
{
	struct cnic_local *cp = dev->cnic_priv;

	if (!pd) {
		CNIC_ERR(dev, "ooo_free_buf_single: pd = NULL!");
		return VMK_BAD_PARAM;
	}
	if (pd->buf) {
		CNIC_INFO(dev, CNIC_MSG_MEM,
			"Freeing pd->buf=%p map=0x%lx size=0x%x",
			pd->buf, pd->mapping, cp->iooo_mgmr.pkt_buf_size);
		ql_vmk_dma_free(dev, pd->buf, cp->iooo_mgmr.pkt_buf_size);
		pd->buf = NULL;
	}
	if (pd->skb) {
		if (cnic_reuse_ooo_pkt(pd->skb, dev))
			CNIC_ERR(dev, "ooo_free_buf_single: Error freeing skb.");
		pd->skb = NULL;
	}
	ql_vmk_free(pd);

	return 0;
}

static void ooo_free_tx_pend(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_mgmt *im = &cp->iooo_mgmr;
	struct iooo_pkt_desc *pd = NULL, *next = NULL;

	ql_vmk_list_each_entry_safe(pd, next, &im->txr.tx_pend_pd_head.list,
			list, struct iooo_pkt_desc) {
		vmk_ListRemove(&pd->list);
		if (ooo_free_buf_single(dev, pd))
			CNIC_ERR(dev, "Error freeing tx pend list.");
		im->txr.tx_pend_pd_cnt--;
	}
	if (im->txr.tx_pend_pd_cnt)
		CNIC_ERR(dev, "tx_pend_pd_cnt = %d", im->txr.tx_pend_pd_cnt);
}

static int ooo_free_blk(struct cnic_dev *dev, struct iooo_block *blk)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_mgmt *im = &cp->iooo_mgmr;
	struct iooo_pkt_desc *pd = NULL, *next = NULL;

	if (vmk_BitVectorTest(im->flags, IOOO_BLK_EMPTY)) {
		CNIC_ERR(dev, "Freeing an empty blk list!");
		return VMK_BAD_PARAM;
	}

	if (blk->pkt_cnt) {
		ql_vmk_list_each_entry_safe(pd, next, &blk->pd_head.list,
				list, struct iooo_pkt_desc) {
			vmk_ListRemove(&pd->list);
			ooo_free_buf_single(dev, pd);
			blk->pkt_cnt--;
		}
		if (blk->pkt_cnt) {
			CNIC_ERR(dev, "blk free error! pkt_cnt=%d", blk->pkt_cnt);
			blk->pkt_cnt = 0;
		}
	}

	im->blk_prod++;
	if (im->blk_prod >= MAX_IOOO_BLOCK_SUPPORTED)
		im->blk_prod = 0;

	im->blk_alloc[im->blk_prod] = blk->id;

	if (im->blk_cons == im->blk_prod)
		vmk_BitVectorSet(im->flags, IOOO_BLK_EMPTY);
	vmk_BitVectorClear(im->flags, IOOO_BLK_FULL);

	return 0;
}

static void ooo_free_tx_buf(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_tx_ring_info *txr = &cp->iooo_mgmr.txr;
	vmk_uint32 hw_tx_cons, sw_cons;
	struct iooo_pkt_desc *pd;

	if (!vmk_BitVectorTest(cp->iooo_mgmr.flags, IOOO_START_TX_FREE))
		return;

	hw_tx_cons = *txr->tx_cons_idx_ptr;
	while (hw_tx_cons != txr->tx_cons) {
		sw_cons = txr->tx_cons % BNX2_TX_DESC_CNT;
		if (sw_cons != BNX2_MAX_TX_DESC_CNT) {
			pd = txr->tx_pkt_desc[txr->tx_cons &
				(txr->tx_desc_cnt_max - 1)];
			txr->tx_pkt_desc[txr->tx_cons &
				(txr->tx_desc_cnt_max - 1)] = NULL;
			txr->tx_desc_cnt++;
			if (pd)
				ooo_free_buf_single(dev, pd);
		}
		txr->tx_cons++;
	}
}

static void ooo_free_all_buf(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_mgmt *im = &cp->iooo_mgmr;
	int i;

	for (i = 0; i < MAX_OOO_RX_DESC_CNT; i++)
		if (im->rxr.rx_pkt_desc[i]) {
			ooo_free_buf_single(dev, im->rxr.rx_pkt_desc[i]);
			im->rxr.rx_pkt_desc[i] = NULL;
		}

	for (i = 0; i < MAX_IOOO_BLOCK_SUPPORTED; i++)
		if (im->blk[i].pkt_cnt)
			ooo_free_blk(dev, &im->blk[i]);

	for (i = 0; i < MAX_OOO_TX_DESC_CNT; i++)
		if (im->txr.tx_pkt_desc[i]) {
			ooo_free_buf_single(dev, im->txr.tx_pkt_desc[i]);
			im->txr.tx_pkt_desc[i] = NULL;
		}

	ooo_free_tx_pend(dev);
}

/*
 * Whenever the rxbd's prod - cons < 1/2 MAX, replenish
 * gfp presents ATOMIC vs. KERNEL | COMP (sleep-able)
 */
static int ooo_alloc_buf_single(struct cnic_dev *dev, int gfp,
				 struct iooo_pkt_desc **pd, int len)
{
	*pd = ql_vmk_alloc(sizeof(struct iooo_pkt_desc));
	if (*pd == NULL) {
		CNIC_ERR(dev, "Failed to alloc rx pkt_desc.");
		return VMK_NO_MEMORY;
	}

	(*pd)->buf = NULL;
	if (vmk_BitVectorTest(dev->flags, CNIC_F_QFLE3_LEGACY_CLASS)) {
		(*pd)->buf = ql_vmk_dma_alloc(dev, &(*pd)->mapping, len);
		if (!(*pd)->buf) {
			CNIC_ERR(dev, "Failed to alloc rx buf.");
			ql_vmk_free(*pd);
			return VMK_NO_MEMORY;
		}
	}

	(*pd)->skb = NULL;

	return 0;
}

static int ooo_alloc_blk(struct iooo_mgmt *im)
{
	int ret;

	if (vmk_BitVectorTest(im->flags, IOOO_BLK_FULL))
		return MAX_IOOO_BLOCK_SUPPORTED;

	ret = im->blk_alloc[im->blk_cons];

	im->blk[ret].pkt_cnt = 0;

	im->blk_cons++;
	if (im->blk_cons >= MAX_IOOO_BLOCK_SUPPORTED)
		im->blk_cons = 0;
	if (im->blk_cons == im->blk_prod)
		vmk_BitVectorSet(im->flags, IOOO_BLK_FULL);
	vmk_BitVectorClear(im->flags, IOOO_BLK_EMPTY);
	return ret;
}

static int ooo_alloc_rx_buf(struct cnic_dev *dev, int gfp)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_rx_ring_info *rxr = &cp->iooo_mgmr.rxr;
	struct iooo_pkt_desc *pd = NULL;
	struct bnx2_rx_bd *rxbd;
	vmk_uint32 ring, sw_rx_prod, want;
	int ret = 0, cnt = 0;

	want = rxr->rx_desc_cnt_max - rxr->rx_desc_cnt -
	       (rxr->rx_desc_cnt_max / BNX2_RX_DESC_CNT);

	while (want > cnt) {
		ring = (rxr->rx_prod & (rxr->rx_desc_cnt_max - 1)) /
			BNX2_RX_DESC_CNT;
		sw_rx_prod = rxr->rx_prod % BNX2_RX_DESC_CNT;
		rxbd = &rxr->rx_desc_ring[ring][sw_rx_prod];
		if (sw_rx_prod != BNX2_MAX_RX_DESC_CNT) {
			if ((gfp != FLG_SLEEPABLE) &&
			    (rxr->rx_desc_cnt >= want >> 1))
				goto done;

			ret = ooo_alloc_buf_single(dev, gfp, &pd,
						   cp->iooo_mgmr.pkt_buf_size);
			if (ret)
				goto done;

			rxr->rx_prod_bseq += cp->iooo_mgmr.pkt_buf_size;
			rxr->rx_pkt_desc[rxr->rx_prod &
				(rxr->rx_desc_cnt_max - 1)] = pd;
			rxbd->rx_bd_haddr_hi = (vmk_uint64) pd->mapping >> 32;
			rxbd->rx_bd_haddr_lo = (vmk_uint64) pd->mapping & 0xffffffff;
			rxr->rx_desc_cnt++;
			cnt++;
		}
		rxr->rx_prod++;
	}
done:
	CNIC_WR16(dev, rxr->rx_bidx_addr, rxr->rx_prod);
	CNIC_WR(dev, rxr->rx_bseq_addr, rxr->rx_prod_bseq);
	return ret;
}

static void ooo_free_ring(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_tx_ring_info *txr = &cp->iooo_mgmr.txr;
	struct iooo_rx_ring_info *rxr = &cp->iooo_mgmr.rxr;
	int i;

	for (i = 0; i < rxr->rx_max_ring; i++) {
		if (rxr->rx_desc_ring[i]) {
			CNIC_INFO(dev, CNIC_MSG_MEM,
				"Freeing rx_desc_ring[%d]=%p map=0x%lx",
				i, rxr->rx_desc_ring[i], rxr->rx_desc_mapping[i]);
			ql_vmk_dma_free(dev, rxr->rx_desc_ring[i], BNX2_PAGE_SIZE);
			rxr->rx_desc_ring[i] = NULL;
		}
	}
	for (i = 0; i < txr->tx_max_ring; i++) {
		if (txr->tx_desc_ring[i]) {
			CNIC_INFO(dev, CNIC_MSG_MEM,
				"Freeing tx_desc_ring[%d]=%p map=0x%lx",
				i, txr->tx_desc_ring[i], txr->tx_desc_mapping[i]);
			ql_vmk_dma_free(dev, txr->tx_desc_ring[i], BNX2_PAGE_SIZE);
			txr->tx_desc_ring[i] = NULL;
		}
	}
}

static int ooo_alloc_ring(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_tx_ring_info *txr = &cp->iooo_mgmr.txr;
	struct iooo_rx_ring_info *rxr = &cp->iooo_mgmr.rxr;
	int i;

	rxr->rx_max_ring = rxr->rx_desc_cnt_max / BNX2_RX_DESC_CNT;
	if (rxr->rx_desc_cnt_max % BNX2_RX_DESC_CNT)
		++(rxr->rx_max_ring);
	for (i = 0; i < rxr->rx_max_ring; i++) {
		rxr->rx_desc_ring[i] = ql_vmk_dma_alloc(dev,
				&rxr->rx_desc_mapping[i], BNX2_PAGE_SIZE);
		if (!rxr->rx_desc_ring[i])
			goto free;
	}

	txr->tx_max_ring = txr->tx_desc_cnt_max / BNX2_TX_DESC_CNT;
	if (txr->tx_desc_cnt_max % TX_BD_TOTAL_PER_PAGE)
		++(txr->tx_max_ring);
	for (i = 0; i < txr->tx_max_ring; i++) {
		txr->tx_desc_ring[i] = ql_vmk_dma_alloc(dev,
				&txr->tx_desc_mapping[i], BNX2_PAGE_SIZE);
		if (!txr->tx_desc_ring[i])
			goto free;
	}
	return 0;

free:
	ooo_free_ring(dev);
	return VMK_NO_MEMORY;
}

static void ooo_init_rings(struct iooo_mgmt *im)
{
	struct iooo_rx_ring_info *rxr = &im->rxr;
	struct iooo_tx_ring_info *txr = &im->txr;
	struct bnx2_tx_bd *txbd;
	vmk_uint32 next;
	int i, j;

	for (i = 0; i < rxr->rx_max_ring; i++) {
		next = i + 1;
		if (next >= rxr->rx_max_ring)
			next = 0;
		for (j = 0; j < BNX2_MAX_RX_DESC_CNT; j++)
			cnic_set_bnx2_rxbd(&rxr->rx_desc_ring[i][j],
					   im->pkt_buf_size,
					   (vmk_IOA) NULL);
		cnic_set_bnx2_rxbd(&rxr->rx_desc_ring[i][j],
				   BNX2_PAGE_SIZE,
				   rxr->rx_desc_mapping[next]); 
	}
	for (i = 0; i < txr->tx_max_ring; i++) {
		next = i + 1;
		if (next >= txr->tx_max_ring)
			next = 0;
		for (j = 0; j < BNX2_MAX_TX_DESC_CNT; j++) {
			txbd = &txr->tx_desc_ring[i][j];
			txbd->tx_bd_vlan_tag_flags = TX_BD_FLAGS_START |
						     TX_BD_FLAGS_END;
		}
		txbd = &txr->tx_desc_ring[i][BNX2_MAX_TX_DESC_CNT];
		txbd->tx_bd_haddr_hi =
				(vmk_uint64) txr->tx_desc_mapping[next] >> 32;
		txbd->tx_bd_haddr_lo =
				(vmk_uint64) txr->tx_desc_mapping[next] & 0xffffffff;
	}
}

/* Actual placement of the pkt to the txbd */
static int ooo_send(struct iooo_tx_ring_info *txr,
		    struct iooo_pkt_desc *pd)
{
	struct bnx2_tx_bd *txbd;
	vmk_uint32 ring, sw_tx_prod;
	int i;

	vmk_IOA txpd_mapping;
	if (!txr->tx_desc_cnt)
		return VMK_NO_MEMORY;

	for (i = 0; i < 2; i++) {
		ring = (txr->tx_prod & (txr->tx_desc_cnt_max - 1)) /
			BNX2_TX_DESC_CNT;
		sw_tx_prod = txr->tx_prod % BNX2_TX_DESC_CNT;
		txbd = &txr->tx_desc_ring[ring][sw_tx_prod];
		if (sw_tx_prod != BNX2_MAX_TX_DESC_CNT) {
			txr->tx_pkt_desc[txr->tx_prod &
				(txr->tx_desc_cnt_max - 1)] = pd;
			txpd_mapping = pd->mapping + BNX2_RX_OFFSET;
			txbd->tx_bd_mss_nbytes = pd->pkt_len;
			txbd->tx_bd_haddr_hi = (vmk_uint64) txpd_mapping >> 32;
			txbd->tx_bd_haddr_lo = (vmk_uint64) txpd_mapping & 0xffffffff;
			txr->tx_prod_bseq += pd->pkt_len;
			txr->tx_prod++;
			txr->tx_desc_cnt--;
			txr->tx_total_pkt_sent++;
			return 0;
		} else
			txr->tx_prod++;
	}
	return 0;
}

static int ooo_send_qfle3(struct cnic_dev *dev, struct iooo_tx_ring_info *txr,
			  struct iooo_pkt_desc *pd)
{
	int ret;

	ret = cnic_send_ooo_pkt(pd->skb, dev);
	/* SV: XXX: Assuming NETDEV_TX_OK=0 */
	if (ret == 0) {
		txr->tx_total_pkt_sent++;
		/* Once sent, cnic no longer owns the skb */
		pd->skb = NULL;
		ret = 0;
	} else
		CNIC_ERR(dev, "send_ooo ret=0x%x", ret);

	ooo_free_buf_single(dev, pd);

	return ret;
}

static void ooo_send_pend(struct cnic_dev *dev, struct iooo_tx_ring_info *txr)
{
	struct iooo_pkt_desc *pd = NULL, *next = NULL;
	int cnt = 0;

	if (vmk_ListIsEmpty(&txr->tx_pend_pd_head.list)) {
		if (txr->tx_pend_pd_cnt)
			CNIC_ERR(dev, "pend cnt out of sync=0x%x", txr->tx_pend_pd_cnt);
		return;
	}

	ql_vmk_list_each_entry_safe(pd, next, &txr->tx_pend_pd_head.list,
			list, struct iooo_pkt_desc) {
		vmk_ListRemove(&pd->list);
		if (vmk_BitVectorTest(dev->flags, CNIC_F_QFLE3_LEGACY_CLASS))
			ooo_send(txr, pd);
		else
			ooo_send_qfle3(dev, txr, pd);

		dev->ooo_tx_count++;
		txr->tx_pend_pd_cnt--;
		cnt++;
	}

	if (vmk_BitVectorTest(dev->flags, CNIC_F_QFLE3_LEGACY_CLASS) && cnt) {
		CNIC_WR16(dev, txr->tx_bidx_addr, txr->tx_prod);
		CNIC_WR(dev, txr->tx_bseq_addr, txr->tx_prod_bseq);
	}
}

static void ooo_send_pkt(struct iooo_tx_ring_info *txr,
	struct iooo_pkt_desc *pd)
{
	vmk_ListInsert(&pd->list, vmk_ListAtRear(&txr->tx_pend_pd_head.list));
	txr->tx_pend_pd_cnt++;
}

static int ooo_send_blk(struct iooo_tx_ring_info *txr,
	struct iooo_block *blk)
{
	struct iooo_pkt_desc *pd = NULL, *next = NULL;

	ql_vmk_list_each_entry_safe(pd, next, &blk->pd_head.list,
			list, struct iooo_pkt_desc) {
		vmk_ListRemove(&pd->list);
		ooo_send_pkt(txr, pd);
		blk->pkt_cnt--;
	}
	return blk->pkt_cnt;
}

static void ooo_engine(struct cnic_dev *dev, struct iooo_pkt_desc *pkt_desc,
	vmk_uint32 l5_cid)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_mgmt *im = &cp->iooo_mgmr;
	struct cnic_context *ctx = &cp->ctx_tbl[l5_cid];
	struct cnic_iscsi *iscsi = ctx->proto.iscsi;
	struct iooo_block *pen = &iscsi->pen;
	struct iooo_block *blk = NULL, *blk_nxt = NULL;
	struct iooo_pkt_desc *tmp = NULL, *tmp_nxt = NULL;
	int blk_idx, new_blk_idx;

	CNIC_INFO(dev, CNIC_MSG_DRV,
		"cnic: cid=0x%x op %d blk %d dblk %d size %d bcnt=%d",
		im->hsi.iscsi_cid, im->hsi.opcode, im->hsi.blk_idx,
		im->hsi.drop_blk_idx, im->hsi.drop_size,
		iscsi->blk_cnt);

	if (im->hsi.drop_size) {
		if (!im->hsi.drop_blk_idx || vmk_ListIsEmpty(&pen->list)) {
			CNIC_ERR(dev, "drop_blk_idx=%d or list empty.",
				im->hsi.drop_blk_idx);
			goto orphan;
		}
		blk_idx = 1;
		ql_vmk_list_each_entry_safe(blk, blk_nxt, &pen->list,
				list, struct iooo_block) {
			if (im->hsi.drop_blk_idx == blk_idx++) {
				vmk_ListRemove(&blk->list);
				ooo_free_blk(dev, blk);
				iscsi->blk_cnt--;
				im->hsi.drop_blk_idx++;
				if (!(--im->hsi.drop_size))
					break;
			}
		}
	}

	blk = NULL;
	switch (im->hsi.opcode) {
	case OOO_OPCODE_ADD_RIGHT:
		blk_idx = 1;
		ql_vmk_list_each_entry(blk, &pen->list, list, struct iooo_block) {
			if (im->hsi.blk_idx == blk_idx++)
				break;
		}
		if (vmk_ListIsEmpty(&pen->list) || im->hsi.blk_idx != --blk_idx) {
			CNIC_ERR(dev, "can't find block to add right to!");
			goto orphan;
		}
		vmk_ListInsert(&pkt_desc->list, vmk_ListAtRear(&blk->pd_head.list));
		blk->pkt_cnt++;
		break;

	case OOO_OPCODE_ADD_LEFT:
		blk_idx = 1;
		ql_vmk_list_each_entry(blk, &pen->list, list, struct iooo_block) {
			if (im->hsi.blk_idx == blk_idx++)
				break;
		}
		if (vmk_ListIsEmpty(&pen->list) || im->hsi.blk_idx != --blk_idx) {
			CNIC_ERR(dev, "can't find block to add left to!");
			goto orphan;
		}
		vmk_ListInsert(&pkt_desc->list, vmk_ListAtFront(&blk->pd_head.list));
		blk->pkt_cnt++;
		break;

	case OOO_OPCODE_ADD_NEW:
		new_blk_idx = ooo_alloc_blk(im);
		if (MAX_IOOO_BLOCK_SUPPORTED == new_blk_idx) {
			CNIC_ERR(dev, "max blk reached!");
			goto orphan;
		}
		/* Find blk to add to */
		if (im->hsi.blk_idx == 1)
			blk = pen;
		else {
			blk_idx = 2;
			ql_vmk_list_each_entry(blk, &pen->list, list, struct iooo_block) {
				if (im->hsi.blk_idx == blk_idx++)
					break;
			}
		}
		vmk_ListInsert(&im->blk[new_blk_idx].list, vmk_ListAtFront(&blk->list));
		iscsi->blk_cnt++;

		/* Attach pkt to blk */
		blk = &im->blk[new_blk_idx];
		vmk_ListInsert(&pkt_desc->list, vmk_ListAtFront(&blk->pd_head.list));
		blk->pkt_cnt++;
		break;

	case OOO_OPCODE_JOIN:
		if (!im->hsi.blk_idx) {
			if (vmk_ListIsEmpty(&pen->list)) {
				CNIC_ERR(dev, "can't find block to join 0!");
				goto orphan;
			}
			blk = VMK_LIST_ENTRY(vmk_ListFirst(&pen->list), struct iooo_block, list);
			vmk_ListInsert(&pkt_desc->list, vmk_ListAtFront(&blk->pd_head.list));
			blk->pkt_cnt++;
			if (ooo_send_blk(&im->txr, blk)) {
				CNIC_ERR(dev, "blk sent err! pkt_cnt=%d", blk->pkt_cnt);
				blk->pkt_cnt = 0;
			}
			ooo_free_blk(dev, blk);
			iscsi->blk_cnt--;
			vmk_ListRemove(&blk->list);
		} else {
			blk_idx = 1;
			blk = blk_nxt = NULL;
			ql_vmk_list_each_entry(blk, &pen->list, list, struct iooo_block) {
				if (im->hsi.blk_idx == blk_idx++) {
					blk_nxt = VMK_LIST_ENTRY(vmk_ListNext(&blk->list),
							struct iooo_block, list);
					break;
				}
			}
			if (!blk || !blk_nxt || blk_nxt == pen) {
				CNIC_ERR(dev, "can't find block to join!");
				goto orphan;
			}
			vmk_ListInsert(&pkt_desc->list, vmk_ListAtRear(&blk->pd_head.list));
			blk->pkt_cnt++;
			/* Append all the pkts from the nxt blk to this blk */
			ql_vmk_list_each_entry_safe(tmp, tmp_nxt, &blk_nxt->pd_head.list,
					list, struct iooo_pkt_desc) {
				vmk_ListRemove(&tmp->list);
				vmk_ListInsert(&tmp->list, vmk_ListAtRear(&blk->pd_head.list));
				blk->pkt_cnt++;
				blk_nxt->pkt_cnt--;
			}
			ooo_free_blk(dev, blk_nxt);
			iscsi->blk_cnt--;
			vmk_ListRemove(&blk_nxt->list);
		}
		break;

	case OOO_OPCODE_ADD_PEN:
		ooo_send_pkt(&im->txr, pkt_desc);
		break;

	default:
		break;
	}

	return;

orphan:
	if (pkt_desc)
		ooo_free_buf_single(dev, pkt_desc);
}

static void ooo_handle_rx_event(struct cnic_dev *dev,
			       struct iooo_pkt_desc *pd)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_mgmt *im = &cp->iooo_mgmr;
	vmk_uint32 l5_cid;

	/* For 0 len placement, just free the pkt */
	if (VMK_UNLIKELY(!pd->pkt_len)) {
		ooo_free_buf_single(dev, pd);
		return;
	}

	if (cnic_get_l5_cid(cp, im->hsi.iscsi_cid, &l5_cid) == 0) {
		if (l5_cid >= MAX_CM_SK_TBL_SZ) {
			CNIC_ERR(dev, "bad l5_cid=0x%x", l5_cid);
			ooo_free_buf_single(dev, pd);
		} else
			ooo_engine(dev, pd, l5_cid);
	} else {
		CNIC_ERR(dev, "get l5_cid failed.");
		ooo_free_buf_single(dev, pd);
	}
}

static void cnic_handle_bnx2_ooo_rx_event(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_mgmt *im = &cp->iooo_mgmr;
	struct iooo_pkt_desc *pkt_desc;
	struct bnx2_ooo_fhdr *pkt_hsi;
	vmk_uint16 hw_rx_cons, sw_cons;

	/* Process only if ready to start handler */
	if (VMK_UNLIKELY(!vmk_BitVectorTest(im->flags, IOOO_START_HANDLER)))
		return;

	/* Handle RX placement */
	hw_rx_cons = *im->rxr.rx_cons_idx_ptr;
	while (hw_rx_cons != im->rxr.rx_cons) {
		sw_cons = im->rxr.rx_cons % BNX2_RX_DESC_CNT;
		if (sw_cons != BNX2_MAX_RX_DESC_CNT) {
			pkt_desc = im->rxr.rx_pkt_desc[im->rxr.rx_cons &
				(im->rxr.rx_desc_cnt_max - 1)];
			if (!pkt_desc) {
				CNIC_ERR(dev, "pkt_desc = NULL?! rx_cons=%d",
					im->rxr.rx_cons &
					(im->rxr.rx_desc_cnt_max - 1));
				goto out;
			}
			im->rxr.rx_pkt_desc[im->rxr.rx_cons &
				(im->rxr.rx_desc_cnt_max - 1)] = NULL;

			pkt_hsi = (struct bnx2_ooo_fhdr *)pkt_desc->buf;
			pkt_desc->pkt_len = pkt_hsi->pkt_len;
			im->hsi.iscsi_cid = pkt_hsi->icid;
			im->hsi.opcode = pkt_hsi->opcode;
			im->hsi.blk_idx = pkt_hsi->blk_idx;
			im->hsi.drop_size = pkt_hsi->drop_size;
			im->hsi.drop_blk_idx = pkt_hsi->drop_blk_idx;

			ooo_handle_rx_event(dev, pkt_desc);

			im->rxr.rx_desc_cnt--;
		}
out:
		im->rxr.rx_cons++;
	}

	/* If already stopped, keep processing the rx queue but do not alloc
	   more buffers */

	if (VMK_UNLIKELY(!vmk_BitVectorTest(im->flags, IOOO_START)))
		return;

	ooo_alloc_rx_buf(dev, FLG_ATOMIC);
}

static void cnic_handle_bnx2_ooo_tx_event(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_mgmt *im = &cp->iooo_mgmr;
	vmk_uint16 hw_cons, sw_cons;

	if (VMK_UNLIKELY(!vmk_BitVectorTest(im->flags, IOOO_START_HANDLER)))
		return;

	/* Handle fwd ring tx completion */
	hw_cons = *im->txr.tx_cons_idx_ptr;
	sw_cons = im->txr.tx_cons;
	if (sw_cons == hw_cons && !im->txr.tx_pend_pd_cnt)
		return;

	/* Must wait for at least 1 tx completion before attempting to free */
	if (!vmk_BitVectorTest(cp->iooo_mgmr.flags, IOOO_START_TX_FREE) &&
		im->txr.tx_desc_cnt < (im->txr.tx_desc_cnt_max >> 1))
		vmk_BitVectorSet(cp->iooo_mgmr.flags, IOOO_START_TX_FREE);

	vmk_SpinlockLock(im->lock);
	ooo_free_tx_buf(dev);
	vmk_SpinlockUnlock(im->lock);

	if (VMK_UNLIKELY(!vmk_BitVectorTest(im->flags, IOOO_START)))
		return;

	ooo_send_pend(dev, &im->txr);
}

static void cnic_handle_qfle3_ooo_rx_event(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_mgmt *im = &cp->iooo_mgmr;
	struct iooo_pkt_desc *pkt_desc;
	struct iooo_hsi_qfle3 *pkt_hsi;
	struct cnic_ooo_cqe ooo_cqe;
	int ret;
	vmk_uint16 hw_cons, sw_cons;

	/* Process only if ready to start */
	if (VMK_UNLIKELY(!vmk_BitVectorTest(im->flags, IOOO_START)))
		return;

	hw_cons = *im->rxr.rx_cons_idx_ptr;
	sw_cons = im->rxr.rx_cons;
	if (sw_cons == hw_cons)
		return;

	im->rxr.rx_cons = hw_cons;

	ooo_cqe.cqe_type = 0xffffffff;
	/* Handle the rx cqe */
	do {
		ret = cnic_get_ooo_cqe(dev, &ooo_cqe);
		if (ret < 0) {
			CNIC_ERR(dev, "ERROR at retrieving OOO CQE.");
			goto error;
		}

		CNIC_INFO(dev, CNIC_MSG_DRV, "OOO: cqe_type=0x%x", ooo_cqe.cqe_type);

		if (0xffffffff == ooo_cqe.cqe_type)
			goto empty;
		else if (OOO_BD_CQE != ooo_cqe.cqe_type) {
			CNIC_ERR(dev, "OOO CQE type=%d!", ooo_cqe.cqe_type);
			goto reuse;
		}
		if (!ooo_alloc_buf_single(dev, FLG_ATOMIC, &pkt_desc, 0)) {
			pkt_desc->skb = ooo_cqe.u.cqe.pkt_desc;
			pkt_desc->pkt_len = vmk_PktFrameLenGet(pkt_desc->skb);
			pkt_hsi = (struct iooo_hsi_qfle3 *)
				   ooo_cqe.u.cqe.raw_data;
			/* Must mask out for port identifier (bit 23) */
			im->hsi.iscsi_cid = QFLE3_SW_CID(pkt_hsi->iscsi_cid);
			im->hsi.opcode = pkt_hsi->opcode;
			im->hsi.blk_idx = pkt_hsi->blk_idx;
			im->hsi.drop_size = pkt_hsi->drop_size;
			im->hsi.drop_blk_idx = pkt_hsi->drop_blk_idx;

			ooo_handle_rx_event(dev, pkt_desc);

		} else {
			CNIC_ERR(dev, "Failed to allocate pk desc!");
reuse:
			cnic_reuse_ooo_pkt(ooo_cqe.u.cqe.pkt_desc, dev);
			return;
		}
	} while (ret);
error:
empty:
	/* Send any tx pending pkt */
	ooo_send_pend(dev, &im->txr);

	return;
}

static void cnic_handle_qfle3_ooo_tx_event(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_mgmt *im = &cp->iooo_mgmr;
	vmk_uint16 hw_cons, sw_cons;

	/* Handle fwd ring tx completion */
	hw_cons = *im->txr.tx_cons_idx_ptr;
	sw_cons = im->txr.tx_cons;
	if (sw_cons == hw_cons)
		return;

	im->txr.tx_cons = hw_cons;

	/* Handle tx completion by sending event to qfle3 */
	if (VMK_UNLIKELY(cp->ethdev->driverState & CNIC_DRV_STATE_NO_ISCSI_OOO))
		return;

	CNIC_INFO(dev, CNIC_MSG_DRV, "OOO: Handle tx completion. "
		"hw_cons=0x%x, sw_cons=0x%x",
		hw_cons, sw_cons);
	cnic_comp_ooo_tx_pkts(dev);

	/* Lastly, send any tx pending pkt */
	if (VMK_UNLIKELY(!vmk_BitVectorTest(im->flags, IOOO_START)))
		return;

	ooo_send_pend(dev, &im->txr);
}

static void cnic_alloc_bnx2_ooo_resc(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	int ret;

	/* Chip specific */
	cp->iooo_mgmr.rxr.rx_desc_cnt_max = MAX_BNX2_OOO_RX_DESC_CNT;
	cp->iooo_mgmr.txr.tx_desc_cnt_max = MAX_BNX2_OOO_TX_DESC_CNT;
	cp->iooo_mgmr.pkt_buf_size = dev->mtu + ETH_HLEN + BNX2_RX_OFFSET + 8;
	/* General */
	ooo_init(&cp->iooo_mgmr);

	if (BNX2_CHIP(cp) != BNX2_CHIP_5709)
		return;

	ret = ooo_alloc_ring(dev);
	if (!ret)
		vmk_BitVectorSet(cp->iooo_mgmr.flags, IOOO_RESC_AVAIL);
}

static void cnic_alloc_qfle3_ooo_resc(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;

	/* General */
	ooo_init(&cp->iooo_mgmr);
	vmk_BitVectorSet(cp->iooo_mgmr.flags, IOOO_RESC_AVAIL);
}

static void cnic_stop_bnx2_ooo_hw(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_mgmt *im = &cp->iooo_mgmr;
	int cnt = 10;
	struct kwqe *wqes[1], l2kwqe;

	if (!(vmk_BitVectorTest(cp->iooo_mgmr.flags, IOOO_START)))
		return;

	vmk_BitVectorClear(im->flags, IOOO_START_HANDLER);

	/* Send kwqe to clean up the L2 OOO rx ring */
	vmk_Memset(&l2kwqe, 0, sizeof(l2kwqe));
	wqes[0] = &l2kwqe;
	l2kwqe.kwqe_op_flag = (L2_LAYER_CODE << KWQE_LAYER_SHIFT) |
			      (L2_KWQE_OPCODE_VALUE_FLUSH <<
			       KWQE_OPCODE_SHIFT) | RX_CATCHUP_CID;
	dev->submit_kwqes(dev, wqes, 1);

	/* Wait for the hardware indexes to match producer */
	while (*im->rxr.rx_cons_idx_ptr && cnt) {
		vmk_CPUMemFenceReadWrite();
		vmk_WorldSleep(10*VMK_USEC_PER_MSEC);
		cnt--;
	}

	if (!cnt)
		CNIC_ERR(dev, "hw rx_cons=%d", *im->rxr.rx_cons_idx_ptr);

	vmk_BitVectorClear(im->flags, IOOO_START);
}

static void cnic_stop_qfle3_ooo_hw(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iooo_mgmt *im = &cp->iooo_mgmr;

	/* Nothing to do here as the free_ooo_resc gets called in stop_hw */
	vmk_BitVectorClear(im->flags, IOOO_START);
}

static void cnic_start_bnx2_ooo_hw(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct status_block *sblk = (struct status_block *)
		((unsigned long)cp->cfp[CNIC_RX_IDX].status_blk.gen & (~VMK_PAGE_MASK));
	struct iooo_rx_ring_info *rxr = &cp->iooo_mgmr.rxr;
	struct iooo_tx_ring_info *txr = &cp->iooo_mgmr.txr;
	vmk_uint32 val;

	if (!(vmk_BitVectorTest(cp->iooo_mgmr.flags, IOOO_RESC_AVAIL)))
		return;

	rxr->rx_cid_addr = GET_CID_ADDR(RX_CATCHUP_CID);
	rxr->rx_bidx_addr = MB_GET_CID_ADDR(RX_CATCHUP_CID) +
			    L5_KRNLQ_HOST_QIDX;
	rxr->rx_bseq_addr = MB_GET_CID_ADDR(RX_CATCHUP_CID) +
			    L5_KRNLQ_HOST_FW_QIDX;
	rxr->rx_prod = 0;
	rxr->rx_prod_bseq = 0;
	rxr->rx_cons = 0;

	rxr->rx_cons_idx_ptr = (vmk_uint16 *) (&sblk->status_rx_quick_consumer_index1);
	cnic_init_bnx2_rx_ring_start(dev, RX_CATCHUP_CID, rxr->rx_cons_idx_ptr,
				     rxr->rx_desc_mapping[0], 0);
	if (*rxr->rx_cons_idx_ptr)
		CNIC_ERR(dev, "stale hw rx_cons=%d", *rxr->rx_cons_idx_ptr);

	txr->tx_cid_addr = GET_CID_ADDR(TX_CATCHUP_CID);
	txr->tx_bidx_addr = MB_GET_CID_ADDR(TX_CATCHUP_CID) +
			    BNX2_L2CTX_TX_HOST_BIDX;
	txr->tx_bseq_addr = MB_GET_CID_ADDR(TX_CATCHUP_CID) +
			    BNX2_L2CTX_TX_HOST_BSEQ;
	txr->tx_cons_idx_ptr = (vmk_uint16 *) (&sblk->status_tx_quick_consumer_index1);
	txr->tx_prod = 0;
	txr->tx_prod_bseq = 0;
	txr->tx_cons = 0;

	cnic_init_bnx2_tx_ring_start(dev, TX_CATCHUP_CID, txr->tx_desc_mapping[0]);

	ooo_init_rings(&cp->iooo_mgmr);

	val = BNX2_L2CTX_CMD_TYPE_TYPE_L2 | (TX_OOO_EST_NBD << 16);
	cnic_ctx_wr(dev, txr->tx_cid_addr, BNX2_L2CTX_CMD_TYPE_XI, val);

	/* Allocate rx buf, no tx buf yet */
	rxr->rx_desc_cnt = 0;
	txr->tx_desc_cnt = txr->tx_desc_cnt_max - txr->tx_desc_cnt_max /
			   BNX2_TX_DESC_CNT;
	ooo_alloc_rx_buf(dev, FLG_SLEEPABLE);
	vmk_BitVectorSet(cp->iooo_mgmr.flags, IOOO_START);
}

static void cnic_start_qfle3_ooo_hw(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct qfle3_adapter *adapter = NULL;
	struct iooo_rx_ring_info *rxr = &cp->iooo_mgmr.rxr;
	struct iooo_tx_ring_info *txr = &cp->iooo_mgmr.txr;
	int i;

	vmk_DeviceGetAttachedDriverData(dev->netdev, (vmk_AddrCookie *) &adapter);

	if (!(vmk_BitVectorTest(cp->iooo_mgmr.flags, IOOO_RESC_AVAIL)))
		return;

	for (i = 0; i < MAX_OOO_RX_DESC_CNT; i++)
		rxr->rx_pkt_desc[i] = NULL;
	for (i = 0; i < MAX_OOO_TX_DESC_CNT; i++)
		txr->tx_pkt_desc[i] = NULL;

	if (QFLE3_CHIP_IS_E2_PLUS(adapter)) {
		/*SV: XXX: Need to revisit which sb is needed for iscsi ooo tx ring? */
		struct host_hc_status_block_e2 *sb = cp->cfp[CNIC_RX_IDX].status_blk.gen;

		txr->tx_cons_idx_ptr =
			&sb->sb.index_values[HC_INDEX_FWD_TX_CQ_CONS];
		rxr->rx_cons_idx_ptr =
			&sb->sb.index_values[HC_INDEX_OOO_RX_CQ_CONS];
	} else {
		/*SV: XXX: Need to revisit which sb is needed for iscsi ooo tx ring? */
		struct host_hc_status_block_e1x *sb = cp->cfp[CNIC_RX_IDX].status_blk.gen;

		txr->tx_cons_idx_ptr =
			&sb->sb.index_values[HC_INDEX_FWD_TX_CQ_CONS];
		rxr->rx_cons_idx_ptr =
			&sb->sb.index_values[HC_INDEX_OOO_RX_CQ_CONS];
	}
	rxr->rx_cons = *rxr->rx_cons_idx_ptr;
	txr->tx_cons = *txr->tx_cons_idx_ptr;

	vmk_BitVectorSet(cp->iooo_mgmr.flags, IOOO_START);
	if (cp->ethdev->driverState & CNIC_DRV_STATE_NO_ISCSI_OOO)
		vmk_BitVectorClear(cp->iooo_mgmr.flags, IOOO_START);
}

static void cnic_free_ooo_resc(struct cnic_dev *dev)
{
	struct cnic_local *cp = dev->cnic_priv;

	if (!(vmk_BitVectorTest(cp->iooo_mgmr.flags, IOOO_RESC_AVAIL)))
		return;

	vmk_BitVectorClear(cp->iooo_mgmr.flags, IOOO_RESC_AVAIL);
	vmk_BitVectorFree(cnic_driver_info.heap_id, cp->iooo_mgmr.flags);
	ooo_free_all_buf(dev);
	ooo_free_ring(dev);
	vmk_SpinlockDestroy(cp->iooo_mgmr.lock);
}

static void cnic_conn_ooo_init(struct cnic_local *cp, vmk_uint32 l5_cid)
{
	struct cnic_context *ctx = &cp->ctx_tbl[l5_cid];
	struct cnic_iscsi *iscsi = ctx->proto.iscsi;

	iscsi->blk_cnt = 0;
	iscsi->pen.pkt_cnt = 0;
	vmk_ListInit(&iscsi->pen.list);
	vmk_ListInit(&iscsi->pen.pd_head.list);
}

/* Flush the associated iooo_block for the connection specified */
static void cnic_flush_ooo(struct cnic_dev *dev, vmk_uint32 l5_cid)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct cnic_context *ctx;
	struct cnic_iscsi *iscsi;
	struct iooo_block *blk = NULL, *blk_nxt = NULL;

	vmk_SpinlockLock(cp->iooo_mgmr.lock);
	if (l5_cid >= MAX_CM_SK_TBL_SZ)
		goto skip;

	ctx = &cp->ctx_tbl[l5_cid];
	if (!ctx)
		goto skip;

	iscsi = ctx->proto.iscsi;

	if (!iscsi->blk_cnt)
		goto skip;

	ql_vmk_list_each_entry_safe(blk, blk_nxt, &iscsi->pen.list,
			list, struct iooo_block) {
		vmk_ListRemove(&blk->list);
		ooo_free_blk(dev, blk);
		iscsi->blk_cnt--;
	}
	if (iscsi->blk_cnt) {
		CNIC_ERR(dev, "blk cnt=%d != 0", iscsi->blk_cnt);
		iscsi->blk_cnt = 0;
	}
skip:
	if (vmk_BitVectorTest(dev->flags, CNIC_F_QFLE3_LEGACY_CLASS))
		ooo_free_tx_buf(dev);

	vmk_BitVectorClear(cp->iooo_mgmr.flags, IOOO_START_TX_FREE);
	vmk_SpinlockUnlock(cp->iooo_mgmr.lock);
}

static void cnic_bnx2_ooo_iscsi_conn_update(struct cnic_dev *dev,
					    struct kwqe *kwqe)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iscsi_kwqe_conn_update *req =
		(struct iscsi_kwqe_conn_update *) kwqe;
	vmk_uint32 l5_cid;
	struct cnic_context *ctx;

	l5_cid = req->reserved2;
	if (l5_cid >= MAX_ISCSI_TBL_SZ)
		return;

	ctx = &cp->ctx_tbl[l5_cid];
	ctx->cid = req->context_id << 7;

	if (!(vmk_BitVectorTest(cp->iooo_mgmr.flags, IOOO_RESC_AVAIL)))
		return;

	if (vmk_BitVectorTest(dev->flags, CNIC_F_ISCSI_OOO_ENABLE)) {
		cnic_reg_wr_ind(dev, BNX2_RXP_SCRATCH_OOO_RX_CID,
				GET_CID_ADDR(RX_CATCHUP_CID));
		cnic_reg_wr_ind(dev, BNX2_RXP_SCRATCH_OOO_FLAGS,
				BNX2_IOOO_FLAGS_OVERRIDE |
				BNX2_IOOO_FLAGS_ENABLE);
	}
}

static void cnic_bnx2_ooo_iscsi_destroy(struct cnic_dev *dev, struct kwqe *kwqe)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iscsi_kwqe_conn_destroy *req =
		(struct iscsi_kwqe_conn_destroy *) kwqe;
	vmk_uint32 l5_cid = req->reserved0;
	struct cnic_context *ctx;

	if (l5_cid >= MAX_ISCSI_TBL_SZ)
		return;

	ctx = &cp->ctx_tbl[l5_cid];
	ctx->cid = 0;
}

static void cnic_qfle3_ooo_iscsi_conn_update(struct cnic_dev *dev,
					     struct kwqe *kwqe)
{
	struct cnic_local *cp = dev->cnic_priv;
	struct iscsi_kwqe_conn_update *req =
		(struct iscsi_kwqe_conn_update *) kwqe;

	if (!(vmk_BitVectorTest(cp->iooo_mgmr.flags, IOOO_RESC_AVAIL)))
		return;

	if (vmk_BitVectorTest(dev->flags, CNIC_F_ISCSI_OOO_ENABLE) &&
		!(cp->ethdev->driverState & CNIC_DRV_STATE_NO_ISCSI_OOO))
		req->conn_flags =
			(req->conn_flags & ~ISCSI_KWQE_CONN_UPDATE_OOO_SUPPORT_MODE) |
			(TCP_TSTORM_OOO_SUPPORTED <<
			 ISCSI_KWQE_CONN_UPDATE_OOO_SUPPORT_MODE_SHIFT);
}

#endif  /* CNIC_ISCSI_OOO_SUPPORT */

/* ===== SV: XXX: New cnic native changes. START ===== */

/*
 * CNIC Driver attach/detach/start/scan
 */

static VMK_ReturnStatus
cnic_removeDevice(vmk_Device device)
{
	DRV_INFO(CNIC_MSG_DRV, "Remove logical device: %p", device);
	return vmk_DeviceUnregister(device);
}

static vmk_DeviceOps cnicDeviceOps = {
	.removeDevice = cnic_removeDevice,
};

static VMK_ReturnStatus
cnic_attachDevice(vmk_Device ldev)
{
	VMK_ReturnStatus status = VMK_OK;
	struct cnic_dev *cdev = NULL;
	vmk_DeviceID *devID = NULL;
	int is_e3 = 0;

	DRV_INFO(CNIC_MSG_DRV, "Entered");

	status = vmk_DeviceGetDeviceID(cnic_driver_info.heap_id, ldev, &devID);
	if (status == VMK_OK) {
		if (vmk_Memcmp(devID->busIdentifier, QFLE3_DEV_IDENTIFIER,
				devID->busIdentifierLen) == 0)
			is_e3 = QFLE3_DRIVER; /* Check if qfle3 */
		else if (vmk_Memcmp(devID->busIdentifier, QFLE3_LEGACY_DEV_IDENTIFIER,
				devID->busIdentifierLen) == 0)
			is_e3 = QFLE3_LEGACY_DRIVER; /* Check if legacy (bnx2) */
	} else
		return VMK_FAILURE;

	vmk_DevicePutDeviceID(cnic_driver_info.heap_id, devID);

	DRV_INFO(CNIC_MSG_INIT, "is_e3=0x%x", is_e3);

	/*****************************************old*****/

	cdev = is_cnic_dev(ldev, is_e3);
	//SV: XXX: adapter->link_vars.link_up?
	if (cdev /*&& (ldev->flags & IFF_UP)*/) {
		vmk_BitVectorSet(cdev->flags, CNIC_F_IF_UP);
		vmk_DeviceSetAttachedDriverData(ldev, cdev);
	} else {
		DRV_ERR("Error: cdev is NULL! Failing attachDevice.");
		return VMK_FAILURE;
	}

	/*****************************************old*****/
	DRV_INFO(CNIC_MSG_DRV, "Returned (VMK_OK)");

	return VMK_OK;
}

static VMK_ReturnStatus cnic_scanDevice(vmk_Device device)
{
	VMK_ReturnStatus status = VMK_OK;
	vmk_BusType bus_type;
	vmk_Name bus_name;
	struct cnic_dev *cdev = NULL;
	qfle3_adapter *adapter = NULL;
	struct cnic_local *cp = NULL;
	struct cnic_eth_dev *ethdev = NULL;
	vmk_DeviceID dev_id_iscsi, dev_id_fcoe;
	vmk_DeviceProps device_props;

	DRV_INFO(CNIC_MSG_DRV, "Entered");

	status = vmk_DeviceGetAttachedDriverData(device,
			(vmk_AddrCookie *)&cdev);
	if (status != VMK_OK) {
		DRV_WARN("Failed to get cnic_dev: %s", vmk_StatusToString(status));
		return status;
	}

	if (cdev == NULL) {
		DRV_ERR("cdev not valid!!!");
		return VMK_FAILURE;
	}

	cp = cdev->cnic_priv;
	ethdev = cp->ethdev;

	status = vmk_DeviceGetAttachedDriverData(cdev->netdev,
			(vmk_AddrCookie *)&adapter);
	if (status != VMK_OK) {
		CNIC_WARN(cdev, "Failed to get qfle3-adapter: %s",
			vmk_StatusToString(status));
		return status;
	}

	status = vmk_NameInitialize(&bus_name, VMK_LOGICAL_BUS_NAME);
	if (status != VMK_OK) {
		CNIC_WARN(cdev, "Failed to initialize cnic name: %s",
			vmk_StatusToString(status));
		return status;
	}

	status = vmk_BusTypeFind(&bus_name, &bus_type);
	if (status != VMK_OK) {
		CNIC_WARN(cdev, "Failed to BusTypeFind: %s",
			vmk_StatusToString(status));
		return status;
	}

	if (ethdev->driverState & CNIC_DRV_STATE_NO_ISCSI) {
		CNIC_WARN(cdev, "iSCSI not configured/enabled.");
		goto skip_iscsi_ldev;
	}

	/* Create Logical device for iSCSI protocol driver. */
	status = vmk_LogicalCreateBusAddress(
			cnic_driver_info.cnic_driver, device, 0,
			&dev_id_iscsi.busAddress, &dev_id_iscsi.busAddressLen);

	if ( status != VMK_OK) {
		DRV_WARN("Failed to created logicalbus: %s",
			vmk_StatusToString(status));
		goto  fail_register_logical_bus;
	}

	dev_id_iscsi.busType = bus_type;
	dev_id_iscsi.busIdentifier = QFLE3I_DEV_IDENTIFIER;
	dev_id_iscsi.busIdentifierLen = vmk_Strnlen(dev_id_iscsi.busIdentifier,
			VMK_MISC_NAME_MAX);
	device_props.registeringDriver = cnic_driver_info.cnic_driver;
	device_props.deviceID = &dev_id_iscsi;
	device_props.deviceOps = &cnicDeviceOps;
	device_props.registeringDriverData.ptr = cdev;
	/*
	 * SV: XXX: Do we need any registering data to be saved?
	 * 			Keep it "cdev" for now.
	 */
	device_props.registrationData.ptr = cdev;

	status = vmk_DeviceRegister(&device_props, device, &cdev->iSCSIDevice);
	if (status != VMK_OK) {
		DRV_WARN("Failed to register device: %s", vmk_StatusToString(status));
	}

	vmk_LogicalFreeBusAddress(cnic_driver_info.cnic_driver,
			dev_id_iscsi.busAddress);


skip_iscsi_ldev:
	if (ethdev->driverState & CNIC_DRV_STATE_NO_FCOE) {
		CNIC_WARN(cdev, "FCoE not configured/enabled.");
		goto fail_register_logical_bus;
	}

	/* Create Logical device for FCoE protocol driver. */
	status = vmk_LogicalCreateBusAddress(
			cnic_driver_info.cnic_driver, device, 1,
			&dev_id_fcoe.busAddress, &dev_id_fcoe.busAddressLen);

	if ( status != VMK_OK) {
		DRV_WARN("Failed to created logicalbus: %s",
			vmk_StatusToString(status));
		goto  fail_register_logical_bus;
	}

	dev_id_fcoe.busType = bus_type;
	dev_id_fcoe.busIdentifier = QFLE3F_DEV_IDENTIFIER;
	dev_id_fcoe.busIdentifierLen = vmk_Strnlen(dev_id_fcoe.busIdentifier,
			VMK_MISC_NAME_MAX);
	device_props.registeringDriver = cnic_driver_info.cnic_driver;
	device_props.deviceID = &dev_id_fcoe;
	device_props.deviceOps = &cnicDeviceOps;
	device_props.registeringDriverData.ptr = cdev;
	/*
	 * SV: XXX: Do we need any registering data to be saved?
	 * 			Keep it "cdev" for now.
	 */
	device_props.registrationData.ptr = cdev;

	status = vmk_DeviceRegister(&device_props, device, &cdev->FCoEDevice);
	if (status != VMK_OK) {
		DRV_WARN("Failed to register device: %s", vmk_StatusToString(status));
	}

	vmk_LogicalFreeBusAddress(cnic_driver_info.cnic_driver,
			dev_id_fcoe.busAddress);

fail_register_logical_bus:
	vmk_BusTypeRelease(bus_type);
	CNIC_INFO(cdev, CNIC_MSG_DRV, "Exiting.");

	return status;
}

void cnic_detach(struct cnic_dev *cdev)
{
	/* SV: XXX: Verify correct order of cleanup. */
#if 0
	vmk_SemaLock(&cnic_driver_info.cnic_lock);
	vmk_ListRemove(&cdev->list);
	vmk_ListInit(&cdev->list);
	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

	if (vmk_BitVectorTest(cdev->flags, CNIC_F_CNIC_UP))
		cnic_stop_hw(cdev);

	cnic_unregister_netdev(cdev);
	/************************************************/
	/* Disable fcoe_rx_dpc. */
	ql_vmk_tasklet_disable(&cdev->fcoe_rx_dpc);
	/* Free fcoe_rx_list. */
	vmk_PktListReleaseAllPkts(cdev->fcoe_rx_list);
	vmk_PktListFree(cdev->fcoe_rx_list);
	/* Destroy fcoe_rx_list lock. */
	vmk_SpinlockDestroy(cdev->fcoe_rx_list_lck);
	/************************************************/
#endif
	vmk_SpinlockDestroy(cdev->ll2_cb_info.ll2_cb_lck);

	cnic_destroy_dma_engines(cdev);
	cnic_free_dev(cdev);
}

static VMK_ReturnStatus
cnic_detachDevice(vmk_Device device)
{
	struct cnic_dev *cdev = NULL;
	VMK_ReturnStatus status = VMK_OK;

	status = vmk_DeviceGetAttachedDriverData(device, (vmk_AddrCookie *) &cdev);
	if (status != VMK_OK) {
		DRV_ERR("Failed to get device handle status=0x%x", status);
		return status;
	}

	cnic_detach(cdev);

	return status;
}

static VMK_ReturnStatus
cnic_quiesceDevice(vmk_Device device)
{
	struct cnic_dev *cdev = NULL;
	VMK_ReturnStatus status = VMK_OK;

	status = vmk_DeviceGetAttachedDriverData(device, (vmk_AddrCookie *) &cdev);
	if (status != VMK_OK) {
		DRV_ERR("Failed to get device handle status=0x%x", status);
		return status;
	}

	DRV_INFO(CNIC_MSG_DRV, "Entered");

	vmk_SemaLock(&cnic_driver_info.cnic_lock);
	vmk_ListRemove(&cdev->list);
	vmk_ListInit(&cdev->list);
	vmk_SemaUnlock(&cnic_driver_info.cnic_lock);

	if (vmk_BitVectorTest(cdev->flags, CNIC_F_CNIC_UP))
		cnic_stop_hw(cdev);

	cnic_unregister_netdev(cdev);
	/************************************************/
	/* Disable fcoe_rx_dpc. */
	ql_vmk_tasklet_disable(&cdev->fcoe_rx_dpc);
	/* Free fcoe_rx_list. */
	vmk_PktListReleaseAllPkts(cdev->fcoe_rx_list);
	vmk_PktListFree(cdev->fcoe_rx_list);
	/* Destroy fcoe_rx_list lock. */
	vmk_SpinlockDestroy(cdev->fcoe_rx_list_lck);
	/************************************************/

	vmk_LogMessage("qcnic(): returned (VMK_OK)");

	return status;
}

static VMK_ReturnStatus
cnic_startDevice(vmk_Device device)
{
	VMK_ReturnStatus status = VMK_OK;
	struct cnic_dev *cdev = NULL;

	DRV_INFO(CNIC_MSG_DRV, "Entered.");

	status = vmk_DeviceGetAttachedDriverData(device,
			(vmk_AddrCookie *)&cdev);
	if (status != VMK_OK) {
		DRV_WARN("Failed to get pci device handle status: %s",
			vmk_StatusToString(status));
		return status;
	}

	if (!cdev) {
		DRV_ERR("Error: cdev is NULL!! Failing startDevice.");
		return VMK_FAILURE;
	}

	/*
	 * Register with net driver once and start hw.
	 * If using MSI-X, we'll request IRQ below in
	 * cnic_start_hw().  During cnic unload or
	 * CNIC_CTL_STOP_CMD, we'll call cnic_stop_hw().
	 * IRQ will only be freed during cnic unload to
	 * prevent context mismatch in VMWare.
	 */
	CNIC_INFO(cdev, CNIC_MSG_DRV, "Before cnic_register_netdev.");
	if ((status = cnic_register_netdev(cdev)) == VMK_OK) {
		CNIC_INFO(cdev, CNIC_MSG_DRV, "Done cnic_register_netdev.");
		if ((status = cnic_start_hw(cdev)) != VMK_OK) {
			CNIC_ERR(cdev, "Failure: cnic_start_hw");
			goto exit;
		}
		CNIC_INFO(cdev, CNIC_MSG_DRV, "Done cnic_start_hw.");
	}

exit:
	CNIC_INFO(cdev, CNIC_MSG_DRV, "Exit (%s)", vmk_StatusToString(status));

	return status;
}

static void cnic_forgetDevice(vmk_Device device)
{
	DRV_INFO(CNIC_MSG_DRV, "Entered");

	DRV_INFO(CNIC_MSG_DRV, "Returned (VMK_OK)");
}

static vmk_DriverOps cnic_DRVOps = {
	.attachDevice = cnic_attachDevice,
	.scanDevice = cnic_scanDevice,
	.detachDevice = cnic_detachDevice,
	.quiesceDevice = cnic_quiesceDevice,
	.startDevice = cnic_startDevice,
	.forgetDevice = cnic_forgetDevice,
};

VMK_ReturnStatus qcnic_get_driver_info(vmk_uint64 cookie, void *buffer_out)
{
	int length = 0;
	int final_length = MAX_KV_STRING_LEN;

	char *buffer = vmk_HeapAlign(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
			(final_length+1), VMK_L1_CACHELINE_SIZE);
	if (!buffer) {
		DRV_ERR("Driver info allocation failed.");
		return VMK_NO_MEMORY;
	}

	vmk_StringFormat(buffer+length, final_length - length, NULL,
			"\nDriver Name: %s\n", CNIC_DRIVER_NAME);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
			"Driver Version: %s\n", CNIC_DRV_MODULE_VERSION);
	length = vmk_Strnlen(buffer, final_length);

	if (cnic_debug & CNIC_MSG_MEM) {
		vmk_StringFormat(buffer+length, final_length - length, NULL,
			"Memory DBG:\n    Total-mem = 0x%lx bytes\n    count = 0x%lx\n",
			cnic_driver_info.mem, cnic_driver_info.count);
		length = vmk_Strnlen(buffer, final_length);
	}

	if (length <= final_length)
		vmk_StringCopy(buffer_out, buffer, length);
	else {
		DRV_WARN("Output data truncated to avoid overflow. "
			"length=0x%x max-length=0x%x", length, final_length);
		vmk_StringCopy(buffer_out, buffer, final_length);
	}

	vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), buffer);

	return VMK_OK;
}

VMK_ReturnStatus qcnic_set_driver_info(vmk_uint64 cookie, void *buffer_in)
{
	int length = vmk_Strnlen("cnic_debug=", MAX_KV_STRING_LEN);

	DRV_INFO(CNIC_MSG_DRV, "Input param: %s", (char *)buffer_in);

	if (vmk_Strncmp(buffer_in, "cnic_debug=", length) == 0) {
		vmk_LogMessage("Before cnic_debug=0x%lx\n", cnic_debug);
		vmk_Sscanf((buffer_in + length), "%lx", &cnic_debug);
		vmk_LogMessage("After cnic_debug=0x%lx\n", cnic_debug);
	}

	return VMK_OK;
}


int init_module(void)
{
	vmk_HeapID heap_id;
	vmk_Name heap_name;
	vmk_HeapCreateProps heap_props;
	VMK_ReturnStatus vmk_stat = VMK_OK;
#if (VMWARE_ESX_DDK_VERSION < 67000)
	vmk_ModuleID module_id;
#endif
	vmk_DriverProps pci_drv_props;
	vmk_LogProperties log_props;
	vmk_TimerQueueProps timerqueue_props;
	vmk_SpinlockCreateProps lock_props;

	vmk_Name kv_name;
	vmk_Name key_name;
	vmk_MgmtProps mgmtProps;

	vmk_LogMessage("qcnic module ver %s\n", CNIC_DRV_MODULE_VERSION);

	/*
	 * Register module with vmkernel
	 */
#if (VMWARE_ESX_DDK_VERSION < 67000)
	vmk_stat = vmk_ModuleRegister(&module_id, VMKAPI_REVISION);

	if (vmk_stat != VMK_OK) {
		DRV_ALERT_MODLOAD_FAIL("vmk_ModuleRegister failed", vmk_stat);
		goto module_register_fail;
	}
	cnic_driver_info.module_id = module_id;
#else
	cnic_driver_info.module_id = vmk_ModuleCurrentID;
#endif
	/*
	 * Set driver name here
	 */
	vmk_stat = vmk_NameInitialize(&cnic_driver_info.driver_name,
			CNIC_DRIVER_NAME);

	/* Creating driver heap */
	heap_props.type = VMK_HEAP_TYPE_SIMPLE;
	vmk_NameFormat(&heap_name, "%s", CNIC_DRIVER_NAME);
	vmk_stat = vmk_NameInitialize(&heap_props.name, (const char *)&heap_name);
	heap_props.module = vmk_ModuleCurrentID;
	heap_props.initial = CNIC_HEAP_INITIAL_SIZE;
	heap_props.max = CNIC_HEAP_MAXIMUM_SIZE;
	heap_props.creationTimeoutMS = VMK_TIMEOUT_NONBLOCKING;

	vmk_stat = vmk_HeapCreate(&heap_props, &heap_id);
	if (vmk_stat != VMK_OK) {
		DRV_ALERT_MODLOAD_FAIL("vmk_HeapCreate failed.", vmk_stat);
		goto heap_create_fail;
	}
	cnic_driver_info.heap_id = heap_id;

	vmk_ModuleSetHeapID(vmk_ModuleCurrentID, heap_id);
	VMK_ASSERT(vmk_ModuleGetHeapID(vmk_ModuleCurrentID) == heap_id);

	/* Creating driver DMA heap */
	heap_props.type = VMK_HEAP_TYPE_CUSTOM;
	vmk_NameFormat(&heap_name, "%s-dma", CNIC_DRIVER_NAME);
	vmk_stat = vmk_NameInitialize(&heap_props.name, (const char *)&heap_name);
	heap_props.module = vmk_ModuleCurrentID;
	heap_props.typeSpecific.custom.physContiguity = VMK_MEM_PHYS_CONTIGUOUS;
	heap_props.typeSpecific.custom.physRange = VMK_PHYS_ADDR_ANY;
	heap_props.initial = CNIC_HEAP_INITIAL_SIZE;
	heap_props.max = CNIC_HEAP_MAXIMUM_SIZE;
	heap_props.creationTimeoutMS = VMK_TIMEOUT_NONBLOCKING;

	vmk_stat = vmk_HeapCreate(&heap_props, &heap_id);
	if (vmk_stat != VMK_OK) {
		DRV_ALERT_MODLOAD_FAIL("vmk_HeapCreate of dma failed.", vmk_stat);
		goto heap_create_fail;
	}
	cnic_driver_info.heap_id_dma = heap_id;

	/*
	 * Register driver log
	 */
	vmk_stat = vmk_NameInitialize(&log_props.name, CNIC_DRIVER_NAME);
	VMK_ASSERT(vmk_stat == VMK_OK);
	if (vmk_stat != VMK_OK) {
		DRV_ALERT_MODLOAD_FAIL(
			"vmk_NameInitialize failed for vmk_LogRegister", vmk_stat);
		goto log_create_fail;
	}
	log_props.module = vmk_ModuleCurrentID;
	log_props.heap = cnic_driver_info.heap_id;
	log_props.defaultLevel = 0;
	log_props.throttle = NULL;
	vmk_stat = vmk_LogRegister(&log_props, &cnic_driver_info.cnic_vmkLog);
	if (vmk_stat != VMK_OK) {
		DRV_ALERT_MODLOAD_FAIL("vmk_LogRegister failed", vmk_stat);
		goto log_create_fail;
	}

	vmk_LogSetCurrentLogLevel(cnic_driver_info.cnic_vmkLog, CNIC_DBG_INFO);

	vmk_stat = vmk_NameInitialize(&timerqueue_props.name, "cnic_timerq");
	VMK_ASSERT(vmk_stat == VMK_OK);
	timerqueue_props.moduleID = vmk_ModuleCurrentID;
	timerqueue_props.heapID = vmk_ModuleGetHeapID(vmk_ModuleCurrentID);
	timerqueue_props.attribs = VMK_TIMER_QUEUE_ATTR_NONE;

	vmk_stat = vmk_TimerQueueCreate(&timerqueue_props,
			&cnic_driver_info.timer_queue);
	if (vmk_stat != VMK_OK) {
		DRV_ALERT_MODLOAD_FAIL("vmk_TimerQueueCreate failed", vmk_stat);
		goto timerqueue_failed;
	}

	vmk_stat = vmk_NameInitialize(&timerqueue_props.name, "cnic_delayed_tq");
	VMK_ASSERT(vmk_stat == VMK_OK);
	timerqueue_props.moduleID = vmk_ModuleCurrentID;
	timerqueue_props.heapID = vmk_ModuleGetHeapID(vmk_ModuleCurrentID);
	timerqueue_props.attribs = VMK_TIMER_QUEUE_ATTR_NONE;

	vmk_stat = vmk_TimerQueueCreate(&timerqueue_props,
			&cnic_driver_info.delayed_tq);
	if (vmk_stat != VMK_OK) {
		DRV_ALERT_MODLOAD_FAIL("vmk_TimerQueueCreate failed", vmk_stat);
		goto delayed_tq_failed;
	}

	/*
	 * Create lock domain
	 */
	vmk_stat = vmk_LockDomainCreate(vmk_ModuleCurrentID,
			cnic_driver_info.heap_id,
			&cnic_driver_info.driver_name,
			&cnic_driver_info.lock_domain);
	if (vmk_stat != VMK_OK) {
		DRV_ALERT_MODLOAD_FAIL("vmk_LockDomainCreate failed", vmk_stat);
		goto lock_dom_fail;
	}

	/*
	 * Create a global driver lock to access global data
	 */
	lock_props.moduleID = vmk_ModuleCurrentID;
	lock_props.heapID = cnic_driver_info.heap_id;
	lock_props.type = VMK_SPINLOCK;
	lock_props.domain = cnic_driver_info.lock_domain;
	lock_props.rank = CNIC_HIGH_LOCK_RANK;
	vmk_NameInitialize(&lock_props.name, "cnic_drv_lock");

	vmk_stat = vmk_SpinlockCreate(&lock_props, &cnic_driver_info.drv_lock);
	if (vmk_stat != VMK_OK) {
		DRV_ALERT_MODLOAD_FAIL("Failed to create global lock for cnic driver",
				vmk_stat);
		goto spin_lock_fail;
	}

	vmk_stat = vmk_BinarySemaCreate(&cnic_driver_info.cnic_lock,
		cnic_driver_info.heap_id, "cnic_lock_mutex");
	if (vmk_stat != VMK_OK) {
		DRV_ALERT_MODLOAD_FAIL("cnic: Failed to create cnic_lock.", vmk_stat);
		goto cnic_lock_fail;
	}

	vmk_Memset(&pci_drv_props, 0, sizeof(pci_drv_props));
	vmk_NameCopy(&pci_drv_props.name, &(cnic_driver_info.driver_name));
	pci_drv_props.moduleID = vmk_ModuleCurrentID;
	pci_drv_props.ops = &cnic_DRVOps;

	/* Register PCI Callback function */
	vmk_stat = vmk_DriverRegister(&pci_drv_props,
			&cnic_driver_info.cnic_driver);

	if (vmk_stat != VMK_OK) {
		DRV_ALERT_MODLOAD_FAIL(
			"cnic: Unable to register PCI callback function", vmk_stat);
		goto pci_callback_register_fail;
	}

	vmk_ListInit(&cnic_driver_info.cnic_dev_list);

	/* key-value pair interface */
	vmk_NameInitialize(&kv_name, "QCNIC");
	vmk_NameInitialize(&cnic_driver_info.kvSig.name,
		vmk_NameToString(&kv_name));

	mgmtProps.modId = vmk_ModuleCurrentID;
	mgmtProps.heapId = cnic_driver_info.heap_id;
	mgmtProps.sig = &cnic_driver_info.kvSig;
	mgmtProps.cleanupFn = NULL;
	mgmtProps.sessionAnnounceFn = NULL;
	mgmtProps.sessionCleanupFn = NULL;
	mgmtProps.handleCookie = 0;
	vmk_stat = vmk_MgmtInit(&mgmtProps, &cnic_driver_info.kv_mgmt_handle);

	VMK_ASSERT(vmk_stat == VMK_OK);

	vmk_NameInitialize(&key_name, "DRIVERINFO");
	vmk_stat = vmk_MgmtAddKey(cnic_driver_info.kv_mgmt_handle,
			VMK_MGMT_KEY_TYPE_STRING,
			&key_name,
			qcnic_get_driver_info,
			qcnic_set_driver_info);
	VMK_ASSERT(vmk_stat == VMK_OK);

	cnic_driver_info.delayed_wq =
		ql_vmk_create_singlethread_workqueue("cnic_delayed_wq",
			VMK_WORLD_SCHED_CLASS_QUICK);


	return VMK_OK;

pci_callback_register_fail:
	vmk_SemaDestroy(&cnic_driver_info.cnic_lock);
cnic_lock_fail:
	vmk_SpinlockDestroy(cnic_driver_info.drv_lock);
spin_lock_fail:
	vmk_LockDomainDestroy(cnic_driver_info.lock_domain);
lock_dom_fail:
	vmk_TimerQueueDestroy(cnic_driver_info.delayed_tq);
delayed_tq_failed:
	vmk_TimerQueueDestroy(cnic_driver_info.timer_queue);
timerqueue_failed:
	vmk_LogUnregister(cnic_driver_info.cnic_vmkLog);
log_create_fail:
	vmk_HeapDestroy(cnic_driver_info.heap_id);
heap_create_fail:
#if (VMWARE_ESX_DDK_VERSION < 67000)
	vmk_ModuleUnregister(vmk_ModuleCurrentID);
module_register_fail:
#endif
	return VMK_FAILURE;
}

void cleanup_module(void)
{
	DRV_INFO(CNIC_MSG_DRV, "Entered");

	if (cnic_driver_info.delayed_wq) {
		ql_vmk_destroy_singlethread_workqueue(cnic_driver_info.delayed_wq);
		cnic_driver_info.delayed_wq = NULL;
	}

	/* Unregister the driver */
	vmk_DriverUnregister(cnic_driver_info.cnic_driver);

	vmk_SemaDestroy(&cnic_driver_info.cnic_lock);
	vmk_SpinlockDestroy(cnic_driver_info.drv_lock);

	vmk_LockDomainDestroy(cnic_driver_info.lock_domain);
	vmk_TimerQueueDestroy(cnic_driver_info.delayed_tq);
	vmk_TimerQueueDestroy(cnic_driver_info.timer_queue);
	DRV_INFO(CNIC_MSG_DRV, "cnic: Unregistering debug log.");
	vmk_LogUnregister(cnic_driver_info.cnic_vmkLog);
	vmk_MgmtDestroy(cnic_driver_info.kv_mgmt_handle);

	vmk_LogMessage("total-mem=0x%lx cnt=0x%lx\n",
		cnic_driver_info.mem, cnic_driver_info.count);
	vmk_HeapDestroy(cnic_driver_info.heap_id_dma);
	vmk_HeapDestroy(cnic_driver_info.heap_id);

#if (VMWARE_ESX_DDK_VERSION < 67000)
	vmk_ModuleUnregister(cnic_driver_info.module_id);
#endif
	return;

	/*
	 * SV: XXX: TODO:
	 * this can be broken down to cnic-dev free in detachDevice.
	 */
#if 0
	cnic_register_release_all_callbacks();
#endif
}


/* ===== SV: XXX: New cnic native changes. END ===== */
