/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#include <vmkapi.h>
#include <qfle3f.h>
#include <qfle3f_vmk.h>
#include <qfle3f_version.h>
#include <qcnic_if.h>
#include "ql_fcoe_mgmt/ql_fcoe.h"

/* extern defines. */
extern VMK_ReturnStatus qfle3f_scsi_scan(void *data);
extern vmk_DeviceOps qfle3fDeviceOps;
extern VMK_ReturnStatus qfle3f_flogi_poll(void *data);


struct qfle3fHBA *qfle3f_vportLookupByPortID(struct qfle3fHBA *phba,
		vmk_uint32 host_portID)
{
	vmk_ListLinks *current = NULL, * nextPtr = NULL;
	struct qfle3fHBA *vhba = NULL;

	vmk_SpinlockLock(phba->vportListLck);
	VMK_LIST_FORALL_SAFE(&phba->vportList, current, nextPtr) {
		vhba = VMK_LIST_ENTRY(current, struct qfle3fHBA, vportList);

		qfle3f_log(vhba, LOG_NPIV, "vport: %p id=0x%x tgd-id=0x%x.",
				vhba, vhba->instance, vhba->initiator_targetPortID);
		if (host_portID == vhba->initiator_targetPortID) {
			qfle3f_log(vhba, LOG_NPIV, "Found vport: %p id=0x%x",
					vhba, vhba->instance);
			break;
		}
	}
	vmk_SpinlockUnlock(phba->vportListLck);

	return vhba;
}

/**
 * Function Name : __qfle3f_vha_release
 *
 * DESCRIPTION:
 * Remove the hba from the func->hba_address
 * and delete the hba and ctlr.
 *
 * PARAMETERS:
 * refcount:
 *
 * RETURN:
 * None.
 *
 * NOTE:
 */
void __qfle3f_vha_release(void *arg)
{
	struct qfle3fHBA *hba = (struct qfle3fHBA *) arg;

	qfle3f_log(hba, LOG_NPIV, "Enter: %p of type = %d", hba, hba->type);
	vmk_LogMessage("Enter: %s: %d", __func__, __LINE__);
	if (hba->type != HBA_VIRTUAL)
		return;

	/* qfle3f_deallocate_vp_id(hba); */
	/* vmk_BitVectorSet(base_vha->dpc_flags, VPORT_CLEANUP_NEEDED); */
	/* qfle3f_wake_dpc(vha); */

	qfle3f_log(hba, LOG_NPIV, "Freeing hba : %p", hba);
	/* qfcoe_ctlr_destroy(&hba->ctlr); */

	qfle3f_log(hba, LOG_NPIV, "Calling vport Delete");

	vmk_SpinlockDestroy(hba->scsiScanListLock);

	vmk_HeapFree(qfle3fDriverInfo.moduleHeapID, hba);
	hba = NULL;
}

struct qfle3fHBA *qfle3f_get_hba(struct qfle3fHBA *hba) {

    /* Sneaking way of checking if the hba is physical or not.
     * Doing this rather than checking the type field in the
     * hba(physical or virtual) as hba can be NULL.
     */
    if(hba->type == HBA_PHYSICAL) return hba;

    int i=0;
    vmk_Bool found = VMK_FALSE;
	/* vmk_SpinlockLock(func->vport_lock); */
    for(i=0; i < 1024; i++) {
        if(!qfle3fDriverInfo.hbaArray[i])
            continue;

        if(qfle3fDriverInfo.hbaArray[i] == hba) {
            ql_vmk_ref_get(&hba->kref);
            found = VMK_TRUE;
        }
    }
	/* vmk_SpinlockUnlock(func->vport_lock); */

    if(found == VMK_TRUE) return hba;

    return NULL;
}

void qfle3f_put_hba(struct qfle3fHBA *hba) {

    if(hba->type == HBA_PHYSICAL) return;

    int i;
	vmk_Bool found = VMK_FALSE;
	/* vmk_SpinlockLock(func->vport_lock); */
    for(i=0; i < 1024; i++) {
        if(qfle3fDriverInfo.hbaArray[i] == hba) {
			qfle3f_warning(hba, "%s: %d : %ld", __func__, i, hba->kref);
			if(vmk_AtomicRead64(&hba->kref) == 1) {
				qfle3f_warning(hba, "%s: Deleting from global array", __func__);
				qfle3fDriverInfo.hbaArray[i] = NULL;
			}
			found = VMK_TRUE;
            break;
        }
    }
	/* vmk_SpinlockUnlock(func->vport_lock); */

	if(found == VMK_TRUE)
		ql_vmk_ref_put(&hba->kref, __qfle3f_vha_release, (void *) hba);
}

VMK_ReturnStatus qfle3f_vadapter_create(vmk_Device vmkdev,
		vmk_VportWwn *wwpn_array, vmk_VportWwn *wwnn_array,
		vmk_ScsiAdapter **vadapter, vmk_uint8 is_pre_boot)
{
	struct qfle3fHBA *hba = NULL, *vhba = NULL;
	VMK_ReturnStatus status = VMK_FAILURE;
	int i = 0;
	vmk_uint64 wwpn = 0, wwnn = 0;

	qfle3f_charArrayToVmku64((vmk_uint8 *)wwpn_array, WWN_SIZE, &wwpn);
	qfle3f_charArrayToVmku64((vmk_uint8 *)wwnn_array, WWN_SIZE, &wwnn);
	qfle3f_notice(hba, "Creating vport: 0x%lx:0x%lx\n", wwpn, wwnn);

	status = vmk_DeviceGetAttachedDriverData(vmkdev, (vmk_AddrCookie *)&hba);
	if (status != VMK_OK) {
		qfle3f_err(hba, "qfle3f: Failed to retrieve hba!\n");
		return VMK_BAD_PARAM;
	}

	vhba = qfle3f_interface_create(hba->cnic, 1);
	if (vhba == NULL) {
		qfle3f_err(hba, "Failed to create vhba.");
		return VMK_FAILURE;
	}

	/* NPIV: Init vhba */
	vhba->type = HBA_VIRTUAL;
	vhba->wwpn = wwpn;
	vhba->wwnn = wwnn;
	vmk_Memset(&vhba->tag, 0, sizeof(16));
	vhba->phba = hba;
	vhba->pdev = hba->cnic->pdev;
	vhba->deviceCreatedByCnicDriver = vmkdev;
	vhba->qDepth = 1024;
	vhba->vmk_pci_addr = hba->vmk_pci_addr;
	vhba->pdev_id = hba->pdev_id;
	vhba->devLossTimeout = qfle3f_devloss_tmo;
	if (is_pre_boot)
		vmk_BitVectorSet(vhba->flags2, QFLE3F_NPIV_PREBOOT);

	/* To be dropped at vport_delete */
	ql_vmk_ref_init(&vhba->kref);

	qfle3f_log(hba, LOG_NPIV, "Memset target list: %ld",
			(sizeof(struct qfle3f_rport *) * QFLE3_FCOE_NUM_CONNECTIONS));
	qfle3f_log(hba, LOG_NPIV, "Initialize target list.");
	for (i = 0; i < QFLE3_FCOE_NUM_CONNECTIONS; i++)
		vhba->targetOffloadList[i] = NULL;

	vhba->num_ofld_sess = 0;


	qfle3f_log(vhba, LOG_NPIV, "Calling scsi adapter alloc.");
	vhba->scsiAdapter = vmk_ScsiAllocateAdapter();
	if (vhba->scsiAdapter == NULL) {
		status = VMK_NO_MEMORY;
		qfle3f_warning(vhba, "Could not obtain memory for scsi vhba %s",
				vmk_StatusToString(status));
		goto alloc_scsi_fail;
	}

	vhba->nicDevice = hba->cnic->ldev;

	vhba->host_no = vhba->scsiAdapter->hostNum;
	vhba->vmhbaName = vmk_ScsiGetAdapterName(vhba->scsiAdapter);
	vmk_StringFormat(vhba->virtualName, 64, NULL,
			"vport-0x%x-%s", vhba->instance, vhba->vmhbaName);
	vhba->vmkScsiAdapterDevice = NULL;

	status = qfle3f_scsiAdapterSetup(vhba);
	if (status != VMK_OK) {
		goto setup_adapter_fail;
	}

	qfle3f_log(vhba, LOG_NPIV, "cnic->fip_mac: ");
	qfle3f_log(vhba, LOG_NPIV, VMK_ETH_ADDR_FMT_STR,
			VMK_ETH_ADDR_FMT_ARGS(hba->cnic->fip_mac));
	vmk_Memcpy(vhba->dataSourceMacAddress, hba->cnic->fip_mac,
			VMK_ETH_ADDR_LENGTH);
	qfle3f_log(vhba, LOG_NPIV, "vhba->dataSourceMacAddress: ");
	qfle3f_log(vhba, LOG_NPIV, VMK_ETH_ADDR_FMT_STR,
			VMK_ETH_ADDR_FMT_ARGS(vhba->dataSourceMacAddress));

	qfle3f_log(vhba, LOG_NPIV, "Setting the ql_fcoe_mgmt Interface");
	vhba->qlfcoeAdapter = hba->qlfcoeAdapter;

	qfle3f_log(vhba, LOG_NPIV, "Creating the scsi scan world");
	vmk_Name name;
	vmk_ListInit(&vhba->scsiScanList);
	vmk_NameFormat(&name, "scsiScanListLock-%u", vhba->instance);
	qfle3f_vmk_spin_lock_init(&vhba->scsiScanListLock, LOCK_RANK_HIGHEST,
			(const char *)&name);

	vmk_AtomicWrite64(&vhba->wakeupPending, VMK_FALSE);
	/* Scsi scan world per scsi vhba */
	vmk_WorldProps scsi_scan_props;
	char scsi_scan_world_name[20];

	vmk_Memset(&scsi_scan_props, 0, sizeof(vmk_WorldProps));
	scsi_scan_props.heapID = qfle3fDriverInfo.moduleHeapID;

	scsi_scan_props.moduleID = vmk_ModuleCurrentID;
	scsi_scan_props.startFunction = (void *)qfle3f_scsi_scan;
	scsi_scan_props.data = (void *)vhba;
	scsi_scan_props.schedClass = VMK_WORLD_SCHED_CLASS_QUICK;
	vmk_StringFormat(scsi_scan_world_name, 20, NULL,
			"qfle3f-scsi-scan-%u", vhba->instance);
	scsi_scan_props.name = scsi_scan_world_name;
	status = vmk_WorldCreate(&scsi_scan_props, &(vhba->scsiScanWorldID));
	if (status != VMK_OK) {
		qfle3f_warning(vhba, "FAILED scsi_scan_world create :%s",
				vmk_StatusToString(status));
		goto scsi_scan_thread_fail;
	}

	qfle3f_log(vhba, LOG_NPIV, "scsi scan world created. world id = %d.",
			vhba->scsiScanWorldID);

	vmk_AtomicWrite64(&vhba->pollwakeupPending, VMK_FALSE);
	/* Flogi poll world (physical port's) */
	vmk_WorldProps flogi_poll_props;
	char flogi_poll_world_name[20];

	vmk_Memset(&flogi_poll_props, 0, sizeof(vmk_WorldProps));
	flogi_poll_props.heapID = qfle3fDriverInfo.moduleHeapID;

	flogi_poll_props.moduleID = vmk_ModuleCurrentID;
	flogi_poll_props.startFunction = (void *)qfle3f_flogi_poll;
	flogi_poll_props.data = (void *)vhba;
	flogi_poll_props.schedClass = VMK_WORLD_SCHED_CLASS_DEFAULT;
	vmk_StringFormat(flogi_poll_world_name, 20, NULL,
			"qfle3f-flogi-poll-%u", vhba->instance);
	flogi_poll_props.name = flogi_poll_world_name;
	status = vmk_WorldCreate(&flogi_poll_props, &(vhba->flogiPollWorldID));
	if (status != VMK_OK) {
		qfle3f_warning(vhba, "FAILED flogi_poll_world create :%s",
				vmk_StatusToString(status));
	} else {
		qfle3f_log(vhba, LOG_INIT, "flogi poll world created. world id = %d.",
						vhba->flogiPollWorldID);
	}

	vmk_BitVectorClear(vhba->flags2, QFLE3F_CNIC_REGISTERED);

	/* NPIV: skip cnic register and ll2 callback registeration. */

	qfle3f_log(vhba, LOG_NPIV, "Setting the timer work queue");
	vhba->timerWorkQueue =
		ql_vmk_create_singlethread_workqueue("qfle3f_timer_wq",
				VMK_WORLD_SCHED_CLASS_QUICK);
	if (VMK_UNLIKELY(!vhba->timerWorkQueue)) {
		qfle3f_err(vhba, "cannot create timer_wq");
		goto timerWorkQueue_creation_failed;
	}

#ifdef __FCOE_IF_RESTART_WQ__
	vhba->fcoeInterfaceRestartWorkQueue =
		ql_vmk_create_singlethread_workqueue("qfle3f_fcoe_if_restart",
				VMK_WORLD_SCHED_CLASS_QUICK);
	if (VMK_UNLIKELY(!vhba->fcoeInterfaceRestartWorkQueue)) {
		qfle3f_err(vhba, "cannot create fcoeInterfaceRestartWorkQueue");
		status = VMK_FAILURE;
		goto fcoeInterfaceRestartWorkQueue_creation_failed;
	}
#endif

	vhba->ioAllowed = VMK_FALSE;

	/* NPIV: Need to share parent hba's cmd-manager and task context. */
	vhba->commandManager = hba->commandManager;
	vhba->taskContext = hba->taskContext;
	//SV: NPIV: XXX: dma addresses may not be required to be shared.
	vmk_Memcpy(&vhba->taskContextDMA, &hba->taskContextDMA,
			(sizeof(struct qla_dmamem) * QFLE3F_TASK_CTX_ARR_SZ));
	vhba->taskContextBDTable = hba->taskContextBDTable;
	vmk_Memcpy(&vhba->taskContextBDDMA, &hba->taskContextBDDMA,
			sizeof(struct qla_dmamem));

	vmk_BitVectorSet(vhba->hbaState, ADAPTER_STATE_READY);

	qfle3f_log(vhba, LOG_NPIV, "vhba->qDepth = %d", vhba->qDepth);

	vmk_HashProperties hashTableProps;
	vmk_Memset(&hashTableProps, 0, sizeof(hashTableProps));

	hashTableProps.moduleID = vmk_ModuleCurrentID;
	hashTableProps.heapID = vmk_ModuleGetHeapID(vmk_ModuleCurrentID);
	hashTableProps.keyType = VMK_HASH_KEY_TYPE_OPAQUE;
	hashTableProps.keyFlags = VMK_HASH_KEY_FLAGS_LOCAL_COPY;
	hashTableProps.keySize = sizeof(vmk_uint32);
	hashTableProps.nbEntries = QFLE3_FCOE_NUM_CONNECTIONS;
	hashTableProps.acquire = NULL;
	hashTableProps.release = NULL;

	int size = vmk_HashGetAllocSize(QFLE3_FCOE_NUM_CONNECTIONS);
	qfle3f_notice(vhba, "Best estimate amount for Hash Table: %d", size);

	status = vmk_HashAlloc(&hashTableProps, &vhba->connIdHashTable);
	if (status != VMK_OK) {
		qfle3f_notice(vhba, "HashTable Alloc failed. status=%s",
				vmk_StatusToString(status));
		goto hashTableAllocFailed;
	}

	*vadapter = vhba->scsiAdapter;

	for (i = 0; i < 1024; i++) {
		if (!qfle3fDriverInfo.hbaArray[i]) {
			qfle3fDriverInfo.hbaArray[i] = hba;
			break;
		}
	}

	/* If link is up, issue vport login */
	if(vhba->cnic->uplinkLinkState.state == VMK_LINK_STATE_UP) {
		vmk_BitVectorSet(vhba->hbaState, ADAPTER_STATE_UP);
		vmk_AtomicWrite64(&vhba->pollwakeupPending, VMK_TRUE);
		vmk_WorldForceWakeup(vhba->flogiPollWorldID);
		qfle3f_log(vhba, LOG_NPIV, "Sleeping for 4 seconds.");

		vmk_WorldSleep(4 * VMK_USEC_PER_SEC);
	}

	qfle3f_notice(vhba, "Vport Created Successfully");

	return VMK_OK;

hashTableAllocFailed:
#ifdef __FCOE_IF_RESTART_WQ__
	ql_vmk_destroy_singlethread_workqueue(hba->fcoeInterfaceRestartWorkQueue);
#endif
fcoeInterfaceRestartWorkQueue_creation_failed:
	ql_vmk_destroy_singlethread_workqueue(vhba->timerWorkQueue);
timerWorkQueue_creation_failed:
	vmk_WorldDestroy(vhba->flogiPollWorldID);
	vmk_WorldWaitForDeath(vhba->flogiPollWorldID);
	vmk_WorldDestroy(vhba->scsiScanWorldID);
	vmk_WorldWaitForDeath(vhba->scsiScanWorldID);
scsi_scan_thread_fail:
setup_adapter_fail:
	vmk_ScsiFreeAdapter(vhba->scsiAdapter);
alloc_scsi_fail:
	qfle3f_vport_interface_destroy(vhba);
	vmk_LogMessage("Failure: (%s)", vmk_StatusToString(status));

	return status;
}

VMK_ReturnStatus qfle3f_vport_create(vmk_Device vmkdev,
		vmk_VportWwn *wwpn_array, vmk_VportWwn *wwnn_array,
		vmk_ScsiAdapter **vadapter)
{
	struct qfle3fHBA *hba = NULL;
	VMK_ReturnStatus status = VMK_FAILURE;

	qfle3f_log(hba, LOG_NPIV, "Enter %s.\n", __func__);

	status = vmk_DeviceGetAttachedDriverData(vmkdev, (vmk_AddrCookie *)&hba);
	if (status != VMK_OK) {
		qfle3f_err(hba, "qfle3f: Failed to retrieve hba!\n");
		return VMK_BAD_PARAM;
	}

	if (hba == NULL || hba->cnic == NULL) {
		qfle3f_warning(hba, "Parent hba is NULL.");
		return VMK_FAILURE;
	}

	if (!vmk_BitVectorTest(hba->initDone, QFLE3F_FW_INIT_DONE)) {
		qfle3f_err(hba, "phba: FW INIT not done. "
				"vn ports cannot be created on this hba.");
		return VMK_FAILURE;
	}

	status = qfle3f_vadapter_create(vmkdev, wwpn_array, wwnn_array, vadapter, 0);

	return status;
}

VMK_ReturnStatus qfle3f_vport_create_from_nvram(struct qfle3fHBA *baseHba)
{
	struct qcnic_fc_npiv_tbl *npiv_tbl = NULL;
	vmk_ScsiAdapter *vadapter = NULL;
	VMK_ReturnStatus status = VMK_FAILURE;
	int i = 0;

	if (baseHba == NULL || baseHba->cnic == NULL ||
			baseHba->cnic->cm_get_fc_npiv_tbl == NULL) {
		qfle3f_err(baseHba, "hba is NULL or cnic-ops not supported.");
		return VMK_BAD_PARAM;
	}

	npiv_tbl = (struct qcnic_fc_npiv_tbl *)
		qfle3f_alloc(sizeof(struct qcnic_fc_npiv_tbl));
	if (npiv_tbl == NULL) {
		qfle3f_err(baseHba, "Failed to allocate npiv_tbl.");
		return VMK_NO_MEMORY;
	}

	qfle3f_log(baseHba, LOG_NPIV, "Get FC NPIV table.");
	status = baseHba->cnic->cm_get_fc_npiv_tbl(baseHba->cnic, npiv_tbl);
	if (status == VMK_OK) {
		/* Check for empty npiv-table. */
		if (npiv_tbl->count == 1) {
			if (npiv_tbl->wwpn[0][0] == 0 && npiv_tbl->wwpn[0][1] == 0 &&
				npiv_tbl->wwpn[0][2] == 0 && npiv_tbl->wwpn[0][3] == 0 &&
				npiv_tbl->wwpn[0][4] == 0 && npiv_tbl->wwpn[0][5] == 0 &&
				npiv_tbl->wwpn[0][6] == 0 && npiv_tbl->wwpn[0][7] == 0) {
				/* This is a zero entry npiv table. */
				npiv_tbl->count = 0;
			}
		}

		qfle3f_log(baseHba, LOG_NPIV, "Create Pre-Boot vports: %d",
				npiv_tbl->count);
		for (i = 0; i < npiv_tbl->count; i++) {
			qfle3f_vadapter_create(baseHba->deviceCreatedByCnicDriver,
					(vmk_VportWwn *)npiv_tbl->wwpn[i],
					(vmk_VportWwn *)npiv_tbl->wwnn[i], &vadapter, 1);
		}
	} else {
		qfle3f_warning(baseHba, "Pre-Boot: status=%s",
				vmk_StatusToString(status));
	}

	qfle3f_free(npiv_tbl);

	return VMK_OK;
}

void qfle3f_disable_vp(struct qfle3fHBA *vhba)
{
	/* struct qfle3f_hba *search_hba = NULL; */
	/* vmk_ListLinks *current, *next_ptr; */
	/* struct qfle3f_rport *rport = NULL; */
	/* vmk_ListLinks *cur_rport; */
	/* int wait_cnt = 0; */

	/* This is essential to disable to NPIV port for the firmware.
	 * No need to do this as all the session for the NPIV port would
	 * have been already onloaded.
	 */

    /* LogoutAllFabrics(vhba->qlfcoeAdapter, ONLOGOUT_HOLD, VMK_FALSE); */

    vmk_uint8 wwpn_array[WWN_SIZE];
    vmk_uint64 npivStatus = 0;

    qfle3f_u64ToCharArray_BE(vhba->wwpn, WWN_SIZE, wwpn_array);

    qfle3f_log(vhba, LOG_NPIV, "Calling LogoutVirtualPort");
    LogoutVirtualPort(vhba->qlfcoeAdapter, wwpn_array, NULL, NULL, &npivStatus);
    qfle3f_log(vhba, LOG_NPIV, "Returned from LogoutVirtualPort");
}

/* this function is to be used for vport only */
void qfle3f_vport_interface_destroy(struct qfle3fHBA *vhba)
{

	if (vhba->type != HBA_VIRTUAL) {
		qfle3f_log(vhba, LOG_NPIV, "%s called for physical port.", __func__ );
		return;
	}

	vmk_HashRelease(vhba->connIdHashTable);

#ifdef __FCOE_IF_RESTART_WQ__
	ql_vmk_destroy_singlethread_workqueue(vhba->fcoeInterfaceRestartWorkQueue);
#endif

	ql_vmk_destroy_singlethread_workqueue(vhba->timerWorkQueue);

	vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, vhba->hbaType);
	vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, vhba->initDone);
	vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, vhba->flags2);
	vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, vhba->flags);
	vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, vhba->hbaState);
	vmk_SemaDestroy(&vhba->hbaMutex);
	vmk_SemaDestroy(&vhba->ioctlMutex);
	vmk_SpinlockDestroy(vhba->hbaLock);
	vmk_SpinlockDestroy(vhba->scsiScanListLock);

	vmk_SemaLock(&qfle3fDriverInfo.drvLock);
	qfle3fDriverInfo.host_count--;
	vmk_BitVectorClear(qfle3fDriverInfo.hostInstanceMap, vhba->instance);
	num_hosts--;
	vmk_ListRemove(&vhba->link);
	vmk_SemaUnlock(&qfle3fDriverInfo.drvLock);

	vmk_SpinlockLock(vhba->phba->vportListLck);
	vhba->vp_id = QFLE3F_NPIV_INVALID;
	vmk_ListRemove(&vhba->vportList);
	vhba->phba->vportsInUse--;
	vmk_SpinlockUnlock(vhba->phba->vportListLck);

}

void qfle3f_vport_quiesce(struct qfle3fHBA *vhba)
{
	vmk_uint32 i = 0;
	struct qfle3f_rport *target = NULL;
	VMK_ReturnStatus status = VMK_OK;

	if(vhba == NULL) {
		qfle3f_warning(vhba, "hba is NULL.");
		return;
	}

	if (vmk_BitVectorTest(vhba->hbaState, ADAPTER_STATE_QUIESCE)) {
        return;
	}
    vmk_BitVectorSet(vhba->hbaState, ADAPTER_STATE_QUIESCE);

    qfle3f_log(vhba, LOG_NPIV, "Ready to upload, num_ofld_sess =%d\n", vhba->num_ofld_sess);
    vmk_SemaLock(&vhba->hbaMutex);
    for (i = 0; i < QFLE3_FCOE_NUM_CONNECTIONS; i++) {
            target = vhba->targetOffloadList[i];
            if(target) {
                if (!(vmk_BitVectorTest(target->flags, QFLE3F_FLAG_OFFLOADED))) {
                    continue;
                }
                vmk_BitVectorClear(target->flags, QFLE3F_FLAG_SESSION_READY);
                qfle3f_uploadSession(vhba, target);
                vhba->num_ofld_sess--;

				status = ql_fcoe_cancel_delayed_work_sync(&target->Sess->RportEvHandlerWorkItem);
				if (status == VMK_OK) {
					qfle3f_log(vhba, LOG_NPIV,
						"successfully Cancelled ql_fcoe_cancel_delayed_work_sync sess = %p", target->Sess);
					DereferenceSession(target->Sess);
				} else {
					qfle3f_log(vhba, LOG_NPIV,
						"Failed to Cancel ql_fcoe_cancel_delayed_work_sync sess = %p status = %s",
						target->Sess,  vmk_StatusToString(status));
				}

				qfle3f_removeConn(vhba, target);
			}
	}
	vmk_SemaUnlock(&vhba->hbaMutex);
	qfle3f_log(vhba, LOG_NPIV, "All sessions are uploaded, num_ofld_sess =%d\n", vhba->num_ofld_sess);
}

void qfle3f_vport_free(void *arg)
{
	int i = 0;

	struct qfle3fHBA *vhba = (struct qfle3fHBA *)arg;

	/* Removal from the list happens later */
	vmk_SpinlockLock(vhba->phba->vportListLck);
	for(i = 0; i < QFLE3F_MAX_NPIV; i++) {
		if(vhba->phba->vhba_address[i] == vhba) {
			vhba->phba->vhba_address[i] = NULL;
		qfle3f_log(vhba, LOG_NPIV, "deleting vport from array");
			break;
		}
	}
	vmk_SpinlockUnlock(vhba->phba->vportListLck);

	qfle3f_vport_interface_destroy(vhba);
	vmk_LogMessage("%s: qfle3f: Vport being freed", __func__);
	qfle3f_free(vhba);
}

VMK_ReturnStatus qfle3f_vport_delete(vmk_Device vmkdev,
		vmk_ScsiAdapter *vadapter)
{
	struct qfle3fHBA *baseHba = NULL, *vhba = NULL;
	vmk_uint8 wwpn_array[WWN_SIZE], wwnn_array[WWN_SIZE];
	VMK_ReturnStatus status = VMK_OK;

	vmk_LogMessage("Enter %s.", __func__);

	status = vmk_DeviceGetAttachedDriverData(vmkdev, (vmk_AddrCookie *)&baseHba);
	if (status != VMK_OK) {
		vmk_LogMessage("qfle3f: Failed to retrieve baseHba!");
		return VMK_BAD_PARAM;
	}

	if (baseHba== NULL || baseHba->cnic == NULL) {
		vmk_LogMessage("qfle3f: Parent hba is NULL.");
		return VMK_FAILURE;
	}

	vhba = (struct qfle3fHBA *)vadapter->clientData;
	VMK_ASSERT(vhba != NULL);

    vmk_BitVectorSet(vhba->hbaState, ADAPTER_STATE_VPORT_DELETE);

	qfle3f_log(vhba, LOG_NPIV, "Vport-Delete: %s=%p vport-no=0x%x",
			vhba->vmhbaName, vhba, vhba->host_no);

	/* Destroy the scsi scan world */
	vmk_WorldDestroy(vhba->scsiScanWorldID);
	status = vmk_WorldWaitForDeath(vhba->scsiScanWorldID);
	if(status != VMK_OK) {
		qfle3f_log(vhba, LOG_NPIV, "Unable to destroy the scsi scan world");
		return VMK_BAD_PARAM;
	}

	vmk_ListInit(&vhba->scsiScanList);
	vhba->scsiScanWorldID = VMK_INVALID_WORLD_ID;

	/* Destroying the flogi poll world */
	status = vmk_WorldDestroy(vhba->flogiPollWorldID);
	if(status != VMK_OK) {
		qfle3f_log(vhba, LOG_NPIV, "Unable to find flogi poll world");
	} else {
		qfle3f_log(vhba, LOG_NPIV, "successfully posted work to destroy flogi poll world");
	}
	status = vmk_WorldWaitForDeath(vhba->flogiPollWorldID);
	if(status != VMK_OK) {
		qfle3f_log(vhba, LOG_NPIV, "Unable to destroy the flogi poll world");
		return VMK_BAD_PARAM;
	}

	vhba->flogiPollWorldID = VMK_INVALID_WORLD_ID;

	vmk_BitVectorClear(vhba->hbaState, ADAPTER_STATE_UP);
	vmk_BitVectorClear(vhba->hbaState, ADAPTER_STATE_GOING_DOWN);
	vmk_BitVectorClear(vhba->hbaState, ADAPTER_STATE_READY);

	/* Disable vport operations. */
	qfle3f_log(vhba, LOG_NPIV, "%s:%d :: Calling qfle3f_disable_vp", __func__, __LINE__);
	qfle3f_disable_vp(vhba);
	qfle3f_log(vhba, LOG_NPIV, "%s:%d :: Returned from qfle3f_disable_vp", __func__, __LINE__);

	qfle3f_log(vhba, LOG_NPIV, "Waiting for all sessions onload, num_rport = %d",
			vhba->num_rport);
	vmk_SemaLock(&vhba->hbaMutex);
	while (vhba->num_rport) {
		vmk_SemaUnlock(&vhba->hbaMutex);
		vmk_WorldSleep(10 * VMK_USEC_PER_MSEC);
		vmk_SemaLock(&vhba->hbaMutex);
	}
	vmk_SemaUnlock(&vhba->hbaMutex);


	qfle3f_vport_quiesce(vhba);
	qfle3f_u64ToCharArray_BE(vhba->wwpn, WWN_SIZE, wwpn_array);
	qfle3f_u64ToCharArray_BE(vhba->wwnn, WWN_SIZE, wwnn_array);
	ql_wait_for_fabric_deletion(vhba->qlfcoeAdapter, wwpn_array, wwnn_array, VIRTUAL_FABRIC);

	vhba->qlfcoeAdapter = NULL;
	qfle3f_log(vhba, LOG_NPIV, "Free mgmtAdapter.t.fcoe is done");
	qfle3f_free(vhba->scsiAdapter->mgmtAdapter.t.fcoe);

	/* ql_vmk_free(vhba->vmk_scsi_adapter->mgmtAdapter.t.fc); */
	vmk_LogMessage("%s:%d :: Call scsi adapter free ", __func__, __LINE__);
	vmk_ScsiFreeAdapter(vhba->scsiAdapter);
	vmk_LogMessage("%s:%d :: Return from scsi adapter free ", __func__, __LINE__);

	ql_vmk_ref_put(&vhba->kref, qfle3f_vport_free, (void *)vhba);

	return status;
}

VMK_ReturnStatus qfle3f_vport_getinfo(vmk_Device vmkdev,
		vmk_uint32 *max_vports, vmk_uint32 *vport_inuse)
{
	struct qfle3fHBA *hba = NULL;
	VMK_ReturnStatus status = VMK_FAILURE;

	vmk_LogMessage("Enter %s.\n", __func__);

	status = vmk_DeviceGetAttachedDriverData(vmkdev, (vmk_AddrCookie *)&hba);
	if (status != VMK_OK) {
		vmk_LogMessage("qfle3f: Failed to retrieve hba!\n");
		return VMK_BAD_PARAM;
	}

	if (hba == NULL || hba->cnic == NULL) {
		qfle3f_warning(hba, "Parent hba is NULL.");
		return VMK_FAILURE;
	}

	*max_vports = QFLE3F_MAX_NPIV;
	*vport_inuse = hba->vportsInUse;

	qfle3f_log(hba, LOG_NPIV, "Max VPORTS = %d : VPORT INUSE = %d",
			*max_vports, *vport_inuse);

	return status;
}

VMK_ReturnStatus qfle3f_registerPreBootVport(struct qfle3fHBA *vhba)
{
	VMK_ReturnStatus status = VMK_FAILURE;
	vmk_BusType bus_type;
	vmk_Name bus_name;
	struct qfle3fHBA *hba = vhba->phba;
    vmk_Device parent = hba->deviceCreatedByCnicDriver;
	vmk_DeviceID devId;
	vmk_DeviceProps deviceProps;


	status = vmk_NameInitialize(&bus_name, VMK_LOGICAL_BUS_NAME);
	if (status != VMK_OK) {
		qfle3f_warning(vhba, "Failed to initialize bus name: %s",
				vmk_StatusToString(status));
		return status;
	}

	status = vmk_BusTypeFind(&bus_name, &bus_type);
	if (status != VMK_OK) {
		qfle3f_warning(vhba,"Failed to BusTypeFind: %s",
				vmk_StatusToString(status));
		return status;
	}

	/* Create Logical device for iSCSI protocol driver. */
	status = vmk_LogicalCreateBusAddress(
			qfle3fDriverInfo.qfle3fDriver, parent, vhba->instance,
			&devId.busAddress, &devId.busAddressLen);

	if (status != VMK_OK) {
		qfle3f_warning(vhba, "Failed to created logicalbus: %s",
				vmk_StatusToString(status));
		goto  fail_register_logical_bus;
	}

	devId.busType = bus_type;
	devId.busIdentifier = VMK_SCSI_PSA_DRIVER_BUS_ID;
	devId.busIdentifierLen = vmk_Strnlen(devId.busIdentifier,
			VMK_MISC_NAME_MAX);
	deviceProps.registeringDriver = qfle3fDriverInfo.qfle3fDriver;
	deviceProps.deviceID = &devId;
	deviceProps.deviceOps = &qfle3fDeviceOps;
	deviceProps.registeringDriverData.ptr = vhba;
	deviceProps.registrationData.ptr = vhba->scsiAdapter;

	status = vmk_DeviceRegister(&deviceProps, parent,
			&vhba->vmkScsiAdapterDevice);
	if (status != VMK_OK) {
		qfle3f_warning(vhba, "Failed to register device: %s",
				vmk_StatusToString(status));
	} else {
		qfle3f_log(vhba, LOG_NPIV,
				"Registered Logical device (local %s:global %s) with device layer.",
				vmk_ScsiGetAdapterName(vhba->scsiAdapter), devId.busAddress);
		qfle3f_log(vhba, LOG_NPIV, "Returned (VMK_OK)");
	}

	vmk_LogicalFreeBusAddress(qfle3fDriverInfo.qfle3fDriver, devId.busAddress);

fail_register_logical_bus:
	vmk_BusTypeRelease(bus_type);

	return status;
}

vmk_ScsiVportOps qfle3f_vportops = {
	.createVport = qfle3f_vport_create,
	.deleteVport = qfle3f_vport_delete,
	.getVportInfo = qfle3f_vport_getinfo,
};
