/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

/*
 * qfle3f_hwi.c: QLogic 10 Gigabit ESX Ethernet FCoE offload driver.
 *
 * This file contains the code that low level functions that interact
 * with 57712 FCoE firmware.
 */

#include "qfle3f.h"
#include "ql_fcoe_mgmt/ql_fcoe.h"
#include "ql_fcoe_mgmt/ql_fcoe_session.h"
#include "ql_fcoe_mgmt/ql_fcoe_fabric.h"
#include "ql_fcoe_mgmt/FCoE_Framing.h"

static void qfle3f_processCQCompletion(struct qfle3f_rport *target, vmk_uint16 wqe);
static void qfle3f_fastpathNotification(struct qfle3fHBA *hba,
					struct fcoe_kcqe *new_cqe_kcqe);
static void qfle3f_processOffloadCompletion(struct qfle3fHBA *hba,
					struct fcoe_kcqe *ofld_kcqe);
static void qfle3f_processConnectionEnableCompletion(struct qfle3fHBA *hba,
						struct fcoe_kcqe *ofld_kcqe);
static void qfle3f_initializationFailure(struct qfle3fHBA *hba, vmk_uint32 err_code);
static void qfle3f_processConnectionDestroyCompletion(struct qfle3fHBA *hba,
					struct fcoe_kcqe *conn_destroy);

void qfle3f_printBuffer(struct qfle3fHBA *hba, vmk_uint8 *b, vmk_uint32 size,
		char *printBanner)
{
	vmk_uint32 cnt;
	vmk_uint8 c;
	char printString[2048];
	char tempString[128];
	size = (size>256)?256:size;

	qfle3f_log(hba, LOG_FRAME, "****** %s ******", printBanner);
	qfle3f_log(hba, LOG_FRAME, "--------------------------------------------"
			"--------------------------");

	printString[0] = '\0';

	cnt = 0;
	vmk_StringFormat(tempString, 128, NULL, "%02x==>", cnt);
	vmk_StringCat(printString, tempString, 2048);

	for (cnt = 0; cnt < size;) {
		c = *b++;
		vmk_StringFormat(tempString, 128, NULL, "  %02x", c);
		vmk_StringCat(printString, tempString, 2048);
		cnt++;

		if ((!(cnt % 16) && (cnt != 0)) || ((cnt) == size)) {
			qfle3f_log(hba, LOG_FRAME, "%s", printString);
			printString[0] = '\0';
			vmk_StringFormat(tempString, 128, NULL, "%02x==>", cnt);
			vmk_StringCat(printString, tempString, 2048);
		}
	}
	qfle3f_log(hba, LOG_FRAME, "--------------------------------------------"
			"--------------------------");
}

int qfle3f_sendStatRequest(struct qfle3fHBA *hba)
{
	struct fcoe_kwqe_stat stat_req;
	struct kwqe *kwqe_arr[2];
	int num_kwqes = 1;
	int rc = 0;
	vmk_uint32 *x;
	int i;

	qfle3f_log(hba, LOG_INIT, "entered qfle3f_sendStatRequest");

	vmk_Memset(&stat_req, 0x00, sizeof(struct fcoe_kwqe_stat));
	stat_req.hdr.op_code = FCOE_KWQE_OPCODE_STAT;
	stat_req.hdr.flags =
		(FCOE_KWQE_LAYER_CODE << FCOE_KWQE_HEADER_LAYER_CODE_SHIFT);

	stat_req.stat_params_addr_lo = U64_LO(GET_DMA_ADDR(hba->statsBufferDMA));
	stat_req.stat_params_addr_hi = U64_HI(GET_DMA_ADDR(hba->statsBufferDMA));

	kwqe_arr[0] = (struct kwqe *) &stat_req;

	qfle3f_log(hba, LOG_INITV, "dumping stat_req KWQE...");
	x = (vmk_uint32 *)&stat_req;
	for (i=0; i<8; i++) {
		qfle3f_log(hba, LOG_INITV, "%8x ", *x++);
	}
	qfle3f_log(hba, LOG_INITV, "\n");

	if (hba->cnic && hba->cnic->submit_kwqes)
		rc = hba->cnic->submit_kwqes(hba->cnic, kwqe_arr, num_kwqes);

	return rc;
}

/**
 * qfle3f_sendFirmwareFCoEInitiMessage - initiates initial handshake with FCoE f/w
 *
 * @hba:	hba structure pointer
 *
 * Send down FCoE firmware init KWQEs which initiates the initial handshake
 *	with the f/w. 
 *
 */
int qfle3f_sendFirmwareFCoEInitiMessage(struct qfle3fHBA *hba)
{
	struct fcoe_kwqe_init1 fcoe_init1;
	struct fcoe_kwqe_init2 fcoe_init2;
	struct fcoe_kwqe_init3 fcoe_init3;
	struct kwqe *kwqe_arr[3];
	int num_kwqes = 3;
	int rc = 0;
	int i;
	vmk_uint32 *x;

	if (!hba->cnic) {
		qfle3f_warning(hba, "hba->cnic NULL during fcoe fw init");
		return -1;
	}

	/* fill init1 KWQE */
	vmk_Memset(&fcoe_init1, 0x00, sizeof(struct fcoe_kwqe_init1));
	fcoe_init1.hdr.op_code = FCOE_KWQE_OPCODE_INIT1;
	fcoe_init1.hdr.flags = (FCOE_KWQE_LAYER_CODE << 
					FCOE_KWQE_HEADER_LAYER_CODE_SHIFT);

	fcoe_init1.num_tasks = QFLE3F_MAX_TASKS;
	fcoe_init1.sq_num_wqes = QFLE3F_SQ_WQES_MAX;
	fcoe_init1.rq_num_wqes = QFLE3F_RQ_WQES_MAX;
	fcoe_init1.rq_buffer_log_size = QFLE3F_RQ_BUF_LOG_SZ;
	fcoe_init1.cq_num_wqes = QFLE3F_CQ_WQES_MAX;
	fcoe_init1.dummy_buffer_addr_lo = U64_LO(GET_DMA_ADDR(hba->dummyBufferDMA));
	fcoe_init1.dummy_buffer_addr_hi = U64_HI(GET_DMA_ADDR(hba->dummyBufferDMA));
	fcoe_init1.task_list_pbl_addr_lo = U64_LO(GET_DMA_ADDR(hba->taskContextBDDMA));
	fcoe_init1.task_list_pbl_addr_hi = U64_HI(GET_DMA_ADDR(hba->taskContextBDDMA));
#if 0
    qfle3f_err(hba, "==========TaskContextBDTable============");
    qfle3f_printBuffer(hba, (vmk_uint8 *)hba->taskContextBDTable, VMK_PAGE_SIZE, "TaskContextBD");
#endif
	fcoe_init1.mtu = QFLE3F_MINI_JUMBO_MTU;

	fcoe_init1.flags = (VMK_PAGE_SHIFT <<
				FCOE_KWQE_INIT1_LOG_PAGE_SIZE_SHIFT);

	fcoe_init1.num_sessions_log = QFLE3F_NUM_MAX_SESS_LOG;
	qfle3f_log(hba, LOG_INITV, "dumping fcoe_init1 KWQE...");
        x = (vmk_uint32 *)&fcoe_init1;
        for (i=0; i<8; i++) {
		qfle3f_log(hba, LOG_INITV, "%8x", *x++);
        }
	qfle3f_log(hba, LOG_ELS, "\n");

	/* fill init2 KWQE */
	vmk_Memset(&fcoe_init2, 0x00, sizeof(struct fcoe_kwqe_init2));
	fcoe_init2.hdr.op_code = FCOE_KWQE_OPCODE_INIT2;
	fcoe_init2.hdr.flags = (FCOE_KWQE_LAYER_CODE <<
					FCOE_KWQE_HEADER_LAYER_CODE_SHIFT);

	fcoe_init2.hash_tbl_pbl_addr_lo = U64_LO(GET_DMA_ADDR(hba->hashTablePBLDMA));
	fcoe_init2.hash_tbl_pbl_addr_hi = U64_HI(GET_DMA_ADDR(hba->hashTablePBLDMA));

	fcoe_init2.t2_hash_tbl_addr_lo = U64_LO(GET_DMA_ADDR(hba->t2HashTable_dma));
	fcoe_init2.t2_hash_tbl_addr_hi = U64_HI(GET_DMA_ADDR(hba->t2HashTable_dma));

	fcoe_init2.t2_ptr_hash_tbl_addr_lo = U64_LO(GET_DMA_ADDR(hba->t2HashTablePointer_dma));
	fcoe_init2.t2_ptr_hash_tbl_addr_hi = U64_HI(GET_DMA_ADDR(hba->t2HashTablePointer_dma));

	fcoe_init2.free_list_count = QFLE3_FCOE_NUM_CONNECTIONS;
	qfle3f_log(hba, LOG_INITV, "dumping fcoe_init2 KWQE...");
        x = (vmk_uint32 *)&fcoe_init2;
        for (i=0; i<8; i++) {
		qfle3f_log(hba, LOG_INITV, "%8x ", *x++);
        }
	qfle3f_log(hba, LOG_INITV, "\n");

	/* fill init3 KWQE */
	vmk_Memset(&fcoe_init3, 0x00, sizeof(struct fcoe_kwqe_init3));
	fcoe_init3.hdr.op_code = FCOE_KWQE_OPCODE_INIT3;
	fcoe_init3.hdr.flags = (FCOE_KWQE_LAYER_CODE <<
					FCOE_KWQE_HEADER_LAYER_CODE_SHIFT);
	fcoe_init3.error_bit_map_lo = 0xffffffff;
	fcoe_init3.error_bit_map_hi = 0xffffffff;

	fcoe_init3.perf_config = 1;

	qfle3f_log(hba, LOG_INITV, "dumping fcoe_init3 KWQE...");
        x = (vmk_uint32 *)&fcoe_init3;
        for (i=0; i<8; i++) {
		qfle3f_log(hba, LOG_INITV, "%8x ", *x++);
        }
	qfle3f_log(hba, LOG_INITV, "\n");

	kwqe_arr[0] = (struct kwqe *) &fcoe_init1;
	kwqe_arr[1] = (struct kwqe *) &fcoe_init2;
	kwqe_arr[2] = (struct kwqe *) &fcoe_init3;

	if (hba->cnic && hba->cnic->submit_kwqes)
		rc = hba->cnic->submit_kwqes(hba->cnic, kwqe_arr, num_kwqes);

    qfle3f_log(hba, LOG_INITV, "return");
	return rc;
}

int qfle3f_sendFirmwareFCoEDestroyMessage(struct qfle3fHBA *hba)
{
	struct cnic_dev *cnic;
	struct fcoe_kwqe_destroy fcoe_destroy;
	struct kwqe *kwqe_arr[2];
	int num_kwqes = 1;
	int rc = -1;

	if (hba) {
		cnic = hba->cnic;
		if (cnic->fw_recov_in_progress) {
			qfle3f_err(hba, "fw error recovery in progress. Exit.");
			return rc;
		}
	} else {
		qfle3f_err(hba, "invalid hba. Exit.");
		return rc;
	}

	/* fill destroy KWQE */
	vmk_Memset(&fcoe_destroy, 0x00, sizeof(struct fcoe_kwqe_destroy));
	fcoe_destroy.hdr.op_code = FCOE_KWQE_OPCODE_DESTROY;
	fcoe_destroy.hdr.flags = (FCOE_KWQE_LAYER_CODE <<
					FCOE_KWQE_HEADER_LAYER_CODE_SHIFT);
	kwqe_arr[0] = (struct kwqe *) &fcoe_destroy;

	qfle3f_log(hba, LOG_INIT, "submitting destroy KWQE");
	if (hba->cnic && hba->cnic->submit_kwqes)
		rc = hba->cnic->submit_kwqes(hba->cnic, kwqe_arr, num_kwqes);
	qfle3f_log(hba, LOG_INIT, "destroy rc = %d", rc);

	return rc;
}

/**
 * qfle3f_sendSessionOffloadRequest - initiates FCoE Session offload process 
 * 
 * @port:		port structure pointer
 * @target:		qfle3f_rport structure pointer
 */
int qfle3f_sendSessionOffloadRequest(struct qfle3fHBA *hba,
					struct qfle3f_rport *target)
{
	struct kwqe *kwqe_arr[4];
	struct fcoe_kwqe_conn_offload1 ofld_req1;
	struct fcoe_kwqe_conn_offload2 ofld_req2;
	struct fcoe_kwqe_conn_offload3 ofld_req3;
	struct fcoe_kwqe_conn_offload4 ofld_req4;
	int num_kwqes = 4;
	vmk_uint32 initiator_targetPortID, target_targetPortID;
	int rc = 0;
	int i;
	vmk_uint32 *x;
    vmk_uint32 fcoe_conn_id;

 	/* Initialize offload request 1 structure */
	vmk_Memset(&ofld_req1, 0x00, sizeof(struct fcoe_kwqe_conn_offload1));

	ofld_req1.hdr.op_code = FCOE_KWQE_OPCODE_OFFLOAD_CONN1;
	ofld_req1.hdr.flags =
		(FCOE_KWQE_LAYER_CODE << FCOE_KWQE_HEADER_LAYER_CODE_SHIFT);

	if (!hba->cnic || !hba->cnic->submit_kwqes)
        return VMK_FAILURE;

    rc = hba->cnic->cm_acquire(hba->cnic, CNIC_ULP_FCOE, &fcoe_conn_id);

    ofld_req1.fcoe_conn_id = fcoe_conn_id;
    target->fcoe_conn_id = fcoe_conn_id;

    qfle3f_addConn(hba, target);

    qfle3f_notice(hba, "conn_id = %d", ofld_req1.fcoe_conn_id);

	ofld_req1.sq_addr_lo = U64_LO(GET_DMA_ADDR(target->sqDMA));
	ofld_req1.sq_addr_hi = U64_HI(GET_DMA_ADDR(target->sqDMA));

	ofld_req1.rq_pbl_addr_lo = U64_LO(GET_DMA_ADDR(target->rqPblDMA));
	ofld_req1.rq_pbl_addr_hi = U64_HI(GET_DMA_ADDR(target->rqPblDMA));

	ofld_req1.rq_first_pbe_addr_lo = U64_LO(GET_DMA_ADDR(target->rqDMA));
	ofld_req1.rq_first_pbe_addr_hi = U64_HI(GET_DMA_ADDR(target->rqDMA));

	ofld_req1.rq_prod = 0x8000;
	qfle3f_log(hba, LOG_SESSV, "dumping ofld_req1 KWQE...");
        x = (vmk_uint32 *)&ofld_req1;
        for (i=0; i<8; i++) {
		qfle3f_log(hba, LOG_SESSV, "%8x  ", *x++);
        }
	qfle3f_log(hba, LOG_SESSV, "\n");

 	/* Initialize offload request 2 structure */
	vmk_Memset(&ofld_req2, 0x00, sizeof(struct fcoe_kwqe_conn_offload2));

	ofld_req2.hdr.op_code = FCOE_KWQE_OPCODE_OFFLOAD_CONN2;
	ofld_req2.hdr.flags = 
		(FCOE_KWQE_LAYER_CODE << FCOE_KWQE_HEADER_LAYER_CODE_SHIFT);

	ofld_req2.tx_max_fc_pay_len = target->Sess->c3_txsize;

	ofld_req2.cq_addr_lo = U64_LO(GET_DMA_ADDR(target->cqDMA));
	ofld_req2.cq_addr_hi = U64_HI(GET_DMA_ADDR(target->cqDMA));

	ofld_req2.xferq_addr_lo = U64_LO(GET_DMA_ADDR(target->xferqDMA));
	ofld_req2.xferq_addr_hi = U64_HI(GET_DMA_ADDR(target->xferqDMA));

	ofld_req2.conn_db_addr_lo = U64_LO(GET_DMA_ADDR(target->connDbDMA));
	ofld_req2.conn_db_addr_hi = U64_HI(GET_DMA_ADDR(target->connDbDMA));
	qfle3f_log(hba, LOG_SESSV, "dumping ofld_req2 KWQE...");
        x = (vmk_uint32 *)&ofld_req2;
        for (i=0; i<8; i++) {
		qfle3f_log(hba, LOG_SESSV, "%8x  ", *x++);
        }
	qfle3f_log(hba, LOG_SESSV, "\n");

 	/* Initialize offload request 3 structure */
	vmk_Memset(&ofld_req3, 0x00, sizeof(struct fcoe_kwqe_conn_offload3));

	ofld_req3.hdr.op_code = FCOE_KWQE_OPCODE_OFFLOAD_CONN3;
	ofld_req3.hdr.flags = 
		(FCOE_KWQE_LAYER_CODE << FCOE_KWQE_HEADER_LAYER_CODE_SHIFT);

	qfle3f_log(hba, LOG_SESSV, "FCoE VLAN ID 0x%x, fcoe_cid (%d)",
			 (target->Sess->Fabric->vlan_and_pri & 0xfff), fcoe_conn_id);

    ofld_req3.vlan_tag =(target->Sess->Fabric->vlan_and_pri & 0xfff) <<
				FCOE_KWQE_CONN_OFFLOAD3_VLAN_ID_SHIFT;

	//ofld_req3.vlan_tag |= 3 << FCOE_KWQE_CONN_OFFLOAD3_PRIORITY_SHIFT;

    hba->vlanID = (target->Sess->Fabric->vlan_and_pri & 0xfff) << FCOE_KWQE_CONN_OFFLOAD3_VLAN_ID_SHIFT;

    initiator_targetPortID = ((vmk_uint32)target->Sess->Fabric->fc_id[0] << 16) |
        ((vmk_uint32)target->Sess->Fabric->fc_id[1] << 8) |
        (vmk_uint32)target->Sess->Fabric->fc_id[2];
    hba->initiator_targetPortID = initiator_targetPortID;
    vmk_LogMessage("fabric port id %x\n", initiator_targetPortID);

	/* 
	 * Store s_id of the initiator for further reference. This will
	 * be used during disable/destroy during linkdown processing as
	 * when the lport is reset, the targetPortID also is reset to 0
	 */
	target->initiatorPortID = initiator_targetPortID;
	ofld_req3.s_id[0] = (initiator_targetPortID & 0x000000FF);
	ofld_req3.s_id[1] = (initiator_targetPortID & 0x0000FF00) >> 8;
	ofld_req3.s_id[2] = (initiator_targetPortID & 0x00FF0000) >> 16;

    /* TODO: Figure out what this is */
    ofld_req3.tx_max_conc_seqs_c3 = MAX_NUM_SEQUENCES;

	target_targetPortID = target->targetPortID;
	ofld_req3.d_id[0] = (target_targetPortID & 0x000000FF);
	ofld_req3.d_id[1] = (target_targetPortID & 0x0000FF00) >> 8;
	ofld_req3.d_id[2] = (target_targetPortID & 0x00FF0000) >> 16;

	ofld_req3.rx_max_fc_pay_len  = target->Sess->c3_txsize;
	ofld_req3.tx_max_conc_seqs_c3 = MAX_NUM_SEQUENCES;

	ofld_req3.rx_total_conc_seqs = QFLE3F_MAX_SEQS;
	ofld_req3.rx_max_conc_seqs_c3 = QFLE3F_MAX_SEQS;
	ofld_req3.rx_open_seqs_exch_c3 = 1;

	ofld_req3.confq_first_pbe_addr_lo = U64_LO(GET_DMA_ADDR(target->confqDMA));
	ofld_req3.confq_first_pbe_addr_hi = U64_HI(GET_DMA_ADDR(target->confqDMA));

	/* set mul_n_targetPortIDs supported flag to 0, until it is supported */
	ofld_req3.flags = 0;

	/* Info from PLOGI response */
    if(target->Sess->Fabric->flogi_data.CommonFeatures[0] & FLOGI_COMMON_FEATURES0_E_D_TOV_Resolution)
	    ofld_req3.flags |= 1 << FCOE_KWQE_CONN_OFFLOAD3_B_E_D_TOV_RES_SHIFT;

    if(target->Sess->Fabric->flogi_data.CommonFeatures[1] & FLOGI_COMMON_FEATURES1_SEQ_COUNT_OR_VENDOR_SPECIFIC)
        ofld_req3.flags |= 1 << FCOE_KWQE_CONN_OFFLOAD3_B_CONT_INCR_SEQ_CNT_SHIFT;

	/* vlan flag */
    if(hba->vlanID)
	    ofld_req3.flags |= 1 << FCOE_KWQE_CONN_OFFLOAD3_B_VLAN_FLAG_SHIFT;

    qfle3f_log(hba, LOG_SESSV, "ofld_req3.flags = %d", ofld_req3.flags);

	/* C2_VALID and ACK flags are not set as they are not suppported */

	qfle3f_log(hba, LOG_SESSV, "dumping ofld_req3 KWQE...");
        x = (vmk_uint32 *)&ofld_req3;
        for (i=0; i<8; i++) {
		qfle3f_log(hba, LOG_SESSV, "%8x  ", *x++);
        }
	qfle3f_log(hba, LOG_SESSV, "");

 	/* Initialize offload request 4 structure */
	vmk_Memset(&ofld_req4, 0x00, sizeof(struct fcoe_kwqe_conn_offload4));
	ofld_req4.hdr.op_code = FCOE_KWQE_OPCODE_OFFLOAD_CONN4;
	ofld_req4.hdr.flags = 
		(FCOE_KWQE_LAYER_CODE << FCOE_KWQE_HEADER_LAYER_CODE_SHIFT);

	ofld_req4.e_d_tov_timer_val = target->Sess->e_d_tov_timer_val / 20;

	/* local mac */
	ofld_req4.src_mac_addr_lo[0] =  target->Sess->Fabric->local_mac[5];
	ofld_req4.src_mac_addr_lo[1] =  target->Sess->Fabric->local_mac[4];
	ofld_req4.src_mac_addr_mid[0] =  target->Sess->Fabric->local_mac[3];
	ofld_req4.src_mac_addr_mid[1] = target->Sess->Fabric->local_mac[2];
	ofld_req4.src_mac_addr_hi[0] =  target->Sess->Fabric->local_mac[1];
	ofld_req4.src_mac_addr_hi[1] =  target->Sess->Fabric->local_mac[0];

	vmk_Memcpy(hba->fpma, target->Sess->Fabric->local_mac, VMK_MAX_ETHERNET_MAC_LENGTH);

	/* fcf mac */
	ofld_req4.dst_mac_addr_lo[0] =  target->Sess->Fabric->fcf_fcp_mac[5];
	ofld_req4.dst_mac_addr_lo[1] =  target->Sess->Fabric->fcf_fcp_mac[4];
	ofld_req4.dst_mac_addr_mid[0] =  target->Sess->Fabric->fcf_fcp_mac[3];
	ofld_req4.dst_mac_addr_mid[1] =  target->Sess->Fabric->fcf_fcp_mac[2];
	ofld_req4.dst_mac_addr_hi[0] =  target->Sess->Fabric->fcf_fcp_mac[1];
	ofld_req4.dst_mac_addr_hi[1] =  target->Sess->Fabric->fcf_fcp_mac[0];

	vmk_Memcpy(hba->fcf_mac, target->Sess->Fabric->fcf_fcp_mac, VMK_MAX_ETHERNET_MAC_LENGTH);

	ofld_req4.lcq_addr_lo = U64_LO(GET_DMA_ADDR(target->lcqDMA));
	ofld_req4.lcq_addr_hi = U64_HI(GET_DMA_ADDR(target->lcqDMA));

	ofld_req4.confq_pbl_base_addr_lo = U64_LO(GET_DMA_ADDR(target->confqPblDMA));
	ofld_req4.confq_pbl_base_addr_hi = U64_HI(GET_DMA_ADDR(target->confqPblDMA));

	kwqe_arr[0] = (struct kwqe *) &ofld_req1;
	kwqe_arr[1] = (struct kwqe *) &ofld_req2;
	kwqe_arr[2] = (struct kwqe *) &ofld_req3;
	kwqe_arr[3] = (struct kwqe *) &ofld_req4;

	qfle3f_log(hba, LOG_SESSV, "dumping ofld_req4 KWQE...");
        x = (vmk_uint32 *)&ofld_req4;
        for (i=0; i<8; i++) {
		qfle3f_log(hba, LOG_SESSV, "%8x  ", *x++);
        }
	qfle3f_log(hba, LOG_SESSV, "\n");

	if (hba->cnic && hba->cnic->submit_kwqes)
		rc = hba->cnic->submit_kwqes(hba->cnic, kwqe_arr, num_kwqes);

	return rc;
}

/**
 * qfle3f_sendSessionEnableRequest - initiates FCoE Session enablement 
 *
 * @port:		port structure pointer
 * @target:		qfle3f_rport structure pointer
 */
int qfle3f_sendSessionEnableRequest(struct qfle3fHBA *hba,
					struct qfle3f_rport *target)
{
	struct kwqe *kwqe_arr[2];
	struct fcoe_kwqe_conn_enable_disable enbl_req;
	int num_kwqes = 1;
	int rc = 0;
	vmk_uint32 targetPortID;
	int i;
	vmk_uint32 *x;

	vmk_Memset(&enbl_req, 0x00, 
	       sizeof(struct fcoe_kwqe_conn_enable_disable));
	enbl_req.hdr.op_code = FCOE_KWQE_OPCODE_ENABLE_CONN;
	enbl_req.hdr.flags = 
		(FCOE_KWQE_LAYER_CODE << FCOE_KWQE_HEADER_LAYER_CODE_SHIFT);

	/* local mac */
	enbl_req.src_mac_addr_lo[0] =  target->Sess->Fabric->local_mac[5];
	enbl_req.src_mac_addr_lo[1] =  target->Sess->Fabric->local_mac[4];
	enbl_req.src_mac_addr_mid[0] = target->Sess->Fabric->local_mac[3];
	enbl_req.src_mac_addr_mid[1] = target->Sess->Fabric->local_mac[2];
	enbl_req.src_mac_addr_hi[0] =  target->Sess->Fabric->local_mac[1];
	enbl_req.src_mac_addr_hi[1] =  target->Sess->Fabric->local_mac[0];

	/* fcf mac */
	enbl_req.dst_mac_addr_lo[0] =  target->Sess->Fabric->fcf_fcp_mac[5];
	enbl_req.dst_mac_addr_lo[1] =  target->Sess->Fabric->fcf_fcp_mac[4];
	enbl_req.dst_mac_addr_mid[0] = target->Sess->Fabric->fcf_fcp_mac[3];
	enbl_req.dst_mac_addr_mid[1] = target->Sess->Fabric->fcf_fcp_mac[2];
	enbl_req.dst_mac_addr_hi[0] =  target->Sess->Fabric->fcf_fcp_mac[1];
	enbl_req.dst_mac_addr_hi[1] =  target->Sess->Fabric->fcf_fcp_mac[0];

    targetPortID = ((vmk_uint32)target->Sess->Fabric->fc_id[0] << 16) |
        ((vmk_uint32)target->Sess->Fabric->fc_id[1] << 8) |
        (vmk_uint32)target->Sess->Fabric->fc_id[2];
    qfle3f_log(hba, LOG_SESSV, "Fabric port id %x", targetPortID);

	enbl_req.s_id[0] = (targetPortID & 0x000000FF);
	enbl_req.s_id[1] = (targetPortID & 0x0000FF00) >> 8;
	enbl_req.s_id[2] = (targetPortID & 0x00FF0000) >> 16;

	targetPortID = target->targetPortID;
	enbl_req.d_id[0] = (targetPortID & 0x000000FF);
	enbl_req.d_id[1] = (targetPortID & 0x0000FF00) >> 8;
	enbl_req.d_id[2] = (targetPortID & 0x00FF0000) >>16;
    hba->vlan_id = (target->Sess->Fabric->vlan_and_pri & 0xfff);

	qfle3f_log(hba, LOG_SESSV, "FCoE VLAN ID 0x%x",
                hba->vlan_id);

	enbl_req.vlan_tag = (target->Sess->Fabric->vlan_and_pri & 0xfff) <<
				FCOE_KWQE_CONN_ENABLE_DISABLE_VLAN_ID_SHIFT;
	enbl_req.vlan_tag |= 3 << FCOE_KWQE_CONN_ENABLE_DISABLE_PRIORITY_SHIFT;

    if(hba->vlanID)
	    enbl_req.vlan_flag = 1;

	qfle3f_log(hba, LOG_SESSV, "enbl_req.vlan_flag = %d", enbl_req.vlan_flag);

	enbl_req.context_id = target->contextID;
	enbl_req.conn_id = target->fcoe_conn_id;

	kwqe_arr[0] = (struct kwqe *) &enbl_req;

	qfle3f_log(hba, LOG_SESSV, "submit enable kwqe - contextID = 0x%x",
		target->contextID);
	qfle3f_log(hba, LOG_SESSV, "dumping enbl_req KWQE...");
        x = (vmk_uint32 *)&enbl_req;
        for (i=0; i<8; i++) {
		qfle3f_log(hba, LOG_SESSV, "%8x  ", *x++);
        }
	qfle3f_log(hba, LOG_SESSV, "\n");
	if (hba->cnic && hba->cnic->submit_kwqes)
		rc = hba->cnic->submit_kwqes(hba->cnic, kwqe_arr, num_kwqes);
	return rc;
}

int qfle3f_sendSessionDisableRequest(struct qfle3fHBA *hba,
				    struct qfle3f_rport *target)
{
	struct cnic_dev *cnic;
	struct fcoe_kwqe_conn_enable_disable disable_req;
	struct kwqe *kwqe_arr[2];
	int num_kwqes = 1;
	int rc = VMK_FAILURE;
	vmk_uint32 targetPortID;

	cnic = hba->cnic;
	if (cnic->fw_recov_in_progress) {
		vmk_BitVectorSet(target->flags, QFLE3F_FLAG_DISABLED); //to allow SessionDestroyRequest
		qfle3f_err(hba, "fw error recovery in progress. Exit.");
		return rc;
	}

	vmk_uint32 *x;
	int i;

	vmk_Memset(&disable_req, 0x00,
	       sizeof(struct fcoe_kwqe_conn_enable_disable));
	disable_req.hdr.op_code = FCOE_KWQE_OPCODE_DISABLE_CONN;
	disable_req.hdr.flags =
		(FCOE_KWQE_LAYER_CODE << FCOE_KWQE_HEADER_LAYER_CODE_SHIFT);

	disable_req.src_mac_addr_lo[0] =  target->Sess->Fabric->local_mac[5];
	disable_req.src_mac_addr_lo[1] =  target->Sess->Fabric->local_mac[4];
	disable_req.src_mac_addr_mid[0] =  target->Sess->Fabric->local_mac[3];
	disable_req.src_mac_addr_mid[1] =  target->Sess->Fabric->local_mac[2];
	disable_req.src_mac_addr_hi[0] =  target->Sess->Fabric->local_mac[1];
	disable_req.src_mac_addr_hi[1] =  target->Sess->Fabric->local_mac[0];

	/* fcf mac */
	disable_req.dst_mac_addr_lo[0] =  target->Sess->Fabric->fcf_fcp_mac[5];
	disable_req.dst_mac_addr_lo[1] =  target->Sess->Fabric->fcf_fcp_mac[4];
	disable_req.dst_mac_addr_mid[0] =  target->Sess->Fabric->fcf_fcp_mac[3];
	disable_req.dst_mac_addr_mid[1] =  target->Sess->Fabric->fcf_fcp_mac[2];
	disable_req.dst_mac_addr_hi[0] =  target->Sess->Fabric->fcf_fcp_mac[1];
	disable_req.dst_mac_addr_hi[1] =  target->Sess->Fabric->fcf_fcp_mac[0];

	targetPortID = target->initiatorPortID;
	qfle3f_log(hba, LOG_SESS, "Host Port id = 0x%x", targetPortID);
	disable_req.s_id[0] = (targetPortID & 0x000000FF);
	disable_req.s_id[1] = (targetPortID & 0x0000FF00) >> 8;
	disable_req.s_id[2] = (targetPortID & 0x00FF0000) >> 16;


	targetPortID = target->targetPortID;
	disable_req.d_id[0] = (targetPortID & 0x000000FF);
	disable_req.d_id[1] = (targetPortID & 0x0000FF00) >> 8;
	disable_req.d_id[2] = (targetPortID & 0x00FF0000) >>16;
	disable_req.context_id = target->contextID;
	disable_req.conn_id = target->fcoe_conn_id;
	disable_req.vlan_tag = (target->Sess->Fabric->vlan_and_pri & 0xfff) <<
				FCOE_KWQE_CONN_ENABLE_DISABLE_VLAN_ID_SHIFT;
	disable_req.vlan_tag |=
			3 << FCOE_KWQE_CONN_ENABLE_DISABLE_PRIORITY_SHIFT;

    if(hba->vlanID)
	    disable_req.vlan_flag = 1;

	qfle3f_log(hba, LOG_SESS, "disable_req.vlan_flag = %d", disable_req.vlan_flag);

	kwqe_arr[0] = (struct kwqe *) &disable_req;

	qfle3f_log(hba, LOG_SESSV, "dumping disable_req KWQE...");
        x = (vmk_uint32 *)&disable_req;
        for (i=0; i<8; i++) {
		qfle3f_log(hba, LOG_SESSV, "%8x ", *x++);
        }
	qfle3f_log(hba, LOG_SESSV, "\n");

	qfle3f_log(hba, LOG_SESSV, "submit disable req - contextID = 0x%x",
		target->contextID);
	if (hba->cnic && hba->cnic->submit_kwqes)
		rc = hba->cnic->submit_kwqes(hba->cnic, kwqe_arr, num_kwqes);

	return rc;
}

int qfle3f_sendSessionDestroyRequest(struct qfle3fHBA *hba, 
					struct qfle3f_rport *target)
{
	struct fcoe_kwqe_conn_destroy destroy_req;
	struct kwqe *kwqe_arr[2];
	int num_kwqes = 1;
	int rc = 0;
	vmk_uint32 *x;
	int i;

	vmk_Memset(&destroy_req, 0x00, sizeof(struct fcoe_kwqe_conn_destroy));
	destroy_req.hdr.op_code = FCOE_KWQE_OPCODE_DESTROY_CONN;
	destroy_req.hdr.flags =
		(FCOE_KWQE_LAYER_CODE << FCOE_KWQE_HEADER_LAYER_CODE_SHIFT);

	destroy_req.context_id = target->contextID;
	destroy_req.conn_id = target->fcoe_conn_id;

	kwqe_arr[0] = (struct kwqe *) &destroy_req;

	qfle3f_log(hba, LOG_SESS, "dumping destroy_req KWQE...");
	x = (vmk_uint32 *)&destroy_req;
	for (i=0; i<8; i++) {
		qfle3f_log(hba, LOG_SESSV, "%8x ", *x++);
	}
	qfle3f_log(hba, LOG_SESSV, "\n");

	qfle3f_log(hba, LOG_SESS, "submit destroy req - contextID = 0x%x",
		target->contextID);
	if (hba->cnic && hba->cnic->submit_kwqes) 
		rc = hba->cnic->submit_kwqes(hba->cnic, kwqe_arr, num_kwqes);

	return rc;
}

void qfle3f_processL2FrameCompletion(struct qfle3f_rport *target,
				   unsigned char *buf,
				   vmk_uint32 frame_len, vmk_uint16 l2_oxid)
{
	struct qfle3fHBA *hba = target->hba;
	struct FC_FRAME_HEADER *fh;
    vmk_EthHdr *eh = NULL;
	vmk_PktHandle *pkt;
	vmk_uint32 payload_len;
    vmk_uint8 *mapped_data = NULL;
	vmk_uint8 op;
	int vmk_PktAlloc_retry;

	qfle3f_log(hba, LOG_SESS, "l2_frame_compl l2_oxid = 0x%x, frame_len = %d",
		        l2_oxid, frame_len);
	payload_len = frame_len - sizeof(struct FC_FRAME_HEADER);

qfle3f_processL2FrameCompletion_retry:
	vmk_PktAlloc_retry = 10;
    vmk_PktAllocForDMAEngine(frame_len + QFC_FRAME_HEADROOM + QFC_FRAME_TAILROOM,
        hba->qlfcoeAdapter->dma_engine, &pkt);

	if (!pkt) {
		qfle3f_log(hba, LOG_SESS, "vmk_PktAlloc_retry = %d",
			vmk_PktAlloc_retry);
		if (vmk_PktAlloc_retry--) {
			qfle3f_log(hba, LOG_SESS, "vmk_PktAlloc_retry = %d",
				vmk_PktAlloc_retry);
			vmk_WorldSleep(10 * VMK_USEC_PER_MSEC);
			goto qfle3f_processL2FrameCompletion_retry;
		} else
			qfle3f_log(hba, LOG_SESS, "All retry failed = %d",
				vmk_PktAlloc_retry);
	}


    mapped_data = (char *)vmk_PktFrameMappedPointerGet(pkt);
    eh = (vmk_EthHdr *)((vmk_uint8 *)mapped_data);

    vmk_Memcpy(eh->daddr, target->Sess->Fabric->local_mac, 6);
    vmk_Memcpy(eh->saddr, target->Sess->Fabric->fcf_fcp_mac, 6);
    eh->type = vmk_CPUToBE16(ETH_P_8021Q);

    vmk_VLANHdr *vlhdr = (vmk_VLANHdr *)(eh + 1);
    vmk_Memset(vlhdr, 0, sizeof(vmk_VLANHdr));

    //vlhdr->priority = 3;
    vlhdr->type = vmk_CPUToBE16(FCOE_ETHERTYPE);
    vlhdr->vlanIDLow = target->Sess->Fabric->vlan_and_pri & 0xfff;

    struct fcoe_header {
        vmk_uint8 version[13];
        vmk_uint8 sof;
    };

    struct fcoe_header *now;

    now = (struct fcoe_header *) (vlhdr + 1);
    vmk_Memset(now, 0, sizeof(struct fcoe_header));
    now->sof = SOFi3;

	fh = (FC_FRAME_HEADER *)(now + 1);

	/* Copy FC Frame header and payload into the frame */
	vmk_Memcpy(fh, buf, frame_len);

	if (l2_oxid != INVALID_EXCHANGE_IDENTIFIER) {
		fh->ox_id[0] = l2_oxid & 0xff;
		fh->ox_id[1] = l2_oxid >> 8;
	}

	if ((fh->r_ctl == FC_R_CTL_EXTENDED_LINK_DATA_COMMAND_REQUEST) ||
	    (fh->r_ctl == FC_R_CTL_EXTENDED_LINK_DATA_RESPONSE)) {

		if (fh->type == FC_TYPE_ELS) {
            vmk_uint8 *temp = (vmk_uint8 *) (fh + 1);

            op = *(vmk_uint32 *) temp;
            qfle3f_log(hba, LOG_SESS, "ELS 0x%x", op);
			if((fh->r_ctl == FC_R_CTL_EXTENDED_LINK_DATA_COMMAND_REQUEST) &&
				(op == FC_ELS_LOGO) &&
				(vmk_BitVectorTest(target->flags, QFLE3F_FLAG_LOGO_REQ_PENDING)))
			{
				qfle3f_err(hba, "received a LOGO Request while we have issued a LOGO Request. Dropping the frame");
				hba->logoLogoRace++;
				vmk_PktRelease(pkt);
				return;
			}

			if ((op == FC_ELS_TEST) ||	(op == FC_ELS_ESTC) ||
			    (op == FC_ELS_FAN) || (op == FC_ELS_CSU)) {
				/*
				 * No need to reply for these
				 * ELS requests
				 */
				qfle3f_err(hba, "dropping ELS 0x%x", op);
				vmk_PktRelease(pkt);
				return;
			}
		}

        // then, calculate CRC for the frame
        frame_len = ((frame_len + 3) &~3);
        PlaceFC_CRC(fh, frame_len);

        vmk_uint8 *now_pointer = (vmk_uint8 *)fh + frame_len;
        now_pointer += 4;
        *now_pointer = EOFt;

#if 1
		qfle3f_log(hba, LOG_SESS, "sizeof(vmk_EthHdr) = %ld, sizeof(vmk_VLANHdr) = %ld,\n"
				"sizeof(struct fcoe_header) = %ld, frame_len =%d",
				sizeof(vmk_EthHdr), sizeof(vmk_VLANHdr),
				sizeof(struct fcoe_header), frame_len);
        qfle3f_printBuffer(hba, (vmk_uint8 *)mapped_data, sizeof(vmk_EthHdr) +
				sizeof(vmk_VLANHdr) + sizeof(struct fcoe_header) +
				frame_len + 8, "Printing L2 Frame Completion After: ");
#endif
        vmk_PktFrameLenSet(pkt, sizeof(vmk_EthHdr) + sizeof(vmk_VLANHdr) +
				sizeof(struct fcoe_header) + frame_len + 8);

        qfle3f_log(hba, LOG_SESS, "vmk_PktFrameLenGet() = %d", vmk_PktFrameLenGet(pkt));
        ql_fcoe_ll2_rx((void *)hba->qlfcoeAdapter, pkt, 0, 0);
	} else {
		qfle3f_log(hba, LOG_FRAME, "r_ctl = 0x%x", fh->r_ctl);
		vmk_PktRelease(pkt);
	}
}

static void qfle3f_processUnsolicilatedCompletion(struct qfle3f_rport *target, vmk_uint16 wqe)
{
	vmk_uint8 num_rq;
	struct fcoe_err_report_entry *err_entry;
	unsigned char *rq_data;
	unsigned char *buf = NULL, *buf1;
	int i;
	vmk_uint16 xid;
	vmk_uint32 frame_len, len;
	struct qfle3fCommand *ioRequest = NULL;
	struct fcoe_task_ctx_entry *task, *task_page;
	struct qfle3fHBA *hba = target->hba;
	int task_idx, index;
	int rc = 0;
	VMK_ReturnStatus status = VMK_OK;

	qfle3f_log(hba, LOG_IOERR, "entered UNSOL COMPLETION wqe = 0x%x", wqe);
	switch (wqe & FCOE_UNSOLICITED_CQE_SUBTYPE) {
	case FCOE_UNSOLICITED_FRAME_CQE_TYPE:
		frame_len = (wqe & FCOE_UNSOLICITED_CQE_PKT_LEN) >>
			     FCOE_UNSOLICITED_CQE_PKT_LEN_SHIFT;

		num_rq = (frame_len + QFLE3F_RQ_BUF_SZ - 1) / QFLE3F_RQ_BUF_SZ;

		rq_data = (unsigned char *)qfle3f_getNextReceiveQueueEntry(target, num_rq);
		if (rq_data) {
			buf = rq_data;
		} else {
			buf1 = buf = qfle3f_alloc((num_rq * QFLE3F_RQ_BUF_SZ));
			for (i = 0; i < num_rq; i++) {
				rq_data = (unsigned char *)
					   qfle3f_getNextReceiveQueueEntry(target, 1);
				len = QFLE3F_RQ_BUF_SZ;
				vmk_Memcpy(buf1, rq_data, len);
				buf1 += len;
			}
		}
		qfle3f_processL2FrameCompletion(target, buf, frame_len,
					      INVALID_EXCHANGE_IDENTIFIER);

		if (buf != rq_data)
			qfle3f_free(buf);
		qfle3f_returnReceiveQueueEntry(target, num_rq);
		break;

	case FCOE_ERROR_DETECTION_CQE_TYPE:
		/*
		 *In case of error reporting CQE a single RQ entry
		 * is consumes.
		 */
		vmk_SpinlockLock(target->targetLock);
		num_rq = 1;
		err_entry = (struct fcoe_err_report_entry *)
			     qfle3f_getNextReceiveQueueEntry(target, 1);
		xid = err_entry->fc_hdr.ox_id;
		qfle3f_log(hba, LOG_UNSOL, "Unsol Error Frame OX_ID = 0x%x", xid);
		qfle3f_log(hba, LOG_UNSOL, "err_warn_bitmap = %08x:%08x",
			err_entry->data.err_warn_bitmap_hi,
			err_entry->data.err_warn_bitmap_lo);
		qfle3f_log(hba, LOG_UNSOL, "buf_offsets - tx = 0x%x, rx = 0x%x",
			err_entry->data.tx_buf_off, err_entry->data.rx_buf_off);

		qfle3f_returnReceiveQueueEntry(target, 1);

		if (xid > QFLE3F_MAX_XID) {
			qfle3f_log(hba, LOG_UNSOL, "xid(0x%x) out of FW range",
				   xid);
			vmk_SpinlockUnlock(target->targetLock);
			break;
		}

		task_idx = xid / QFLE3F_TASKS_PER_PAGE;
		index = xid % QFLE3F_TASKS_PER_PAGE;
		task_page = (struct fcoe_task_ctx_entry *)hba->taskContext[task_idx];
		task = &(task_page[index]);

		ioRequest = (struct qfle3fCommand *)hba->commandManager->cmds[xid];
		if (!ioRequest) {
			vmk_SpinlockUnlock(target->targetLock);
			break;
		}

#ifdef IOERROR_DEBUGGING
		ioRequest->printIORelease = VMK_TRUE;
#endif
		if (ioRequest->cmd_type != QFLE3F_SCSI_CMD) {
			qfle3f_err(hba, "err_warn: Not a SCSI cmd");
			vmk_SpinlockUnlock(target->targetLock);
			break;
		}

		if (vmk_BitVectorAtomicTestAndClear(ioRequest->req_flags,
						QFLE3F_FLAG_IO_CLEANUP)) {
			qfle3f_err(hba, "unsol_err: cleanup in "
				   "progress.. ignore unsol err");
			vmk_SpinlockUnlock(target->targetLock);
			break;
		}

		/*
		 * If ABTS is already in progress, and FW error is
		 * received after that, do not cancel the timeout_work
		 * and let the error recovery continue by explicitly
		 * logging out the target, when the ABTS eventually
		 * times out.
		 */
		if (!vmk_BitVectorAtomicTestAndSet(ioRequest->req_flags,
						QFLE3F_FLAG_ISSUE_ABTS)) {
			/*
			 * Cancel the timeout_work, as we received IO
			 * completion with FW error.
			 */
			if (ql_vmk_cancel_delayed_work(&ioRequest->timeout_work) == VMK_OK)
				ql_vmk_ref_put(&ioRequest->refcount,
					 qfle3f_commandRelease, ioRequest); // timer hold

			qfle3f_notice(hba, "initiate_abts for xid = 0x%x", ioRequest->xid);
			status = qfle3f_initiateABTS(ioRequest);
			if (status != VMK_OK) {
				qfle3f_err(hba, "err_warn: initiate_abts "
					   "failed xid = 0x%x. issue cleanup : %s",
					   ioRequest->xid, vmk_StatusToString(status));
				rc = qfle3f_initiateCleanup(ioRequest);
				//BUG_ON(rc);
			}
		} else
			qfle3f_err(hba, "err_warn: ioRequest (0x%x) already "
				   "in ABTS processing.", xid);
		vmk_SpinlockUnlock(target->targetLock);
		break;

	case FCOE_WARNING_DETECTION_CQE_TYPE:
		/*
		 *In case of warning reporting CQE a single RQ entry
		 * is consumes.
		 */
		num_rq = 1;
		err_entry = (struct fcoe_err_report_entry *)
			     qfle3f_getNextReceiveQueueEntry(target, 1);
		xid = vmk_CPUToBE16(err_entry->fc_hdr.ox_id);
		qfle3f_log(hba, LOG_UNSOL, "Unsol Warning Frame OX_ID = 0x%x",
			xid);
		qfle3f_log(hba, LOG_UNSOL, "err_warn_bitmap = %08x:%08x",
			err_entry->data.err_warn_bitmap_hi,
			err_entry->data.err_warn_bitmap_lo);
		qfle3f_log(hba, LOG_UNSOL, "buf_offsets - tx = 0x%x, rx = 0x%x",
			err_entry->data.tx_buf_off, err_entry->data.rx_buf_off);

		qfle3f_returnReceiveQueueEntry(target, 1);
		break;

	default:
		qfle3f_err(hba, "Unsol Compl: Invalid CQE Subtype 0x%x", wqe);
		break;
	}
}

static void qfle3f_processCQCompletion(struct qfle3f_rport *target, vmk_uint16 wqe)
{
	struct fcoe_task_ctx_entry *task;
	struct fcoe_task_ctx_entry *task_page;
	struct qfle3fHBA *hba = target->hba;
	struct qfle3fCommand *ioRequest;
	int task_idx, index;
	vmk_uint16 xid;
	vmk_uint8  cmd_type;
	vmk_uint8 rx_state = 0;
	vmk_uint8 num_rq;

	vmk_SpinlockLock(target->targetLock);
	xid = wqe & FCOE_PEND_WQ_CQE_TASK_ID;
	if (xid >= QFLE3F_MAX_TASKS) {
		qfle3f_warning (hba, "ERROR:xid out of range");
		vmk_SpinlockUnlock(target->targetLock);
		return;
	}
	task_idx = xid / QFLE3F_TASKS_PER_PAGE;
	index = xid % QFLE3F_TASKS_PER_PAGE;
	task_page = (struct fcoe_task_ctx_entry *)hba->taskContext[task_idx];
	task = &(task_page[index]);

	num_rq = ((task->rxwr_txrd.var_ctx.rx_flags &
		   FCOE_TCE_RX_WR_TX_RD_VAR_NUM_RQ_WQE) >>
		   FCOE_TCE_RX_WR_TX_RD_VAR_NUM_RQ_WQE_SHIFT);

	ioRequest = (struct qfle3fCommand *)hba->commandManager->cmds[xid];

	if (VMK_UNLIKELY(ioRequest == NULL)) {
		qfle3f_err(hba, "ERROR? cq_compl - ioRequest is NULL");
		vmk_SpinlockUnlock(target->targetLock);
		return;
	}

	/* Timestamp IO completion time */
	cmd_type = ioRequest->cmd_type;

	/* optimized completion path */
	if (cmd_type == QFLE3F_SCSI_CMD) {
		rx_state = ((task->rxwr_txrd.var_ctx.rx_flags &
			    FCOE_TCE_RX_WR_TX_RD_VAR_RX_STATE) >>
			    FCOE_TCE_RX_WR_TX_RD_VAR_RX_STATE_SHIFT);

		if (rx_state == FCOE_TASK_RX_STATE_COMPLETED) {
			qfle3f_processScsiCommandCompletion(ioRequest, task, num_rq);
			vmk_SpinlockUnlock(target->targetLock);
			return;
		}
	}

	/* Process other IO completion types */
	switch (cmd_type) {
	case QFLE3F_SCSI_CMD:
		qfle3f_log(hba, LOG_IOERR, "SCSI_CMD(0x%x) "\
			"complete - rx_state = %d", ioRequest->xid, rx_state);

		if (rx_state == FCOE_TASK_RX_STATE_ABTS_COMPLETED)
			qfle3f_processABTSCompletion(ioRequest, task, num_rq);
		else if (rx_state ==
			 FCOE_TASK_RX_STATE_EXCHANGE_CLEANUP_COMPLETED)
			qfle3f_processCleanupCompletion(ioRequest, task, num_rq);
		else
			qfle3f_err(hba, "invalid rx state - %d", rx_state);
		break;

	case QFLE3F_TASK_MGMT_CMD:
		qfle3f_log(hba, LOG_IOERR, "processing TM complete (0x%x)",
			   ioRequest->xid);
		qfle3f_processTMCompletion(ioRequest, task, num_rq);
		break;

	case QFLE3F_ABTS:
		/*
		 * ABTS request received by firmware. ABTS response
		 * will be delivered to the task belonging to the IO
		 * that was aborted
		 */
		qfle3f_log(hba, LOG_IOERR, "cq_compl(0x%x) - ABTS sent out "
			   "by fw", ioRequest->xid);
		ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
		break;

	case QFLE3F_ELS:
		qfle3f_log(hba, LOG_ELS, "cq_compl(0x%x) - call"
			   " process_els_compl", ioRequest->xid);
		qfle3f_process_els_compl(ioRequest, task, num_rq);
		break;

	case QFLE3F_CLEANUP:
		qfle3f_log(hba, LOG_IOERR, "cq_compl(0x%x) - cleanup resp rcvd",
			   ioRequest->xid);

		if (ioRequest->cleanup_tmr_active == CLEANUP_TIMER_ACTIVE) {
			qfle3f_log(hba, LOG_IOERR, "cleanup-tmr-active prev=0x%x now=0x%x",
					ioRequest->cleanup_tmr_active, CLEANUP_TIMER_INPROCESS);
			ioRequest->cleanup_tmr_active = CLEANUP_TIMER_INPROCESS;
		} else {
			/* cleanup tmo handler will process all. */
			break;
		}

		/*
		 * 1. TimerCancel return val = VMK_OK - means timer deleted before firing.
		 *    As such perform cleanup ourselves.
		 * 2. TimerCancel return val = VMK_NOT_FOUND - means timer has already
		 *    fired and tmo handler will take care of it. So skip processing it.
		 */
		if (vmk_TimerCancel(ioRequest->cleanup_tmr, VMK_FALSE) == VMK_OK) {
			qfle3f_log(hba, LOG_IOERR, "cleanup-timer cancelled.\n");
			/* Put cleanup_tmr reference. */
			ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
			ioRequest->cleanup_tmr_active = CLEANUP_TIMER_INACTIVE;
		}

		/* Put cleanup_ioreq reference. */
		ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
		break;

	default:
		qfle3f_err(hba, "invalid cmd_type %d", cmd_type);
		break;
	}
	vmk_SpinlockUnlock(target->targetLock);
}

void qfle3f_armCQ(struct qfle3f_rport *target)
{
	struct b577xx_fcoe_rx_doorbell *rxDb = &target->rxDb;
	vmk_uint32 msg;

	//vmk_CPUMemFenceWrite();
	vmk_CPUMemFenceWrite();
	rxDb->doorbell_cq_cons = target->completionQueueConstIndex | (target->completionQueueCurrentToggleBit <<
						  FCOE_CQE_TOGGLE_BIT_SHIFT);
	msg = *((vmk_uint32 *)rxDb);
	WRT_REG_DWORD(target->ctx_base, QL_CPU_TO_LE32(msg));
	//vmk_CPUMemFenceWrite();
	vmk_CPUMemFenceWrite();

}

int qfle3f_processNewMultipleCQ(struct qfle3f_rport *target)
{
	struct fcoe_cqe *cq;
	vmk_uint32 cq_cons;
	struct fcoe_cqe *cqe;
	vmk_uint32 num_free_sqes = 0;
	vmk_uint32 num_cqes = 0;
	vmk_uint16 wqe;
	struct qfle3fHBA *hba = target->hba;

	/*
	 * completionQueueLock is a low contention lock used to protect
	 * the CQ data structure from being freed up during
	 * the upload operation
	 */
	vmk_SpinlockLock(target->completionQueueLock);

	if (!target->cq) {
		qfle3f_warning(hba, "process_new_cqes: cq is NULL");
		vmk_SpinlockUnlock(target->completionQueueLock);
		return 0;
	}
	cq = target->cq;
	cq_cons = target->completionQueueConstIndex;
	cqe = &cq[cq_cons];

	while (((wqe = cqe->wqe) & FCOE_CQE_TOGGLE_BIT) ==
	       (target->completionQueueCurrentToggleBit <<
	       FCOE_CQE_TOGGLE_BIT_SHIFT)) {

		/* new entry on the cq */
		if (wqe & FCOE_CQE_CQE_TYPE) {
			/* Unsolicited event notification */
			qfle3f_processUnsolicilatedCompletion(target, wqe);
		} else {
			/* Pending work request completion */
			qfle3f_processCQCompletion(target, wqe);
			num_free_sqes++;
		}
		cqe++;
		target->completionQueueConstIndex++;
		num_cqes++;

		if (target->completionQueueConstIndex == QFLE3F_CQ_WQES_MAX) {
			target->completionQueueConstIndex = 0;
			cqe = cq;
			target->completionQueueCurrentToggleBit =
				1 - target->completionQueueCurrentToggleBit;
		}
	}
	if (num_cqes) {
		/* Arm CQ only if doorbell is mapped */
		if (target->ctx_base)
			qfle3f_armCQ(target);
		//atomic_add(num_free_sqes, &target->free_sqes);
		vmk_AtomicAdd64(&target->freeSendQueues, num_free_sqes);
	}

	vmk_SpinlockUnlock(target->completionQueueLock);

	return 0;
}

/**
 * qfle3f_fastpathNotification - process global event queue (KCQ)
 *
 * @hba:		hba structure pointer
 * @new_cqe_kcqe:	pointer to newly DMA'd KCQ entry
 *
 * Fast path event notification handler
 */
static void qfle3f_fastpathNotification(struct qfle3fHBA *hba,
					struct fcoe_kcqe *new_cqe_kcqe)
{
	vmk_uint32 conn_id = new_cqe_kcqe->fcoe_conn_id;
    struct qfle3f_rport *target;

    target = qfle3f_findTargetByConnectionID(hba, conn_id, VMK_FALSE);

	if (!target) {
		qfle3f_warning(hba, "conn_id 0x%x not valid", conn_id);
		return;
	}
	qfle3f_processNewMultipleCQ(target);
}

/**
 * qfle3f_processOffloadCompletion - process FCoE session offload completion
 *
 * @hba:	hba structure pointer
 * @ofld_kcqe:	connection offload kcqe pointer
 *
 * handle session offload completion, enable the session if offload is
 * successful.
 */
static void qfle3f_processOffloadCompletion(struct qfle3fHBA *hba,
					struct fcoe_kcqe *ofld_kcqe)
{
	struct qfle3f_rport		*target;
	vmk_uint32				conn_id;
	vmk_uint32				contextID;
	int				rc;
    VMK_ReturnStatus status;

	conn_id = ofld_kcqe->fcoe_conn_id;
	contextID = ofld_kcqe->fcoe_conn_context_id;
	qfle3f_log(hba, LOG_SESS, "entered ofld compl - contextID = 0x%x",
		ofld_kcqe->fcoe_conn_context_id);
    target = qfle3f_findTargetByConnectionID(hba, conn_id, VMK_FALSE);
	if (!target) {
		qfle3f_warning(hba, "ERROR:ofld_cmpl: No pending ofld req");
		return;
	}
	if (hba != target->hba) {
		qfle3f_warning(hba, "ERROR:ofld_cmpl: HBA mis-match");
		goto ofld_cmpl_err;
	}
	/*
	 * cnic has allocated a contextID for this session; use this
	 * while enabling the session.
	 */
	target->contextID = contextID;
	qfle3f_log(hba, LOG_SESS, "entered ofld compl - ofld_kcqe->completion_status = 0x%x",
		ofld_kcqe->completion_status);
	if (ofld_kcqe->completion_status) {
		if (ofld_kcqe->completion_status ==
				FCOE_KCQE_COMPLETION_STATUS_CTX_ALLOC_FAILURE) {
			qfle3f_err(hba, "unable to allocate FCoE context "
				   "resources");
			vmk_BitVectorSet(target->flags, QFLE3F_FLAG_CTX_ALLOC_FAILURE);
		}
		goto ofld_cmpl_err;
	} else {

		/* now enable the session */
		rc = qfle3f_sendSessionEnableRequest(hba, target);
		if (rc) {
			qfle3f_warning(hba, "enable session failed");
			goto ofld_cmpl_err;
		}
	}
	return;
ofld_cmpl_err:
	vmk_BitVectorSet(target->flags, QFLE3F_FLAG_OFLD_REQ_CMPL);
	//wake_up_interruptible(&target->offloadWait);
    qfle3f_log(hba, LOG_SESS, "waking up offloadWait event: %p",
                &(target->offloadWait));
	status = vmk_WorldWakeup((vmk_WorldEventID)&(target->offloadWait));
	if(status != VMK_OK) {
		qfle3f_err(hba, "could not wake the world waiting on "
			"event offloadWait: %s.", vmk_StatusToString(status));
	}
}

/**
 * qfle3f_processConnectionEnableCompletion - process FCoE session enable completion
 * 
 * @hba:	hba structure pointer
 * @ofld_kcqe:	connection offload kcqe pointer
 *
 * handle session enable completion, mark the rport as ready 
 * Algorithm:
 * o Mark rport as ready if offload is successful.
 * o Add rport to the list of rports
 */

static void qfle3f_processConnectionEnableCompletion(struct qfle3fHBA *hba,
						struct fcoe_kcqe *ofld_kcqe)
{
	struct qfle3f_rport 		*target;
	vmk_uint32				conn_id;
	vmk_uint32				contextID;
    VMK_ReturnStatus status;

	contextID = ofld_kcqe->fcoe_conn_context_id;
	conn_id = ofld_kcqe->fcoe_conn_id;
    target = qfle3f_findTargetByConnectionID(hba, conn_id, VMK_FALSE);

	qfle3f_log(hba, LOG_SESS, "enable compl - contextID = 0x%x",
		ofld_kcqe->fcoe_conn_context_id);
	if (!target) {
		qfle3f_warning(hba, "ERROR:enbl_cmpl: No pending ofld req");
		return;
	}
	/*
	 * contextID should be the same for this target during offload
	 * and enable
	 */
	if (target->contextID != contextID) {
		qfle3f_warning(hba, "context id mis-match");
		return;
	}
	if (hba != target->hba) {
		qfle3f_warning(hba, "qfle3f-enbl_cmpl: HBA mis-match");
		goto enbl_cmpl_err;
	}
	if (ofld_kcqe->completion_status) {
		goto enbl_cmpl_err;
	} else {
		/* enable successful - rport ready for issuing IOs */
		vmk_BitVectorSet(target->flags, QFLE3F_FLAG_OFFLOADED);
		vmk_BitVectorSet(target->flags, QFLE3F_FLAG_OFLD_REQ_CMPL);
		//wake_up_interruptible(&target->offloadWait);
        qfle3f_log(hba, LOG_SESS, "waking up offloadWait event: %p",
                &(target->offloadWait));
        status = vmk_WorldWakeup((vmk_WorldEventID)&(target->offloadWait));
        if(status != VMK_OK) {
            qfle3f_err(hba, "could not wake the world waiting on "
                    "event offloadWait: %s.", vmk_StatusToString(status));
        }
	}
	return;

enbl_cmpl_err:
	vmk_BitVectorSet(target->flags, QFLE3F_FLAG_OFLD_REQ_CMPL);
	//wake_up_interruptible(&target->offloadWait);
    qfle3f_log(hba, LOG_SESS, "waking up offloadWait event: %p",
                &(target->offloadWait));
    status = vmk_WorldWakeup((vmk_WorldEventID)&(target->offloadWait));
    if(status != VMK_OK) {
        qfle3f_err(hba, "could not wake the world waiting on "
                "event offloadWait: %s.", vmk_StatusToString(status));
    }
}

static void qfle3f_processConnectionDisableCompletion(struct qfle3fHBA *hba,
					struct fcoe_kcqe *disable_kcqe)
{

	struct qfle3f_rport 		*target;
	vmk_uint32				conn_id;
    VMK_ReturnStatus status;

	conn_id = disable_kcqe->fcoe_conn_id;
    target = qfle3f_findTargetByConnectionID(hba, conn_id, VMK_FALSE);
	qfle3f_notice(hba, "disable_cmpl: conn_id %d target %p",
		   conn_id, target);
	if (!target) {
		qfle3f_warning(hba, "ERROR: disable_cmpl: No disable req");
		return;
	}

	if (disable_kcqe->completion_status) {
		qfle3f_warning(hba, "ERROR: Disable failed with cmpl status %d",
			disable_kcqe->completion_status);
		vmk_BitVectorSet(target->flags, QFLE3F_FLAG_DISABLE_FAILED);
		vmk_BitVectorSet(target->flags, QFLE3F_FLAG_UPLD_REQ_COMPL);
	} else {
		/* disable successful */
		qfle3f_notice(hba, "disable successful");
		vmk_BitVectorClear(target->flags, QFLE3F_FLAG_OFFLOADED);
		vmk_BitVectorSet(target->flags, QFLE3F_FLAG_DISABLED);
		vmk_BitVectorSet(target->flags, QFLE3F_FLAG_UPLD_REQ_COMPL);
	}
	//wake_up_interruptible(&target->uploadWait);
    qfle3f_log(hba, LOG_SESS, "waking up uploadWait event: %p",
                &(target->uploadWait));
    status = vmk_WorldWakeup((vmk_WorldEventID)&(target->uploadWait));
    if(status != VMK_OK) {
        qfle3f_err(hba, "could not wake the world waiting on "
                    "event uploadWait: %s.",
                    vmk_StatusToString(status));
    }
}

static void qfle3f_processConnectionDestroyCompletion(struct qfle3fHBA *hba,
					struct fcoe_kcqe *destroy_kcqe)
{
	struct qfle3f_rport 		*target;
	vmk_uint32				conn_id;
    VMK_ReturnStatus status;

	conn_id = destroy_kcqe->fcoe_conn_id;
    target = qfle3f_findTargetByConnectionID(hba, conn_id, VMK_FALSE);
	qfle3f_log(hba, LOG_SESS, "destroy_cmpl: conn_id %d target %p",
		conn_id, target);
	if (!target) {
		qfle3f_warning(hba, "destroy_cmpl: No destroy req");
		return;
	}

	if (destroy_kcqe->completion_status) {
		qfle3f_warning(hba, "destroy conn failed, cmpl status %d",
			destroy_kcqe->completion_status);
		return;
	} else {
		/* destroy successful */
		qfle3f_log(hba, LOG_SESS, "upload successful");
		vmk_BitVectorSet(target->flags, QFLE3F_FLAG_DESTROYED);
		vmk_BitVectorSet(target->flags, QFLE3F_FLAG_UPLD_REQ_COMPL);
		vmk_BitVectorClear(target->flags, QFLE3F_FLAG_DISABLED);
		//wake_up_interruptible(&target->uploadWait);
        qfle3f_log(hba, LOG_SESS, "waking up uploadWait event: %p",
                &target->uploadWait);
        status = vmk_WorldWakeup((vmk_WorldEventID)&(target->uploadWait));
        if(status != VMK_OK) {
            qfle3f_err(hba, "could not wake the world waiting on "
                        "event uploadWait: %s.",
                        vmk_StatusToString(status));
        }
	}
}

static void qfle3f_initializationFailure(struct qfle3fHBA *hba, vmk_uint32 err_code)
{
	switch(err_code) {
	case FCOE_KCQE_COMPLETION_STATUS_INVALID_OPCODE:
		qfle3f_err(hba, "init_failure due to invalid opcode");
		break;

	case FCOE_KCQE_COMPLETION_STATUS_CTX_ALLOC_FAILURE:
		qfle3f_err(hba, "init failed due to ctx alloc failure");
		break;

	case FCOE_KCQE_COMPLETION_STATUS_NIC_ERROR:
		qfle3f_err(hba, "init_failure due to NIC error");
		break;

	default:
		qfle3f_err(hba, "Unknown Error code %d", err_code);
	}
}

/**
 * qfle3f_indicae_kcqe - process KCQE
 *
 * @hba:	hba structure pointer
 * @kcqe:	kcqe pointer
 * @num_cqe:	Number of completion queue elements
 *
 * Generic KCQ event handler
 */
void qfle3f_indicateKCQE(void *context, struct kcqe *kcq[],
					vmk_uint32 num_cqe)
{
	struct qfle3fHBA *hba = (struct qfle3fHBA *)context;
	int i = 0;
	struct fcoe_kcqe *kcqe = NULL;
	VMK_ReturnStatus status;

	while (i < num_cqe) {
		kcqe = (struct fcoe_kcqe *) kcq[i++];
		hba = (struct qfle3fHBA *)context;

		/* NPIV: Find the hba/vhba for this kcqe. */
		//SV: NPIV: TODO: put this in routine after testing.
		if (kcqe->op_code >= FCOE_KCQE_OPCODE_OFFLOAD_CONN &&
				kcqe->op_code <= FCOE_KCQE_OPCODE_CQ_EVENT_NOTIFICATION) {
			struct qfle3f_rport *target = NULL;
			struct qfle3fHBA *vhba = NULL;
			vmk_uint16 conn_id = kcqe->fcoe_conn_id;
			vmk_ListLinks *current = NULL;

			qfle3f_log(hba, LOG_INFO, "find hba for this kcqe for conn_id = %d.", conn_id);
			target = qfle3f_findTargetByConnectionID(hba, conn_id, VMK_FALSE);

			/* NPIV: If conn not found in parent, try to find in vhba. */
			if (target == NULL &&
					!vmk_BitVectorTest(hba->flags2, QFLE3F_NPIV_PREBOOT)) {
				qfle3f_log(hba, LOG_INFO, "find vhba for this kcqe for conn_id = %d.", conn_id);
				vmk_SpinlockLock(hba->vportListLck);
				VMK_LIST_FORALL(&hba->vportList, current) {
					vhba = VMK_LIST_ENTRY(current, struct qfle3fHBA, vportList);
					target = qfle3f_findTargetByConnectionID(vhba, conn_id,
							VMK_FALSE);
					if (target)
						break;
				}
				vmk_SpinlockUnlock(hba->vportListLck);

				if (target == NULL) {
					qfle3f_warning(hba, "conn_id 0x%x not valid!!!", conn_id);
					return;
				} else {
					/* Assign vhba for all further kcqe processing. */
					qfle3f_log(hba, LOG_INFO, "found %s for kcqe cid=0x%x.", vhba->vmhbaName, conn_id);
					hba = vhba;
				}
			}
		}

		switch (kcqe->op_code) {
		case FCOE_KCQE_OPCODE_CQ_EVENT_NOTIFICATION:
			qfle3f_fastpathNotification(hba, kcqe);
			break;

		case FCOE_KCQE_OPCODE_OFFLOAD_CONN:
			qfle3f_processOffloadCompletion(hba, kcqe);
			break;

		case FCOE_KCQE_OPCODE_ENABLE_CONN:
			qfle3f_processConnectionEnableCompletion(hba, kcqe);
			break;

		case FCOE_KCQE_OPCODE_INIT_FUNC:
			if (kcqe->completion_status !=
					FCOE_KCQE_COMPLETION_STATUS_SUCCESS) {
				qfle3f_initializationFailure(hba,
						kcqe->completion_status);
			} else {
				vmk_BitVectorSet(hba->hbaState, ADAPTER_STATE_UP);
                // TODO
				//qfle3f_get_link_state(hba);
				qfle3f_warning(hba, "[%.2x]: FCOE_INIT passed",
					(vmk_uint8)hba->vmk_pci_addr.bus);
			}
			break;

		case FCOE_KCQE_OPCODE_DESTROY_FUNC:
			if (kcqe->completion_status !=
					FCOE_KCQE_COMPLETION_STATUS_SUCCESS) {

				qfle3f_err(hba, "destroy  failed");
			} else {
				qfle3f_notice(hba, "destroy success");
			}
			vmk_BitVectorSet(hba->flags, QFLE3F_FLAG_DESTROY_CMPL);
            qfle3f_log(hba, LOG_SESS, "waking up destroyFunctionEvent event: 0x%p",
                &hba->destroyFunctionEvent);
			status = vmk_WorldWakeup((vmk_WorldEventID)&(hba->destroyFunctionEvent));
			if(status != VMK_OK) {
				qfle3f_err(hba, "could not wake the world waiting on "
							"event destroyFunctionEvent: %s.",
						vmk_StatusToString(status));
			}
			break;

		case FCOE_KCQE_OPCODE_DISABLE_CONN:
			qfle3f_processConnectionDisableCompletion(hba, kcqe);
			break;

		case FCOE_KCQE_OPCODE_DESTROY_CONN:
			qfle3f_processConnectionDestroyCompletion(hba, kcqe);
			break;

		case FCOE_KCQE_OPCODE_STAT_FUNC:
			if (kcqe->completion_status !=
			    FCOE_KCQE_COMPLETION_STATUS_SUCCESS)
				qfle3f_err(hba, "STAT failed");

            hba->statRequestCompleted = VMK_TRUE;
            qfle3f_log(hba, LOG_SESS, "waking up statRequestDone event: 0x%p",
                &hba->statRequestDone);
			status = vmk_WorldWakeup((vmk_WorldEventID)&(hba->statRequestDone));
			if(status != VMK_OK) {
				qfle3f_log(hba, LOG_INFO, "could not wake the world waiting on "
							"event statRequestDone: %s.",
						vmk_StatusToString(status));
			}
			break;

		case FCOE_KCQE_OPCODE_FCOE_ERROR:
			/* fall thru */
		default:
			qfle3f_warning(hba, "unknown opcode 0x%x",
								kcqe->op_code);
		}
	}
}

void qfle3f_addToSQ(struct qfle3f_rport *target, vmk_uint16 xid)
{
	struct fcoe_sqe *sqe;

	sqe = &target->sq[target->sendQueueProducerIndex];

	/* Fill SQ WQE */
	sqe->wqe = xid << FCOE_SQE_TASK_ID_SHIFT;
	sqe->wqe |= target->sendQueueCurrentToggleBit << FCOE_SQE_TOGGLE_BIT_SHIFT;

	/* Advance SQ Prod Idx */
	if (++target->sendQueueProducerIndex == QFLE3F_SQ_WQES_MAX) {
		target->sendQueueProducerIndex = 0;
		target->sendQueueCurrentToggleBit = 1 - target->sendQueueCurrentToggleBit;
	}
}

void qfle3f_ringDoorbell(struct qfle3f_rport *target)
{
	struct b577xx_doorbell_set_prod *sq_db = &target->sq_db;
	vmk_uint32 msg;

	//vmk_CPUMemFenceWrite();
	vmk_CPUMemFenceWrite();
	sq_db->prod = target->sendQueueProducerIndex |
				(target->sendQueueCurrentToggleBit << 15);
	msg = *((vmk_uint32 *)sq_db);
	WRT_REG_DWORD(target->ctx_base, QL_CPU_TO_LE32(msg));
	//vmk_CPUMemFenceWrite();
	vmk_CPUMemFenceWrite();
}

int qfle3f_mapDoorbell(struct qfle3f_rport *target)
{
	vmk_uint32 contextID = target->contextID;
	struct qfle3fHBA   *hba = target->hba;
	vmk_uint32 reg_off;
	vmk_uint64 reg_base = 0;

    qfle3f_notice(hba, "hba->cdev->doorbells = %p", hba->cnic->doorbells);
    reg_base = (vmk_uint64)hba->cnic->doorbells;
    qfle3f_notice(hba, "reg_base = 0x%lx", reg_base);

	//reg_base = pci_resource_start(port->hba->pcidev,
	//				BNX2X_DOORBELL_PCI_BAR);
	//target->ctx_base = ioremap_nocache(reg_base + reg_off, 4);

	reg_off =  (1 << BNX2X_DB_SHIFT) * (contextID & 0x1FFFF);
	qfle3f_notice(hba, "reg_off = 0x%x", reg_off);
	target->ctx_base = (reg_base + reg_off);
	qfle3f_notice(hba, "target->ctx_base(reg_bas+reg_off) = 0x%lx", target->ctx_base);
	if (!target->ctx_base)
		return -1;
	return 0;
}

char *qfle3f_getNextReceiveQueueEntry(struct qfle3f_rport *target, vmk_uint8 num_items)
{
	char *buf = (char *)target->rq + (target->receiveQueueConsumerIndex * QFLE3F_RQ_BUF_SZ);

	if (target->receiveQueueConsumerIndex + num_items > QFLE3F_RQ_WQES_MAX)
		return NULL;

	target->receiveQueueConsumerIndex += num_items;

	if (target->receiveQueueConsumerIndex >= QFLE3F_RQ_WQES_MAX)
		target->receiveQueueConsumerIndex -= QFLE3F_RQ_WQES_MAX;

	return buf;
}

void qfle3f_returnReceiveQueueEntry(struct qfle3f_rport *target, vmk_uint8 num_items)
{
	/* return the rq buffer */
	vmk_uint32 next_prod_idx = target->receiveQueueProducerIndex + num_items;
	if ((next_prod_idx & 0x7fff) == QFLE3F_RQ_WQES_MAX) {
		/* Wrap around RQ */
		next_prod_idx += 0x8000 - QFLE3F_RQ_WQES_MAX;
	}
	target->receiveQueueProducerIndex = next_prod_idx;
	target->connDb->rq_prod = target->receiveQueueProducerIndex;
}

void qfle3f_initializeCleanupTask(struct qfle3fCommand *ioRequest,
			      struct fcoe_task_ctx_entry *task,
			      vmk_uint16 orig_xid)
{
	vmk_uint8 task_type = FCOE_TASK_TYPE_EXCHANGE_CLEANUP;
	struct qfle3f_rport *target = ioRequest->target;
	vmk_uint32 contextID = target->contextID;

	vmk_Memset(task, 0, sizeof(struct fcoe_task_ctx_entry));

	/* Tx Write Rx Read */
	/* init flags */
	task->txwr_rxrd.const_ctx.init_flags = task_type <<
				FCOE_TCE_TX_WR_RX_RD_CONST_TASK_TYPE_SHIFT;
	task->txwr_rxrd.const_ctx.init_flags |= FCOE_TASK_CLASS_TYPE_3 <<
				FCOE_TCE_TX_WR_RX_RD_CONST_CLASS_TYPE_SHIFT;
	task->txwr_rxrd.const_ctx.init_flags |= FCOE_TASK_DEV_TYPE_DISK <<
				FCOE_TCE_TX_WR_RX_RD_CONST_DEV_TYPE_SHIFT;
	task->txwr_rxrd.union_ctx.cleanup.ctx.cleaned_task_id = orig_xid;

	/* Tx flags */
	task->txwr_rxrd.const_ctx.tx_flags = FCOE_TASK_TX_STATE_EXCHANGE_CLEANUP <<
				FCOE_TCE_TX_WR_RX_RD_CONST_TX_STATE_SHIFT;

	/* Rx Read Tx Write */
	task->rxwr_txrd.const_ctx.init_flags = contextID <<
				FCOE_TCE_RX_WR_TX_RD_CONST_CID_SHIFT;
	task->rxwr_txrd.var_ctx.rx_flags |= 1 <<
				FCOE_TCE_RX_WR_TX_RD_VAR_EXP_FIRST_FRAME_SHIFT;
}

void qfle3f_initializeMPTask(struct qfle3fCommand *ioRequest,
				struct fcoe_task_ctx_entry *task)
{
	struct qfle3fHBA *hba = ioRequest->vhba;
	struct qfle3f_mp_req *mp_req = &(ioRequest->mp_req);
	struct qfle3f_rport *target = ioRequest->target;
	FC_FRAME_HEADER *fc_hdr;
	vmk_uint8 task_type = 0;
	vmk_uint64 *hdr;
	vmk_uint64 temp_hdr[3];
	vmk_uint32 contextID;

	qfle3f_log(hba, LOG_IOERR, "initializing MP task for cmd_type = %d",
		ioRequest->cmd_type);

	/* Obtain task_type */
	if ((ioRequest->cmd_type == QFLE3F_TASK_MGMT_CMD) ||
	    (ioRequest->cmd_type == QFLE3F_ELS)) {
		task_type = FCOE_TASK_TYPE_MIDPATH;
	} else if (ioRequest->cmd_type == QFLE3F_ABTS) {
		task_type = FCOE_TASK_TYPE_ABTS;
	}

	vmk_Memset(task, 0, sizeof(struct fcoe_task_ctx_entry));

	/* Setup the task from ioRequest for easy reference */
	ioRequest->task = task;

	qfle3f_log(hba, LOG_IOERR, "task type = %d", task_type);

	/* Tx only */
	if ((task_type == FCOE_TASK_TYPE_MIDPATH) ||
	    (task_type == FCOE_TASK_TYPE_UNSOLICITED)) {
		vmk_uint64 dmaAddr = GET_DMA_ADDR(mp_req->mp_req_bd_dma);
		task->txwr_only.sgl_ctx.sgl.mul_sgl.cur_sge_addr.lo =
				(vmk_uint32)dmaAddr;
		task->txwr_only.sgl_ctx.sgl.mul_sgl.cur_sge_addr.hi =
				(vmk_uint32)((vmk_uint64)dmaAddr >> 32);
		task->txwr_only.sgl_ctx.sgl.mul_sgl.sgl_size = 1;
		qfle3f_log(hba, LOG_IOERR, "init_mp_task - bd_dma = 0x%lx",
			GET_DMA_ADDR(mp_req->mp_req_bd_dma));
	}

	/* Tx Write Rx Read */
	/* init flags */
	task->txwr_rxrd.const_ctx.init_flags = task_type <<
				FCOE_TCE_TX_WR_RX_RD_CONST_TASK_TYPE_SHIFT;
	task->txwr_rxrd.const_ctx.init_flags |= FCOE_TASK_DEV_TYPE_DISK <<
				FCOE_TCE_TX_WR_RX_RD_CONST_DEV_TYPE_SHIFT;
	task->txwr_rxrd.const_ctx.init_flags |= FCOE_TASK_CLASS_TYPE_3 <<
				FCOE_TCE_TX_WR_RX_RD_CONST_CLASS_TYPE_SHIFT;

	/* tx flags */
	task->txwr_rxrd.const_ctx.tx_flags = FCOE_TASK_TX_STATE_INIT <<
				FCOE_TCE_TX_WR_RX_RD_CONST_TX_STATE_SHIFT;

	task->txwr_rxrd.const_ctx.verify_tx_seq = 0;


	/* Rx Write Tx Read */
	task->rxwr_txrd.const_ctx.data_2_trns = ioRequest->data_xfer_len;

	/* rx flags */
	task->rxwr_txrd.var_ctx.rx_flags |= 1 <<
				FCOE_TCE_RX_WR_TX_RD_VAR_EXP_FIRST_FRAME_SHIFT;

	contextID = target->contextID;
	task->rxwr_txrd.const_ctx.init_flags = contextID <<
				FCOE_TCE_RX_WR_TX_RD_CONST_CID_SHIFT;

	fc_hdr = &(mp_req->req_fc_hdr);
	if (task_type == FCOE_TASK_TYPE_MIDPATH) {
		//fc_hdr->ox_id = vmk_CPUToBE16(ioRequest->xid);
		fc_hdr->ox_id[0] = ioRequest->xid >> 8;
		fc_hdr->ox_id[1] = (ioRequest->xid & 0xFF);

		//fc_hdr->rx_id = vmk_CPUToBE16(0xffff);
		fc_hdr->rx_id[0] = 0xff;
		fc_hdr->rx_id[1] = 0xff;
		task->rxwr_txrd.var_ctx.rx_id = 0xffff;
	} else if (task_type == FCOE_TASK_TYPE_UNSOLICITED) {
		//fc_hdr->rx_id = vmk_CPUToBE16(ioRequest->xid);
		fc_hdr->rx_id[0] = ioRequest->xid  >> 8;
		fc_hdr->rx_id[1] = (ioRequest->xid & 0xFF);
	}

	/* Fill FC Header into middle path buffer */
	hdr = (vmk_uint64 *) &task->txwr_rxrd.union_ctx.tx_frame.fc_hdr;
	vmk_Memcpy(temp_hdr, fc_hdr, sizeof(temp_hdr));
	hdr[0] = vmk_CPUToBE64(temp_hdr[0]);
	hdr[1] = vmk_CPUToBE64(temp_hdr[1]);
	hdr[2] = vmk_CPUToBE64(temp_hdr[2]);

	/* Rx Only */
	if (task_type == FCOE_TASK_TYPE_MIDPATH) {
		task->rxwr_only.union_ctx.read_info.sgl_ctx.sgl.mul_sgl.cur_sge_addr.lo =
				U64_LO(GET_DMA_ADDR(mp_req->mp_resp_bd_dma));
		task->rxwr_only.union_ctx.read_info.sgl_ctx.sgl.mul_sgl.cur_sge_addr.hi =
				U64_HI(GET_DMA_ADDR(mp_req->mp_resp_bd_dma));
		task->rxwr_only.union_ctx.read_info.sgl_ctx.sgl.mul_sgl.sgl_size = 1;
	}
}

void qfle3f_initializeTask(struct qfle3fCommand *ioRequest,
		  	     struct fcoe_task_ctx_entry *task)
{
	vmk_uint8 task_type;
	vmk_ScsiCommand *sc_cmd = ioRequest->sc_cmd;
	struct io_bdt *bd_tbl = ioRequest->bd_tbl;
	struct qfle3f_rport *target = ioRequest->target;
	struct fcoe_read_flow_info *rinfo=NULL;
	vmk_uint64 *fcp_cmnd;
	vmk_uint64 tmp_fcp_cmnd[4];
	vmk_uint32 contextID;
	int cnt, i;
	int bd_count;

	vmk_Memset(task, 0, sizeof(struct fcoe_task_ctx_entry));

	/* Setup the task from ioRequest for easy reference */
	ioRequest->task = task;

	if (sc_cmd->dataDirection == VMK_SCSI_COMMAND_DIRECTION_WRITE)
		task_type = FCOE_TASK_TYPE_WRITE;
	else
		task_type = FCOE_TASK_TYPE_READ;

	/* Tx only */
	bd_count = bd_tbl->bd_valid;
	if (task_type == FCOE_TASK_TYPE_WRITE) {
		if (bd_count == 1) {
			struct fcoe_bd_ctx *fcoe_bd_tbl = bd_tbl->bd_tbl;

			task->txwr_only.sgl_ctx.cached_sge.cur_buf_addr.lo =
					fcoe_bd_tbl->buf_addr_lo;
			task->txwr_only.sgl_ctx.cached_sge.cur_buf_addr.hi =
					fcoe_bd_tbl->buf_addr_hi;
			task->txwr_only.sgl_ctx.cached_sge.cur_buf_rem =
					fcoe_bd_tbl->buf_len;
			/* T7.2 only */
			rinfo = &task->rxwr_only.union_ctx.read_info;
			rinfo->sgl_ctx.cached_sge.cur_buf_addr.lo=
					fcoe_bd_tbl->buf_addr_lo;
			rinfo->sgl_ctx.cached_sge.cur_buf_addr.hi=
					fcoe_bd_tbl->buf_addr_hi;
			rinfo->sgl_ctx.cached_sge.cur_buf_rem=
					fcoe_bd_tbl->buf_len;

			task->txwr_rxrd.const_ctx.init_flags |= 1 <<
				FCOE_TCE_TX_WR_RX_RD_CONST_CACHED_SGE_SHIFT;
		} else {
			task->txwr_only.sgl_ctx.sgl.mul_sgl.cur_sge_addr.lo =
					U64_LO(GET_DMA_ADDR(bd_tbl->bd_tbl_dma));
			task->txwr_only.sgl_ctx.sgl.mul_sgl.cur_sge_addr.hi =
					U64_HI(GET_DMA_ADDR(bd_tbl->bd_tbl_dma));
			task->txwr_only.sgl_ctx.sgl.mul_sgl.sgl_size =
					bd_tbl->bd_valid;
		}
	}

	/*Tx Write Rx Read */
	/* Init state to NORMAL */
	task->txwr_rxrd.const_ctx.init_flags |= task_type <<
				FCOE_TCE_TX_WR_RX_RD_CONST_TASK_TYPE_SHIFT;
	task->txwr_rxrd.const_ctx.init_flags |= FCOE_TASK_DEV_TYPE_DISK <<
				FCOE_TCE_TX_WR_RX_RD_CONST_DEV_TYPE_SHIFT;
	task->txwr_rxrd.const_ctx.init_flags |= FCOE_TASK_CLASS_TYPE_3 <<
				FCOE_TCE_TX_WR_RX_RD_CONST_CLASS_TYPE_SHIFT;
	/* tx flags */
	task->txwr_rxrd.const_ctx.tx_flags = FCOE_TASK_TX_STATE_NORMAL <<
				FCOE_TCE_TX_WR_RX_RD_CONST_TX_STATE_SHIFT;

	/* Set initiative ownership */
	task->txwr_rxrd.const_ctx.verify_tx_seq = 0;

	/* Set initial seq counter */
	task->txwr_rxrd.union_ctx.tx_seq.ctx.seq_cnt = 1;

	/* Fill FCP_CMND IU */
	fcp_cmnd = (vmk_uint64 *)
		    task->txwr_rxrd.union_ctx.fcp_cmd.opaque;
	qfle3f_buildFCPCommand(ioRequest, (FCP_CMND_PAYLOAD *)&tmp_fcp_cmnd);

	/* swap fcp_cmnd */
	cnt = sizeof(FCP_CMND_PAYLOAD) / sizeof(vmk_uint64);

	for (i=0; i < cnt; i++) {
		*fcp_cmnd = vmk_CPUToBE64(tmp_fcp_cmnd[i]);
		fcp_cmnd++;
	}

	/* Rx Write Tx Read */
	task->rxwr_txrd.const_ctx.data_2_trns = ioRequest->data_xfer_len;

	contextID = target->contextID;
	task->rxwr_txrd.const_ctx.init_flags = contextID <<
				FCOE_TCE_RX_WR_TX_RD_CONST_CID_SHIFT;

	/* rx flags */
	/* Set state to "waiting for the first packet" */
	task->rxwr_txrd.var_ctx.rx_flags |= 1 <<
				FCOE_TCE_RX_WR_TX_RD_VAR_EXP_FIRST_FRAME_SHIFT;

	task->rxwr_txrd.var_ctx.rx_id = 0xffff;

	/* Rx Only */
	if (task_type == FCOE_TASK_TYPE_READ) {

		bd_count = bd_tbl->bd_valid;
		if (bd_count == 1) {

			struct fcoe_bd_ctx *fcoe_bd_tbl = bd_tbl->bd_tbl;

			task->rxwr_only.union_ctx.read_info.sgl_ctx.cached_sge.cur_buf_addr.lo
					= fcoe_bd_tbl->buf_addr_lo;
			task->rxwr_only.union_ctx.read_info.sgl_ctx.cached_sge.cur_buf_addr.hi
					= fcoe_bd_tbl->buf_addr_hi;
			task->rxwr_only.union_ctx.read_info.sgl_ctx.cached_sge.cur_buf_rem
					= fcoe_bd_tbl->buf_len;
			task->txwr_rxrd.const_ctx.init_flags |= 1 << 
				FCOE_TCE_TX_WR_RX_RD_CONST_CACHED_SGE_SHIFT;
		} else if (bd_count == 2) {
			struct fcoe_bd_ctx *fcoe_bd_tbl = bd_tbl->bd_tbl;

			task->rxwr_only.union_ctx.read_info.sgl_ctx.cached_sge.cur_buf_addr.lo
					= fcoe_bd_tbl->buf_addr_lo;
			task->rxwr_only.union_ctx.read_info.sgl_ctx.cached_sge.cur_buf_addr.hi
					= fcoe_bd_tbl->buf_addr_hi;
			task->rxwr_only.union_ctx.read_info.sgl_ctx.cached_sge.cur_buf_rem
					= fcoe_bd_tbl->buf_len;

			fcoe_bd_tbl++;
			task->rxwr_only.union_ctx.read_info.sgl_ctx.cached_sge.second_buf_addr.lo
					= fcoe_bd_tbl->buf_addr_lo;
			task->rxwr_only.union_ctx.read_info.sgl_ctx.cached_sge.second_buf_addr.hi
					= fcoe_bd_tbl->buf_addr_hi;
			task->rxwr_only.union_ctx.read_info.sgl_ctx.cached_sge.second_buf_rem
					= fcoe_bd_tbl->buf_len;
			task->txwr_rxrd.const_ctx.init_flags |= 1 << 
				FCOE_TCE_TX_WR_RX_RD_CONST_CACHED_SGE_SHIFT;
		} else { 

			task->rxwr_only.union_ctx.read_info.sgl_ctx.sgl.mul_sgl.cur_sge_addr.lo =
					U64_LO(GET_DMA_ADDR(bd_tbl->bd_tbl_dma));
			task->rxwr_only.union_ctx.read_info.sgl_ctx.sgl.mul_sgl.cur_sge_addr.hi =
					U64_HI(GET_DMA_ADDR(bd_tbl->bd_tbl_dma));
			task->rxwr_only.union_ctx.read_info.sgl_ctx.sgl.mul_sgl.sgl_size =
					bd_count;
		}
	}
}

/**
 * qfle3f_setupTaskContext - allocate and map task context
 *
 * @hba:	pointer to hba structure
 *
 * allocate memory for task context, and associated BD table to be used
 * by firmware
 *
 */
int qfle3f_setupTaskContext(struct qfle3fHBA *hba)
{
	int rc = 0;
	struct regpair *taskContext_bdt;
	vmk_IOA addr;
	int i;

	/* 
	 * Allocate task context bd table. A page size of bd table
	 * can map 256 buffers. Each buffer contains 32 task context
	 * entries. Hence the limit with one page is 8192 task context
	 * entries.
	 */
	hba->taskContextBDTable = qfle3f_dma_alloc(hba,
							&hba->taskContextBDDMA,
						    VMK_PAGE_SIZE);
	if (VMK_UNLIKELY(!hba->taskContextBDTable)) {
		qfle3f_err(hba, "unable to allocate task context BDT");
		rc = -1;
		goto out;
	}
	vmk_Memset(hba->taskContextBDTable, 0, VMK_PAGE_SIZE);

	/*
	 * Allocate taskContext which is an array of pointers pointing to
	 * a page containing 32 task contexts
	 */
	hba->taskContext = qfle3f_alloc(QFLE3F_TASK_CTX_ARR_SZ * sizeof(void *));
	if (VMK_UNLIKELY(!hba->taskContext)) {
		qfle3f_err(hba, "unable to allocate task context array");
		rc = -1;
		goto out1;
	}

	/*
	 * Allocate taskContextDMA which is an array of dma addresses
	 */
	taskContext_bdt = (struct regpair *)hba->taskContextBDTable;
	for (i=0; i < QFLE3F_TASK_CTX_ARR_SZ; i++) {

		hba->taskContext[i] = qfle3f_dma_alloc(hba,
								&hba->taskContextDMA[i],
								VMK_PAGE_SIZE);

		if (VMK_UNLIKELY(!hba->taskContext[i])) {
			qfle3f_err(hba, "unable to alloc task context");
			rc = -1;
			goto out3;
		}

		vmk_Memset(hba->taskContext[i], 0, VMK_PAGE_SIZE);
		//addr = (vmk_uint64)hba->taskContextDMA[i];
		addr = GET_DMA_ADDR(hba->taskContextDMA[i]);
		taskContext_bdt->hi = QL_CPU_TO_LE32((vmk_uint64)addr >> 32);
		taskContext_bdt->lo = QL_CPU_TO_LE32((vmk_uint32)addr);
		taskContext_bdt++;
	}
	return 0;

out3:
	for (i=0; i < QFLE3F_TASK_CTX_ARR_SZ; i++) {
		if (hba->taskContext[i]) {

			qfle3f_dma_free(hba, &hba->taskContextDMA[i]);
			hba->taskContext[i] = NULL;
		}
	}

	qfle3f_free(hba->taskContext);
	hba->taskContext = NULL;
out1:
	qfle3f_dma_free(hba, &hba->taskContextBDDMA);
	hba->taskContextBDTable = NULL;
out:
	return rc;
}

void qfle3f_freeTaskContext(struct qfle3fHBA *hba)
{
	int i;

	if (hba->taskContextBDTable) {
		qfle3f_dma_free(hba, &hba->taskContextBDDMA);
		hba->taskContextBDTable = NULL;
	}

	if (hba->taskContext) {
		for (i=0; i < QFLE3F_TASK_CTX_ARR_SZ; i++) {
			if (hba->taskContext[i]) {
				qfle3f_dma_free(hba,
						    &hba->taskContextDMA[i]);
				hba->taskContext[i] = NULL;
			}
		}
		qfle3f_free(hba->taskContext);
		hba->taskContext = NULL;
	}
}

static void qfle3f_freeHashTable(struct qfle3fHBA *hba)
{
	int i;
	int segment_count;
	int hash_table_size;
	vmk_uint32 *pbl;

	segment_count = hba->hashTableSegmentCount;
	hash_table_size = QFLE3_FCOE_NUM_CONNECTIONS* QFLE3F_MAX_ROWS_IN_HASH_TBL *
		sizeof(struct fcoe_hash_table_entry);

	pbl = hba->hashTablePBL;
	for (i = 0; i < segment_count; ++i) {
		vmk_IOA dma_address;

		dma_address = *pbl;
		++pbl;
		dma_address += ((vmk_uint64)(*pbl)) << 32;
		++pbl;
		qfle3f_dma_free(hba, hba->hashTableSegmentsDMA[i]);
	}

	if (hba->hashTablePBL) {
		qfle3f_dma_free(hba, &hba->hashTablePBLDMA);
		hba->hashTablePBL = NULL;
	}

	if (hba->hashTableSegmentsDMA[0]) {
		qfle3f_free(hba->hashTableSegmentsDMA[0]);
		hba->hashTableSegmentsDMA[0] = NULL;
	}
	if (hba->hashTableSegmentsDMA) {
		qfle3f_free(hba->hashTableSegmentsDMA);
		hba->hashTableSegmentsDMA = NULL;
	}
	if (hba->hashTableSegments) {
		qfle3f_free(hba->hashTableSegments);
		hba->hashTableSegments = NULL;
	}
}

static int qfle3f_allocateHashTable(struct qfle3fHBA *hba)
{
	int i;
	int hash_table_size;
	int segment_count;
	int segment_array_size;
	int dma_segment_array_size;
	vmk_uint32 *pbl;

	hash_table_size = QFLE3_FCOE_NUM_CONNECTIONS * QFLE3F_MAX_ROWS_IN_HASH_TBL *
		sizeof(struct fcoe_hash_table_entry);

	segment_count = hash_table_size + QFLE3F_HASH_TBL_CHUNK_SIZE - 1;
	segment_count /= QFLE3F_HASH_TBL_CHUNK_SIZE;
	hba->hashTableSegmentCount = segment_count;

	segment_array_size = segment_count * sizeof(*hba->hashTableSegments);
	hba->hashTableSegments = qfle3f_alloc(segment_array_size);
	if (VMK_UNLIKELY(!hba->hashTableSegments)) {
		qfle3f_err(hba, "hash table pointers alloc failed");
		return -1;
	}
	dma_segment_array_size = segment_count;

    qfle3f_log(hba, LOG_INIT, "dma_segment_array_size = %d", dma_segment_array_size);

	hba->hashTableSegmentsDMA = qfle3f_alloc(dma_segment_array_size * sizeof(struct qla_dmamem *));
	if (VMK_UNLIKELY(!hba->hashTableSegmentsDMA)) {
		qfle3f_err(hba, "hash table pointers (dma) alloc failed");
		return -1;
	}

    void *temp = qfle3f_alloc(dma_segment_array_size * sizeof(struct qla_dmamem));
	if (VMK_UNLIKELY(!temp)) {
		qfle3f_err(hba, "hash table qla_dmamem (dma) alloc failed");
		return -1;
	}

    for (i = 0; i < dma_segment_array_size; i++) {
		qfle3f_log(hba, LOG_INIT, "hash table qla_dmamem (dma) : i = %d", i);
        hba->hashTableSegmentsDMA[i] = temp + (sizeof(struct qla_dmamem) * i);
    }

	for (i = 0; i < segment_count; ++i) {
		hba->hashTableSegments[i] =
			qfle3f_dma_alloc(hba,
						hba->hashTableSegmentsDMA[i],
						QFLE3F_HASH_TBL_CHUNK_SIZE);
		qfle3f_log(hba, LOG_INIT, "hash table segment[%i]: %p: %lx",
		       i, hba->hashTableSegments[i],
		       GET_DMA_ADDR(*hba->hashTableSegmentsDMA[i]));
		if (VMK_UNLIKELY(!hba->hashTableSegments[i])) {
			qfle3f_err(hba, "hash segment alloc failed");
			while (--i >= 0) {
				qfle3f_dma_free(hba, hba->hashTableSegmentsDMA[i]);
				hba->hashTableSegments[i] = NULL;
			}
			qfle3f_free(hba->hashTableSegments);
			hba->hashTableSegments = NULL;
			qfle3f_free(temp);
			hba->hashTableSegmentsDMA[0] = NULL;
			qfle3f_free(hba->hashTableSegmentsDMA);
			hba->hashTableSegmentsDMA = NULL;
			return -1;
		}
		vmk_Memset(hba->hashTableSegments[i], 0,
		       QFLE3F_HASH_TBL_CHUNK_SIZE);
	}

	hba->hashTablePBL = qfle3f_dma_alloc(hba,
						 &hba->hashTablePBLDMA,
						 VMK_PAGE_SIZE);
	qfle3f_log(hba, LOG_INIT, "hash table pbl: %p: %lx",
		       hba->hashTablePBL,
				GET_DMA_ADDR(hba->hashTablePBLDMA));
	if (VMK_UNLIKELY(!hba->hashTablePBL)) {
		qfle3f_err(hba, "hash table pbl alloc failed");
		for (i = 0; i < segment_count; ++i) {
			qfle3f_dma_free(hba, hba->hashTableSegmentsDMA[i]);
			hba->hashTableSegments[i] = NULL;
		}
		qfle3f_free(hba->hashTableSegments);
		hba->hashTableSegments = NULL;
		qfle3f_free(temp);
		hba->hashTableSegmentsDMA[0] = NULL;
		qfle3f_free(hba->hashTableSegmentsDMA);
		hba->hashTableSegmentsDMA = NULL;
		return -1;
	}
	vmk_Memset(hba->hashTablePBL, 0, VMK_PAGE_SIZE);

	pbl = hba->hashTablePBL;
	for (i = 0; i < segment_count; ++i) {
		vmk_uint64 paddr = GET_DMA_ADDR(*hba->hashTableSegmentsDMA[i]);
		*pbl = QL_CPU_TO_LE32((vmk_uint32) paddr);
		++pbl;
		*pbl = QL_CPU_TO_LE32((vmk_uint32) (paddr >> 32));
		++pbl;
	}
	pbl = hba->hashTablePBL;
	i = 0;
	while (*pbl && *(pbl + 1)) {
		vmk_uint32 lo;
		vmk_uint32 hi;
		lo = *pbl;
		++pbl;
		hi = *pbl;
		++pbl;
		qfle3f_log(hba, LOG_INIT, "hash table[%d]: 0x%08x%08x", i, hi, lo);
		++i;
	}
	return 0;
}

/**
 * qfle3f_setupFWResources - Allocate and map hash table and dummy buffer
 *
 * @hba:	Pointer to hba structure
 *
 */
int qfle3f_setupFWResources(struct qfle3fHBA *hba)
{
	unsigned long addr;
	vmk_uint32 mem_size;
	int i;

	if (qfle3f_allocateHashTable(hba))
		return -1;

	mem_size = QFLE3_FCOE_NUM_CONNECTIONS * sizeof(struct regpair);
	hba->t2HashTablePointer = qfle3f_dma_alloc(hba,
						    &hba->t2HashTablePointer_dma, mem_size);
	if (VMK_UNLIKELY(!hba->t2HashTablePointer)) {
		qfle3f_err(hba, "unable to allocate t2 hash table ptr");
		qfle3f_freeFWResources(hba);
		return -1;
	}
	vmk_Memset(hba->t2HashTablePointer, 0x00, mem_size);
	qfle3f_log(hba, LOG_INIT, "t2HashTablePointer = 0x%p, DMA = 0x%lx",
		hba->t2HashTablePointer, GET_DMA_ADDR(hba->t2HashTablePointer_dma));

	mem_size = QFLE3_FCOE_NUM_CONNECTIONS * sizeof(struct fcoe_t2_hash_table_entry);
	hba->t2HashTable = qfle3f_dma_alloc(hba,
						&hba->t2HashTable_dma, mem_size);
	if (VMK_UNLIKELY(!hba->t2HashTable)) {
		qfle3f_err(hba, "unable to allocate t2 hash table");
		qfle3f_freeFWResources(hba);
		return -1;
	}
	vmk_Memset(hba->t2HashTable, 0x00, mem_size);
	for (i=0; i < QFLE3_FCOE_NUM_CONNECTIONS; i++) {
		addr = (unsigned long) GET_DMA_ADDR(hba->t2HashTable_dma) +
			 ((i+1) * sizeof(struct fcoe_t2_hash_table_entry));
		hba->t2HashTable[i].next.lo = addr & 0xffffffff;
		hba->t2HashTable[i].next.hi = addr >> 32;
	}
	qfle3f_log(hba, LOG_INIT, "t2HashTable = 0x%p, DMA = 0x%lx",
		hba->t2HashTable, GET_DMA_ADDR(hba->t2HashTable_dma));

	hba->dummyBuffer = qfle3f_dma_alloc(hba, &hba->dummyBufferDMA,
                                     VMK_PAGE_SIZE);
    if (VMK_UNLIKELY(!hba->dummyBuffer)) {
        qfle3f_err(hba, "unable to alloc MP Dummy Buffer");
		qfle3f_freeFWResources(hba);
		return -1;
    }

	hba->statsBuffer = qfle3f_dma_alloc(hba,
						 &hba->statsBufferDMA, VMK_PAGE_SIZE);
	if (VMK_UNLIKELY(!hba->statsBuffer)) {
		qfle3f_err(hba, "unable to alloc Stats Buffer");
		qfle3f_freeFWResources(hba);
		return -1;
	}
	vmk_Memset(hba->statsBuffer, 0x00, VMK_PAGE_SIZE);

	return 0;
}

void qfle3f_freeFWResources(struct qfle3fHBA *hba)
{
	if (hba->statsBuffer) {
		qfle3f_dma_free(hba, &hba->statsBufferDMA);
		hba->statsBuffer = NULL;
	}

	if (hba->dummyBuffer) {
		qfle3f_dma_free(hba, &hba->dummyBufferDMA);
		hba->dummyBuffer = NULL;
	}

	if (hba->t2HashTablePointer) {
		qfle3f_dma_free(hba, &hba->t2HashTablePointer_dma);
		hba->t2HashTablePointer = NULL;
	}

	if (hba->t2HashTable) {
        qfle3f_dma_free(hba, &hba->t2HashTable_dma);
        hba->t2HashTable = NULL;
    }
    qfle3f_freeHashTable(hba);
}
