/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

/*
 * qfle3f_fcoe.c: QLogic 10 Gigabit ESX Ethernet FCoE offload driver.
 *
 */

#include "qfle3f.h"
#include "qfle3f_vmk.h"
#include "qfle3f_version.h"
#include "qcnic_if.h"
#include "ql_fcoe_mgmt/ql_fcoe.h"
#include "ql_fcoe_mgmt/ql_fcoe_disc.h"


struct qfle3f_rport *qfle3f_targetLookupByPortID(struct qfle3fHBA *, vmk_uint32);

vmk_uint32 hba_count;
static volatile vmk_uint32 in_prog;
//vmk_atomic64 qfle3f_reg_device;
//static DEFINE_MUTEX(qfle3f_dev_lock);
//DEFINE_PER_CPU(struct fcoe_percpu_s, qfle3f_percpu);
vmk_uint64 qfle3f_get_wwpn(struct qfle3fHBA *hba);
vmk_uint64 qfle3f_get_wwnn(struct qfle3fHBA *hba);

#define FCOE_MAX_QUEUE_DEPTH 256
#define FCOE_LOW_QUEUE_DEPTH	32
#define FCOE_GW_ADDR_MODE           0x00
#define FCOE_FCOUI_ADDR_MODE        0x01

#define FCOE_WORD_TO_BYTE  4

int num_hosts;
//static struct qfle3fHBA *qfle3fHBA_lookup(struct net_device *phys_dev);
//static int qfle3f_rcv(vmk_PktHandle *, struct qfle3fHBA *hba);
struct qfle3fHBA *qfle3f_find_hba_for_dev(struct cnic_dev *dev);
void qfle3f_restart_fcoe_disc(struct qfle3fHBA *hba);
extern void qfle3f_vport_free(void *);

extern struct cnic_ulp_ops qfle3f_cnic_cb;

#if defined(__FIP_VLAN_DISC_CMPL__)
static int qfle3f_vlan_disc_cmpl(struct net_device *netdev);
#endif
//static vmk_uint8 *qfle3f_get_src_mac(struct qfle3fHBA *hba);

//static int qfle3f_fcoe_xport_registered;

#if defined(__VMKLNX__) && (VMWARE_ESX_DDK_VERSION == 50000)
/**
 * is_fip_mode() - returns true if FIP mode selected.
 * @fip:        FCoE controller.
 */
static inline bool is_fip_mode(struct fcoe_ctlr *fip)
{
	return fip->state == FIP_ST_ENABLED;
}
#endif

static int qfle3f_link_ok_wait(struct qfle3fHBA *hba);

#define VALID_VLANID_TIMEOUT_VALUE	20*1000 /* 20 seconds */


#if 0
/**
 * Function Name : qfle3f_vlan_disc_timeout
 * 
 * DESCRIPTION:
 *  In case the VLAN Discovery fails, set the VLAN ID to 1002.
 *  In case the VLAN ID is set to 1002 and we don't get VLAN Advertisement,
 *  set the VLAN ID to 0 and start the VLAN Discovery.
 *
 * Timer for each of this operation to complete is 20 seconds.
 *
 * We would keep on altering between VLAN ID 1002 and VLAN Discovery, until
 * a fabric is found.
 *
 * PARAMETERS:
 * work - The vlan_disc_timeout_work.work work associated
 * 	  with the hba.
 *
 * RETURN:
 *  None.
 */
static void qfle3f_vlan_disc_timeout(struct work_struct *work) {
	struct qfle3fHBA *hba = container_of(work, struct qfle3fHBA,
						 vlan_disc_timeout_work.work);
	struct fcoe_ctlr *fip = &hba->ctlr;
        struct fcoe_fcf *fcf, *next;
	struct fc_lport *lport;
	int vlanid, found = 0;

	lport = hba->ctlr.lp;

	if(!test_and_vmk_BitVectorClear(QFLE3F_VLAN_TIMEOUT, &hba->flags2)) {
		qfle3f_log(hba, LOG_INIT, "QFLE3F_VLAN_TIMEOUT is not set.");
		goto end;
	}

	if (qfle3f_link_ok_wait(lport)) {
		qfle3f_log(hba, LOG_INIT, "link is not ok.");
		goto end;
	}

	vlanid = vmklnx_cna_get_vlan_tag(hba->fcoe_net_dev) & VLAN_VID_MASK;

	qfle3f_log(hba, LOG_INIT, "VLAN ID = %d", vlanid);

	if(vlanid == 0) {
		qfle3f_log(hba, LOG_INIT, "VLAN Discovery Failed."
				"Trying default VLAN 1002");

		fcoe_ctlr_link_down(&hba->ctlr);

		fip->vlan_id = FCOE_FIP_NO_VLAN_DISCOVERY;
		vmklnx_cna_set_vlan_tag(hba->fcoe_net_dev, 1002);

		qfle3f_start_disc(hba);

	} else if(vlanid == 1002) {
		//list_for_each_entry_safe(fcf, next, &fip->fcfs, list) {
		VMK_LIST_FORALL_SAFE(&fip->fcfs, current, nextPtr) {
			fcf = VMK_LIST_ENTRY(current, struct , list);
			/* Check if we got a FIP Discovery Adv. as a response to
			 * to FIP Discovery Solicitation and also FCF is available
			 * for login.
			 */
			if((fcf->flags & FCOE_FIP_SOLICITED) && (fcf->flags & FCOE_FIP_AVAILABLE))
				found = 1;
		}

		if(found == 1) {
			qfle3f_log(hba, LOG_INIT, "Successful");
			return;
		}

		fcoe_ctlr_link_down(&hba->ctlr);
		qfle3f_log(hba, LOG_INIT, "VLAN 1002 failed. Trying VLAN Discovery.");
		fip->vlan_id = 0;
		vmklnx_cna_set_vlan_tag(hba->fcoe_net_dev, 0);
		qfle3f_start_disc(hba);
	}

end:
	return;
}
#endif /* #if 0 */

/*
 * NOTE : This function would also be called if driver sets the
 * 	  vlan by calling vmklnx_cna_set_vlan_tag() function.
 *
 *	  We do call vmklnx_cna_set_vlan_tag() from
 *	  qfle3f_vlan_disc_timeout() function.
 */
#if defined(__FIP_VLAN_DISC_CMPL__)
static int qfle3f_vlan_disc_cmpl(void *handle)
{
	struct net_device *netdev = dev_get_by_name(cna_dev->name);
	struct qfle3fHBA *hba = handle;
	struct cnic_dev *cnic_dev;

	cnic_dev = hba->cnic;
	if ((cnic_dev->mf_mode == MULTI_FUNCTION_SD) &&
	    (cnic_dev->mf_sub_mode == SUB_MF_MODE_FFA)) {
		hba->vlan_id = 0;
		vmk_BitVectorClear(hba->flags2, QFLE3F_VLAN_ENABLED);
	} else  {
#ifdef __FCOE_ENABLE_BOOT__
		hba->vlan_id = vmklnx_cna_get_vlan_tag(cna_dev) & VLAN_VID_MASK;
#else
		hba->vlan_id = vmklnx_cna_get_vlan_tag(cna_dev);
#endif
		vmk_BitVectorSet(hba->flags2, QFLE3F_VLAN_ENABLED);
	}

	qfle3f_err(hba, "%s: vlan_disc_cmpl: hba is on vlan_id %d",
		   hba->netdev->name, hba->vlan_id);

	return 0;
}

#endif

static VMK_ReturnStatus qfle3f_bind_pcidev(struct qfle3fHBA *hba);
static int qfle3f_fw_init(struct qfle3fHBA *hba);

static void qfle3f_port_shutdown(struct qfle3fHBA *hba);

static inline int qfle3f_start_io(struct qfle3fHBA *hba, vmk_PktHandle *skb)
{
	//vmk_PktHandle *nskb;
	//int rc;

	qfle3f_log(hba, LOG_FRAME, "skb %p", skb);


    /* TODO
     * Harman -- Call the new CNIC interface for queing this skb in the
     *           fcoe tx/rx reserved for the FIP/FCoE traffic.
     */

	return 0;
}

#ifdef FDMI_SUPPORT
#define FDMI_WAIT_FOR_VALUE	1*1000 /* 1 second */

static int qfle3f_backgroundThread(void *arg)
{
	struct qfle3fHBA *hba = (struct qfle3fHBA *)arg;

	qfle3f_log(hba, LOG_ELS, "background thread started.");

	set_user_nice(current, -20);
	set_current_state(TASK_INTERRUPTIBLE);
	while (!kthread_should_stop()) {
		schedule();
		set_current_state(TASK_RUNNING);

		/* This is a workaround.
		 *
		 * Ideally, PLOGI to Dir. Server should be
		 * send before sending PLOGI to Mgmt Server.
		 *
		 * As, driver only hooks up to FLOGI response, driver
		 * does not know if PLOGI to Dir. Server has been send.
		 * Therefore driver is sleeping for 1 second here.
		 *
		 * one second should be enough for libfc to send
		 * the PLOGI to Dir. Server.
		 *
		 * NOTE : Probably we don't need to sleep here.
		 */
		msleep(FDMI_WAIT_FOR_VALUE);
		/* Check if need to login to fabric management
		 * server and perform the RHBA and RPA commands.
		 */
		vmk_SemaLock(&hba->backgroundMutex);

		qfle3f_log(hba, LOG_ELS, "backgroundThread woken up."
			" FDMICurrentState = 0x%x",
			hba->FDMICurrentState);

		hba->FDMICurrentState = PLOGI_SEND_MGMT_SERVER;
		qfle3f_bk_send_els_cmd(hba);

		vmk_SemaUnlock(&hba->backgroundMutex);
		set_current_state(TASK_INTERRUPTIBLE);
	}
	set_current_state(TASK_RUNNING);
	return 0;
}
#endif

vmk_Bool checkStatReqComplFlag(void *data) {
    struct qfle3fHBA *hba = (struct qfle3fHBA *) data;
    return hba->statRequestCompleted;
}

void qfle3f_get_host_stats(struct qfle3fHBA *hba)
{
	struct fcoe_statistics_params *fw_stats;
    VMK_ReturnStatus status;

	fw_stats = (struct fcoe_statistics_params *)hba->statsBuffer;
	if (!fw_stats)
		return;

	//vmk_SemaLock(&hba->ioctlMutex);

	/* init_completion(&hba->statRequestDone); */
    hba->statRequestCompleted = VMK_FALSE;

	if (qfle3f_sendStatRequest(hba)) {
		//vmk_SemaUnlock(&hba->ioctlMutex);
		return;
	}
    status = qfle3f_sleepWithCondition(hba, hba, (vmk_WorldEventID)&(hba->statRequestDone),
                                        VMK_LOCK_INVALID, 2, "Waiting from stats RAMROD",
                                        checkStatReqComplFlag);
	if(status != VMK_OK) {
		qfle3f_err(hba, "ERROR! Stat RAMROD Request timedout: %s.",
						vmk_StatusToString(status));
	}

    hba->statRequestCompleted = VMK_TRUE;

	//vmk_SemaUnlock(&hba->ioctlMutex);
}

static int qfle3f_link_ok(struct qfle3fHBA *hba)
{
    /* vmk_Device nicDevice = hba->nicDevice; */
    struct cnic_dev *cnic = hba->cnic;
	int rc = 0;

	/* if ((dev->flags & IFF_UP) && netif_carrier_ok(dev)) { */
    if (cnic->uplinkLinkState.state == VMK_LINK_STATE_UP) {
		qfle3f_log(hba, LOG_INIT, "Link is up.");
		vmk_BitVectorClear(hba->hbaState, ADAPTER_STATE_LINK_DOWN);
	} else {
		vmk_BitVectorSet(hba->hbaState, ADAPTER_STATE_LINK_DOWN);
		rc = -1;
	}

    qfle3f_log(hba, LOG_INIT, "returning %d", rc);
	return rc;
}

static int qfle3f_link_ok_wait(struct qfle3fHBA *hba) {
	int wait_cnt = 0;
	int rc = -1;

	rc = qfle3f_link_ok(hba);
	while (rc) {
		vmk_WorldSleep(250 * VMK_USEC_PER_MSEC);
		/* give up after 3 secs */
		if (++wait_cnt > 12) {
			qfle3f_log(hba, LOG_INIT, "link_ok_wait: Link wait expired");
			break;
		}
		rc = qfle3f_link_ok(hba);
	}
	return rc;
}

vmk_uint64 qfle3f_get_wwnn(struct qfle3fHBA *hba)
{
	vmk_uint64 wwnn = 0;

	wwnn = hba->wwnn;
	return wwnn;
}

vmk_uint64 qfle3f_get_wwpn(struct qfle3fHBA *hba)
{
	vmk_uint64 wwpn = 0;

	wwpn = hba->wwpn;

	return wwpn;
}

static void qfle3f_netevent_handler(struct qfle3fHBA *hba, unsigned long event)
{
	vmk_uint32 link_possible = 1;

	qfle3f_log(hba, LOG_DEV_EVT, "netevent handler - event=%s %ld",
				vmk_NameToString(&hba->cnic->name), event);

	switch (event) {
	case NETDEV_UP:
		qfle3f_log(hba, LOG_DEV_EVT, "Port up, hbaState.");
		if (!vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_UP)) {
			qfle3f_err(hba, "indicate_netevent:  hba is not UP!!");
		} else if (vmk_BitVectorTest(hba->hbaType, QFLE3F_ADAPTER_BOOT)) {
			/* For boot devices VLAN ID is obtained from the boot
			 * table and not through VLAN DISCOVERY process. If this
			 * dev is marked as boot hba, use previously set VID
			 */
			vmk_BitVectorSet(hba->flags2, QFLE3F_VLAN_ENABLED);
		}
		break;
	case NETDEV_GOING_DOWN:
		qfle3f_log(hba, LOG_DEV_EVT, "Port going down");
		link_possible = 0;
		break;
	default:
		qfle3f_err(hba, "Unknown netevent %ld", event);
		return;
	}

	if (link_possible && !qfle3f_link_ok_wait(hba)) {
		qfle3f_notice(hba, "netevent_handler: call ctlr_link_up");

		if (vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_QUIESCE)) {
			qfle3f_notice(hba, "return: quiesceDevice is in progress.");
			 return;
		}

		hba->qlfcoeAdapter->LinkStatus = STATUS_LINK_UP;
		qfle3f_restart_fcoe_disc(hba);
	} else {
	    if(vmk_BitVectorTest(hba->initDone, QFLE3F_CREATE_DONE) &&
			hba->qlfcoeAdapter->LinkStatus == STATUS_LINK_UP) {
			hba->qlfcoeAdapter->LinkStatus = STATUS_LINK_DOWN;
			hba->qlfcoeAdapter->FIPVlanDiscTries = 0; //Reset the FIPVlanDiscTries.
            CancelAllFipTimeoutWorkitems(hba->qlfcoeAdapter);
	        qfle3f_log(hba, LOG_DEV_EVT, "Calling LogoutAllFabrics");
            LogoutAllFabrics(hba->qlfcoeAdapter, ONLOGOUT_DELETE, VMK_FALSE);
	    }
    }

	qfle3f_log(hba, LOG_DEV_EVT, "%s - event %ld END!!",
		vmk_NameToString(&hba->cnic->name), event);

}

#ifdef __FCOE_IF_RESTART_WQ__
static void qfle3f_indicateNetworkEvent_wq(void *data)
{
	ql_vmk_singlethread_workitem_t *work = (ql_vmk_singlethread_workitem_t *) data;
	struct qfle3fHBA *hba = (struct qfle3fHBA*) work->data1;
	unsigned long *eventP = (unsigned long *) work->data2;

	qfle3f_log(hba, LOG_DEV_EVT, "enter");

	if (VMK_UNLIKELY(!hba)) {
		qfle3f_err(hba, "Invalid arg!! hba: %p", hba);
		return;
	}

	qfle3f_log(hba, LOG_DEV_EVT, "netdev_wq processing event=%ld",
				*eventP);
	qfle3f_netevent_handler(hba, *eventP);

	qfle3f_free(work->data2);
	qfle3f_free(work);
}
#endif
/**
 * qfle3f_indicateNetworkEvent - Generic netdev event handler
 *
 * @context:	hba structure pointer
 * @event:	event type
 *
 * Handles NETDEV_UP, NETDEV_DOWN, NETDEV_GOING_DOWN,NETDEV_CHANGE and
 * NETDEV_CHANGE_MTU events
 */
static void qfle3f_indicateNetworkEvent(void *context, unsigned long event)
{
	struct qfle3fHBA *hba = (struct qfle3fHBA *)context;
#ifdef __FCOE_IF_RESTART_WQ__
	//struct qfle3f_netdev_work *work = NULL;
	ql_vmk_singlethread_workitem_t *item;
#endif

	qfle3f_notice(hba, "enter indicate_netevent - event=%ld", event);

	/*
	 * ASSUMPTION:
	 * indicate_netevent cannot be called from cnic unless qfle3f
	 * does register_device
	 */

	qfle3f_log(hba, LOG_DEV_EVT, "indicate event = %ld", event);

#ifdef __FCOE_IF_RESTART_WQ__
	item = qfle3f_alloc(sizeof(*item));
	if (VMK_UNLIKELY(!item)) {
		qfle3f_err(hba, "Unable to allocate work item event ");
		return ;
	}

	unsigned long *eventP = qfle3f_alloc(sizeof(unsigned long));
	*eventP = event;

	ql_vmk_init_singlethread_workitem(item, qfle3f_indicateNetworkEvent_wq,
									hba, eventP);

	ql_vmk_singlethread_queue_work(hba->fcoeInterfaceRestartWorkQueue, item);
#else
	//qfle3f_netevent_handler(hba, event);
#endif
}

/**
 * qfle3f_ulpGetStats - cnic callback to populate FCoE stats
 *
 * @handle:	transport handle pointing to hba struture
 */
 static int qfle3f_ulpGetStats(void *handle)
 {
	struct qfle3fHBA *hba = handle;
	struct cnic_dev *cnic;
	struct fcoe_stats_info *stats_addr;

	if (!hba)
		return -1;

	cnic = hba->cnic;

	if (cnic->fw_recov_in_progress) {
		qfle3f_err(hba, "fw error recovery in progress. Exit.");
		return -1;
	}

	stats_addr = &cnic->stats_addr->fcoe_stat;
	if (!stats_addr)
		return -1;

	vmk_StringCopy(stats_addr->version, QFLE3F_VERSION,
		sizeof(stats_addr->version));
	stats_addr->txq_size = QFLE3F_SQ_WQES_MAX;
	stats_addr->rxq_size = QFLE3F_CQ_WQES_MAX;

	return 0;
 }

static void qfle3f_interface_release(void *arg)
{
	struct qfle3fHBA *hba = (struct qfle3fHBA *) arg;
	//struct net_device *netdev;
	//struct net_device *phys_dev;

	qfle3f_log(hba, LOG_INIT, "Interface is being released hba = 0x%p", hba);

	//netdev = hba->netdev;
	//phys_dev = hba->phys_dev;

	/* tear-down FIP controller */
	if (vmk_BitVectorAtomicTestAndClear(hba->initDone, QFLE3F_CTLR_INIT_DONE)) {
		/* TODO: Call ql_fcoe_mgmt to destroy the ctlr */
		//fcoe_ctlr_destroy(&hba->ctlr);
		qfle3f_log(hba, LOG_INIT, "Interface released.");
	}
}

static inline void qfle3f_interface_get(struct qfle3fHBA *hba)
{
	ql_vmk_ref_get(&hba->kref);
}

static inline void qfle3f_interface_put(struct qfle3fHBA *hba)
{
	ql_vmk_ref_put(&hba->kref, qfle3f_interface_release, (void *) hba);
}

void qfle3f_interface_destroy(struct qfle3fHBA *hba)
{
	vmk_WarningMessage("qfle3f: entered %s.\n", __func__);

	if (hba->type == HBA_VIRTUAL) {
		qfle3f_warning(hba, "%s is to be called for physical port only.", __func__);
		return;
	}

	vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, hba->hbaType);
	vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, hba->initDone);
	vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, hba->flags2);
	vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, hba->flags);
	vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, hba->hbaState);

	if (hba->hbaMutex)
		vmk_SemaDestroy(&hba->hbaMutex);

	if (hba->ioctlMutex)
		vmk_SemaDestroy(&hba->ioctlMutex);

	vmk_WarningMessage("qfle3f: Destroying scsiScanListLock");
	if (hba->scsiScanListLock)
		vmk_SpinlockDestroy(hba->scsiScanListLock);

	if (hba->hbaLock)
		vmk_SpinlockDestroy(hba->hbaLock);

	if (hba->vportMutex)
		vmk_SemaDestroy(&hba->vportMutex);

	if (hba->vportListLck)
		vmk_SpinlockDestroy(hba->vportListLck);

	vmk_SemaLock(&qfle3fDriverInfo.drvLock);
	qfle3fDriverInfo.host_count--;
	vmk_BitVectorClear(qfle3fDriverInfo.hostInstanceMap, hba->instance);
	num_hosts--;
	vmk_ListRemove(&hba->link);
	vmk_SemaUnlock(&qfle3fDriverInfo.drvLock);

	if (hba->qlfcoeAdapter)
		qfle3f_free(hba->qlfcoeAdapter);

	qfle3f_free(hba);
}

/*
 * qfle3f_interface_create - create a new fcoe instance
 *
 * @cnic:	pointer to cnic device
 *
 * Creates a new FCoE instance on the given device which include
 * allocating hba structure.
 */
struct qfle3fHBA *qfle3f_interface_create(struct cnic_dev *cnic,
		vmk_uint8 is_npiv)
{
	struct qfle3fHBA *hba = NULL, *phba = NULL;
	VMK_ReturnStatus status;
	vmk_Bool ret;
	vmk_Name name;
	int i = 0;

	qfle3f_log(hba, LOG_INIT, "entering");

	/* NPIV: */
	if (is_npiv) {
		phba = qfle3f_find_hba_for_cnic(cnic);
		if (phba == NULL) {
			qfle3f_warning(hba, "Parent hba not found/invalid!");
			return NULL;
		}
		qfle3f_log(hba, LOG_NPIV, "phba=%p instance=0x%x host_no=0x%x",
			phba, phba->instance, phba->host_no);
	}

	hba = qfle3f_alloc(sizeof(*hba));
	if (VMK_UNLIKELY(!hba)) {
		qfle3f_err(hba, "Unable to allocate hba structure");
		return NULL;
	}

	//SV: NPIV: XXX: check if we can avoid blind memcpy of parent.
	//if (is_npiv)
	//	vmk_Memcpy(hba, phba, sizeof(struct qfle3fHBA));

	qfle3f_log(hba, LOG_INIT, "Finding the hba instance");
	/* Add HBA to the hba list */
	vmk_SemaLock(&qfle3fDriverInfo.drvLock);
	vmk_ListInsert(&hba->link, vmk_ListAtRear(&qfle3fDriverInfo.qfle3fHostList));
	ret = vmk_BitVectorNextBit(qfle3fDriverInfo.hostInstanceMap,
			0, VMK_FALSE, &hba->instance);
	if (ret == VMK_FALSE) {
		qfle3f_err(hba, "Host instance exhausted");
		qfle3f_free(hba);
		return NULL;
	}


	vmk_BitVectorSet(qfle3fDriverInfo.hostInstanceMap, hba->instance);
	qfle3fDriverInfo.host_count++;
	num_hosts++;
	vmk_SemaUnlock(&qfle3fDriverInfo.drvLock);

	if (is_npiv) {
		/* NPIV: Add vhba to parent's vportList and array of pointer */
		vmk_ListInitElement(&hba->vportList);
		vmk_SpinlockLock(phba->vportListLck);
		vmk_ListInsert(&hba->vportList, vmk_ListAtRear(&phba->vportList));
		/*
		 * The array is used for traversal where vportListLck
		 * cannot be held thru the traversal.
		 */
		for(i = 0; i < QFLE3F_MAX_NPIV; i++) {
			if(phba->vhba_address[i] == NULL) {
				phba->vhba_address[i] = hba;
				hba->vp_id = i + 1;
				break;
			}
		}
		phba->vportsInUse++;
		vmk_SpinlockUnlock(phba->vportListLck);
	} else {
		vmk_ListInit(&hba->vportList);
		vmk_NameFormat(&name, "vportListLck-%u", hba->instance);
		status = qfle3f_vmk_spin_lock_init(&hba->vportListLck,
				LOCK_RANK_HIGHEST, vmk_NameToString(&name));
		if(status != VMK_OK) {
			qfle3f_err(hba, "Failed to create vportListLck.");
			goto vport_list_lock_creation_failed;
		}

		/* Initialize array */
		for(i = 0; i < QFLE3F_MAX_NPIV; i++) {
			hba->vhba_address[i] = NULL;
		}

		hba->vp_id = 0;

		status = vmk_BinarySemaCreate(&hba->vportMutex,
					vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
					"vportMutex");
		if (status != VMK_OK) {
			qfle3f_err(hba, "qfle3f: Unable to create vportMutex. "
				"status=%s", vmk_StatusToString(status));
			goto vport_mutex_creation_failed;
		}
	}

	qfle3f_log(hba, LOG_INIT, "hba = %p instance = %d",
			hba, hba->instance);

	vmk_NameFormat(&name, "hbaLock-%u", hba->instance);
	status = qfle3f_vmk_spin_lock_init(&hba->hbaLock, LOCK_RANK_HIGHEST,
			vmk_NameToString(&name));
    if(status != VMK_OK) {
        qfle3f_err(hba, "Unable to allocate hba structure");
        goto hba_lock_creation_failed;
    }

	status = vmk_BinarySemaCreate(&hba->ioctlMutex,
				vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
				"ioctlMutex");
	if (status != VMK_OK) {
		qfle3f_err(hba, "qfle3f: Unable to create ioctlMutex. "
			"status=%s", vmk_StatusToString(status));
		goto ioctlMutex_creation_failed;
	}


	status = vmk_BinarySemaCreate(&hba->hbaMutex,
				vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
				"hbaMutex");
	if (status != VMK_OK) {
		qfle3f_err(hba, "qfle3f: Unable to create hbaMutex. "
			"status=%s", vmk_StatusToString(status));
		goto hba_mutex_creation_failed;
	}

    hba->hbaState = vmk_BitVectorAlloc(qfle3fDriverInfo.moduleHeapID, 32);
    if(hba->hbaState == NULL) {
        qfle3f_err(hba, "Unable to allocate hba state flags ");
        goto hbaState_alloc_failed;
    }
	vmk_BitVectorZap(hba->hbaState);

    hba->flags = vmk_BitVectorAlloc(qfle3fDriverInfo.moduleHeapID, 32);
    if(hba->flags == NULL) {
        qfle3f_err(hba, "Unable to allocate hba flags ");
        goto flags_alloc_failed;
    }
	vmk_BitVectorZap(hba->flags);

    hba->flags2 = vmk_BitVectorAlloc(qfle3fDriverInfo.moduleHeapID, 32);
    if(hba->flags2 == NULL) {
        qfle3f_err(hba, "Unable to allocate hba flags2");
        goto flags2_alloc_failed;
    }
	vmk_BitVectorZap(hba->flags2);

    hba->initDone = vmk_BitVectorAlloc(qfle3fDriverInfo.moduleHeapID, 32);
    if(hba->initDone == NULL) {
        qfle3f_err(hba, "Unable to allocate hba init done");
        goto initDone_alloc_failed;
    }
	vmk_BitVectorZap(hba->initDone);

    hba->hbaType = vmk_BitVectorAlloc(qfle3fDriverInfo.moduleHeapID, 32);
    if(hba->hbaType == NULL) {
        qfle3f_err(hba, "Unable to allocate hba type");
        goto hbaType_alloc_failed;
    }
	vmk_BitVectorZap(hba->hbaType);

	hba->cnic = cnic;

	status = qfle3f_bind_pcidev(hba);
	if (status != VMK_OK)
		goto bind_err;

	/* will get overwritten after we do vlan discovery */
	//hba->netdev = hba->phys_dev;

	//qfle3f_set_capabilities(hba);

#ifdef FDMI_SUPPORT
	char wq_name[60];
	char backgroundThread_name[60];

	vmk_Memset(wq_name, 0, 60);

	/* SV: TODO: Should be done after vmhbaName has been populated. */
	sprintf(wq_name, "%s:els_timeout_wq",
				hba->vmhbaName);
	sprintf(backgroundThread_name, "%s:qfle3f_l2_thread",
				hba->vmhbaName);

	mutex_init(&hba->backgroundMutex);
	hba->els_timeout_wq =
		create_singlethread_workqueue(wq_name);

	qfle3f_log(hba, LOG_INIT, "starting the background thread (%s).", backgroundThread_name);
	hba->backgroundThread = kthread_create(qfle3f_backgroundThread,
			(void *)hba,
			backgroundThread_name);
	if (IS_ERR(hba->backgroundThread)) {
		rc = PTR_ERR(hba->backgroundThread);
	}
	wake_up_process(hba->backgroundThread);
#endif

	qfle3f_log(hba, LOG_INIT, "returning");
	return hba;

	qfle3f_err(hba, "create_interface: bind error");
bind_err:
    vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, hba->hbaType);
hbaType_alloc_failed:
    vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, hba->initDone);
initDone_alloc_failed:
    vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, hba->flags2);
flags2_alloc_failed:
    vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, hba->flags);
flags_alloc_failed:
    vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, hba->hbaState);
hbaState_alloc_failed:
	vmk_SemaDestroy(&hba->hbaMutex);
hba_mutex_creation_failed:
	vmk_SemaDestroy(&hba->ioctlMutex);
ioctlMutex_creation_failed:
    vmk_SpinlockDestroy(hba->hbaLock);
hba_lock_creation_failed:
	if (!is_npiv) {
		vmk_SemaDestroy(&hba->vportMutex);
	}
vport_mutex_creation_failed:
	if (!is_npiv) {
    	vmk_SpinlockDestroy(hba->vportListLck);
	}
vport_list_lock_creation_failed:
	if (is_npiv) {
		vmk_SpinlockLock(phba->vportListLck);
		vmk_ListRemove(&hba->vportList);
		vmk_SpinlockUnlock(phba->vportListLck);
	}
	qfle3f_free(hba);
	return NULL;
}

void qfle3f_printPktContent(struct qfle3fHBA *hba, vmk_PktHandle *pkt, char* printBanner) {
    vmk_uint32 size;
    vmk_uint8 *b = (vmk_uint8 * ) vmk_PktFrameMappedPointerGet(pkt);

    size = vmk_PktFrameLenGet(pkt);

    qfle3f_printBuffer(hba, b, size, printBanner);
}

void print_fip_frame(struct qfle3fHBA *hba, vmk_PktHandle *pkt, char* printBanner) {
	FCOE_FIP_DISCOVERY_HEADER *fiph = NULL;
    FC_FRAME_HEADER *fh;
    char *Frame = NULL;
    vmk_uint32 length;
    vmk_uint32 opcode;
    char *mapped_data = NULL;

    mapped_data = (char *)vmk_PktFrameMappedPointerGet(pkt);

	Frame = (char *)vmk_PktFrameMappedPointerGet(pkt);
	fiph = (FCOE_FIP_DISCOVERY_HEADER *)((vmk_uint8 *)Frame);
	fh = (FC_FRAME_HEADER *)
		((vmk_uint8 *)Frame + sizeof(FC_OE_FRAME_HEADER));
	length = vmk_PktFrameLenGet(pkt);

	vmk_uint16 vlan_id = vmk_PktVlanIDGet(pkt);
    vmk_uint16 pri = vmk_PktPriorityGet(pkt);
    qfle3f_log(hba, LOG_INIT, "pkt vlan id = 0x%x pri = 0x%x", vlan_id, pri);

    qfle3f_printPktContent(hba, pkt, printBanner);

	opcode = (fiph->OperationCode[0] << 8) + fiph->OperationCode[1];

    qfle3f_log(hba, LOG_INIT, "pkt operation code = 0x%x sub code = 0x%x",
                opcode, fiph->SubCode);

    if(vmk_Memcmp(printBanner, "Recieved Frame ", 15))
        return;

    if(opcode == FCOE_FIP_OPERATION_CODE_DISCOVERY) {
        if(fiph->SubCode == FCOE_FIP_SUBCODE_DISCOVERY_SOLICITATION) {
            qfle3f_log(hba, LOG_INIT, "Discovery Solicitaion");
        } else if(fiph->SubCode == FCOE_FIP_SUBCODE_DISCOVERY_ADVERTISEMENT) {
            qfle3f_log(hba, LOG_INIT, "Discovery Advertisement");
        }
    } else if (opcode == FCOE_FIP_OPERATION_CODE_LINK_SERVICE) {
        if(fiph->SubCode == FCOE_FIP_SUBCODE_LINK_SERVICE_REQUEST) {
            qfle3f_log(hba, LOG_INIT, "FLOGI Request");
        } else if(fiph->SubCode == FCOE_FIP_SUBCODE_LINK_SERVICE_REPLY) {
            qfle3f_log(hba, LOG_INIT, "FLOGI Response");
        }
    } else if (opcode == FCOE_FIP_OPERATION_CODE_LINK_CONTROL) {
        if(fiph->SubCode == FCOE_FIP_SUBCODE_LINK_KEEPALIVE) {
            qfle3f_log(hba, LOG_INIT, "FIP Keep Alive");
        } else if(fiph->SubCode == FCOE_FIP_SUBCODE_CLEAR_VIRTUAL_LINKS) {
            qfle3f_log(hba, LOG_INIT, "FIP CVL");
        }
    } else if (opcode == FCOE_FIP_OPERATION_CODE_VLAN_DISCOVERY) {
        if(fiph->SubCode == FCOE_FIP_SUBCODE_VLAN_DISCOVERY_REQUEST) {
            qfle3f_log(hba, LOG_INIT, "VLAN Discovery Request");
        } else if(fiph->SubCode == FCOE_FIP_SUBCODE_VLAN_DISCOVERY_RESPONSE) {
            qfle3f_log(hba, LOG_INIT, "VLAN Discovery Response");
        }
    }
}

VMK_ReturnStatus qfle3f_rx_cb(struct cnic_dev *cdev, vmk_PktHandle *pkt) {

	struct qfle3fHBA *hba = qfle3f_find_hba_for_cnic(cdev);

	if (!vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_UP)) {
        qfle3f_log(hba, LOG_FRAME_ERR, "Adapter DOWN: hence dropping ll2 packet");
		vmk_PktRelease(pkt);
        return VMK_OK;
	}

    if (hba->cnic->uplinkLinkState.state == VMK_LINK_STATE_DOWN) {
        qfle3f_log(hba, LOG_FRAME_ERR, "LINK DOWN: hence dropping ll2 packet");
		vmk_PktRelease(pkt);
        return VMK_OK;
    }

	if (vmk_BitVectorTest(hba->flags, QFLE3F_FLAG_UNLOADING)) {
		qfle3f_log(hba, LOG_FRAME_ERR, "Unloading module qfle3f: hence dropping ll2 packet");
		vmk_PktRelease(pkt);
		return VMK_OK;
	}

#if 1
	vmk_uint8 *mapped_data =(char *)vmk_PktFrameMappedPointerGet(pkt);
	int size = vmk_PktFrameLenGet(pkt);
	qfle3f_printBuffer(hba, mapped_data, size, "Recv. Frame");
#endif

	if(hba)
		ql_fcoe_ll2_rx((void *)hba->qlfcoeAdapter, pkt, 0, 0);

	return VMK_OK;
}

void qfle3f_packet_send(struct ql_fcoe_adapter *qlfcoeAdapter, vmk_PktHandle *pkt)
{
    /* Call the new CNIC interface for queing this skb in the
     * fcoe tx/rx reserved for the FIP/FCoE traffic.
     *
     *
     * NOTE: This vmk_Pkt() is submitted through fcoe tx in nic driver.
     * So, the tx ack interrupt also is received by nic which should free the pkt.
     * No need for us to release the pkt here.
     */
	struct qfle3fHBA *hba = (struct qfle3fHBA *)qlfcoeAdapter->fcoe_drv_handle;
	VMK_ReturnStatus status;

#if 1
    /* print_fip_frame(hba, pkt, "Send Frame "); */
	vmk_uint8 *mapped_data =(char *)vmk_PktFrameMappedPointerGet(pkt);
	int size = vmk_PktFrameLenGet(pkt);
	qfle3f_printBuffer(hba, mapped_data, size, "Send Frame");
#endif

	if (hba && vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_UP)) {
		qfle3f_log(hba, LOG_FRAME, "Sending Frame");
		if (hba->cnic->uplinkLinkState.state == VMK_LINK_STATE_DOWN) {
			qfle3f_log(hba, LOG_FRAME_ERR, "LINK DOWN: hence not sending ll2 packet");
			vmk_PktRelease(pkt);
			return;
		}

		if (vmk_BitVectorTest(hba->flags, QFLE3F_FLAG_UNLOADING)) {
			qfle3f_log(hba, LOG_FRAME_ERR, "Unloading module qfle3f: hence dropping ll2 packet");
			vmk_PktRelease(pkt);
			return;
		}

		{
			struct qfle3fHBA *els_hba = hba;
			vmk_uint8 *mapped_data = (char *)vmk_PktFrameMappedPointerGet(pkt);
			vmk_EthHdr *eh = NULL;
			struct FC_FRAME_HEADER *fh = NULL;

			eh = (vmk_EthHdr *) mapped_data;
			if(eh->type == vmk_CPUToBE16(ETH_P_FCOE)) {
				struct fcoe_header {
					vmk_uint8 version[13];
					vmk_uint8 sof;
				};
				struct fcoe_header *now = NULL;
				vmk_uint8 *opcode =  NULL;

				now = (struct fcoe_header  *) (eh+1);
				fh = (struct FC_FRAME_HEADER *)(now + 1);
				opcode = (vmk_uint8 *)(fh + 1);

				if((fh->r_ctl == FC_R_CTL_EXTENDED_LINK_DATA_COMMAND_REQUEST)
						&& (fh->type == FC_TYPE_ELS) && *opcode == FC_ELS_LOGO)
				{
					struct qfle3f_rport *target =  NULL;
					int rc = 0;
					vmk_uint32 portID = ((((vmk_uint32)fh->d_id[0]) << 16) |
							(((vmk_uint32)fh->d_id[1]) << 8) |
							(vmk_uint32)fh->d_id[2]);
					struct LOGO_DATA *data = (struct LOGO_DATA *) (fh + 1);
					vmk_uint32 host_portID =
						((((vmk_uint32)data->fc_id[0]) << 16) |
							(((vmk_uint32)data->fc_id[1]) << 8) |
							(vmk_uint32)data->fc_id[2]);

					qfle3f_log(els_hba, LOG_INIT,
							"tgt-id=0x%x fcid=0x%x hba->fcid=0x%x",
							portID, host_portID,
							els_hba->initiator_targetPortID);
					if (host_portID != hba->initiator_targetPortID) {
						els_hba = qfle3f_vportLookupByPortID(els_hba, host_portID);
					}

					if (els_hba == NULL) {
						qfle3f_err(hba, "Failure: Invalid host_portID=0x%x\n",
								host_portID);
						goto use_ll2_send;
					}

					vmk_SpinlockLock(els_hba->hbaLock);
					target = qfle3f_targetLookupByPortID(els_hba, portID);
					qfle3f_log(els_hba, LOG_INIT, "target=%p logo-flag=0x%x\n",
							target,
							target ? (vmk_BitVectorTest(target->flags,
									QFLE3F_FLAG_EXPL_LOGO)) : 0);

					if (target) {
						if ((!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_DESTROYED) &&
							!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_DISABLED) &&
							vmk_BitVectorTest(target->flags, QFLE3F_FLAG_EXPL_LOGO)) ||
							!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_EXPL_LOGO)) {
							vmk_SpinlockUnlock(els_hba->hbaLock);

							rc = qfle3f_send_els_logo(target, pkt);
							if (rc == 0) { // Success Case.
								return;
							} else {
								// If we fail to send the frame via offloaded session,
								// let the frame go through the LL2 path.
								goto use_ll2_send;
							}
						} else {
							vmk_SpinlockUnlock(els_hba->hbaLock);
							qfle3f_warning(els_hba, "Skip explicit-logo: "
									"sess onload in progress / completed.\n");
							return;
						}
					} else {
						vmk_SpinlockUnlock(els_hba->hbaLock);
						qfle3f_warning(els_hba,
								"Fallback to ll2: sess not offloaded.\n");
					}
				}
			}
		}

use_ll2_send:
		status = hba->cnic->ll2_ops.ll2_send(hba->cnic, pkt);
		qfle3f_log(hba, LOG_FRAME, "Frame send: %s",
				vmk_StatusToString(status));
		if(status != VMK_OK) {
			qfle3f_err(hba, "Failed to send the frame: %s",
					vmk_StatusToString(status));
		}
	}
}

VMK_ReturnStatus qfle3f_interface_setup(struct qfle3fHBA *hba)
{
    struct ql_fcoe_adapter *qlfcoeAdapter = NULL;
	int i;

    qfle3f_log(hba, LOG_INIT, "enter");
	ql_vmk_ref_init(&hba->kref);

    /* Create hba structure for ql_fcoe */
    qlfcoeAdapter = qfle3f_alloc(sizeof(struct ql_fcoe_adapter));
    if (!qlfcoeAdapter)
        goto qlfcoeAdapter_alloc_fail;

    hba->qlfcoeAdapter = qlfcoeAdapter;

    vmk_Memcpy(qlfcoeAdapter->FipMacAddress, hba->dataSourceMacAddress, VMK_ETH_ADDR_LENGTH);
    vmk_Memcpy(qlfcoeAdapter->SPMacAddress, hba->dataSourceMacAddress, VMK_ETH_ADDR_LENGTH);

    qfle3f_log(hba, LOG_INIT, "FipMacAddress: ");
    qfle3f_log(hba, LOG_INIT, VMK_ETH_ADDR_FMT_STR,
                    VMK_ETH_ADDR_FMT_ARGS(qlfcoeAdapter->FipMacAddress));

    qfle3f_log(hba, LOG_INIT, "SPMacAddress: ");
    qfle3f_log(hba, LOG_INIT, VMK_ETH_ADDR_FMT_STR,
                    VMK_ETH_ADDR_FMT_ARGS(qlfcoeAdapter->SPMacAddress));

    /* Store in big endian format */
    SixtyFourBitToBigEndian(qlfcoeAdapter->PortWWN, qfle3f_get_wwpn(hba));
    SixtyFourBitToBigEndian(qlfcoeAdapter->NodeWWN, qfle3f_get_wwnn(hba));
    qlfcoeAdapter->dma_engine = hba->ioDMAEngine;
    qlfcoeAdapter->fcoe_drv_handle = (void *)hba;
    qlfcoeAdapter->vmk_scsi_adapter = hba->scsiAdapter;
    qlfcoeAdapter->packet_send = qfle3f_packet_send;
    qlfcoeAdapter->rport_add = qfle3f_remotePortAdd;
    qlfcoeAdapter->rport_delete = qfle3f_remotePortDelete;
    qlfcoeAdapter->rport_event_callback = qfle3f_rportEventHandler;

    qlfcoeAdapter->pri = 3;

    if ((hba->cnic->mf_mode == MULTI_FUNCTION_SD) && (hba->cnic->mf_sub_mode == SUB_MF_MODE_FFA))
    {
        // Bette Davis mode
        qfle3f_log(hba, LOG_INIT, "GetFCoEInfo mf_mode %x sd_mode %x, Bronco/VC mode is detected",
            hba->cnic->mf_mode, hba->cnic->mf_sub_mode);
        qlfcoeAdapter->AdapterFlags |= ADAPTER_FLAGS_BD_MODE;
    }
    else
    if ((hba->cnic->mf_mode == MULTI_FUNCTION_SD) && (hba->cnic->mf_sub_mode == SUB_MF_MODE_UFP))
    {
        // UFP mode
        qfle3f_log(hba, LOG_INIT, "GetFCoEInfo mf_mode %x sd_mode %x, UFP mode is detected, UFPVlanID %d",
            hba->cnic->mf_mode, hba->cnic->mf_sub_mode, hba->cnic->e1hov_tag);
        qlfcoeAdapter->AdapterFlags |= ADAPTER_FLAGS_UFP_MODE;
        qlfcoeAdapter->UFPVlanID = hba->cnic->e1hov_tag;
    }

    vmk_Memcpy(&(qlfcoeAdapter->vmk_pci_addr), &(hba->vmk_pci_addr), sizeof(vmk_PCIDeviceAddr));

    qfle3f_log(hba, LOG_INIT, "qlfcoeAdapter->FipMacAddress: ");
    qfle3f_log(hba, LOG_INIT, VMK_ETH_ADDR_FMT_STR,
                    VMK_ETH_ADDR_FMT_ARGS(qlfcoeAdapter->FipMacAddress));

    qfle3f_log(hba, LOG_INIT, "Memset target list : %ld",
                    (sizeof(struct qfle3f_rport *) * QFLE3_FCOE_NUM_CONNECTIONS));
    qfle3f_log(hba, LOG_INIT, "Initialize target list");
	for (i = 0; i < QFLE3_FCOE_NUM_CONNECTIONS; i++) {
		hba->targetOffloadList[i] = NULL;
	}

	if (hba->type == HBA_PHYSICAL)
		vmk_AtomicWrite64(&hba->qlfcoeAdapter->unload_flags, VMK_FALSE);

	hba->num_ofld_sess = 0;

    qfle3f_log(hba, LOG_INIT, "Return");
	return VMK_OK;

qlfcoeAdapter_alloc_fail:
	qfle3f_interface_put(hba);
	return VMK_NO_MEMORY;
}

static void qfle3f_unbind_hba_devices(struct qfle3fHBA *hba)
{
	qfle3f_log(hba, LOG_INIT, "ENTERED unbind_hba");
	qfle3f_freeFWResources(hba);
	qfle3f_freeTaskContext(hba);
}

/**
 * qfle3f_bind_hba_devices - binds qfle3f hba with the associated
 *			pci structure
 * @hba:		Adapter instance
 **/
static int qfle3f_bind_hba_devices(struct qfle3fHBA *hba)
{
	if (qfle3f_setupTaskContext(hba))
		goto mem_err;

	if (qfle3f_setupFWResources(hba))
		goto mem_err;

	return 0;
mem_err:
	qfle3f_unbind_hba_devices(hba);
	return -1;
}

static VMK_ReturnStatus qfle3f_bind_pcidev(struct qfle3fHBA *hba)
{
	struct cnic_dev *cnic;
    vmk_PCIDeviceID devID;
    VMK_ReturnStatus status;

	if (VMK_UNLIKELY(!hba->cnic)) {
		qfle3f_err(hba, "cnic is NULL");
		return VMK_FAILURE;
	}
	cnic = hba->cnic;
	if (VMK_UNLIKELY(!hba->cnic->pdev)) {
		qfle3f_err(hba, "!hba->pdev is NULL");
		return VMK_FAILURE;
	}

    status = vmk_PCIQueryDeviceID(cnic->pdev, &devID);
    if(status != VMK_OK) {
        qfle3f_err(hba, "vmk_PCIQueryDeviceID failed: %s",
                    vmk_StatusToString(status));
        return status;
    }

	switch (devID.deviceID) {
	case PCI_DEVICE_ID_NX2_57712:
	case PCI_DEVICE_ID_NX2_57712E:
		vmk_StringCopy(hba->sym_name, "57712", QFLE3F_SYM_NAME_LEN);
		break;
	case PCI_DEVICE_ID_NX2_57800:
	case PCI_DEVICE_ID_NX2_57800_MF:
		vmk_StringCopy(hba->sym_name, "57800", QFLE3F_SYM_NAME_LEN);
		break;
	case PCI_DEVICE_ID_NX2_57810:
	case PCI_DEVICE_ID_NX2_57810_MF:
		vmk_StringCopy(hba->sym_name, "57810", QFLE3F_SYM_NAME_LEN);
		break;
	case PCI_DEVICE_ID_NX2_57840_MF:
	case PCI_DEVICE_ID_NX2_57840_2_20:
	case PCI_DEVICE_ID_NX2_57840_4_10:
		vmk_StringCopy(hba->sym_name, "57840", QFLE3F_SYM_NAME_LEN);
		break;
	default:
		qfle3f_err(hba, "Unknown device id 0x%x", devID.deviceID);
		break;
	}

	return VMK_OK;
}

VMK_ReturnStatus qfle3f_check_flogi_completion(struct qfle3fHBA *vhba)
{
	struct ql_fcoe_adapter *Adapter = vhba->qlfcoeAdapter;
	struct ql_fcoe_fabric *PhysFabric =
		GetFabricByFlags(Adapter,
				FABRIC_FLAGS_NPIV_SUPPORTED | FABRIC_FLAGS_LOGGED_IN |
				FABRIC_FLAGS_NPIV_OWNER,
				FABRIC_FLAGS_NPIV_SUPPORTED | FABRIC_FLAGS_LOGGED_IN |
				FABRIC_FLAGS_NPIV_OWNER);
	if (PhysFabric) {
		ReleaseFabricReference(PhysFabric);
		return VMK_OK;
	} else
		return VMK_FAILURE;
}

VMK_ReturnStatus qfle3f_flogi_poll(void *data)
{
	struct qfle3fHBA *vhba = (struct qfle3fHBA *)data, *hba = NULL;
	VMK_ReturnStatus status = VMK_FAILURE;
#define QFLE3F_FLOGI_POLL_TIME       (100 * VMK_USEC_PER_MSEC)

	qfle3f_log(vhba, LOG_NPIV, "entering flogi poll world =0x%x",
				vhba->flogiPollWorldID);

	if (vhba && vhba->type == HBA_VIRTUAL)
		hba = vhba->phba;
	else {
		qfle3f_err(vhba, "Error: Invalid vhba.");
		return VMK_BAD_PARAM;
	}

	while (1) {

        if(vmk_AtomicRead64(&vhba->pollwakeupPending) == VMK_FALSE) {
		    status = vmk_WorldWait(VMK_EVENT_NONE, VMK_LOCK_INVALID,
				VMK_TIMEOUT_UNLIMITED_MS, "flogi poll world sleeping");

		    /* Only exit the world if VMK_DEATH_PENDING is returned */
		    if(status == VMK_DEATH_PENDING)
			    goto exit_poll_world;

		    if(status != VMK_OK)
			    continue;
        }

		qfle3f_log(vhba, LOG_NPIV, "waking flogi-poll world=0x%x status = %s.",
			vhba->flogiPollWorldID, vmk_StatusToString(status));

        vmk_AtomicWrite64(&vhba->pollwakeupPending, VMK_FALSE);

	/* Poll until parent hba's FLOGI is completed. */
	/* SV: TODO: Specify a finite time to wait for Flogi completion. */
		while (1) {
			/* Serialize vport access */
			vmk_SemaLock(&hba->vportMutex);
			if (qfle3f_check_flogi_completion(vhba) == VMK_OK) {
				vmk_uint32 event = NETDEV_DOWN;

				qfle3f_log(vhba, LOG_NPIV,
						"Link state is uplinkLinkState.state = %d",
						vhba->cnic->uplinkLinkState.state);

				if(vhba->cnic->uplinkLinkState.state == VMK_LINK_STATE_UP)
					event = NETDEV_UP;
				else if(vhba->cnic->uplinkLinkState.state == VMK_LINK_STATE_DOWN)
					event = NETDEV_DOWN;

				vmk_SemaUnlock(&hba->vportMutex);
				qfle3f_cnic_cb.indicate_netevent(vhba, event);
				break;
			}
			vmk_SemaUnlock(&hba->vportMutex);

			vmk_WorldSleep(QFLE3F_FLOGI_POLL_TIME);
		}
	} /* end unconditional while */

exit_poll_world:
	qfle3f_log(vhba, LOG_NPIV, "Exiting flogi poll world.");
	return status;
}



void qfle3f_restart_fcoe_disc(struct qfle3fHBA *hba)
{
	struct qfle3fHBA *vhba = NULL;
	vmk_ListLinks *current = NULL, *nextPtr = NULL;
    VMK_ReturnStatus status;
    int rc = 0;
	int i = 0;

	/* First time post bootup */
	if(hba->type == HBA_PHYSICAL &&
			!vmk_BitVectorAtomicTestAndSet(hba->initDone, QFLE3F_CREATE_DONE)) {
		qfle3f_notice(hba, "creating ql_fcoe_mgmt interface");
		status = ql_fcoe_interface_create(hba->qlfcoeAdapter);
		if(status != VMK_OK) {
			qfle3f_err(hba, "ql_fcoe_interface_create failed");
		}

		qfle3f_notice(hba, "Init fw");
		rc = qfle3f_fw_init(hba);
		if (rc)
			qfle3f_err(hba, "qfle3f_fw_init failed : %d", rc);

		vmk_SpinlockLock(hba->vportListLck);
		VMK_LIST_FORALL_SAFE(&hba->vportList, current, nextPtr) {
			vhba = VMK_LIST_ENTRY(current, struct qfle3fHBA, vportList);

			/* Initiate pre-boot vport login. */
			if (vmk_BitVectorTest(vhba->flags2, QFLE3F_NPIV_PREBOOT)) {
				vmk_WorldProps flogi_poll_props;
				char flogi_poll_world_name[20];

				/* Re-assign taskContext for pre-boot vports. */
				vhba->taskContext = hba->taskContext;
				//SV: NPIV: XXX: dma addresses may not be required to be shared.
				vmk_Memcpy(&vhba->taskContextDMA, &hba->taskContextDMA,
						(sizeof(struct qla_dmamem) * QFLE3F_TASK_CTX_ARR_SZ));
				vhba->taskContextBDTable = hba->taskContextBDTable;
				vmk_Memcpy(&vhba->taskContextBDDMA, &hba->taskContextBDDMA,
						sizeof(struct qla_dmamem));

				vmk_AtomicWrite64(&vhba->pollwakeupPending, VMK_TRUE);
				vmk_Memset(&flogi_poll_props, 0, sizeof(vmk_WorldProps));
				flogi_poll_props.heapID = qfle3fDriverInfo.moduleHeapID;

				flogi_poll_props.moduleID = vmk_ModuleCurrentID;
				flogi_poll_props.startFunction = (void *)qfle3f_flogi_poll;
				flogi_poll_props.data = (void *)vhba;
				flogi_poll_props.schedClass = VMK_WORLD_SCHED_CLASS_DEFAULT;
				vmk_StringFormat(flogi_poll_world_name, 20, NULL,
						"qfle3f-flogi-poll-%u", vhba->instance);
				flogi_poll_props.name = flogi_poll_world_name;
				vmk_SpinlockUnlock(hba->vportListLck);
				status = vmk_WorldCreate(&flogi_poll_props,
						&(vhba->flogiPollWorldID));
				vmk_SpinlockLock(hba->vportListLck);
				if (status != VMK_OK) {
					qfle3f_warning(vhba, "FAILED flogi_poll_world create :%s",
							vmk_StatusToString(status));
				}

				qfle3f_log(vhba, LOG_INIT, "flogi poll world created. world id = %d.",
						vhba->flogiPollWorldID);
			}
		}
		vmk_SpinlockUnlock(hba->vportListLck);
	} else {
		/* Subsequent link toggles */
		if (hba->type == HBA_PHYSICAL) {
			qfle3f_log(hba, LOG_INIT, "Start Fabric Discovery: ");
			StartFabricDiscovery(hba->qlfcoeAdapter);

			vmk_SpinlockLock(hba->vportListLck);
			for(i = 0; i < QFLE3F_MAX_NPIV; i++) {

				vhba = hba->vhba_address[i];
				if(vhba == NULL)
					continue;

				if (!ql_vmk_ref_get(&vhba->kref))
					continue;

				if (vmk_BitVectorTest(vhba->hbaState, ADAPTER_STATE_VPORT_DELETE)) {
					ql_vmk_ref_put(&vhba->kref, qfle3f_vport_free, (void *)vhba);
					continue;
				}

				vmk_SpinlockUnlock(hba->vportListLck);
				vmk_AtomicWrite64(&vhba->pollwakeupPending, VMK_TRUE);
				status = vmk_WorldForceWakeup(vhba->flogiPollWorldID);
				qfle3f_log(vhba, LOG_NPIV, "flogiPollWorldID status = %s", vmk_StatusToString(status));
				ql_vmk_ref_put(&vhba->kref, qfle3f_vport_free, (void *)vhba);
				vmk_SpinlockLock(hba->vportListLck);
			}
			vmk_SpinlockUnlock(hba->vportListLck);

		} else if (hba->type == HBA_VIRTUAL) {
			vmk_uint8 npivStatus = 0;
			vmk_uint8 wwpn_array[WWN_SIZE], wwnn_array[WWN_SIZE];

			qfle3f_u64ToCharArray_BE(hba->wwpn, WWN_SIZE, wwpn_array);
			qfle3f_u64ToCharArray_BE(hba->wwnn, WWN_SIZE, wwnn_array);

			qfle3f_log(hba, LOG_NPIV, "Start NPIV Fabric Discovery: "
					"wwpn: 0x%lx wwnn:0x%lx", hba->wwpn, hba->wwnn);
			qfle3f_log(hba, LOG_NPIV,
					"WWPN = 0x%02x %02x %02x %02x %02x %02x %02x %02x",
					wwpn_array[0], wwpn_array[1], wwpn_array[2], wwpn_array[3],
					wwpn_array[4], wwpn_array[5], wwpn_array[6], wwpn_array[7]);
			qfle3f_log(hba, LOG_NPIV,
					"WWNN = 0x%02x %02x %02x %02x %02x %02x %02x %02x",
					wwnn_array[0], wwnn_array[1], wwnn_array[2], wwnn_array[3],
					wwnn_array[4], wwnn_array[5], wwnn_array[6], wwnn_array[7]);
			/* Serialize vport login */
			/* This is being done because our ref count handling for fabric structure
 			 * is not completely atomic and simultaneous access is leading to issue
 			 * around the fabric being freed incorrectly. This logic can be changed once
 			 * changes to address the "atomicity" issue is in place.
 			 */ 
			vmk_SemaLock(&hba->phba->vportMutex);
			status = LoginVirtualPort(hba->qlfcoeAdapter, hba, hba->scsiAdapter, wwpn_array,
					wwnn_array, hba->tag, hba->virtualName, NULL, NULL,
					&npivStatus);
			vmk_SemaUnlock(&hba->phba->vportMutex);
			if (npivStatus == NPIV_SUCCESS) {
				qfle3f_log(hba, LOG_NPIV, "vport login successfull. vport=0x%x",
						hba->host_no);
			} else {
				qfle3f_warning(hba, "Error: vport login failed. "
						"vport=0x%x npivStatus=0x%x status=%s",
						hba->host_no, npivStatus, vmk_StatusToString(status));
			}
		}
	}
}

#ifdef __FCOE_IF_RESTART_WQ__
static void qfle3f_restart_fcoe_init(void *data)
{
    ql_vmk_singlethread_workitem_t *work = (ql_vmk_singlethread_workitem_t *) data;
    struct qfle3fHBA *hba = (struct qfle3fHBA *) work->data1;

	qfle3f_restart_fcoe_disc(hba);
}
#endif

/**
 * qfle3f_ulp_start - cnic callback to initialize & start hba instance
 *
 * @handle:	transport handle pointing to hba struture
 *
 * This function maps hba structure to pcidev structure and initiates
 *	firmware handshake to enable/initialize on-chip FCoE components.
 *	This qfle3f - cnic interface api callback is used after following
 *	conditions are met -
 *	a) underlying network interface is up (marked by event NETDEV_UP
 *		from netdev
 *	b) qfle3f adatper structure is registered.
 */
static void qfle3f_ulp_start(void *handle)
{
	struct qfle3fHBA *hba = handle;

	qfle3f_log(hba, LOG_INIT, "entered: hba = %p", hba);

	vmk_SemaLock(&qfle3fDriverInfo.drvLock);

	if (vmk_BitVectorTest(hba->initDone, QFLE3F_FW_INIT_DONE))
		goto start_disc;

	if (vmk_BitVectorTest(hba->initDone, QFLE3F_CREATE_DONE))
		qfle3f_fw_init(hba);
start_disc:
	vmk_SemaUnlock(&qfle3fDriverInfo.drvLock);

    vmk_BitVectorSet(hba->hbaState, ADAPTER_STATE_READY);
	qfle3f_log(hba, LOG_INIT, "qfle3f started.");

	/* Kick off Fabric discovery*/
	if (vmk_BitVectorTest(hba->initDone, QFLE3F_CREATE_DONE)) {
#ifdef __FCOE_IF_RESTART_WQ__
		qfle3f_notice(hba, "ulp_init: start discovery, "
			   "queue fcoeInterfaceRestartWorkQueue");

		ql_vmk_queue_delayed_work(qfle3fDriverInfo.delayedTQ,
					hba->fcoeInterfaceRestartWorkQueue,
					&hba->fcoeDiscoveryWork,
					qfle3f_restart_fcoe_init, hba, NULL, 2000);
#else
		qfle3f_restart_fcoe_disc(hba);
#endif
	}
	qfle3f_log(hba, LOG_INIT, "exit: hba = %p", hba);
}

static void qfle3f_port_shutdown(struct qfle3fHBA *hba)
{
	qfle3f_log(hba, LOG_INIT, "ENTERED: flush fcoeInterfaceRestartWorkQueue");
}

/**
 * qfle3f_stop - cnic callback to shutdown hba instance
 * @handle:	transport handle pointing to hba structure
 *
 * Driver checks if hba is already in shutdown mode, if not start
 * 	the shutdown process.
 **/
void qfle3f_stop(struct qfle3fHBA *hba)
{
	struct cnic_dev *cnic = hba->cnic;
	vmk_uint32 event;
	vmk_uint8 wwpn_array[WWN_SIZE], wwnn_array[WWN_SIZE];
	int stop = (vmk_BitVectorTest(hba->initDone, QFLE3F_FW_INIT_DONE) &&
		    vmk_BitVectorTest(hba->initDone, QFLE3F_CREATE_DONE));

	qfle3f_log(hba, LOG_INIT, "ENTERED: initDone = %d",
				vmk_BitVectorNumBitsSet(hba->initDone));
	if (stop) {
		qfle3f_port_shutdown(hba);
		qfle3f_log(hba, LOG_INIT, "qfle3f_stop: waiting for %d "
				"offloaded sessions",
				hba->num_ofld_sess);
		vmk_BitVectorClear(hba->flags2, QFLE3F_VLAN_ENABLED);

		qfle3f_err(hba, "Clean-up FCoE CNA Queues");
	}
#if defined (__FCOE_CNA_QUEUE_CLEANUP__)
	if (vmk_BitVectorTest(hba->flags2, QFLE3F_CNA_QUEUES_ALLOCED)) {
		qfle3f_err(hba, "dev->netq_state 0x%lx",
			   hba->fcoe_net_dev->netq_state);
		vmknetddi_queueops_getset_state(hba->fcoe_net_dev, false);
		vmklnx_cna_cleanup_queues(hba->fcoe_net_dev);
		hba->cnaCleanupQueue++;

	}
#endif /* !defined (__VMKLNX__) */
	if (stop) {
		vmk_SemaLock(&hba->hbaMutex);
		vmk_BitVectorClear(hba->hbaState, ADAPTER_STATE_UP);
		vmk_BitVectorClear(hba->hbaState, ADAPTER_STATE_GOING_DOWN);

		vmk_BitVectorClear(hba->hbaState, ADAPTER_STATE_READY);
		vmk_SemaUnlock(&hba->hbaMutex);
		qfle3f_notice(hba, "Number of Offloaded Sessions =%d", hba->num_ofld_sess);

		/* Initiate a link down to kick off session upload/cleanup */
		if (cnic->fw_recov_in_progress ||
			 cnic->uplinkLinkState.state == VMK_LINK_STATE_UP) {
			if (cnic->uplinkLinkState.state == VMK_LINK_STATE_UP) {
				event = NETDEV_GOING_DOWN;
				qfle3f_cnic_cb.indicate_netevent(hba, event);
			}
			qfle3f_wait_for_upload(hba);
			qfle3f_notice(hba, "all sessions are uploaded, num_ofld_sess =%d", hba->num_ofld_sess);
			qfle3f_notice(hba, "wait for all Fabric to delete");
			qfle3f_u64ToCharArray_BE(hba->wwpn, WWN_SIZE, wwpn_array);
			qfle3f_u64ToCharArray_BE(hba->wwnn, WWN_SIZE, wwnn_array);
			ql_wait_for_fabric_deletion(hba->qlfcoeAdapter, wwpn_array, wwnn_array, PHYSICAL_FABRIC);
		}
	}
}

static int qfle3f_fw_init(struct qfle3fHBA *hba)
{
#define QFLE3F_INIT_POLL_TIME		(1000 / vmk_TimerCyclesPerSecond())
	int rc = -1;
	int i = vmk_TimerCyclesPerSecond();

	rc = qfle3f_bind_hba_devices(hba);
	if (rc) {
		qfle3f_warning(hba,
			"qfle3f_bind_hba_devices failed - rc = %d", rc);
		goto err_out;
	}

	rc = qfle3f_sendFirmwareFCoEInitiMessage(hba);
	if (rc) {
		qfle3f_warning(hba,
			"qfle3f_sendFirmwareFCoEInitiMessage failed - rc = %d", rc);
		goto err_unbind;
	}

	/*
	 * Wait until the hba init message is complete, and hba
	 * state is UP.
	 */
	while (!vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_UP) && i--)
		vmk_WorldSleep(QFLE3F_INIT_POLL_TIME);

	if (!vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_UP)) {
		qfle3f_err(hba, "qfle3f_start: failed to initialize. Ignoring...");
		rc = -1;
		goto err_unbind;
	}


	/* Mark HBA to indicate that the FW INIT is done */
	vmk_BitVectorSet(hba->initDone, QFLE3F_FW_INIT_DONE);
	return 0;

err_unbind:
	qfle3f_unbind_hba_devices(hba);
err_out:
	return rc;
}

vmk_Bool checkDestroyFuncReqComplFlag(void *data) {
    struct qfle3fHBA *hba = (struct qfle3fHBA *) data;

    return vmk_BitVectorTest(hba->flags, QFLE3F_FLAG_DESTROY_CMPL);
}

void qfle3f_fw_destroy(struct qfle3fHBA *hba)
{
	VMK_ReturnStatus status;

	if (vmk_BitVectorAtomicTestAndClear(hba->initDone, QFLE3F_FW_INIT_DONE)) {
		if (qfle3f_sendFirmwareFCoEDestroyMessage(hba) == 0) {
            status = qfle3f_sleepWithCondition(hba, (void *)hba,
                                    (vmk_WorldEventID)&(hba->destroyFunctionEvent),
                                    VMK_LOCK_INVALID, (QFLE3F_FW_TIMEOUT * VMK_MSEC_PER_SEC),
							        "destroy func wait", checkDestroyFuncReqComplFlag);

            qfle3f_notice(hba, "destroy func: %s", vmk_StatusToString(status));
            if(status != VMK_OK) {
                qfle3f_err(hba, "ERROR! Destroy Func wait failed: %s.",
                        vmk_StatusToString(status));
            }

			vmk_BitVectorClear(hba->flags, QFLE3F_FLAG_DESTROY_CMPL);
			/* This should never happen */
			qfle3f_log(hba, LOG_INIT, "hba->flags = ");
		}
		qfle3f_unbind_hba_devices(hba);
	}
}

/**
 * qfle3f_ulp_stop - cnic callback to shutdown hba instance
 * @handle:	transport handle pointing to hba structure
 *
 * Driver checks if hba is already in shutdown mode, if not start
 *	the shutdown process.
 */
static void qfle3f_ulp_stop(void *handle)
{
	struct qfle3fHBA *hba = (struct qfle3fHBA *)handle;

	qfle3f_notice(hba, "ULP_STOP");

	vmk_SemaLock(&qfle3fDriverInfo.drvLock);
	qfle3f_stop(hba);
	qfle3f_fw_destroy(hba);
	vmk_SemaUnlock(&qfle3fDriverInfo.drvLock);

	qfle3f_notice(hba, "ULP_STOP exit");
}

/**
 * qfle3f_cnic_fw_recovery_notify
 *
 * @context:	hba structure pointer
 * @event:	event type
 *
 */
static void qfle3f_cnic_fw_recovery_notify(void *context, int event)
{
	struct qfle3fHBA *hba = (struct qfle3fHBA *)context;

	qfle3f_notice(hba, "enter %s- event=%d", __func__, event);

}


/*
 * qfle3f_cnic_cb - global template of qfle3f - cnic driver interface
 *			structure carrying callback function pointers
 */
struct cnic_ulp_ops qfle3f_cnic_cb = {
	.version		= CNIC_ULP_OPS_VER,
	.cnic_init		= NULL,
	.cnic_exit		= NULL,
	.cnic_start		= qfle3f_ulp_start,
	.cnic_stop		= qfle3f_ulp_stop,
	.cnic_fw_recovery_notify = qfle3f_cnic_fw_recovery_notify,
	.indicate_kcqes		= qfle3f_indicateKCQE,
	.indicate_netevent	= qfle3f_indicateNetworkEvent,
	.cnic_get_stats         = qfle3f_ulpGetStats,
};
