/******************************************************************************
* The information contained in this file is confidential and proprietary to
* QLogic Corporation. No part of this file may be reproduced or
* distributed, in any form or by any means for any purpose, without the
* express written permission of QLogic Corporation.
*
*              Copyright (c) 2015 QLogic Corporation
*                        All rights reserved
*******************************************************************************/
#include <vmkapi.h>
#include "qfle3.h"
#include "qfle3_mgmt.h"

vmk_MgmtHandle vmkmgmt_api_handler;
qfle3_module_param_t qfle3_mod_param_array[] = {
   {"debug_mask", "Enabled debug mask (default: 0)",
    param_int, 0, 0, 0xFFFFFFFF, {QFLE3_MOD_PARAM_VALID_VALS_TERM}, 0, 0},
   {"mtu", "MTU when the driver is loaded. Range: 0-9000. (default: 1500)",
    param_int, 1500, 0, 9000, { QFLE3_MOD_PARAM_VALID_VALS_TERM}, 0, 0},
   {"intr_mode", "Interrupt Mode: 0 (AUTO), 1(IRQ), 2(MSI), and 3(MSI-X). Default: Auto",
    param_int, 0, 0, 3, {0, 1, 2, 3, QFLE3_MOD_PARAM_VALID_VALS_TERM}, 0, 0},
   {"txqueue_nr", "Number of Tx Queues: 0(Auto) or 1 to 8 (fixed queue number). Default: 4",
    param_int, 4, 1, 8, {1, 2, 3, 4, 5, 6, 7, 8, QFLE3_MOD_PARAM_VALID_VALS_TERM}, 0, 0},
   {"rxqueue_nr", "Number of Rx Queues: 0(Auto) or 1 to 8 (fixed queue number). Default: 4",
    param_int, 4, 1, 8, {1, 2, 3, 4, 5, 6, 7, 8, QFLE3_MOD_PARAM_VALID_VALS_TERM}, 0, 0},
   {"rssq_nr", "Number of RSS Queues: 0(Auto) or 1 to 4 (fixed queue number). Default: Auto",
    param_int, 0, 1, 4, {1, 2, 3, 4, QFLE3_MOD_PARAM_VALID_VALS_TERM}, 0, 0},
   {"hw_vlan", "Enable/Disable VLAN removal/insertion by hardware. 0: Disable and 1: Enabled. Default: 1",
    param_bool, 1, 0, 1, {0, 1, QFLE3_MOD_PARAM_VALID_VALS_TERM}, 0, 0},
   {"offload_flags", "Offload flags: 1(cso) 2(tso) 4(vxlan offload) 8(Geneve offload). Default: 15",
    param_int, 7, 0, 7, {0, 1, 2, 3, 4, 5, 6, 7, QFLE3_MOD_PARAM_VALID_VALS_TERM}, 0, 0},
   {"rx_filters", "Define the number of RX filters per NetQueue -1: use the default number of Rx filters based on availability; 0: Disable use of multiple RX filters; 1,2,3...: Will force the number of RX filters to use for NetQueue; Default: -1",
    param_int, -1, -1, 256, {-1, QFLE3_MOD_PARAM_VALID_VALS_TERM}, 0, 0},
   {"enable_lro", "Enable/Disable the TPA (LRO) feature. Set to 0 to disable TPA; Set to 1 to enable TPA(Default)",
    param_bool, 1, 0, 1, {0, 1,  QFLE3_MOD_PARAM_VALID_VALS_TERM}, 0, 0},
   {"rxring_bd_nr", "Number of RX BD Buffers: 4096(min) 16384(max), will round up to nearest power of two. Default: 4096",
    param_int, 4096, 4096, 16384, {4096, 8192, 16384, QFLE3_MOD_PARAM_VALID_VALS_TERM}, 0, 0},
   {"txring_bd_nr", "Number of TX BD Buffers: 4096(min) 16384(max), will round up to nearest power of two. Default: 4096",
    param_int, 4096, 4096, 16384, {4096, 8192, 16384, QFLE3_MOD_PARAM_VALID_VALS_TERM}, 0, 0},

};

const static struct {
   QFLE3_MGMT_CMD cmd;
   int (*fnptr) (qfle3_adapter * adapter, qfle3_mgmt_data_t * data);
} mgmt_fntbl[QFLE3_CMD_MAX] = {
   {
   QFLE3_CMD_NONE, NULL},     // should not be used
   {
   QFLE3_CMD_PCI_CFG_READ, qfle3_read_pci_cfg},     // Read and write PCI config registers
   {
   QFLE3_CMD_PCI_CFG_WRITE, qfle3_write_pci_cfg},   // sizes 1, 2, or 4 bytes
   {
   QFLE3_CMD_REG_READ, qfle3_read_reg},     // Read a register
   {
   QFLE3_CMD_REG_WRITE, qfle3_write_reg},   // Write to a register
   {
   QFLE3_CMD_TRCBUF_INFO, NULL}, //qfle3_get_trc_buff_info}, // get trace buffer INFO
   {
   QFLE3_CMD_TRCBUF_READ, NULL}, // qfle3_get_trc_buff},      // read a section of the trace buffer
   {
   QFLE3_CMD_MINIDUMP_SIZE, qfle3_get_minidump_size},       // get minidump size
   {
   QFLE3_CMD_MINIDUMP_READ, qfle3_get_minidump},    // read a section of the minidump
   {
   QFLE3_CMD_DIAG_TEST, qfle3_diag_test},   // do diagnostic test, test cases are in the payload.
   {
   QFLE3_CMD_DCBX_VALUES_SIZE, qfle3_get_dcbx_size},        // get the size(length) of the DCBX values
   {
   QFLE3_CMD_DCBX_VALUES_READ, qfle3_get_dcbx},     // read all of the DCBX values
   {
   QFLE3_CMD_MOD_PARAMS_INFO, qfle3_get_modparam_info},     // get the number of module parameters
   {
   QFLE3_CMD_MOD_PARAMS_READ, qfle3_get_modparam},  // read module parameters
   {
   QFLE3_CMD_WOL_GET, qfle3_wol_get},       // read WOL parameters
   {
   QFLE3_CMD_WOL_SET, qfle3_wol_set},       // change WOL parameters
   {
   QFLE3_PAUSE_GET, qfle3_pause_get},       // read pause parameters
   {
   QFLE3_PAUSE_SET, qfle3_pause_set},       // change pause parameters
   {
   QFLE3_COALESCE_GET, qfle3_coalesce_get}, // read coalesce parameters
   {
   QFLE3_COALESCE_SET, qfle3_coalesce_set}, // change coalesce parameters
   {
   QFLE3_PHYSICAL_ID, qfle3_physical_id},   // blink LED to physically identify port
   {
   QFLE3_OFFLOAD_GET, qfle3_offload_get},   // read offload parameters
   {
   QFLE3_OFFLOAD_SET, qfle3_offload_set},   // change offload parameters
   {
   QFLE3_RESTART_NEG, qfle3_restart_autoneg},       // restart negotiation
   {
   QFLE3_RING_GET, qfle3_ring_get}, // read ring parameters
   {
   QFLE3_RING_SET, qfle3_ring_set}, // change ring parameters
};

/* Selftest implementing functions for qedrntv_selftest_cmd_t */
//qfle3_selftest_fntbl_t qfle3_selftest_ecore[QFLE3_SELFTEST_MAX] = {
//   {
//   ecore_selftest_memory},    // QFLE3_SELFTEST_MEMORY
//   {
//   ecore_selftest_interrupt}, // QFLE3_SELFTEST_INTERRUPT
//   {
//   ecore_selftest_register},  // QFLE3_SELFTEST_REGISTER
//   {
//   ecore_selftest_clock},     // QFLE3_SELFTEST_CLOCK
//   {
//   ecore_selftest_nvram},     // QFLE3_SELFTEST_NVRAM
//};

void
qfle3_callback(vmk_MgmtCookies * cookie, vmk_MgmtEnvelope * envelope,
                 qfle3_mgmt_data_t * data)
{
   vmk_uint64 instance = envelope->instanceId;

   qfle3_adapter *adapter = NULL;
   QFLE3_MGMT_CMD cmd;
   adapter = (qfle3_adapter *) instance;

   if (adapter == NULL) {
      data->rv = QFLE3_CMD_RT_ERROR;
      return;
   }

   cmd = data->cmd;
   if ((cmd == 0) || (cmd >= QFLE3_CMD_MAX)) {
      data->rv = QFLE3_CMD_RT_ERROR;
      return;
   }

   if (mgmt_fntbl[cmd].fnptr != NULL) {
      QFLE3_DBG(QFLE3_DBG_MGMT,"qfle3_callback called for %d\n", cmd);
      (mgmt_fntbl[cmd].fnptr) (adapter, data);
      return;
   }
   data->rv = QFLE3_CMD_RT_ERROR;
   return;
}

void
qfle3_npar_callback(vmk_MgmtCookies * cookie, vmk_MgmtEnvelope * envelope,
                      qfle3_npar_sriov_cmd_t * npar_cmd,
                      qfle3_pf_data_t * data)
{
   vmk_uint64 instance = envelope->instanceId;

   qfle3_adapter *adapter = (qfle3_adapter *) instance;
//   int rc;

   if ((instance == 0) || (data == NULL))
      return;

   switch (npar_cmd->npar_sriov_cmd) {
   case QFLE3_PF_BW_GET:
      data->min_bw = (adapter->mf_config[QFLE3_VN(adapter)] & FUNC_MF_CFG_MIN_BW_MASK)
                     >> FUNC_MF_CFG_MIN_BW_SHIFT;
      data->max_bw = (adapter->mf_config[QFLE3_VN(adapter)] & FUNC_MF_CFG_MAX_BW_MASK)
                     >> FUNC_MF_CFG_MAX_BW_SHIFT;
      break;
   case QFLE3_PF_BW_SET:
         if (IS_MF_SD(adapter) || !IS_MF(adapter)) {
                QFLE3_ERR("Device is in SF/MF_SD mode, BW setting is not supported \n");
                return;
        }
        vmk_uint32 line_speed = adapter->link_vars.line_speed;

        /* use 10G if no link detected */
        if (!line_speed)
                line_speed = 10000;

        vmk_uint16 curr_speed = qfle3_get_mf_speed(adapter);

     /*   if (req_type == QFLE3_GET_NIC_SPEED && (bp->state == QFLE3_STATE_OPEN)
                        && bp->link_vars.link_up && !(bp->flags & MF_FUNC_DIS)) {
                if (!BP_NOMCP(bp))
                        req.cmd_req.config_speed.current_speed = curr_speed;
                else
                        req.cmd_req.config_speed.current_speed = bp->link_vars.line_speed;
                DP(NETIF_MSG_LINK, "GET_NIC_SPEED: curr_speed = %d\n", req.cmd_req.config_speed.current_speed);
                break;
        }*/

        if (IS_MF_SI(adapter)) {
                vmk_uint32 part = 0;
                vmk_uint32 new_speed = data->max_bw * 100;

                QFLE3_DBG(QFLE3_DBG_MGMT, "SET_SPEED: current_speed = %d, new_speed = %d, line_speed = %d\n", curr_speed, new_speed, line_speed);
                adapter->pending_max = 0;

                if (adapter->hw_info.bc_ver < REQ_BC_VER_4_SET_MF_BW) {
                        QFLE3_ERR("To set speed BC %X or higher is required, please upgrade BC\n",
                                       REQ_BC_VER_4_SET_MF_BW);
                        return;
                }

                part = (new_speed * 100) / line_speed;

                if (line_speed < new_speed || !part) {
                        QFLE3_ERR("Speed setting should be in a range from 1%% to 100%% of actual line speed\n");
                        return;
                }

                if(vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_QUIESCED)) {
                        /* store value for following "load" */
                        adapter->pending_max = part;
                }
                else
                        qfle3_update_max_mf_config(adapter, part);
                return;
        }

      break;
   default:
      QFLE3_ERR("NPAR Callback issued for invalid CMD %d\n",
             npar_cmd->npar_sriov_cmd);
      break;
   }

   return;
}
void
qfle3_sriov_callback(vmk_MgmtCookies * cookie, vmk_MgmtEnvelope * envelope,
                       qfle3_npar_sriov_cmd_t * sriov_cmd,
                       qfle3_sriov_vf_data_t * data)
{
   vmk_uint64 instance = envelope->instanceId;
   qfle3_adapter *adapter = (qfle3_adapter *) instance;
   vmk_int32 vf_id;
//   vmk_int32 j, i = 0;
//   u32 abs_vf_id;

   if ((instance == 0) || (data == NULL)) {
      return;
   }

   switch (sriov_cmd->npar_sriov_cmd) {
   case QFLE3_SRIOV_GET:
      QFLE3_DBG(QFLE3_DBG_MGMT, "SRIOV Callback issued for Get Information, Callback CMD %d\n",
              sriov_cmd->npar_sriov_cmd);
      
//         struct ecore_vf_info *vf_info;

         vf_id = (vmk_int32) data->vfid;
         struct qfle3_virtf *vf_info = QFLE3_VF(adapter, vf_id);
         if (vf_info == NULL) {
            data->vf_status = -1;
            QFLE3_DBG(QFLE3_DBG_MGMT,"SRIOV VF Not Initialized\n");
            break;
         }
         
         struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf_info->index];
         int Index = 0;
         int Bitindex = 0;
         vmk_uint64 Bit = 0;
         
         data->pfid = adapter->pf_id;
         data->min_bw = 0;
         data->max_bw = 0;
         data->max_tx_queues = vf_txq_count(vf_info);
         data->max_rx_queues = vf_rxq_count(vf_info);
//         data->max_filters_allowed = vf_info->nu
//         data->pvid = qfle3_hwfn->bc_info_buff_per_vf[abs_vf_id].pvlan;
          data->spoofing_check_enabled = VMK_FALSE;
         data->vf_status = vf_info->state;
         
         vmk_Bool started_block = VMK_FALSE;
         data->num_vlan_blocks = 0;
         int vlanid = 0;
         for (vlanid = 0; vlanid < 4096; vlanid++){
            Index = (vlanid >> 6);
            Bitindex = vlanid & 0x3F;
            Bit = ((vmk_uint64)1 << Bitindex);
            
            if (adapter_esx_vf->requested_gvlans[Index] & Bit){ // found a vlan
               if (!started_block) { // starting a new block
                  started_block = VMK_TRUE;
                  data->vlan_blocks[data->num_vlan_blocks].start_vlan_id = vlanid;
                  data->vlan_blocks[data->num_vlan_blocks].num_vlanid = 1;
                  data->num_vlan_blocks ++;
               } else { // continueing with an existing block
                  data->vlan_blocks[data->num_vlan_blocks-1].num_vlanid++;
               }

            }else {  // found a vlan not programmed
               if (started_block) { // this is the end of the block
                  started_block = VMK_FALSE;
                  // stop searching if we have 32 blocks already
                  if (data->num_vlan_blocks >= 32)
                     break;
               }// else do nothing
            }
         }

         break;
      break;
   case QFLE3_SRIOV_SET:
      QFLE3_DBG(QFLE3_DBG_MGMT,"SRIOV Callback issued for Set Information, Not yet Implemented\n");
      break;
   default:
      QFLE3_DBG(QFLE3_DBG_MGMT,"SRIOV Callback issued for invalid CMD %d\n",
              sriov_cmd->npar_sriov_cmd);
      break;
   }

   return;
}

void
qfle3_module_data(vmk_MgmtCookies * cookie, vmk_MgmtEnvelope * envelope,
                    qfle3_module_data_t * data)
{
   vmk_uint64 instance = envelope->instanceId;
   qfle3_adapter *adapter = (qfle3_adapter *) instance;

//              vmk_LogMessage("QFLE3_MODULE_DATA");

   if ((instance == 0) || (data == NULL))
      return;

   vmk_NameCopy(&data->drv_version, &adapter->uplinkSharedData.driverInfo.version);
   vmk_NameCopy(&data->drv_name, &adapter->uplinkSharedData.driverInfo.driver);
   qfle3_fill_fw_str(adapter, (char *) &data->fw_version, sizeof(data->fw_version));
   vmk_Snprintf(data->pci_bus_info.string, VMK_MISC_NAME_MAX, "%d:%d:%d:%d",
                adapter->pdev_addr.seg, adapter->pdev_addr.bus, adapter->pdev_addr.dev,
                adapter->pdev_addr.fn);
   data->nvm_size = adapter->hw_info.flash_size;
   data->vmkapi = VMKAPI_REVISION;
#if 0
   
   qfle3_mfw_command(adapter, 0x72000000, 0);
#endif
   return;
}

void
qfle3_instance_info(vmk_MgmtCookies * cookie, vmk_MgmtEnvelope * envelope,
                      qfle3_instance_info_t * data)
{
   vmk_uint64 instance = envelope->instanceId;
   qfle3_adapter *adapter = (qfle3_adapter *) instance;
   vmk_LinkStatus link_info;
   vmk_Memset(&link_info, 0, sizeof(link_info));
//              vmk_LogMessage("QFLE3_INSTANCE_INFO");

   if ((instance == 0) || (data == NULL))
      return;
  
   link_info = vmk_UplinkLinkStatusGet(adapter->uplink);
   
   vmk_Memcpy(&data->MAC, &adapter->hwMacAddr, sizeof (vmk_EthAddress));
   data->link_speed = adapter->link_vars.line_speed;
   data->duplexity = adapter->link_vars.duplex;
   data->link_status = link_info.state;
   data->phy_addr = adapter->pdev_addr.fn;
   data->MTU = adapter->mtu;
   data->supportedModesArraySize = adapter->supportedModesArraySz;
   vmk_CopyToUser((vmk_VA) data->supportedModesArray,
                  (vmk_VA) adapter->supportedModes,
                  sizeof(vmk_UplinkSupportedMode) *
                  adapter->supportedModesArraySz);
   data->autonegotiationSupported = VMK_TRUE;
   data->autonegotiation = VMK_TRUE;
   data->supportedCableTypes = CABLE_TYPE_FIBRE;
   data->port_type = CABLE_TYPE_FIBRE;
   return;
}


void
qfle3_statistics(vmk_MgmtCookies * cookie, vmk_MgmtEnvelope * envelope,
                   qfle3_statistics_t * data)
{

   vmk_uint64 instance = envelope->instanceId;
   qfle3_adapter *adapter = (qfle3_adapter *) instance;
   vmk_AddrCookie drv_data;
   vmk_UplinkStats stats;
//   int q_id, tc;

   if ((instance == 0) || (data == NULL))
      return;

   drv_data.addr = (vmk_uint64) adapter;
   if (qfle3_uplink_stats_get(drv_data, &stats) != VMK_OK) {
      vmk_Memset(data, 0, sizeof(vmk_UplinkStats));
   }
   data->rxPkts = stats.rxPkts;
   data->txPkts = stats.txPkts;
   data->rxBytes = stats.rxBytes;
   data->txBytes = stats.txBytes;
   data->rxErrors = stats.rxErrors;
   data->txErrors = stats.txErrors;
   data->rxDrops = stats.rxDrops;
   data->txDrops = stats.txDrops;
   data->rxMulticastPkts = stats.rxMulticastPkts;
   data->txMulticastPkts = stats.txMulticastPkts;
   data->rxBroadcastPkts = stats.rxBroadcastPkts;
   data->txBroadcastPkts = stats.txBroadcastPkts;
   data->collisions = stats.collisions;
   data->rxLengthErrors = stats.rxLengthErrors;
   data->rxOverflowErrors = stats.rxOverflowErrors;
   data->rxCRCErrors = stats.rxCRCErrors;
   data->rxFrameAlignErrors = stats.rxFrameAlignErrors;
   data->rxFifoErrors = stats.rxFifoErrors;
   data->rxMissErrors = stats.rxMissErrors;
   data->txAbortedErrors = stats.txAbortedErrors;
   data->txCarrierErrors = stats.txCarrierErrors;
   data->txFifoErrors = stats.txFifoErrors;
   data->txHeartbeatErrors = stats.txHeartbeatErrors;
   data->txWindowErrors = stats.txWindowErrors;
   data->num_fp_queues = QFLE3_TOTAL_ETH_QUEUES(adapter);
//   data->num_tc = edev->num_tc;

   /* Print queue specific rx statistics. */
//   for (q_id = 0; (q_id < adapter->num_ethqs) && (q_id < QFLE3_MAX_NO_QUEUES);
//        q_id++) {
//      struct qfle3_fastpath *fp = &adapter->fp[q_id];
//      qfle3_netq *rx_netq = &adapter->queues[fp->rxq->qid];
//      if (vmk_BitVectorTest(fp->state= QFLE3_)) {
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.q_state = 1;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.rx_packets =
//            fp->drv_stats.rx_pkts;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.rx_bytes =
//            fp->drv_stats.rx_bytes;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.rx_drops = fp->drv_stats.rx_Drops;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.drop_err = fp->drv_stats.rx_Errors;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.drop_fcs = fp->drv_stats.rx_FrameAlignErrors;/*FIX ME:Populate or remove */
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.rx_Buf_Alloc_fail = fp->drv_stats.r rx_netq->rx_stats.rx_Buf_Alloc_fail;     /*FIX ME:Populate or remove */
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.b_type_h = rx_netq->rx_stats.b_type_h;       /*FIX ME:Populate or remove */
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.b_type_b = rx_netq->rx_stats.b_type_b;       /*FIX ME:Populate or remove */
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.rx_hw_errors =
//            rx_netq->rx_stats.rx_hw_errors;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.rx_alloc_errors =
//            rx_netq->rx_stats.rx_alloc_errors;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.netpoll_rx_errors =
//            rx_netq->rx_stats.netpoll_rx_errors;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.cso_pkt =
//            rx_netq->rx_stats.cso_pkt;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.lro_pkts =
//            rx_netq->rx_stats.lro_pkts;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.lro_bytes =
//            rx_netq->rx_stats.lro_bytes;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.encap_stats.
//            rx_bytes = rx_netq->rx_stats.encap_stats.rx_bytes;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.encap_stats.rx_pkts =
//            rx_netq->rx_stats.encap_stats.rx_pkts;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.encap_stats.
//            lro_bytes = rx_netq->rx_stats.encap_stats.lro_bytes;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.encap_stats.
//            lro_pkts = rx_netq->rx_stats.encap_stats.lro_pkts;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.encap_stats.
//            outer_l4_csummed = rx_netq->rx_stats.encap_stats.outer_l4_csummed;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.encap_stats.
//            outer_l3_csummed = rx_netq->rx_stats.encap_stats.outer_l3_csummed;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.encap_stats.
//            inner_l4_csummed = rx_netq->rx_stats.encap_stats.inner_l4_csummed;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.encap_stats.
//            inner_l3_csummed = rx_netq->rx_stats.encap_stats.inner_l3_csummed;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.encap_stats.
//            outer_l4_csum_err = rx_netq->rx_stats.encap_stats.outer_l4_csum_err;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.encap_stats.
//            outer_l3_csum_err = rx_netq->rx_stats.encap_stats.outer_l3_csum_err;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.encap_stats.
//            inner_l4_csum_err = rx_netq->rx_stats.encap_stats.inner_l4_csum_err;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.encap_stats.
//            inner_l3_csum_err = rx_netq->rx_stats.encap_stats.inner_l3_csum_err;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.null_pkt =
//            rx_netq->rx_stats.null_pkt;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.rx_frame_len_errors =
//            rx_netq->rx_stats.rx_frame_len_errors;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.rx_offset_errors =
//            rx_netq->rx_stats.rx_offset_errors;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.
//            rx_dev_not_rdy_errors = rx_netq->rx_stats.rx_dev_not_rdy_errors;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.cqe_type_slow_path =
//            rx_netq->rx_stats.cqe_type_slow_path;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.cqe_type_tpa_start =
//            rx_netq->rx_stats.cqe_type_tpa_start;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.cqe_type_tpa_cont =
//            rx_netq->rx_stats.cqe_type_tpa_cont;
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.rx_vlan_pkt =
//            rx_netq->rx_stats.rx_vlan_pkts;
//      } else {
//         data->qfle3_q_stats_t[q_id].qfle3_rxq_stats_t.q_state = 0;
//      }

//      /* Print queue specific tx statistics. */
//      for (tc = 0; (tc < edev->num_tc) && (tc < QFLE3_MAX_NO_TC); tc++) {
//         qfle3_netq *tx_queue = fp->txq[tc]->q_data;
//         if (qfle3_state_test
//             (fp->txq[tc]->state_bv, QFLE3_Q_STATE_ALLOCATED)) {
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].q_state = 1;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].tx_packets =
//               tx_queue->tx_stats.tx_packets;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].tx_bytes =
//               tx_queue->tx_stats.tx_bytes;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].tx_drops =
//               tx_queue->tx_stats.tx_drops;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].tx_compl =
//               tx_queue->tx_stats.tx_compl;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].
//               frag_get_error = tx_queue->tx_stats.frag_get_error;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].
//               frag_cpy_error = tx_queue->tx_stats.frag_cpy_error;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].
//               too_many_frags = tx_queue->tx_stats.too_many_frags;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].ring_error =
//               tx_queue->tx_stats.ring_error;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].
//               sg_elem_error = tx_queue->tx_stats.sg_elem_error;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].dma_error =
//               tx_queue->tx_stats.dma_error;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].drop_bad_q =
//               tx_queue->tx_stats.drop_bad_q;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].q_stopped =
//               tx_queue->tx_stats.q_stopped;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].ring_full =
//               tx_queue->tx_stats.ring_full;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].cso_pkt =
//               tx_queue->tx_stats.cso_pkt;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].cso_bytes =
//               tx_queue->tx_stats.cso_bytes;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].tso_pkts =
//               tx_queue->tx_stats.tso_pkts;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].tso_bytes =
//               tx_queue->tx_stats.tso_bytes;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].
//               drop_hdr_inspect_err = tx_queue->tx_stats.drop_hdr_inspect_err;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               xmitencapcalled = tx_queue->tx_stats.encap_stats.xmitencapcalled;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               tso_bytes = tx_queue->tx_stats.encap_stats.tso_bytes;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               tso_pkts = tx_queue->tx_stats.encap_stats.tso_pkts;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               encap_tso = tx_queue->tx_stats.encap_stats.encap_tso;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               inr_ipv4 = tx_queue->tx_stats.encap_stats.inr_ipv4;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               inr_ipv6 = tx_queue->tx_stats.encap_stats.inr_ipv6;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               outer_ipv4 = tx_queue->tx_stats.encap_stats.outer_ipv4;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               outer_ipv6 = tx_queue->tx_stats.encap_stats.outer_ipv6;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               inr_tcp = tx_queue->tx_stats.encap_stats.inr_tcp;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               inr_udp = tx_queue->tx_stats.encap_stats.inr_udp;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               outer_udp = tx_queue->tx_stats.encap_stats.outer_udp;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               outer_tcp = tx_queue->tx_stats.encap_stats.outer_tcp;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               drop_malformed = tx_queue->tx_stats.encap_stats.drop_malformed;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].encap_stats.
//               no_offload = tx_queue->tx_stats.encap_stats.no_offload;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].
//               unsupported_proto = tx_queue->tx_stats.unsupported_proto;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].no_csum =
//               tx_queue->tx_stats.no_csum;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].IPv4_csum =
//               tx_queue->tx_stats.IPv4_csum;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].IPv4_tso_pkt =
//               tx_queue->tx_stats.IPv4_tso_pkt;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].IPv6_csum =
//               tx_queue->tx_stats.IPv6_csum;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].IPv6_tso_pkt =
//               tx_queue->tx_stats.IPv6_tso_pkt;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].tcp_pkt =
//               tx_queue->tx_stats.tcp_pkt;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].udp_pkt =
//               tx_queue->tx_stats.udp_pkt;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].
//               pkt_larger_than_mtu = tx_queue->tx_stats.pkt_larger_than_mtu;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].
//               vlan_insertion = tx_queue->tx_stats.vlan_insertion;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].tso_error =
//               tx_queue->tx_stats.tso_error;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].
//               dma_map_error = tx_queue->tx_stats.dma_map_error;
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].
//               map_length_gt_hlength = tx_queue->tx_stats.map_length_gt_hlength;
//         } else {
//            data->qfle3_q_stats_t[q_id].qfle3_txq_stats_t[tc].q_state = 0;
//         }
//      }
//   }
   return;
}

int
qfle3_read_pci_cfg(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{

   vmk_uint32 status;
   vmk_uint32 reg_value32;
   vmk_uint32 offset;

   QFLE3_DBG(QFLE3_DBG_MGMT, "QFLE3_CMD_PCI_CFG_READ");
   if (adapter == NULL) {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 1;
   }

   offset = data->off;
   switch (data->size) {
   case 1:
      status = vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev,
                                 VMK_PCI_CONFIG_ACCESS_8,
                                 offset, &reg_value32);
      *(vmk_uint32 *) & (data->u[0]) = reg_value32;
      break;
   case 2:
      status = vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev,
                                 VMK_PCI_CONFIG_ACCESS_16, 
                                 offset, &reg_value32);
      *(vmk_uint32 *) & (data->u[0]) = reg_value32;
      break;
   case 4:
      status = vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev,
                                 VMK_PCI_CONFIG_ACCESS_32, 
                                 offset, &reg_value32);
      *(vmk_uint32 *) & (data->u[0]) = reg_value32;
      break;
   default:
      data->rv = QFLE3_CMD_RT_INVALID_PARAM;
      return 1;
   }
   if (status != 0)
      data->rv = QFLE3_CMD_RT_ERROR;
   else {
      data->rv = QFLE3_CMD_RT_SUCCESS;
   }
   return status;
}

int
qfle3_write_pci_cfg(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{

   vmk_uint32 status;
   vmk_uint32 reg_value32;
   vmk_uint32 offset;
   QFLE3_DBG(QFLE3_DBG_MGMT, "QFLE3_CMD_PCI_CFG_WRITE");

   if (adapter == NULL) {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 1;
   }

   offset = data->off;
   switch (data->size) {
   case 1:
      reg_value32 = *(vmk_int8 *) & data->u;
      status =
         vmk_PCIWriteConfig(vmk_ModuleCurrentID, adapter->pdev,
                            VMK_PCI_CONFIG_ACCESS_8, data->off, reg_value32);
      QFLE3_DBG(QFLE3_DBG_MGMT, "write %d bytes at 0x%lx, value 0x%x\n", data->size, data->off, reg_value32);
      break;
   case 2:
      reg_value32 = *(vmk_int16 *) & data->u;
      if ((data->off & 0x1) != 0) {
         data->rv = QFLE3_CMD_RT_INVALID_PARAM;
         return 1;
      }
      status =
         vmk_PCIWriteConfig(vmk_ModuleCurrentID, adapter->pdev,
                            VMK_PCI_CONFIG_ACCESS_16, data->off, reg_value32);
      QFLE3_DBG(QFLE3_DBG_MGMT, "write %d bytes at 0x%lx, value 0x%x\n", data->size, data->off, reg_value32);
      break;
   case 4:
      reg_value32 = *(vmk_int32 *) & data->u;
      if ((data->off & 0x3) != 0) {
         data->rv = QFLE3_CMD_RT_INVALID_PARAM;
         return 1;
      }
      status =
         vmk_PCIWriteConfig(vmk_ModuleCurrentID, adapter->pdev,
                            VMK_PCI_CONFIG_ACCESS_32, data->off, reg_value32);
      QFLE3_DBG(QFLE3_DBG_MGMT, "write %d bytes at 0x%lx, value 0x%x\n", data->size, data->off, reg_value32);
      break;
   default:
      data->rv = QFLE3_CMD_RT_INVALID_PARAM;
      return 1;
   }
   if (status != VMK_OK)
      data->rv = QFLE3_CMD_RT_ERROR;
   else {
      data->rv = QFLE3_CMD_RT_SUCCESS;
   }
   return status;
}

int
qfle3_read_reg(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{

   vmk_uint32 status = 0;
   vmk_uint32 reg_value32;

   QFLE3_DBG(QFLE3_DBG_MGMT, "QFLE3_CMD_REG_READ");
   if (adapter == NULL) {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 1;
   }

   reg_value32 = REG_RD(adapter, data->off);
   vmk_Memcpy((void *)&(data->u), &reg_value32, sizeof(vmk_uint32));
   data->size = 4;
   data->rv = QFLE3_CMD_RT_SUCCESS;
   return status;
}

int
qfle3_write_reg(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{

   vmk_uint32 status = 0;
   vmk_uint32 reg_value32;

   QFLE3_DBG(QFLE3_DBG_MGMT, "QFLE3_CMD_REG_WRITE");
   if (adapter == NULL) {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 1;
   }

   reg_value32 = *(vmk_int32 *) & data->u;
   REG_WR(adapter, data->off, reg_value32);

   data->rv = QFLE3_CMD_RT_SUCCESS;
   return status;
}

int
qfle3_get_trc_buff_info(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{

//   qfle3_trc_buff_info_t info;
//              vmk_LogMessage("QFLE3_CMD_TRCBUF_SIZE");
//   if (adapter == NULL) {
//      data->rv = QFLE3_CMD_RT_ERROR;
//      return 1;
//   }

//   info.lineSize = sizeof(struct trace_s);
//   info.lineNum = 4096;
//   info.startLine = 0;
//   info.endLine = 4095;
//   data->size = sizeof(qfle3_trc_buff_info_t);
//              vmk_LogMessage("size = %d\n", data->size);
//   // TODO: add trace buffer to data structure
//   vmk_Memcpy((void *)data->u, &info, data->size);
//   data->rv = QFLE3_CMD_RT_ERROR;
   return 0;
}

int
qfle3_get_trc_buff(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
//   vmk_uint32 size;
//   vmk_uint32 offset;
//   int i;

//   struct trace_s t;

//   if (adapter == NULL) {
//      data->rv = QFLE3_CMD_RT_ERROR;
//      return 1;
//   }
//   t.fn_ptr = (vmk_uintptr_t) qfle3_get_trc_buff;
//   t.ts = vmk_GetTimerCycles();
//   t.cpu = 0;
//   t.line = 0;
//   t.val = 0xbadbadba;

//   // TODO: add trace buffer to data structure
//              vmk_LogMessage("QFLE3_CMD_TRCBUF");
//   size = data->size;
//   offset = data->off;

//   if (offset >= 4096) {
//      data->rv = QFLE3_CMD_RT_INVALID_PARAM;
//      return 0;
//   }

//   if ((offset + size) > 4096) {
//      size = 4096 - offset;
//   }

//   for (i = 0; i < size; i++) {
//      t.line = i + offset;
//      vmk_CopyToUser((data->ptr + i * sizeof(struct trace_s)),
//                     (vmk_uintptr_t) & t, sizeof(struct trace_s));
//   }
//   data->size = size;
//   data->rv = QFLE3_CMD_RT_ERROR;
   return 0;
}

int
qfle3_get_minidump_size(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   vmk_uint32 size;

   if (adapter == NULL) {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 1;
   }
   size = DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0 * 2;// for 2 ports

   data->size = 4;
   data->hwfn = 0;

   vmk_Memcpy((void *)data->u, &size, 4);
   data->rv = QFLE3_CMD_RT_SUCCESS;
   return 0;
}

int
qfle3_get_minidump(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   struct chip_core_dmp *dmp;
   vmk_ListLinks *link = NULL;
   vmk_uint32 size = 0;
   vmk_int32 times = 0;
   
   vmk_VA dest = data->ptr;

   if (adapter == NULL) {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 1;
   }

   QFLE3_DBG(QFLE3_DBG_MGMT, "QFLE3_CMD_MINIDUMP_READ, size = %d", data->size);
   
   VMK_LIST_FORALL(&qfle3_mod_info.adapterList, link){
      if (times == 2)
         break;
      
      adapter = VMK_LIST_ENTRY(link, struct qfle3_adapter, adapterLink);
      if (adapter->fwdmp_buf) {
         data->u[0] = 1;  // indicate the dump was previously captured.
      } else {
         adapter->fwdmp_buf = qfle3_heap_alloc(DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0);
         if (adapter->fwdmp_buf == NULL){
            QFLE3_ERR("Failed to allocate heap for grcdump.");
            goto error;
         }
      }
      dmp = (struct chip_core_dmp *)adapter->fwdmp_buf;
      qfle3_get_grcdump(adapter, adapter->fwdmp_buf, VMK_TRUE);

      QFLE3_INFO("Writing %u bytes to dump device. Instance %i\n", dmp->fw_hdr.dmp_size, ++times);
      vmk_CopyToUser(dest, (vmk_uintptr_t) dmp, dmp->fw_hdr.dmp_size);
      dest += dmp->fw_hdr.dmp_size;
      size += dmp->fw_hdr.dmp_size;
      qfle3_heap_free(adapter->fwdmp_buf);
      
   }
   QFLE3_INFO("Writing %u bytes size. \n", size);
   data->size = size;
   data->rv = QFLE3_CMD_RT_SUCCESS;
   return 0;
error:
   
   QFLE3_INFO("Writing nothing to size. \n");
   data->rv = QFLE3_CMD_RT_ERROR;
   return 1;
}

int
qfle3_lb_test(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{

   if (adapter == NULL) {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 1;
   }
   // do loopback test
   data->rv = QFLE3_CMD_RT_SUCCESS;
   return 0;
}

#if 0
int
qfle3_diag_test(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)

void qfle3_self_test(struct net_device *dev,
			    struct ethtool_test *etest, u64 *buf)
{
	struct qfle3_adapter *bp = netdev_priv(dev);
	u8 is_serdes, link_up;
	int rc, cnt = 0;

	if (pci_num_vf(bp->pdev)) {
		DP(QFLE3_MSG_IOV,
		   "VFs are enabled, can not perform self test\n");
		return;
	}

	if (bp->recovery_state != QFLE3_RECOVERY_DONE) {
		netdev_err(bp->dev,
			   "Handling parity error recovery. Try again later\n");
		etest->flags |= ETH_TEST_FL_FAILED;
		return;
	}

	DP(QFLE3_MSG_ETHTOOL,
	   "Self-test command parameters: offline = %d, external_lb = %d\n",
	   (etest->flags & ETH_TEST_FL_OFFLINE),
	   (etest->flags & ETH_TEST_FL_EXTERNAL_LB)>>2);

	memset(buf, 0, sizeof(u64) * QFLE3_NUM_TESTS(bp));

	if (qfle3_test_nvram(bp) != 0) {
		if (!IS_MF(bp))
			buf[4] = 1;
		else
			buf[0] = 1;
		etest->flags |= ETH_TEST_FL_FAILED;
	}

#ifdef __VMKLNX__ /* ! QFLE3_UPSTREAM */
	if (bp->state == QFLE3_STATE_ERROR) {
		QFLE3_ERR("Cannot perform self tests in current HW state: 0x%x. Try again later.\n",
			  bp->state);
		return;
	}
#endif
	if (!netif_running(dev)) {
		DP(QFLE3_MSG_ETHTOOL, "Interface is down\n");
		return;
	}

	is_serdes = (bp->link_vars.link_status & LINK_STATUS_SERDES_LINK) > 0;
	link_up = bp->link_vars.link_up;
	/* offline tests are not supported in MF mode */
	if ((etest->flags & ETH_TEST_FL_OFFLINE) && !IS_MF(bp)) {
		int port = BP_PORT(bp);
		u32 val;

		/* save current value of input enable for TX port IF */
		val = REG_RD(bp, NIG_REG_EGRESS_UMP0_IN_EN + port*4);
		/* disable input for TX port IF */
		REG_WR(bp, NIG_REG_EGRESS_UMP0_IN_EN + port*4, 0);

		qfle3_nic_unload(bp, UNLOAD_NORMAL, false);
		rc = qfle3_nic_load(bp, LOAD_DIAG);
		if (rc) {
			etest->flags |= ETH_TEST_FL_FAILED;
			DP(QFLE3_MSG_ETHTOOL,
			   "Can't perform self-test, nic_load (for offline) failed\n");
			return;
		}

		/* wait until link state is restored */
		qfle3_wait_for_link(bp, 1, is_serdes);

		if (qfle3_test_registers(bp) != 0) {
			buf[0] = 1;
			etest->flags |= ETH_TEST_FL_FAILED;
		}
		if (qfle3_test_memory(bp) != 0) {
			buf[1] = 1;
			etest->flags |= ETH_TEST_FL_FAILED;
		}

		buf[2] = qfle3_test_loopback(bp); /* internal LB */
		if (buf[2] != 0)
			etest->flags |= ETH_TEST_FL_FAILED;

		if (etest->flags & ETH_TEST_FL_EXTERNAL_LB) {
			buf[3] = qfle3_test_ext_loopback(bp); /* external LB */
			if (buf[3] != 0)
				etest->flags |= ETH_TEST_FL_FAILED;
			etest->flags |= ETH_TEST_FL_EXTERNAL_LB_DONE;
		}

		qfle3_nic_unload(bp, UNLOAD_NORMAL, false);

		/* restore input for TX port IF */
		REG_WR(bp, NIG_REG_EGRESS_UMP0_IN_EN + port*4, val);
		rc = qfle3_nic_load(bp, LOAD_NORMAL);
		if (rc) {
			etest->flags |= ETH_TEST_FL_FAILED;
			DP(QFLE3_MSG_ETHTOOL,
			   "Can't perform self-test, nic_load (for online) failed\n");
			return;
		}
		/* wait until link state is restored */
		qfle3_wait_for_link(bp, link_up, is_serdes);
	}

	if (qfle3_test_intr(bp) != 0) {
		if (!IS_MF(bp))
			buf[5] = 1;
		else
			buf[1] = 1;
		etest->flags |= ETH_TEST_FL_FAILED;
	}

	if (link_up) {
		cnt = 100;
		while (qfle3_link_test(bp, is_serdes) && --cnt)
			msleep(20);
	}

	if (!cnt) {
		if (!IS_MF(bp))
			buf[6] = 1;
		else
			buf[2] = 1;
		etest->flags |= ETH_TEST_FL_FAILED;
	}
#ifndef QFLE3_UPSTREAM /* ! _UPSTREAM */
	/* run the idle check twice */
	qfle3_idle_chk(bp);
	rc = qfle3_idle_chk(bp);
	if (!IS_MF(bp))
		buf[7] = rc;
	else
		buf[3] = rc;
	if (rc)
		etest->flags |= ETH_TEST_FL_FAILED;
#endif
}
#endif
int
qfle3_diag_test(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   qfle3_selftest_cmd_t selftest_cmd;
   VMK_ReturnStatus status;

   if (adapter == NULL) {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 1;
   }

   selftest_cmd = *(vmk_int32 *)&data->u[0];
   
   if ((selftest_cmd < 0) || (selftest_cmd >= QFLE3_SELFTEST_MAX)) {
      data->rv = QFLE3_CMD_RT_INVALID_PARAM;
      return 1;
   }

   if (qfle3_get_selftest(selftest_cmd) != NULL)
      status = qfle3_get_selftest(selftest_cmd)->fn_ptr(adapter);
   else
      status = VMK_FAILURE;

   if (status != VMK_OK)
      data->rv = QFLE3_CMD_RT_ERROR;
   else
      data->rv = QFLE3_CMD_RT_SUCCESS;

   return 0;
}

int
qfle3_get_dcbx_size(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   vmk_uint32 size;

   QFLE3_DBG(QFLE3_DBG_MGMT,"QFLE3_CMD_DCBX_VALUES_SIZE");

   if (adapter == NULL) {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 1;
   }

   size = sizeof(struct qfle3_dcbx_params_get);
   data->size = sizeof(vmk_uint32);
   vmk_CopyToUser((vmk_VA)data->u, (vmk_VA)&size, data->size);
   data->rv = QFLE3_CMD_RT_SUCCESS;
   return 0;
}

#define LLDP_ADMIN_MIB_OFFSET(adpater)   (PORT_MAX*sizeof(struct lldp_params) + \
                      QFLE3_PORT(adapter)*sizeof(struct lldp_admin_mib))

static void qfle3_dcbx_get_bw_percentage_tbl(qfle3_adapter * adapter,
                u32 * set_configuration_bw,
                u32 * pg_bw_tbl)
{
    int i;

    for(i = 0 ;i < DCBX_MAX_NUM_PG_BW_ENTRIES ; i++)
        set_configuration_bw[i] = DCBX_PG_BW_GET(pg_bw_tbl, i);
}

/* assumes that app_array is not smaller than num_entries */
static void qfle3_dcbx_get_priority_app_table(qfle3_adapter * adapter,
        struct qfle3_admin_priority_app_table *set_priority_app,
        int num_entries,
        struct dcbx_app_priority_entry *app_array)
{
    int i;

    for (i = 0; i < num_entries; i++) {
        set_priority_app[i].valid =
           GET_FLAGS(app_array[i].appBitfield, DCBX_APP_ENTRY_VALID) ?
                       VMK_TRUE : VMK_FALSE;

        if(GET_FLAGS(app_array[i].appBitfield,DCBX_APP_SF_ETH_TYPE))
            set_priority_app[i].traffic_type = TRAFFIC_TYPE_ETH;
        else
            set_priority_app[i].traffic_type = TRAFFIC_TYPE_PORT;

        set_priority_app[i].priority = app_array[i].pri_bitmap;

        set_priority_app[i].app_id = app_array[i].app_id;
    }
}

int
qfle3_get_dcbx(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   struct qfle3_dcbx_params_get *dcbx_params;
   struct lldp_admin_mib       admin_mib;
   struct lldp_dcbx_stat       lldp_dcbx_stat;
   u32 offset = 0, stat_offset, dcbx_err_state = 0, rc = VMK_OK;
   int mfw_configured = SHMEM2_HAS(adapter, drv_flags) &&
                 GET_FLAGS(SHMEM2_RD(adapter, drv_flags),
                       1 << DRV_FLAGS_DCB_MFW_CONFIGURED);

   QFLE3_DBG(QFLE3_DBG_MGMT,"QFLE3_CMD_DCBX_READ, size = %d, ptr %ld, u %x, cmd %d \n",
                   data->size, data->ptr, data->u[0], data->cmd);

   if (adapter == NULL) {
     data->rv = QFLE3_CMD_RT_ERROR;
     return VMK_FAILURE;
   }

   dcbx_params = vmk_HeapAlloc(qfle3_mod_info.heapID, sizeof(struct qfle3_dcbx_params_get));
   vmk_Memset(dcbx_params, 0, sizeof(struct qfle3_dcbx_params_get));

    stat_offset = SHMEM2_RD(adapter, dcbx_lldp_dcbx_stat_offset);
    offset = SHMEM2_RD(adapter,dcbx_lldp_params_offset);

    if (offset == SHMEM_LLDP_DCBX_PARAMS_NONE ||
        stat_offset == SHMEM_LLDP_DCBX_STAT_NONE) {
        QFLE3_DBG(QFLE3_DBG_MGMT, "DCBX not supported by BC\n");
        data->rv = QFLE3_CMD_RT_ERROR;
        rc = VMK_FAILURE;
        goto out;
    }

    dcbx_params->ver_num = adapter->dcbx_sb.dcb_version;
    dcbx_params->dcbx_enabled = adapter->dcbx_sb.state;
    dcbx_params->dcb_state = adapter->dcbx_sb.enabled;

    offset += LLDP_ADMIN_MIB_OFFSET(adapter);
    /* Read the data first */
    qfle3_read_data(adapter, (u32 *)&admin_mib, offset,
            sizeof(struct lldp_admin_mib));

    dcbx_params->config_dcbx_params.admin_dcbx_version =
            (admin_mib.ver_cfg_flags & DCBX_CEE_VERSION_MASK)
                        >> DCBX_CEE_VERSION_SHIFT;

    dcbx_params->config_dcbx_params.admin_ets_enable =
                    admin_mib.features.ets.enabled;

    dcbx_params->config_dcbx_params.admin_pfc_enable =
                    admin_mib.features.pfc.enabled;

    /* FOR IEEE adapter->dcbx_config_params.admin_tc_supported_tx_enable */
    dcbx_params->config_dcbx_params.admin_ets_configuration_tx_enable =
        GET_FLAGS(admin_mib.ver_cfg_flags, DCBX_ETS_CONFIG_TX_ENABLED) ?
        VMK_TRUE : VMK_FALSE;

    /* For IEEE admin_ets_recommendation_tx_enable */

    dcbx_params->config_dcbx_params.admin_pfc_tx_enable =
        GET_FLAGS(admin_mib.ver_cfg_flags, DCBX_PFC_CONFIG_TX_ENABLED) ?
        VMK_TRUE : VMK_FALSE;

    dcbx_params->config_dcbx_params.admin_application_priority_tx_enable =
        GET_FLAGS(admin_mib.ver_cfg_flags, DCBX_APP_CONFIG_TX_ENABLED) ?
        VMK_TRUE : VMK_FALSE;

    dcbx_params->config_dcbx_params.admin_ets_willing =
        GET_FLAGS(admin_mib.ver_cfg_flags,DCBX_ETS_WILLING) ?
        VMK_TRUE : VMK_FALSE;

    /* For IEEE admin_ets_reco_valid */
    dcbx_params->config_dcbx_params.admin_pfc_willing =
        GET_FLAGS(admin_mib.ver_cfg_flags,DCBX_PFC_WILLING) ?
        VMK_TRUE : VMK_FALSE;

    dcbx_params->config_dcbx_params.admin_app_priority_willing =
        GET_FLAGS(admin_mib.ver_cfg_flags,DCBX_APP_WILLING) ?
        VMK_TRUE : VMK_FALSE;

    qfle3_dcbx_get_ets_pri_pg_tbl(adapter,
       dcbx_params->config_dcbx_params.admin_configuration_ets_pg,
       admin_mib.features.ets.pri_pg_tbl);

    qfle3_dcbx_get_bw_percentage_tbl(adapter,
       dcbx_params->config_dcbx_params.admin_configuration_bw_percentage,
       admin_mib.features.ets.pg_bw_tbl);

   /* For IEEE admin_recommendation_bw_percentage
     * For IEEE admin_recommendation_ets_pg */
    dcbx_params->config_dcbx_params.admin_pfc_bitmap =
            admin_mib.features.pfc.pri_en_bitmap;

    qfle3_dcbx_get_priority_app_table(adapter,
       dcbx_params->config_dcbx_params.admin_priority_app_table,
       DCBX_CONFIG_MAX_APP_PROTOCOL,
       admin_mib.features.app.app_pri_tbl);

    dcbx_params->config_dcbx_params.admin_default_priority =
            admin_mib.features.app.default_pri;

   /* Get negotiation results MIB data */
    offset = SHMEM2_RD(adapter, dcbx_neg_res_offset);

    if (SHMEM_DCBX_NEG_RES_NONE != offset) {
        struct lldp_local_mib local_mib;

        if(qfle3_dcbx_read_mib(adapter,
                       (u32 *)&local_mib,
                       offset,
                       DCBX_READ_LOCAL_MIB)) {
            data->rv = QFLE3_CMD_RT_ERROR;
            rc = VMK_FAILURE;
            goto out;
        }

        dcbx_params->local_tc_supported =
            local_mib.features.app.tc_supported;
        dcbx_params->local_pfc_caps    =
            local_mib.features.pfc.pfc_caps;
        dcbx_params->local_ets_enable  =
            local_mib.features.ets.enabled;
        dcbx_params->local_pfc_enable  =
            local_mib.features.pfc.enabled;

        qfle3_dcbx_get_bw_percentage_tbl(adapter,
            dcbx_params->local_configuration_bw_percentage,
            local_mib.features.ets.pg_bw_tbl);

        qfle3_dcbx_get_ets_pri_pg_tbl(adapter,
            dcbx_params->local_configuration_ets_pg,
            local_mib.features.ets.pri_pg_tbl);

        dcbx_params->local_pfc_bitmap =
            local_mib.features.pfc.pri_en_bitmap;

        qfle3_dcbx_get_priority_app_table(adapter,
            dcbx_params->local_priority_app_table,
            DCBX_MAX_APP_PROTOCOL,
            local_mib.features.app.app_pri_tbl);

        dcbx_params->pfc_mismatch =
            GET_FLAGS(local_mib.error, DCBX_LOCAL_PFC_MISMATCH) ?
            VMK_TRUE : VMK_FALSE;
        dcbx_params->priority_app_mismatch =
            GET_FLAGS(local_mib.error, DCBX_LOCAL_APP_MISMATCH) ?
            VMK_TRUE : VMK_FALSE;

        if ((GET_FLAGS(local_mib.error, DCBX_REMOTE_MIB_ERROR)) &&
            !mfw_configured)
            dcbx_err_state |= 1;
    }

    offset = SHMEM2_RD(adapter, dcbx_remote_mib_offset);
    if (SHMEM_DCBX_NEG_RES_NONE != offset) {
        struct lldp_remote_mib remote_mib;

        if (qfle3_dcbx_read_mib(adapter,
                    (u32*)&remote_mib,
                    offset,
                    DCBX_READ_REMOTE_MIB)){
            data->rv = QFLE3_CMD_RT_ERROR;
            rc = VMK_FAILURE;
            goto out;
        }

        dcbx_params->remote_tc_supported =
            remote_mib.features.app.tc_supported;
        dcbx_params->remote_pfc_cap =
            remote_mib.features.pfc.pfc_caps;

        dcbx_params->remote_ets_reco_valid =
            GET_FLAGS(remote_mib.flags,DCBX_REMOTE_ETS_RECO_VALID) ?
            VMK_TRUE : VMK_FALSE;

        dcbx_params->remote_ets_willing =
            GET_FLAGS(remote_mib.flags,DCBX_ETS_REM_WILLING) ?
            VMK_TRUE : VMK_FALSE;

        dcbx_params->remote_pfc_willing =
            GET_FLAGS(remote_mib.flags,DCBX_PFC_REM_WILLING) ?
            VMK_TRUE : VMK_FALSE;

        dcbx_params->remote_app_priority_willing =
            GET_FLAGS(remote_mib.flags,DCBX_APP_REM_WILLING) ?
            VMK_TRUE : VMK_FALSE;

        qfle3_dcbx_get_bw_percentage_tbl(adapter,
                dcbx_params->remote_configuration_bw_percentage,
                remote_mib.features.ets.pg_bw_tbl);

        qfle3_dcbx_get_ets_pri_pg_tbl(adapter,
                dcbx_params->remote_configuration_ets_pg,
                remote_mib.features.ets.pri_pg_tbl);

        dcbx_params->remote_pfc_bitmap =
            remote_mib.features.pfc.pri_en_bitmap;

        qfle3_dcbx_get_priority_app_table(adapter,
                dcbx_params->remote_priority_app_table,
                DCBX_MAX_APP_PROTOCOL,
                remote_mib.features.app.app_pri_tbl);
    }

    /* handle the stats */
    qfle3_read_data(adapter, (u32*)&lldp_dcbx_stat, stat_offset,
             sizeof(struct lldp_dcbx_stat));

    dcbx_params->dcbx_frames_sent = lldp_dcbx_stat.num_tx_dcbx_pkts;
    dcbx_params->dcbx_frames_received = lldp_dcbx_stat.num_rx_dcbx_pkts;

    qfle3_acquire_phy_lock(adapter);
    elink_pfc_statistic(&adapter->link_params, &adapter->link_vars,
                dcbx_params->pfc_frames_sent,
                dcbx_params->pfc_frames_received);
    qfle3_release_phy_lock(adapter);

    if (!(SHMEM2_HAS(adapter, drv_flags) &&
          GET_FLAGS(SHMEM2_RD(adapter, drv_flags),
            1 << DRV_FLAGS_DCB_CONFIGURED)))
        dcbx_err_state |= 1;

    if (!dcbx_err_state) {
        dcbx_params->dcb_current_state |=
                QFLE3_DCBX_CURRENT_STATE_IS_SYNC;
        if (adapter->dcbx_sb.port_params.app.enabled)
            dcbx_params->dcb_current_state |=
                QFLE3_PRIORITY_TAGGING_IS_CURRENTLY_OPERATIONAL;
        if (adapter->dcbx_sb.port_params.pfc.enabled)
            dcbx_params->dcb_current_state |=
                QFLE3_PFC_IS_CURRENTLY_OPERATIONAL;
        if (adapter->dcbx_sb.port_params.ets.enabled)
            dcbx_params->dcb_current_state |=
                QFLE3_ETS_IS_CURRENTLY_OPERATIONAL;
    }

   vmk_CopyToUser(data->ptr, (vmk_uintptr_t) dcbx_params,
                  data->size);
   data->rv = QFLE3_CMD_RT_SUCCESS;

out:
   vmk_HeapFree(qfle3_mod_info.heapID, dcbx_params);


   return rc;
}


int
qfle3_get_modparam_info(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{

   qfle3_mod_param_info_t info;
   QFLE3_DBG(QFLE3_DBG_MGMT, "QFLE3_CMD_MOD_PARAMS_INFO");

   info.modParamNum =
      sizeof(qfle3_mod_param_array) / sizeof(qfle3_module_param_t);
   info.modParamSize = sizeof(qfle3_module_param_t);
   data->size = sizeof(qfle3_mod_param_info_t);
   QFLE3_DBG(QFLE3_DBG_MGMT, "size = %d\n", data->size);
   vmk_Memcpy((void *)data->u, &info, data->size);
   data->rv = QFLE3_CMD_RT_SUCCESS;
   return 0;
}

int
qfle3_get_modparam(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   vmk_uint32 size;
   vmk_uint32 offset;

   QFLE3_DBG(QFLE3_DBG_MGMT, "QFLE3_CMD_GET_MOD_PARAMS");
   size = data->size;
   offset = data->off;

   if (offset >= sizeof(qfle3_mod_param_array)) {
      data->rv = QFLE3_CMD_RT_SUCCESS;
      return 1;
   }

   if ((offset + size) > sizeof(qfle3_mod_param_array)) {
      size = sizeof(qfle3_mod_param_array) - offset;
   }

   vmk_CopyToUser((data->ptr), (vmk_uintptr_t) & qfle3_mod_param_array[0],
                  size);
   data->size = size;
   data->rv = QFLE3_CMD_RT_SUCCESS;
   return 0;
}

// key-value callbacks
void
qfle3_kvcleanup(vmk_uint64 cookie)
{
   //qfle3_adapter *adapter = (qfle3_adapter *)cookie;

}

VMK_ReturnStatus
get_rxrs(vmk_uint64 cookie, void *keyVal)
{
   *(vmk_int64 *) keyVal = 64;
   return VMK_OK;
}

VMK_ReturnStatus
set_rxrs(vmk_uint64 cookie, void *keyVal)
{

   return VMK_OK;
}

VMK_ReturnStatus
get_txrs(vmk_uint64 cookie, void *keyVal)
{
   *(vmk_int64 *) keyVal = 64;
   return VMK_OK;
}

VMK_ReturnStatus
set_txrs(vmk_uint64 cookie, void *keyVal)
{

   return VMK_OK;
}

VMK_ReturnStatus
get_noq(vmk_uint64 cookie, void *keyVal)
{
//   extern s32 num_queues;
//   *(vmk_int64 *) keyVal = num_queues;
   return VMK_OK;
}

VMK_ReturnStatus
set_noq(vmk_uint64 cookie, void *keyVal)
{
//   extern s32 num_queues;

//   if ((*(vmk_uint64 *) keyVal >= 0) && (*(vmk_uint64 *) keyVal <= 11)) {
//      num_queues = *(vmk_uint64 *) keyVal;
      return VMK_OK;
//   } else {
//      return VMK_BAD_PARAM;
//   }
}

VMK_ReturnStatus
get_dll(vmk_uint64 cookie, void *keyVal)
{

//   extern u32 debug;
//   *(vmk_uint64 *) keyVal = debug;
   return VMK_OK;
}

VMK_ReturnStatus
set_dll(vmk_uint64 cookie, void *keyVal)
{
//   extern u32 debug;
//   debug = *(vmk_uint64 *) keyVal;
   return VMK_OK;
}

VMK_ReturnStatus
get_hwvlan(vmk_uint64 cookie, void *keyVal)
{
//   extern int hw_vlan;
//   *(vmk_uint64 *) keyVal = hw_vlan;
   return VMK_OK;
}

VMK_ReturnStatus
set_hwvlan(vmk_uint64 cookie, void *keyVal)
{
//   extern int hw_vlan;

//   if ((*(vmk_uint64 *) keyVal == 0) || (*(vmk_uint64 *) keyVal == 1)) {
//      hw_vlan = *(vmk_uint64 *) keyVal;
      return VMK_OK;
//   } else {
//      return VMK_BAD_PARAM;
//   }
}

VMK_ReturnStatus
get_tpa(vmk_uint64 cookie, void *keyVal)
{
   *(vmk_int64 *) keyVal = 1;
   return VMK_OK;
}

VMK_ReturnStatus
set_tpa(vmk_uint64 cookie, void *keyVal)
{

   return VMK_OK;
}

VMK_ReturnStatus
qfle3_init_uplink_mgmt(qfle3_adapter * adapter, vmk_Uplink uplink)
{
   VMK_ReturnStatus status = VMK_OK;

//   extern vmk_DriverProps qfle3_drvprops;
//   vmk_Name keyName;
//   vmk_MgmtProps mgmt_props;

   status =
      vmk_MgmtRegisterInstanceCallbacks(vmkmgmt_api_handler, (vmk_uint64)adapter,
                                        vmk_ModuleCurrentID,
                                        qfle3_mod_info.heapID,
                                        &adapter->pdev_name, QFLE3_NUM_CALLBACKS,
                                        qfle3_mgmt_callbacks);
   if (status != VMK_OK) {
      QFLE3_ERR("Error in registering instance to the management handler.\n");
   }
//              vmk_LogMessage("Management Interface Instance registered successfully, %s\n", udev->dev_name.string);

//   udev->kvSig.version = VMK_REVISION_FROM_NUMBERS(1, 0, 0, 0);
//   vmk_NameCopy(&udev->kvSig.name, &udev->dev_name);
//   vmk_NameInitialize(&udev->kvSig.vendor, "QLogic");
//   udev->kvSig.numCallbacks = 0;
//   udev->kvSig.callbacks = NULL;
//   vmk_Memset(&mgmt_props, 0, sizeof(mgmt_props));
//   mgmt_props.modId = QFLE3_MOD_ID;
//   mgmt_props.heapId = QFLE3_HEAP_ID;
//   mgmt_props.sig = &udev->kvSig;
//   mgmt_props.cleanupFn = qfle3_kvcleanup;

//   status = vmk_MgmtInit(&mgmt_props, &udev->kvMgmtHandle);

//   if (status != VMK_OK) {
//      pr_err("Error initializing key-value instance.\n");
//   }
//   vmk_NameInitialize(&keyName, "rxrs");
//   status =
//      vmk_MgmtAddKey(udev->kvMgmtHandle, VMK_MGMT_KEY_TYPE_LONG, &keyName,
//                     get_rxrs, set_rxrs);
//   if (status != VMK_OK) {
//      pr_err("Error adding key-value pair rxrs.\n");
//   }
//   vmk_NameInitialize(&keyName, "txrs");
//   status =
//      vmk_MgmtAddKey(udev->kvMgmtHandle, VMK_MGMT_KEY_TYPE_LONG, &keyName,
//                     get_txrs, set_txrs);
//   if (status != VMK_OK) {
//      pr_err("Error adding key-value pair txrs.\n");
//   }
//   vmk_NameInitialize(&keyName, "noq");
//   status =
//      vmk_MgmtAddKey(udev->kvMgmtHandle, VMK_MGMT_KEY_TYPE_LONG, &keyName,
//                     get_noq, set_noq);
//   if (status != VMK_OK) {
//      pr_err("Error adding key-value pair noq.\n");
//   }
//   vmk_NameInitialize(&keyName, "dll");
//   status =
//      vmk_MgmtAddKey(udev->kvMgmtHandle, VMK_MGMT_KEY_TYPE_LONG, &keyName,
//                     get_dll, set_dll);
//   if (status != VMK_OK) {
//      pr_err("Error adding key-value pair dll.\n");
//   }
//   vmk_NameInitialize(&keyName, "hwvlan");
//   status =
//      vmk_MgmtAddKey(udev->kvMgmtHandle, VMK_MGMT_KEY_TYPE_LONG, &keyName,
//                     get_hwvlan, set_hwvlan);
//   if (status != VMK_OK) {
//      pr_err("Error adding key-value pair hwvlan.\n");
//   }
//   vmk_NameInitialize(&keyName, "tpa");
//   status =
//      vmk_MgmtAddKey(udev->kvMgmtHandle, VMK_MGMT_KEY_TYPE_LONG, &keyName,
//                     get_tpa, set_tpa);
//   if (status != VMK_OK) {
//      pr_err("Error adding key-value pair tpa.\n");
//   }
//   vmk_LogMessage("Key-value Instance initialized successfully\n");
   return VMK_OK;

}

void
qfle3_diag_callback(vmk_MgmtCookies * cookiep, vmk_MgmtEnvelope * envelope,
                      qfle3_diag_data_t * data)
{
//   vmk_uint64 instance = envelope->instanceId;
//   vmk_uint64 cookie = cookiep->handleCookie;

//   qfle3_adapter *adapter = NULL;

//   adapter = (qfle3_adapter *) instance;
//   vmk_LogMessage("->>>cookie 0x%lx, instance 0x%lx, cmd code %d\n", cookie,
//                  instance, data->cmd);
   return ;
//   switch (data->cmd) {
//   case QFLE3_DIAG_SET_APP_VERSION:
//      qfle3_diag_set_app_version(adapter, data);
//      break;
//   case QFLE3_DIAG_IDLE_CHK_SIZE:
//      qfle3_diag_idle_chk_size(adapter, data);
//      break;
//   case QFLE3_DIAG_IDLE_CHK:
//      qfle3_diag_idle_chk(adapter, data);
//      break;
//   case QFLE3_DIAG_IDLE_CHK_RESULT_SIZE:
//      qfle3_diag_idle_chk_result_size(adapter, data);
//      break;
//   case QFLE3_DIAG_IDLE_CHK_RESULT:
//      qfle3_diag_idle_chk_result(adapter, data);
//      break;
//   case QFLE3_DIAG_GRC_DUMP_SIZE:
//      qfle3_diag_grc_dump_size(adapter, data);
//      break;
//   case QFLE3_DIAG_GRC_DUMP:
//      qfle3_diag_grc_dump(adapter, data);
//      break;
//   case QFLE3_DIAG_FW_FUNC_VER:
//      qfle3_diag_fw_func_ver(adapter, data);
//      break;
//   case QFLE3_DIAG_BUS_RESET:
//      qfle3_diag_bus_reset(adapter, data);
//      break;
//   case QFLE3_DIAG_SET_PCI_OUTPUT:
//      qfle3_diag_set_pci_output(adapter, data);
//      break;
//   case QFLE3_DIAG_SET_NW_OUTPUT:
//      qfle3_diag_set_nw_output(adapter, data);
//      break;
//   case QFLE3_DIAG_ENABLE_BLOCK:
//      qfle3_diag_enable_block(adapter, data);
//      break;
//   case QFLE3_DIAG_ENABLE_STORM:
//      qfle3_diag_enable_storm(adapter, data);
//      break;
//   case QFLE3_DIAG_ENABLE_TIMESTAMP:
//      qfle3_diag_enable_timestamp(adapter, data);
//      break;
//   case QFLE3_DIAG_ADD_EID_RANGE:
//      qfle3_diag_add_eid_range_sem_filter(adapter, data);
//      break;
//   case QFLE3_DIAG_ADD_EID_MASK:
//      qfle3_diag_add_eid_mask_sem_filter(adapter, data);
//      break;
//   case QFLE3_DIAG_ENABLE_FILTER:
//      qfle3_diag_enable_filter(adapter, data);
//      break;
//   case QFLE3_DIAG_ENABLE_TRIGGER:
//      qfle3_diag_enable_trigger(adapter, data);
//      break;
//   case QFLE3_DIAG_ADD_TRIGGER_STATE:
//      qfle3_diag_add_trigger_state(adapter, data);
//      break;
//   case QFLE3_DIAG_START:
//      qfle3_diag_start(adapter, data);
//      break;
//   case QFLE3_DIAG_STOP:
//      qfle3_diag_stop(adapter, data);
//      break;
//   default:
//      break;
//   }
}

//u32
//qfle3_diag_set_app_version(qfle3_adapter * adapter,
//                             qfle3_diag_data_t * data)
//{
//              vmk_LogMessage("QFLE3_DIAG_SET_APP_VERSION");
//   ecore_dbg_set_app_ver(data->v);
//   data->rv = 0;
//   return 0;

//}

//u32
//qfle3_diag_idle_chk_size(qfle3_adapter * adapter, qfle3_diag_data_t * data)
//{

//   vmk_uint32 reg_value32 = 0;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;

//              vmk_LogMessage("QFLE3_DIAG_IDLE_CHK_SIZE");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }
//              if (data->hwfn >= adapter->edev.cdev.num_hwfns) {
//                      data->rv = QFLE3_DIAG_ERROR;
//                      return 1;
//              }

//   p_hwfn = &adapter->edev.cdev.hwfns[0];

//   p_ptt = ecore_ptt_acquire(p_hwfn);
//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }
//   ecore_dbg_idle_chk_get_dump_buf_size(p_hwfn, p_ptt, &reg_value32);

//   ecore_ptt_release(p_hwfn, p_ptt);
//   vmk_LogMessage("idle_chk size %d", reg_value32);
//   data->rv = 0;
//   data->size = reg_value32;
//   return 0;
//}

//u32
//qfle3_diag_idle_chk_result_size(qfle3_adapter * adapter,
//                                  qfle3_diag_data_t * data)
//{

//   vmk_uint32 reg_value32 = 0;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_dev *p_dev;
//              u32 dwords;
//   vmk_uint32 *buff;

//              vmk_LogMessage("QFLE3_DIAG_IDLE_CHK_RESULT_SIZE");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }
//   p_dev = &adapter->edev.cdev;

//              if (data->hwfn >= adapter->edev.cdev.num_hwfns) {
//                      data->rv = QFLE3_DIAG_ERROR;
//                      return 1;
//              }
//   if (data->ptr == 0) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   p_hwfn = &adapter->edev.cdev.hwfns[0];

//   buff = bcm_osal_alloc(p_dev, data->size * 4);
//   if (buff == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   vmk_CopyFromUser((vmk_uintptr_t) buff, data->ptr, data->size * 4);

//   ecore_get_idle_chk_results_buf_size(p_hwfn, buff, data->size, &reg_value32);
//   vmk_LogMessage("size %d", reg_value32);
//   data->rv = 0;
//   data->size = reg_value32;

//   bcm_osal_free(p_dev, buff);
//   return 0;
//}

//u32
//qfle3_diag_idle_chk(qfle3_adapter * adapter, qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;
//   struct ecore_dev *p_dev;
//   vmk_uint32 *buff;
//   vmk_uint32 dumped_words = 0;
//              vmk_LogMessage("QFLE3_DIAG_IDLE_CHK");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }
//              if (data->hwfn >= adapter->edev.cdev.num_hwfns) {
//                      data->rv = QFLE3_DIAG_ERROR;
//                      return 1;
//              }

//   p_hwfn = &adapter->edev.cdev.hwfns[0];
//   p_dev = &adapter->edev.cdev;

//   p_ptt = ecore_ptt_acquire(p_hwfn);
//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }

//   buff = bcm_osal_alloc(p_dev, data->size * 4);
//   vmk_LogMessage("idle_chk allocated  %d dwords \n", data->size);
//   status =
//      ecore_dbg_idle_chk_dump(p_hwfn, p_ptt, buff, data->size, &dumped_words);
//   ecore_ptt_release(p_hwfn, p_ptt);

//   vmk_LogMessage("ecore_dbg_idle_chk_dumped %d dwords  status %d\n",
//                  dumped_words, status);
//   vmk_CopyToUser(data->ptr, (vmk_uintptr_t) buff, dumped_words * 4);

//   bcm_osal_free(p_dev, buff);
//   data->rv = status;
//   data->size = dumped_words;
//   return 0;
//}

//u32
//qfle3_diag_idle_chk_result(qfle3_adapter * adapter,
//                             qfle3_diag_data_t * data)
//{
//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;
//   struct ecore_dev *p_dev;
//   vmk_uint32 *buff;
//   vmk_uint8 *result_buff;
//   vmk_uint32 num_errors, num_warnings;

//   vmk_uint32 dumped_words = 0;
//              vmk_LogMessage("QFLE3_DIAG_IDLE_CHK");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }
//              if (data->hwfn >= adapter->edev.cdev.num_hwfns) {
//                      data->rv = QFLE3_DIAG_ERROR;
//                      return 1;
//              }

//   p_hwfn = &adapter->edev.cdev.hwfns[0];
//   p_dev = &adapter->edev.cdev;

//   buff = bcm_osal_alloc(p_dev, data->size2 * 4);       // dump buff
//   vmk_CopyFromUser((vmk_uintptr_t) buff, data->ptr2, data->size2 * 4);
//   result_buff = bcm_osal_alloc(p_dev, data->size);

//   num_errors = num_warnings = 0;

//   p_ptt = ecore_ptt_acquire(p_hwfn);
//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      bcm_osal_free(p_dev, buff);
//      bcm_osal_free(p_dev, result_buff);
//      return 0;
//   }

//   status =
//      ecore_print_idle_chk_results(p_hwfn, buff, data->size2, result_buff,
//                                   &num_errors, &num_warnings);

//   ecore_ptt_release(p_hwfn, p_ptt);

//   vmk_CopyToUser(data->ptr, (vmk_uintptr_t) result_buff, data->size);
//   bcm_osal_free(p_dev, buff);
//   bcm_osal_free(p_dev, result_buff);

//   data->rv = status;
//   data->size = dumped_words;
//   return 0;
//}

//u32
//qfle3_diag_grc_dump_size(qfle3_adapter * adapter, qfle3_diag_data_t * data)
//{

//   vmk_uint32 reg_value32;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;

//              vmk_LogMessage("QFLE3_DIAG_GRC_DUMP_SIZE");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   if (data->hwfn >= adapter->edev.cdev.num_hwfns) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   p_hwfn = &adapter->edev.cdev.hwfns[data->hwfn];

//   p_ptt = ecore_ptt_acquire(p_hwfn);
//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }
//   ecore_dbg_grc_get_dump_buf_size(p_hwfn, p_ptt, &reg_value32);

//   ecore_ptt_release(p_hwfn, p_ptt);
//   data->rv = 0;
//   data->size = reg_value32;
//   return 0;
//}

//u32
//qfle3_diag_grc_dump(qfle3_adapter * adapter, qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;
//   struct ecore_dev *p_dev;
//   vmk_uint32 *buff;
//   vmk_uint32 dumped_words = 0;

//              vmk_LogMessage("QFLE3_DIAG_GRC_DUMP");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   p_hwfn = &adapter->edev.cdev.hwfns[0];
//   p_dev = &adapter->edev.cdev;
//   buff = bcm_osal_alloc(p_dev, data->size * 4);
//   p_ptt = ecore_ptt_acquire(p_hwfn);

//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }
//   status = ecore_dbg_grc_dump(p_hwfn, p_ptt, buff, data->size, &dumped_words);
//   ecore_ptt_release(p_hwfn, p_ptt);

//              vmk_LogMessage("ecore_dbg_grc_dump %d dwords  status %d\n", dumped_words, status);
//   vmk_CopyToUser(data->ptr, (vmk_uintptr_t) buff, dumped_words * 4);
//   data->rv = status;
//   data->size = dumped_words;
//   return 0;
//}

//u32
//qfle3_diag_fw_func_ver(qfle3_adapter * adapter, qfle3_diag_data_t * data)
//{

//   vmk_uint32 reg_value32;

//              vmk_LogMessage("QFLE3_DIAG_FW_FUNC_VER");
//   reg_value32 = ecore_dbg_get_fw_func_ver();
//   data->rv = reg_value32;
//   data->v = reg_value32;
//   return 0;
//}

//u32
//qfle3_diag_bus_reset(qfle3_adapter * adapter, qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;

//              vmk_LogMessage("QFLE3_DIAG_BUS_RESET");

//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }
//   p_hwfn = &adapter->edev.cdev.hwfns[0];
//   p_ptt = ecore_ptt_acquire(p_hwfn);

//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }

//   status = ecore_dbg_bus_reset(p_hwfn,
//                                p_ptt,
//                                (data->
//                                 flags & QFLE3_DIAG_BUS_RESET_ONE_SHOT ? 1 :
//                                 0), data->size,
//                                (data->
//                                 flags & QFLE3_DIAG_BUS_RESET_UNIFY_INPUT ? 1
//                                 : 0),
//                                (data->
//                                 flags & QFLE3_DIAG_BUS_RESET_GRC_INPUT ? 1 :
//                                 0));
//   ecore_ptt_release(p_hwfn, p_ptt);
//   data->rv = status;
//   return 0;
//}

//u32
//qfle3_diag_set_pci_output(qfle3_adapter * adapter,
//                            qfle3_diag_data_t * data)
//{
//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;

//              vmk_LogMessage("QFLE3_DIAG_SET_PCI_OUTPUT");

//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }
//   p_hwfn = &adapter->edev.cdev.hwfns[0];
//   p_ptt = ecore_ptt_acquire(p_hwfn);

//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }
//   status = ecore_dbg_bus_set_pci_output(p_hwfn, p_ptt, data->size);
//   ecore_ptt_release(p_hwfn, p_ptt);
//   data->rv = status;
//   return 0;
//}

//u32
//qfle3_diag_set_nw_output(qfle3_adapter * adapter, qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;

//              vmk_LogMessage("QFLE3_DIAG_SET_NW_OUTPUT");

//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }
//   p_hwfn = &adapter->edev.cdev.hwfns[0];
//   p_ptt = ecore_ptt_acquire(p_hwfn);

//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }
//   status = ecore_dbg_bus_set_nw_output(p_hwfn, p_ptt, data->off,
//                                        data->v & 0xffffffff,
//                                        (data->v >> 32) & 0xffff,
//                                        data->size,
//                                        (data->
//                                         flags &
//                                         QFLE3_DIAG_SEND_TO_OTHER_ENGINE ? 1 :
//                                         0),
//                                        (data->
//                                         flags &
//                                         QFLE3_DIAG_RECV_FROM_OTHER_ENGINE ? 1
//                                         : 0)
//      );
//   ecore_ptt_release(p_hwfn, p_ptt);
//   data->rv = status;
//   return 0;
//}

//u32
//qfle3_diag_enable_block(qfle3_adapter * adapter, qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;
//   qfle3_diag_enable_block_t t;

//              vmk_LogMessage("QFLE3_DIAG_ENABLE_BLOCK");

//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }
//   p_hwfn = &adapter->edev.cdev.hwfns[0];

//   if (!data->ptr) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }

//   if (data->size != sizeof(qfle3_diag_enable_block_t)) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }

//   vmk_CopyFromUser((vmk_uintptr_t) & t, data->ptr, data->size);
//   p_ptt = ecore_ptt_acquire(p_hwfn);
//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }
//   status = ecore_dbg_bus_enable_block(p_hwfn, p_ptt,
//                                       t.block_id,
//                                       t.line_num,
//                                       t.dword_en,
//                                       t.right_shift,
//                                       t.force_valid, t.force_frame);
//   ecore_ptt_release(p_hwfn, p_ptt);
//   data->rv = status;
//   return 0;
//}

//u32
//qfle3_diag_enable_storm(qfle3_adapter * adapter, qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;

//              vmk_LogMessage("QFLE3_DIAG_ENABLE_STORM");

//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }
//   p_hwfn = &adapter->edev.cdev.hwfns[0];

//   status = ecore_dbg_bus_enable_storm(p_hwfn, data->off, data->v);
//   data->rv = status;
//   return 0;
//}

//u32
//qfle3_diag_enable_timestamp(qfle3_adapter * adapter,
//                              qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;

//              vmk_LogMessage("QFLE3_DIAG_ENABLE_TIMESTAMP");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   p_hwfn = &adapter->edev.cdev.hwfns[0];
//   p_ptt = ecore_ptt_acquire(p_hwfn);

//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }

//   status = ecore_dbg_bus_enable_timestamp(p_hwfn,
//                                           p_ptt,
//                                           (data->v >> 16) & 0xff,
//                                           (data->v >> 8) & 0xff,
//                                           (data->v) & 0xff);
//   ecore_ptt_release(p_hwfn, p_ptt);
//   data->rv = status;
//   return 0;

//}

//u32
//qfle3_diag_add_eid_range_sem_filter(qfle3_adapter * adapter,
//                                      qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;

//              vmk_LogMessage("QFLE3_DIAG_ADD_EID_RANGE");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   p_hwfn = &adapter->edev.cdev.hwfns[0];

//   status = ecore_dbg_bus_add_eid_range_sem_filter(p_hwfn,
//                                                   data->off,
//                                                   (data->v) & 0xff,
//                                                   (data->v >> 8) & 0xff);
//   data->rv = status;
//   return 0;

//}

//u32
//qfle3_diag_add_eid_mask_sem_filter(qfle3_adapter * adapter,
//                                     qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;

//              vmk_LogMessage("QFLE3_DIAG_ADD_EID_MASK");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   p_hwfn = &adapter->edev.cdev.hwfns[0];

//   status = ecore_dbg_bus_add_eid_mask_sem_filter(p_hwfn,
//                                                  data->off,
//                                                  (data->v >> 8) & 0xff,
//                                                  (data->v) & 0xff);
//   data->rv = status;
//   return 0;

//}

//u32
//qfle3_diag_add_cid_sem_filter(qfle3_adapter * adapter,
//                                qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;

//              vmk_LogMessage("QFLE3_DIAG_ADD_CID");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   p_hwfn = &adapter->edev.cdev.hwfns[0];

//   status = ecore_dbg_bus_add_cid_sem_filter(p_hwfn,
//                                             data->off, (data->v) & 0xff);
//   data->rv = status;
//   return 0;

//}

//u32
//qfle3_diag_enable_filter(qfle3_adapter * adapter, qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;

//              vmk_LogMessage("QFLE3_DIAG_ENABLE_FILTER");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   p_hwfn = &adapter->edev.cdev.hwfns[0];
//   p_ptt = ecore_ptt_acquire(p_hwfn);

//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }

//   status = ecore_dbg_bus_enable_filter(p_hwfn, p_ptt, data->off, data->size);
//   ecore_ptt_release(p_hwfn, p_ptt);
//   data->rv = status;
//   return 0;

//}

//u32
//qfle3_diag_enable_trigger(qfle3_adapter * adapter,
//                            qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;

//              vmk_LogMessage("QFLE3_DIAG_ENABLE_TRIGGER");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   p_hwfn = &adapter->edev.cdev.hwfns[0];
//   p_ptt = ecore_ptt_acquire(p_hwfn);

//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }

//   status = ecore_dbg_bus_enable_trigger(p_hwfn,
//                                         p_ptt,
//                                         (data->
//                                          flags & QFLE3_DIAG_REC_PRE_TRIGGER ?
//                                          1 : 0), data->off,
//                                         (data->
//                                          flags & QFLE3_DIAG_REC_POST_TRIGGER
//                                          ? 1 : 0), data->size,
//                                         (data->
//                                          flags &
//                                          QFLE3_DIAG_FILTER_PRE_TRIGGER ? 1 :
//                                          0),
//                                         (data->
//                                          flags &
//                                          QFLE3_DIAG_FILTER_POST_TRIGGER ? 1 :
//                                          0));
//   ecore_ptt_release(p_hwfn, p_ptt);
//   data->rv = status;
//   return 0;

//}

//u32
//qfle3_diag_add_trigger_state(qfle3_adapter * adapter,
//                               qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;

//              vmk_LogMessage("QFLE3_DIAG_ADD_TRIGGER_STATE");
//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   p_hwfn = &adapter->edev.cdev.hwfns[0];
//   p_ptt = ecore_ptt_acquire(p_hwfn);

//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }

//   status = ecore_dbg_bus_add_trigger_state(p_hwfn,
//                                            p_ptt,
//                                            data->off, data->size, data->v);
//   ecore_ptt_release(p_hwfn, p_ptt);
//   data->rv = status;
//   return 0;

//}

//u32
//qfle3_diag_add_constraint(qfle3_adapter * adapter,
//                            qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;
//   qfle3_diag_add_constraint_t t;

//              vmk_LogMessage("QFLE3_DIAG_ADD_CONSTRAINT");

//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }
//   p_hwfn = &adapter->edev.cdev.hwfns[0];

//   if (!data->ptr) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }

//   if (data->size != sizeof(qfle3_diag_add_constraint_t)) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }

//   vmk_CopyFromUser((vmk_uintptr_t) & t, data->ptr, data->size);
//   p_ptt = ecore_ptt_acquire(p_hwfn);

//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }
//   status = ecore_dbg_bus_add_constraint(p_hwfn,
//                                         p_ptt,
//                                         t.constraint_op,
//                                         t.data,
//                                         t.data_mask,
//                                         t.b_compare_frame,
//                                         t.frame_bit,
//                                         t.cycle_offset,
//                                         t.dword_offset, t.b_is_mandatory);
//   ecore_ptt_release(p_hwfn, p_ptt);
//   data->rv = status;
//   return 0;
//}

//u32
//qfle3_diag_start(qfle3_adapter * adapter, qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;

//              vmk_LogMessage("QFLE3_DIAG_START");

//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }

//   if (!data->ptr) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }
//   p_hwfn = &adapter->edev.cdev.hwfns[0];
//   p_ptt = ecore_ptt_acquire(p_hwfn);

//   if (!p_ptt) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }
//   status = ecore_dbg_bus_start(p_hwfn, p_ptt);
//   ecore_ptt_release(p_hwfn, p_ptt);
//   data->rv = status;
//   return 0;
//}

//u32
//qfle3_diag_stop(qfle3_adapter * adapter, qfle3_diag_data_t * data)
//{

//   vmk_uint32 status;
//   struct ecore_hwfn *p_hwfn;
//   struct ecore_ptt *p_ptt;

//              vmk_LogMessage("QFLE3_DIAG_STOP");

//   if (adapter == NULL) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 1;
//   }
//   p_hwfn = &adapter->edev.cdev.hwfns[0];
//   p_ptt = ecore_ptt_acquire(p_hwfn);

//   if (!p_ptt || !data->ptr) {
//      data->rv = QFLE3_DIAG_ERROR;
//      return 0;
//   }
//   status = ecore_dbg_bus_stop(p_hwfn, p_ptt);
//   ecore_ptt_release(p_hwfn, p_ptt);
//   data->rv = status;
//   return 0;
//}

void
qfle3_nvm_callback(vmk_MgmtCookies * cookie, vmk_MgmtEnvelope * envelope,
                     qfle3_nvm_data_t * data)
{
   vmk_uint64 instance = envelope->instanceId;
   vmk_uint32 status = VMK_OK;
   qfle3_adapter *adapter = (qfle3_adapter *) instance;
   vmk_uint32 totalSizeRequested,totalSizeCompleted = 0;
   vmk_uint32 offset;
   vmk_uint8 *buffer = NULL;

   QFLE3_DBG(QFLE3_DBG_MGMT,"QFLE3_NVM_CALLBACK");

   if (adapter == NULL) {
      data->rv = QFLE3_CMD_RT_ERROR;
      return;
   }

   switch (data->cmd) {
   case DRV_MSG_CODE_NVM_READ_NVRAM:

      QFLE3_DBG(QFLE3_DBG_NVM,"QFLE3_NVM_CALLBACK cmd %x, param size %x, offset %x",
                     data->cmd, data->param_size, data->param_offset);
      status = ECORE_SUCCESS;
      totalSizeRequested = data->param_size;
      totalSizeCompleted = 0;
      buffer = vmk_HeapAlloc(qfle3_mod_info.heapID, totalSizeRequested);
      if (!buffer) {
         data->rv = QFLE3_CMD_RT_ERROR;
         break;
      }
      data->txn_size = 0;
      if (!qfle3_is_nvm_accessible(adapter)) {
         QFLE3_ERR("Cannot access eeprom. Try again later.\n");
         data->rv = QFLE3_CMD_RT_ERROR;
         return;
      }
      
      QFLE3_DBG(QFLE3_DBG_NVM, "eeprom: offset 0x%x len 0x%x\n",
         data->param_offset, data->param_size);
      
      status = qfle3_nvram_read(adapter, data->param_offset, buffer, 
         totalSizeRequested, &totalSizeCompleted);


      QFLE3_DBG(QFLE3_DBG_NVM,"ecore_mcp_nvm_rd_cmd  total size read %d response %x\n",
                     totalSizeCompleted, data->mcp_resp);
      data->txn_size = totalSizeCompleted;
      vmk_CopyToUser(data->buf, (vmk_uintptr_t) buffer, totalSizeCompleted);
      vmk_HeapFree(qfle3_mod_info.heapID, buffer);
      if (totalSizeCompleted == totalSizeRequested)
         data->rv = QFLE3_CMD_RT_SUCCESS;
      else
         data->rv = QFLE3_CMD_RT_ERROR;
      break;
   case DRV_MSG_CODE_NVM_WRITE_NVRAM:


      QFLE3_DBG(QFLE3_DBG_NVM,"QFLE3_NVM_CALLBACK cmd %x, param size %x, offset %x",
                     data->cmd, data->param_size, data->param_offset);
                              vmk_LogMessage("QFLE3_NVM_CALLBACK error code %x,  response %x, mcp param %x o size %d", status, data->mcp_resp, data->mcp_param, data->txn_size);

      totalSizeRequested = data->param_size;

      offset = data->param_offset;

      buffer = vmk_HeapAlloc(qfle3_mod_info.heapID, totalSizeRequested);
      if (!buffer) {
         data->rv = QFLE3_CMD_RT_ERROR;
         break;
      }
      vmk_CopyFromUser((vmk_uintptr_t) buffer, data->buf, totalSizeRequested);
      status = qfle3_nvram_write(adapter, offset, buffer, totalSizeRequested);

      
      vmk_HeapFree(qfle3_mod_info.heapID, buffer);

      if (status == VMK_OK){
         data->rv = QFLE3_CMD_RT_SUCCESS;
         data->txn_size = totalSizeRequested;
      }else{
         data->rv = QFLE3_CMD_RT_ERROR;
         data->txn_size = 0;
      }
      break;
   default:
      data->rv = QFLE3_CMD_RT_INVALID_PARAM;
      break;
   }

   return;
}

int
qfle3_pause_get(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   qfle3_pause_params_t *pause;
   vmk_UplinkPauseParams vmPause;
   vmk_AddrCookie drv_data;
   VMK_ReturnStatus status;

   drv_data.ptr = adapter;
   pause = (qfle3_pause_params_t *) data->u;
   status = qfle3_pauseparamsget(drv_data, &vmPause);
   if (status == VMK_OK) {
      pause->autoNegotiate = vmPause.autoNegotiate;
      pause->rxPauseEnabled = vmPause.rxPauseEnabled;
      pause->txPauseEnabled = vmPause.txPauseEnabled;
      data->rv = QFLE3_CMD_RT_SUCCESS;
      return 0;
   } else {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 0;
   }
}
int
qfle3_pause_set(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   qfle3_pause_params_t *pause;
   vmk_UplinkPauseParams vmPause;
   vmk_AddrCookie drv_data;
   VMK_ReturnStatus status;

   drv_data.ptr = adapter;
   pause = (qfle3_pause_params_t *) data->u;

   vmPause.autoNegotiate = pause->autoNegotiate;
   vmPause.rxPauseEnabled = pause->rxPauseEnabled;
   vmPause.txPauseEnabled = pause->txPauseEnabled;
   status = qfle3_pauseparamsset(drv_data, vmPause);
   if (status == VMK_OK) {
      data->rv = QFLE3_CMD_RT_SUCCESS;
      return 1;
   } else {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 0;
   }
}

int
qfle3_wol_get(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   vmk_UplinkWolState wolState;
   vmk_AddrCookie drv_data;
   VMK_ReturnStatus status;
   qfle3_wol_param_t *wol;

   drv_data.ptr = adapter;
   wol = (qfle3_wol_param_t *) data->u;
   status = qfle3_getwolstate(drv_data, &wolState);
   if (status == VMK_OK) {
      vmk_CopyToUser((vmk_VA)&wol->secureONPassword,(vmk_VA)&wolState.secureONPassword,sizeof(qfle3_wol_param_t));
      data->rv = QFLE3_CMD_RT_SUCCESS;
      return 1;
   } else {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 0;
   }
}
int
qfle3_wol_set(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   vmk_UplinkWolState wolState;
   vmk_AddrCookie drv_data;
   VMK_ReturnStatus status;
   qfle3_wol_param_t *wol;

   drv_data.ptr = adapter;
   wol = (qfle3_wol_param_t *) data->u;
   wolState.enabled = wol->enabled;
   wolState.supported = wol->supported;
   vmk_CopyFromUser((vmk_VA)&wolState.secureONPassword,
                    (vmk_VA)&wol->secureONPassword, sizeof(qfle3_wol_param_t));
   status = qfle3_setwolstate(drv_data, &wolState);
   if (status == VMK_OK) {
      data->rv = QFLE3_CMD_RT_SUCCESS;
      return 1;
   } else {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 0;
   }
}
int
qfle3_coalesce_get(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   qfle3_coalesce_param_t *param;
   vmk_UplinkCoalesceParams coalesce;
   vmk_AddrCookie drv_data;
   //VMK_ReturnStatus status;

   drv_data.ptr = adapter;
   param = (qfle3_coalesce_param_t *) data->u;
   qfle3_getcoalesceparam(drv_data, &coalesce);
   
   vmk_CopyToUser((vmk_VA) param,
                  (vmk_VA) &coalesce, sizeof(qfle3_coalesce_param_t));

   data->rv = QFLE3_CMD_RT_SUCCESS;
   return 0;
}

int
qfle3_coalesce_set(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   vmk_UplinkCoalesceParams coalesce;
   vmk_AddrCookie drv_data;
   VMK_ReturnStatus status;
   qfle3_coalesce_param_t *params;

   drv_data.ptr = adapter;
   params = (qfle3_coalesce_param_t *) data->u;

   vmk_CopyFromUser((vmk_VA) &coalesce,
                    (vmk_VA) params, sizeof(qfle3_coalesce_param_t));
   status = qfle3_setcoalesceparam(drv_data, &coalesce);
   if (status == VMK_OK) {
      data->rv = QFLE3_CMD_RT_SUCCESS;
      return 1;
   } else {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 0;
   }
}

int
qfle3_physical_id(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
    int i;
    vmk_uint32 duration = *(vmk_uint32 *)&data->u[0];

    if (!qfle3_is_nvm_accessible(adapter))
        return 0;

    if (duration == 0)
        duration = 2;

    for (i = 0; i < (duration * 2); i++) {
        qfle3_acquire_phy_lock(adapter);
        if ((i % 2) == 0)
            elink_set_led(&adapter->link_params, &adapter->link_vars,
                      ELINK_LED_MODE_ON, ELINK_SPEED_1000);
        else
            elink_set_led(&adapter->link_params, &adapter->link_vars,
                      ELINK_LED_MODE_FRONT_PANEL_OFF, 0);

        qfle3_release_phy_lock(adapter);
        vmk_WorldSleep(500000);
    }

    qfle3_acquire_phy_lock(adapter);
    elink_set_led(&adapter->link_params, &adapter->link_vars,
              ELINK_LED_MODE_OPER, adapter->link_vars.line_speed);
    qfle3_release_phy_lock(adapter);

    data->rv = QFLE3_CMD_RT_SUCCESS;

    return 0;
}

int
qfle3_offload_get(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   qfle3_offload_param_t *param;
   vmk_AddrCookie drv_data;
   qfle3_offload_param_t offload;

   drv_data.ptr = adapter;
   param = (qfle3_offload_param_t *) data->u;
   vmk_Memset(&offload, 0, sizeof(qfle3_offload_param_t));
   if (adapter->offloadflags & QFLE3_FLAGS_CSUM_OFFLOAD_ENABLED){
      offload.csum= 1;
   }
   if (adapter->offloadflags & QFLE3_FLAGS_TSO_ENABLED)
      offload.tso = 1;
   if (adapter->offloadflags & QFLE3_FLAGS_VXLAN_OFFLOAD_ENABLED)
      offload.vxlan = 1;

   
   vmk_CopyToUser((vmk_VA) param,
                  (vmk_VA) &offload, sizeof(qfle3_offload_param_t));
   

   data->rv = QFLE3_CMD_RT_SUCCESS;
   return 0;
}

int
qfle3_offload_set(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   data->rv = QFLE3_CMD_RT_ERROR;
   return 0;
}

int
qfle3_restart_autoneg(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   vmk_AddrCookie drv_data;
   VMK_ReturnStatus status = VMK_FAILURE;

   drv_data.ptr = adapter;

//   status = qfle3_restart_neg(drv_data);
   if (status == VMK_OK) {
      data->rv = QFLE3_CMD_RT_SUCCESS;
      return 1;
   } else {
      data->rv = QFLE3_CMD_RT_ERROR;
      return 0;
   }
}
int
qfle3_ring_get(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   qfle3_ring_param_t *param;
   vmk_AddrCookie drv_data;
   vmk_UplinkRingParams ring;
   //VMK_ReturnStatus status;

   drv_data.ptr = adapter;
   param = (qfle3_ring_param_t *) data->u;

   qfle3_ringparamsget(drv_data, &ring);
   vmk_CopyToUser((vmk_VA) param,
                  (vmk_VA) &ring, sizeof(qfle3_ring_param_t));
   data->rv = QFLE3_CMD_RT_SUCCESS;
   return 0;
}

int
qfle3_ring_set(qfle3_adapter * adapter, qfle3_mgmt_data_t * data)
{
   data->rv = QFLE3_CMD_RT_ERROR;
   return 0;
}

void qfle3_phy_reg_callback(vmk_MgmtCookies * cookie, vmk_MgmtEnvelope * envelope,
                         qfle3_phy_cmd_t * phy_cmd, qfle3_phy_data_t * phy_data)
{
   vmk_uint64 instance = envelope->instanceId;
   qfle3_adapter *adapter = (qfle3_adapter *) instance;
   u32 ext_phy_config, reg_offset = 0, rv = 0;
   int prtad = 0, devad = 0;
   vmk_uint16 reg = 0;
   vmk_uint32 port = QFLE3_PORT(adapter);
   if (adapter == NULL) {
      phy_data->rv = QFLE3_CMD_RT_ERROR;
      return;
   }
   reg_offset = phy_data->phy_id;
   reg      = ( reg_offset & 0xffff )    >> QFLE3_PHY_REG_SHIFT ;      // 0
   devad = ( reg_offset & 0x1f0000  ) >> QFLE3_PHY_DEV_ADDR_SHIFT ; // 16
   prtad = ( reg_offset & 0x3e00000 ) >> QFLE3_PHY_PHY_ADDR_SHIFT ; // 21

   switch (phy_cmd->qfle3_phy_cmd) {
   case QFLE3_CMD_PHY_REG_READ:
    phy_data->val_out = qfle3_phy_read(adapter, prtad, devad, reg);
    if (phy_data->val_out < 0)
       phy_data->rv = QFLE3_CMD_RT_ERROR;
    else     
       phy_data->rv = QFLE3_CMD_RT_SUCCESS;
    return ;

   case QFLE3_CMD_PHY_REG_WRITE:
    ext_phy_config =
        SHMEM_RD(adapter,
             dev_info.port_hw_config[port].external_phy_config);
    if (phy_data->magic  == 1) {
    if (phy_data->phy_id == 0x50485950) {
        /* 'PHYP' (0x50485950): prepare phy for FW upgrade */
        qfle3_stats_handle(adapter, STATS_EVENT_STOP);

        qfle3_acquire_phy_lock(adapter);
        elink_link_reset(&adapter->link_params, &adapter->link_vars, 0);
        if (ELINK_XGXS_EXT_PHY_TYPE(ext_phy_config) ==
                    PORT_HW_CFG_XGXS_EXT_PHY_TYPE_SFX7101)
            elink_cb_gpio_write(adapter, MISC_REGISTERS_GPIO_0,
                       MISC_REGISTERS_GPIO_HIGH, port);
        qfle3_release_phy_lock(adapter);
        qfle3_report_link(adapter);
        phy_data->rv = QFLE3_CMD_RT_SUCCESS;
        return ;
    } else if (phy_data->phy_id == 0x50485952) {
        /* 'PHYR' (0x50485953): re-init link after FW upgrade */
            if (adapter->link_vars.link_up) {
            qfle3_acquire_phy_lock(adapter);
            elink_link_reset(&adapter->link_params, &adapter->link_vars, 1);

            elink_phy_init(&adapter->link_params,
                         &adapter->link_vars);
            qfle3_release_phy_lock(adapter);
            qfle3_calc_fc_adv(adapter);
            phy_data->rv = QFLE3_CMD_RT_SUCCESS;
            return ;
        }
    } else if (phy_data->phy_id == 0x53985943) {
        /* 'PHYC' (0x53985943): PHY FW upgrade completed */
        if (ELINK_XGXS_EXT_PHY_TYPE(ext_phy_config) ==
                       PORT_HW_CFG_XGXS_EXT_PHY_TYPE_SFX7101) {

            /* DSP Remove Download Mode */
            elink_cb_gpio_write(adapter, MISC_REGISTERS_GPIO_0,
                       MISC_REGISTERS_GPIO_LOW, port);

            qfle3_acquire_phy_lock(adapter);

            elink_sfx7101_sp_sw_reset(adapter,
                        &adapter->link_params.phy[ELINK_EXT_PHY1]);

            /* wait 0.5 sec to allow it to run */
            vmk_WorldSleep(500000);
            elink_ext_phy_hw_reset(adapter, port);
            vmk_WorldSleep(500000);
            qfle3_release_phy_lock(adapter);
            phy_data->rv = QFLE3_CMD_RT_SUCCESS;
            return ;
        }
    }
   }
   rv = qfle3_phy_write(adapter, prtad, devad, reg, phy_data->val_in);
   if (rv) { 
      phy_data->rv = QFLE3_CMD_RT_ERROR;
      return;
   }
   break;
   default:
      QFLE3_DBG(QFLE3_DBG_MGMT,"unknown cmd \n");
   }
   phy_data->rv = QFLE3_CMD_RT_SUCCESS;
   return;
}

