// Copyright 2022 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifndef FUZZTEST_GRAMMAR_CODEGEN_ANTLR_FRONTEND_H_
#define FUZZTEST_GRAMMAR_CODEGEN_ANTLR_FRONTEND_H_

#include <string>
#include <vector>

#include "absl/strings/string_view.h"
#include "./grammar_codegen/generated_antlr_parser/ANTLRv4Parser.h"
#include "./grammar_codegen/generated_antlr_parser/ANTLRv4ParserBaseListener.h"
#include "./grammar_codegen/grammar_info.h"

namespace fuzztest::internal::grammar {

using antlr4_grammar::ANTLRv4Parser;
// The GrammarInfoBuilder parses the input grammar specification files and
// constructs the intermediate represenation (`Grammar` in
// grammar_info.h), for code generation. It is based on the antlr4 grammar
// file parser generated by antlr4. Given a input grammar file, the parser
// builds the AST for the file. The builder traverses the AST, collects the
// information for every symbol and constructs the IR.
class GrammarInfoBuilder : public antlr4_grammar::ANTLRv4ParserBaseListener {
 public:
  Grammar BuildGrammarInfo(
      const std::vector<std::string>& input_grammar_specs,
      std::optional<std::string> grammar_name = std::nullopt,
      bool insert_space_between_blocks = false);

  // Every symbol in the grammar has a handler function in the listener, which
  // will be called when such a symbol is visited during tree traversal. The
  // node being visited is passed as the `ctx` argument. `LexerRuleSpec` and
  // `ParserRule` is the symbol for lexer rule and parser rule. From them we
  // build the grammar infomation. The `GrammarSpec` contains the name of the
  // grammar and we collect it for generating the name of the namespace.
  // These APIs should not be used by users but just used by the tree traversal.
  void enterLexerRuleSpec(ANTLRv4Parser::LexerRuleSpecContext* ctx) override;
  void enterParserRuleSpec(ANTLRv4Parser::ParserRuleSpecContext* ctx) override;
  void enterGrammarSpec(ANTLRv4Parser::GrammarSpecContext* ctx) override;

 private:
  Range ParseRange(absl::string_view s);

  Block ConstructBlock(ANTLRv4Parser::BlockContext*);
  Block ConstructBlock(ANTLRv4Parser::AtomContext*);
  Block ConstructBlock(ANTLRv4Parser::ElementContext*);
  Block ConstructBlock(ANTLRv4Parser::LexerAtomContext*);
  ProductionRule ConstructProductionRule(ANTLRv4Parser::LexerAltContext*);
  ProductionRule ConstructProductionRule(ANTLRv4Parser::AlternativeContext*);
  GrammarRule ConstructGrammarRule(ANTLRv4Parser::LexerRuleSpecContext*);
  GrammarRule ConstructGrammarRule(ANTLRv4Parser::ParserRuleSpecContext*);

  std::string grammar_name_;
  std::vector<GrammarRule> rules_;

  bool insert_space_between_blocks_;
};

}  // namespace fuzztest::internal::grammar

#endif  // FUZZTEST_GRAMMAR_CODEGEN_ANTLR_FRONTEND_H_
